/*
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * Copyright (C) 1999  Oleg Drokin (green@ccssu.ccssu.crimea.ua)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include <ctype.h>
#include <errno.h>
#include "compat/strerror.h"
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <netdb.h>
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#include <termios.h>
#include <sys/types.h>
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include "compat/netinet_in.h"
#include <arpa/inet.h>
#include "compat/inet_ntoa.h"
#include "packet.h"
#include "init_protocol.h"
#include "udp_protocol.h"
#include "bnet_protocol.h"
#include "file_protocol.h"
#include "tag.h"
#include "bn_type.h"
#include "field_sizes.h"
#include "network.h"
#include "bnethash.h"
#include "bnethashconv.h"
#include "ansi_term.h"
#include "version.h"
#include "util.h"
#ifdef CLIENTDEBUG
#include "eventlog.h"
#endif
#include "client.h"


#define COMPNAMELEN 128
#define SCREEN_WIDTH 80 /* FIXME need to determine real terminal width here */

#ifdef CLIENTDEBUG
#define dprintf printf
#else
#define dprintf if (0) printf
#endif

#define CHANNEL_STARCRAFT "Starcraft"
#define CHANNEL_BROODWARS "Brood War"
#define CHANNEL_SHAREWARE "Starcraft Shareware"
#define CHANNEL_DIABLORTL "Diablo Retail"
#define CHANNEL_DIABLOSHR "Diablo Shareware" /* FIXME: is this one right? */

static int print_file(struct sockaddr_in * saddr, char const * filename, char const * progname);
static void usage(char const * progname);


static int print_file(struct sockaddr_in * saddr, char const * filename, char const * progname)
{
    int          sd;
    t_packet *   packet;
    t_packet *   rpacket;
    t_packet *   fpacket;
    unsigned int currsize;
    unsigned int filelen;
    
    if (!saddr || !filename || !progname)
	return -1;
    
    if ((sd = socket(PF_INET,SOCK_STREAM,0))<0)
    {
	fprintf(stderr,"%s: could not create socket (socket: %s)\n",progname,strerror(errno));
	return -1;
    }
    
    if (connect(sd,(struct sockaddr *)saddr,sizeof(*saddr))<0)
    {
	fprintf(stderr,"%s: could not connect to server (connect: %s)\n",progname,strerror(errno));
	return -1;
    }
    
    {
	t_connect_header temp;
	
	bn_byte_set(&temp.class,CONNECT_CLASS_FILE);
	if (write(sd,&temp,sizeof(temp))!=sizeof(temp))
	{
	    fprintf(stderr,"%s: could not write to server (write: %s)\n",progname,strerror(errno));
	    return -1;
	}
    }
    
    if (!(rpacket = packet_create(packet_class_file)))
    {
	fprintf(stderr,"%s: could not create packet\n",progname);
	return -1;
    }
    
    if (!(fpacket = packet_create(packet_class_raw)))
    {
	fprintf(stderr,"%s: could not create packet\n",progname);
	return -1;
    }
    
    if (!(packet = packet_create(packet_class_file)))
    {
	fprintf(stderr,"%s: could not create packet\n",progname);
	return -1;
    }
    packet_set_size(packet,sizeof(t_client_file_req));
    packet_set_type(packet,CLIENT_FILE_REQ);
    bn_int_tag_set(&packet->u.client_file_req.archtag,ARCHTAG_INTEL);
    bn_int_tag_set(&packet->u.client_file_req.clienttag,CLIENTTAG_STARCRAFT);
    bn_int_set(&packet->u.client_file_req.adid,0);
    bn_int_set(&packet->u.client_file_req.extensiontag,0);
    bn_int_set(&packet->u.client_file_req.unknown2,0x00000000);
    bn_long_set_a_b(&packet->u.client_file_req.timestamp,0x00000000,0x00000000);
    packet_append_string(packet,filename);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    do
	if (client_blockrecv_packet(sd,rpacket)<0)
	{
	    fprintf(stderr,"%s: server closed file connection\n",progname);
	    packet_del_ref(fpacket);
	    packet_del_ref(rpacket);
	    return -1;
	}
    while (packet_get_type(rpacket)!=SERVER_FILE_REPLY);
    
    filelen = bn_int_get(rpacket->u.server_file_reply.filelen);
    packet_del_ref(rpacket);
    
    for (currsize=0; currsize+MAX_PACKET_SIZE<=filelen; currsize+=MAX_PACKET_SIZE)
    {
	if (client_blockrecv_raw_packet(sd,fpacket,MAX_PACKET_SIZE)<0)
	{
	    fflush(stdout);
	    fprintf(stderr,"%s: server closed file connection\n",progname);
	    packet_del_ref(fpacket);
	    return -1;
	}
	str_print_term(stdout,packet_get_raw_data_const(fpacket,0),MAX_PACKET_SIZE,1);
    }
    filelen -= currsize;
    if (filelen)
    {
	if (client_blockrecv_raw_packet(sd,fpacket,filelen)<0)
	{
	    fflush(stdout);
	    fprintf(stderr,"%s: server closed file connection\n",progname);
	    packet_del_ref(fpacket);
	    return -1;
	}
	str_print_term(stdout,packet_get_raw_data_const(fpacket,0),filelen,1);
    }
    fflush(stdout);
    
    close(sd);
    
    packet_del_ref(fpacket);
    
    return 0;
}


static void usage(char const * progname)
{
    fprintf(stderr,"usage: %s [<options>] [<servername> [<TCP portnumber>]]\n"
            "    -a, --ansi-color            use ANSI colors\n"
            "    -n, --new-account           create a new account\n"
            "    -c, --change-password       change account password\n"
            "    -b, --client=SEXP           report client as Brood Wars\n"
            "    -d, --client=DRTL           report client as Diablo Retail\n"
            "    --client=DSHR               report client as Diablo Shareware\n"
            "    -s, --client=STAR           report client as Starcraft (default)\n"
            "    --client=SSHR               report client as Starcraft Shareware\n"
	    "    -o NAME, --owner=NAME       report CD owner as NAME\n"
	    "    -k KEY, --cdkey=KEY         report CD key as KEY\n"
            "    -h, --help, --usage         show this information and exit\n"
            "    -v, --version               print version number and exit\n",
	    progname);
    exit(1);
}


int main(int argc, char * argv[])
{
    int                a;
    int                useansi=0;
    int                newacct=0;
    int                changepass=0;
    int                sd;
    struct sockaddr_in saddr;
    t_packet *         packet;
    t_packet *         rpacket;
    char               compname[COMPNAMELEN];
    char const *       cdowner=NULL;
    char const *       cdkey=NULL;
    char const *       clienttag=NULL;
    char const *       channel=NULL;
    char const *       servname=NULL;
    unsigned short     servport=0;
    char const *       username;
    char               player[MAX_MESSAGE_LEN];
    char               text[MAX_MESSAGE_LEN];
    struct hostent *   host;
    unsigned int       currsize;
    unsigned int       commpos;
    struct termios     in_attr_old;
    struct termios     in_attr_new;
    int                changed_in;
    unsigned int       count;
    unsigned int       sessionkey;
    int                fd_stdin;
    
    if (argc<1 || !argv || !argv[0])
    {
	fprintf(stderr,"bad arguments\n");
	return 1;
    }
    
    for (a=1; a<argc; a++)
	if (servname && isdigit((int)argv[a][0]) && a+1>=argc)
	{
            if (str_to_ushort(argv[a],&servport)<0)
            {
                fprintf(stderr,"%s: \"%s\" should be a positive integer\n",argv[0],argv[a]);
                usage(argv[0]);
            }
	}
	else if (!servname && argv[a][0]!='-' && a+2>=argc)
	    servname = argv[a];
        else if (strcmp(argv[a],"-a")==0 || strcmp(argv[a],"--use-ansi")==0)
	    useansi = 1;
        else if (strcmp(argv[a],"-n")==0 || strcmp(argv[a],"--new-account")==0)
	{
	    if (changepass)
	    {
		fprintf(stderr,"%s: can not create new account when changing passwords\n",argv[0]);
		usage(argv[0]);
	    }
	    newacct = 1;
	}
        else if (strcmp(argv[a],"-c")==0 || strcmp(argv[a],"--change-password")==0)
	{
	    if (newacct)
	    {
		fprintf(stderr,"%s: can not change passwords when creating a new account\n",argv[0]);
		usage(argv[0]);
	    }
	    changepass = 1;
	}
        else if (strcmp(argv[a],"-b")==0 || strcmp(argv[a],"--client=SEXP")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_BROODWARS;
	    channel = CHANNEL_BROODWARS;
	}
        else if (strcmp(argv[a],"-d")==0 || strcmp(argv[a],"--client=DRTL")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_DIABLORTL;
	    channel = CHANNEL_DIABLORTL;
	}
        else if (strcmp(argv[a],"--client=DSHR")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_DIABLOSHR;
	    channel = CHANNEL_DIABLOSHR;
	}
        else if (strcmp(argv[a],"-s")==0 || strcmp(argv[a],"--client=STAR")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_STARCRAFT;
	    channel = CHANNEL_STARCRAFT;
	}
        else if (strcmp(argv[a],"--client=SSHR")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_SHAREWARE;
	    channel = CHANNEL_SHAREWARE;
	}
	else if (strncmp(argv[a],"--client=",9)==0)
	{
	    fprintf(stderr,"%s: unknown client tag \"%s\"\n",argv[0],&argv[a][9]);
	    usage(argv[0]);
	}
	else if (strcmp(argv[a],"-o")==0)
	{
	    if (a+1>=argc)
            {
                fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
                usage(argv[0]);
            }
	    if (cdowner)
	    {
		fprintf(stderr,"%s: CD owner was already specified as \"%s\"\n",argv[0],cdowner);
		usage(argv[0]);
	    }
	    cdowner = argv[++a];
	}
	else if (strncmp(argv[a],"--owner=",8)==0)
	{
	    if (cdowner)
	    {
		fprintf(stderr,"%s: CD owner was already specified as \"%s\"\n",argv[0],cdowner);
		usage(argv[0]);
	    }
	    cdowner = &argv[a][8];
	}
	else if (strcmp(argv[a],"-k")==0)
	{
	    if (a+1>=argc)
            {
                fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
                usage(argv[0]);
            }
	    if (cdkey)
	    {
		fprintf(stderr,"%s: CD key was already specified as \"%s\"\n",argv[0],cdkey);
		usage(argv[0]);
	    }
	    cdkey = argv[++a];
	}
	else if (strncmp(argv[a],"--cdkey=",8)==0)
	{
	    if (cdkey)
	    {
		fprintf(stderr,"%s: CD key was already specified as \"%s\"\n",argv[0],cdkey);
		usage(argv[0]);
	    }
	    cdkey = &argv[a][8];
	}
        else if (strcmp(argv[a],"--client")==0)
	{
	    fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
        else if (strcmp(argv[a],"--owner")==0)
	{
	    fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
        else if (strcmp(argv[a],"--cdkey")==0)
	{
	    fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
	else if (strcmp(argv[a],"-v")==0 || strcmp(argv[a],"--version")==0)
	{
            printf("version "BNETD_VERSION"\n");
            return 0;
	}
	else if (strcmp(argv[a],"-h")==0 || strcmp(argv[a],"--help")==0 || strcmp(argv[a],"--usage")
==0)
            usage(argv[0]);
	else
	{
	    fprintf(stderr,"%s: unknown option \"%s\"\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
    
    if (servport==0)
	servport = BNETD_SERV_PORT;
    if (!cdowner)
	cdowner = BNETD_DEFAULT_OWNER;
    if (!cdkey)
	cdkey = BNETD_DEFAULT_KEY;
    if (!clienttag)
    {
	clienttag = CLIENTTAG_STARCRAFT;
	channel = CHANNEL_STARCRAFT;
    }
    if (!servname)
	servname = BNETD_DEFAULT_HOST;
    
    if (!(host = gethostbyname(servname)))
    {
	fprintf(stderr,"%s: unknown host \"%s\"\n",argv[0],servname);
	return 1;
    }
    
    if (host->h_addrtype!=AF_INET)
    {
	fprintf(stderr,"%s: host is not in IPv4 address family\n",argv[0]);
	return 1;
    }
    
    if (gethostname(compname,COMPNAMELEN)<0)
    {
	fprintf(stderr,"%s: could not get host name (gethostname: %s)\n",argv[0],strerror(errno));
	return 1;
    }
    if (!(username = getlogin()))
    {
	fprintf(stderr,"%s: could not get login name (getlogin: %s)\n",argv[0],strerror(errno));
	return 1;
    }
    
    fd_stdin = fileno(stdin);
    if (tcgetattr(fd_stdin,&in_attr_old)>=0)
    {
        in_attr_new = in_attr_old;
        in_attr_new.c_lflag &= ~(ECHO | ICANON); /* turn off ECHO and ICANON */
	in_attr_new.c_cc[VMIN]  = 0; /* don't require reads to return data */
        in_attr_new.c_cc[VTIME] = 1; /* timeout = .1 seconds */
        tcsetattr(fd_stdin,TCSANOW,&in_attr_new);
        changed_in = 1;
    }
    else
    {
	fprintf(stderr,"%s: could not set terminal attributes for stdin\n",argv[0]);
	changed_in = 0;
    }
    
    if ((sd = socket(PF_INET,SOCK_STREAM,0))<0)
    {
	fprintf(stderr,"%s: could not create socket (socket: %s)\n",argv[0],strerror(errno));
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    
    memset(&saddr,0,sizeof(saddr));
    saddr.sin_family = AF_INET;
    saddr.sin_port   = htons(servport);
    memcpy(&saddr.sin_addr.s_addr,host->h_addr_list[0],host->h_length);
    if (connect(sd,(struct sockaddr *)&saddr,sizeof(saddr))<0)
    {
	fprintf(stderr,"%s: could not connect to server \"%s\" port %hu (connect: %s)\n",argv[0],servname,servport,strerror(errno));
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    
    if (useansi)
    {
	ansi_text_reset();
	ansi_screen_clear();
	ansi_cursor_move_home();
    }
    printf("Connected to %s:%hu.\n",inet_ntoa(saddr.sin_addr),servport);
    
#ifdef CLIENTDEBUG
    eventlog_set(stderr);
#endif
    
    /* wait for udptest */
    {
	int                lsock;
	struct sockaddr_in laddr;
	char               buff[BNET_UDPTEST_SIZE];
	int                ok=1;
	time_t             start;
	
	if ((lsock = socket(PF_INET,SOCK_DGRAM,0))<0)
	{
	    fprintf(stderr,"%s: could not create UDP socket (socket: %s)\n",argv[0],strerror(errno));
	    ok = 0;
	}
	else
	{
	    if (fcntl(lsock,F_SETFL,O_NONBLOCK)<0)
		fprintf(stderr,"%s: could not set UDP socket to non-blocking mode (fcntl: %s)\n",argv[0],strerror(errno));
	    memset(&laddr,0,sizeof(laddr));
	    laddr.sin_family = AF_INET;
	    laddr.sin_port = htons(BNETD_TEST_PORT);
	    laddr.sin_addr.s_addr = htonl(INADDR_ANY);
	    if (bind(lsock,(struct sockaddr *)&laddr,sizeof(laddr))<0)
	    {
		fprintf(stderr,"%s: could bind to UDP port %d (bind: %s)\n",argv[0],BNETD_TEST_PORT,strerror(errno));
		close(lsock);
		ok = 0;
	    }
	}
	
	{
	    t_connect_header temp;
	    
	    bn_byte_set(&temp.class,CONNECT_CLASS_NORMAL);
	    if (write(sd,&temp,sizeof(temp))!=sizeof(temp))
	    {
		fprintf(stderr,"%s: could not write to server (write: %s)\n",argv[0],strerror(errno));
		if (changed_in)
		    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		return 1;
	    }
	}
	
	if (ok)
	{
	    start = time(NULL);
	    count = 0;
	    while (start+(time_t)5>=time(NULL))  /* wait 5 seconds */
	    {
#ifdef HAVE_RECV /* some recvfrom()s don't handle NULL, but recv is sorta depreciated */
	        if (recv(lsock,buff,BNET_UDPTEST_SIZE,0)!=BNET_UDPTEST_SIZE)
#else
	        if (recvfrom(lsock,buff,BNET_UDPTEST_SIZE,0,NULL,NULL)!=BNET_UDPTEST_SIZE)
#endif
		{
		    if (errno!=EAGAIN && errno!=EWOULDBLOCK)
			fprintf(stderr,"%s: failed to receive UDPTEST (recvfrom: %s)\n",argv[0],strerror(errno));
		    continue;
		}
	        start = time(NULL);
		if (memcmp(buff,BNET_UDPTEST,BNET_UDPTEST_SIZE)!=0)
		{
		    fprintf(stderr,"%s: got bad data in UDPTEST\n",argv[0]);
		    continue;
		}
		count++;
		if (count>=2)
		   break;
	    }
	    close(lsock);
	    if (count<2)
		printf("Could not receive UDP packets. Connection may be slow or firewalled.\n");
	}
    }
    dprintf("Got UDP data\n");
    
    /* reuse this same packet over and over */
    if (!(rpacket = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    
    if (strcmp(clienttag,CLIENTTAG_DIABLOSHR)==0 ||
        strcmp(clienttag,CLIENTTAG_DIABLORTL)==0)
    {
	if (!(packet = packet_create(packet_class_normal)))
	{
	    fprintf(stderr,"%s: could not create packet\n",argv[0]);
	    if (changed_in)
		tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	    return 1;
	}
	packet_set_size(packet,sizeof(t_client_unknown_1b));
	packet_set_type(packet,CLIENT_UNKNOWN_1B);
	bn_short_set(&packet->u.client_unknown_1b.unknown1,CLIENT_UNKNOWN_1B_UNKNOWN3);
	bn_short_nset(&packet->u.client_unknown_1b.port,BNETD_GAME_PORT);
	bn_int_nset(&packet->u.client_unknown_1b.ip,0); /* FIXME */
	bn_int_set(&packet->u.client_unknown_1b.unknown2,CLIENT_UNKNOWN_1B_UNKNOWN3);
	bn_int_set(&packet->u.client_unknown_1b.unknown3,CLIENT_UNKNOWN_1B_UNKNOWN3);
	client_blocksend_packet(sd,packet);
	packet_del_ref(packet);
    }
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
#ifdef OLD_PROT
    packet_set_size(packet,sizeof(t_client_compinfo1));
    packet_set_type(packet,CLIENT_COMPINFO1);
    bn_int_set(&packet->u.client_compinfo1.reg_version,CLIENT_COMPINFO1_REG_VERSION);
    bn_int_set(&packet->u.client_compinfo1.reg_auth,CLIENT_COMPINFO1_REG_AUTH);
    bn_int_set(&packet->u.client_compinfo1.client_id,CLIENT_COMPINFO1_CLIENT_ID);
    bn_int_set(&packet->u.client_compinfo1.client_token,CLIENT_COMPINFO1_CLIENT_TOKEN);
    packet_append_string(packet,compname);
    packet_append_string(packet,username);
#else
    packet_set_size(packet,sizeof(t_client_compinfo2));
    packet_set_type(packet,CLIENT_COMPINFO2);
    bn_int_set(&packet->u.client_compinfo2.unknown1,CLIENT_COMPINFO2_UNKNOWN1);
    bn_int_set(&packet->u.client_compinfo2.unknown2,CLIENT_COMPINFO2_UNKNOWN2);
    bn_int_set(&packet->u.client_compinfo2.reg_auth,CLIENT_COMPINFO2_REG_AUTH);
    bn_int_set(&packet->u.client_compinfo2.client_id,CLIENT_COMPINFO2_CLIENT_ID);
    bn_int_set(&packet->u.client_compinfo2.client_token,CLIENT_COMPINFO2_CLIENT_TOKEN);
    packet_append_string(packet,compname);
    packet_append_string(packet,username);
#endif
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_countryinfo));
    packet_set_type(packet,CLIENT_COUNTRYINFO);
    bn_long_set_a_b(&packet->u.client_countryinfo.timestamp1,CLIENT_COUNTRYINFO_TIMESTAMP1_A,CLIENT_COUNTRYINFO_TIMESTAMP1_B);
    bn_long_set_a_b(&packet->u.client_countryinfo.timestamp2,CLIENT_COUNTRYINFO_TIMESTAMP2_A,CLIENT_COUNTRYINFO_TIMESTAMP2_B);
    bn_int_set(&packet->u.client_countryinfo.unknown5,CLIENT_COUNTRYINFO_UNKNOWN5);
    bn_int_set(&packet->u.client_countryinfo.langid1,CLIENT_COUNTRYINFO_LANGID_USENGLISH);
    bn_int_set(&packet->u.client_countryinfo.langid2,CLIENT_COUNTRYINFO_LANGID_USENGLISH);
    bn_int_set(&packet->u.client_countryinfo.langid3,CLIENT_COUNTRYINFO_LANGID_USENGLISH);
    packet_append_string(packet,CLIENT_COUNTRYINFO_LANGSTR_USENGLISH);
    packet_append_string(packet,CLIENT_COUNTRYINFO_COUNTRYCODE_USA);
    packet_append_string(packet,CLIENT_COUNTRYINFO_COUNTRYABB_USA);
    packet_append_string(packet,CLIENT_COUNTRYINFO_COUNTRYNAME_USA);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    /* FIXME: did old clients send this? */
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_unknown_2b));
    packet_set_type(packet,CLIENT_UNKNOWN_2B);
    bn_int_set(&packet->u.client_unknown_2b.unknown1,CLIENT_UNKNOWN_2B_UNKNOWN1);
    bn_int_set(&packet->u.client_unknown_2b.unknown2,CLIENT_UNKNOWN_2B_UNKNOWN2);
    bn_int_set(&packet->u.client_unknown_2b.unknown3,CLIENT_UNKNOWN_2B_UNKNOWN3);
    bn_int_set(&packet->u.client_unknown_2b.unknown4,CLIENT_UNKNOWN_2B_UNKNOWN4);
    bn_int_set(&packet->u.client_unknown_2b.unknown5,CLIENT_UNKNOWN_2B_UNKNOWN5);
    bn_int_set(&packet->u.client_unknown_2b.unknown6,CLIENT_UNKNOWN_2B_UNKNOWN6);
    bn_int_set(&packet->u.client_unknown_2b.unknown7,CLIENT_UNKNOWN_2B_UNKNOWN7);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_progident));
    packet_set_type(packet,CLIENT_PROGIDENT);
    bn_int_tag_set(&packet->u.client_progident.archtag,ARCHTAG_INTEL);
    bn_int_tag_set(&packet->u.client_progident.clienttag,clienttag);
    bn_int_set(&packet->u.client_progident.versionid,0xBD);
    bn_int_set(&packet->u.client_progident.unknown1,CLIENT_PROGIDENT_UNKNOWN1);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	   fprintf(stderr,"%s: server closed connection\n",argv[0]);
	   if (changed_in)
	       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	   return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_COMPREPLY);
    dprintf("Got COMPREPLY\n");
    
#ifdef OLD_PROT
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	   fprintf(stderr,"%s: server closed connection\n",argv[0]);
	   if (changed_in)
	       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	   return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_SESSIONKEY1);
    sessionkey = bn_int_get(rpacket->u.server_sessionkey1.sessionkey);
    dprintf("Got SESSIONKEY1 (0x%08x)\n",sessionkey);
#else
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	   fprintf(stderr,"%s: server closed connection\n",argv[0]);
	   if (changed_in)
	       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	   return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_SESSIONKEY2);
    sessionkey = bn_int_get(rpacket->u.server_sessionkey2.sessionkey);
    dprintf("Got SESSIONKEY2 (0x%08x)\n",sessionkey);
#endif
    
#if 0
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	   fprintf(stderr,"%s: server closed connection\n",argv[0]);
	   if (changed_in)
	       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	   return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_AUTHREQ);
    dprintf("Got AUTHREQ\n");
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_authreq));
    packet_set_type(packet,CLIENT_AUTHREQ);
    bn_int_tag_set(&packet->u.client_authreq.archtag,ARCHTAG_INTEL);
    bn_int_tag_set(&packet->u.client_authreq.clienttag,clienttag);
    bn_int_set(&packet->u.client_authreq.versionid,0xBD);
    bn_byte_set(&packet->u.client_authreq.v4,0x00);
    bn_byte_set(&packet->u.client_authreq.v3,0x05);
    bn_byte_set(&packet->u.client_authreq.v2,0x00);
    bn_byte_set(&packet->u.client_authreq.v1,0x01);
    bn_int_set(&packet->u.client_authreq.answer,CLIENT_AUTHREQ_ANSWER);
    if (strcmp(clienttag,CLIENTTAG_STARCRAFT)==0)
    {
	packet_append_string(packet,CLIENT_AUTHREQ_EXEINFO_STAR);
	client_blocksend_packet(sd,packet);
    }
    if (strcmp(clienttag,CLIENTTAG_SHAREWARE)==0)
    {
	packet_append_string(packet,CLIENT_AUTHREQ_EXEINFO_SSHR);
	client_blocksend_packet(sd,packet);
    }
    else if (strcmp(clienttag,CLIENTTAG_BROODWARS)==0)
    {
	packet_append_string(packet,CLIENT_AUTHREQ_EXEINFO_SEXP);
	client_blocksend_packet(sd,packet);
    }
    packet_del_ref(packet);
#endif
    
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	   fprintf(stderr,"%s: server closed connection\n",argv[0]);
	   if (changed_in)
	       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	   return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_AUTHREPLY);
    dprintf("Got AUTHREPLY\n");
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_iconreq));
    packet_set_type(packet,CLIENT_ICONREQ);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	    fprintf(stderr,"%s: server closed connection\n",argv[0]);
	    if (changed_in)
		tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	    return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_ICONREPLY);
    dprintf("Got ICONREPLY\n");
    
    if (strcmp(clienttag,CLIENTTAG_STARCRAFT)==0 ||
	strcmp(clienttag,CLIENTTAG_BROODWARS)==0)
    {
	if (!(packet = packet_create(packet_class_normal)))
	{
	    fprintf(stderr,"%s: could not create packet\n",argv[0]);
	    if (changed_in)
		tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	    return 1;
	}
	packet_set_size(packet,sizeof(t_client_cdkey));
	packet_set_type(packet,CLIENT_CDKEY);
	bn_int_set(&packet->u.client_cdkey.unknown1,CLIENT_CDKEY_UNKNOWN1);
	packet_append_string(packet,cdkey);
	packet_append_string(packet,cdowner);
	client_blocksend_packet(sd,packet);
	packet_del_ref(packet);
	
	do
	    if (client_blockrecv_packet(sd,rpacket)<0)
	    {
		fprintf(stderr,"%s: server closed connection\n",argv[0]);
		if (changed_in)
		    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		return 1;
	    }
	while (packet_get_type(rpacket)!=SERVER_CDKEYREPLY);
	dprintf("Got CDKEYREPLY (%u)\n",bn_int_get(rpacket->u.server_cdkeyreply.message));
    }
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_tosreq));
    packet_set_type(packet,CLIENT_TOSREQ);
    bn_int_set(&packet->u.client_tosreq.unknown1,CLIENT_TOSREQ_UNKNOWN1);
    bn_int_set(&packet->u.client_tosreq.unknown2,CLIENT_TOSREQ_UNKNOWN2);
    packet_append_string(packet,CLIENT_TOSREQ_FILEUSA);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	   fprintf(stderr,"%s: server closed connection\n",argv[0]);
	   if (changed_in)
	       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	   return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_TOSREPLY);
    
    /* real client would also send statsreq on past logins here */
    
    printf("----\n");
    if (newacct)
    {
	if (useansi)
	    ansi_text_color_fore(ansi_text_color_red);
	printf("###### Terms Of Service ######\n");
	print_file(&saddr,packet_get_str_const(rpacket,sizeof(t_server_tosreply),1024),argv[0]);
	printf("##############################\n\n");
	if (useansi)
	    ansi_text_reset();
    }
	    
    for (;;)
    {
        char         password[MAX_MESSAGE_LEN];
	unsigned int i;
	int          status;
        
	if (newacct)
	{
	    char   passwordvrfy[MAX_MESSAGE_LEN];
            t_hash passhash1;
	    
	    printf("Enter information for your new account\n");
	    commpos = 0;
	    player[0] = '\0';
	    while ((status = client_get_comm("Username: ",player,MAX_MESSAGE_LEN,&commpos,1,SCREEN_WIDTH))==0);
	    printf("\n");
	    if (status<0)
		continue;
	    if (strchr(player,' '))
	    {
		printf("Spaces not allowed in username. Try again.\n");
		continue;
	    }
	    
	    commpos = 0;
	    password[0] = '\0';
	    while ((status = client_get_comm("Password: ",password,MAX_MESSAGE_LEN,&commpos,0,SCREEN_WIDTH))==0);
	    printf("\n");
	    if (status<0)
		continue;
	    for (i=0; i<strlen(password); i++)
		password[i] = tolower((int)password[i]);
	    
	    commpos = 0;
	    passwordvrfy[0] = '\0';
	    while ((status = client_get_comm("Retype password: ",passwordvrfy,MAX_MESSAGE_LEN,&commpos,0,SCREEN_WIDTH))==0);
	    printf("\n");
	    if (status<0)
		continue;
	    for (i=0; i<strlen(passwordvrfy); i++)
		passwordvrfy[i] = tolower((int)passwordvrfy[i]);
	    
	    if (strcmp(password,passwordvrfy)!=0)
	    {
		printf("Passwords do not match. Try again.\n");
		continue;
	    }
            
	    bnet_hash(&passhash1,strlen(password),password); /* do the single hash */
	    
	    if (!(packet = packet_create(packet_class_normal)))
	    {
		fprintf(stderr,"%s: could not create packet\n",argv[0]);
		if (changed_in)
		    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		return 1;
	    }
	    packet_set_size(packet,sizeof(t_client_createacctreq));
	    packet_set_type(packet,CLIENT_CREATEACCTREQ);
	    hash_to_bnhash((t_hash const *)&passhash1,packet->u.client_createacctreq.password_hash1); /* avoid warning */
	    packet_append_string(packet,player);
            client_blocksend_packet(sd,packet);
	    packet_del_ref(packet);
	    
	    do
		if (client_blockrecv_packet(sd,rpacket)<0)
		{
		   fprintf(stderr,"%s: server closed connection\n",argv[0]);
		   if (changed_in)
		       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		   return 1;
		}
	    while (packet_get_type(rpacket)!=SERVER_CREATEACCTREPLY);
	    dprintf("Got CREATEACCTREPLY\n");
	    if (bn_int_get(rpacket->u.server_createacctreply.result)==SERVER_CREATEACCTREPLY_RESULT_NO)
	    {
		printf("Could not create an account under that name. Try another one.\n");
		continue;
	    }
	    printf("Account created.\n");
	}
	else if (changepass)
	{
	    char         passwordprev[MAX_MESSAGE_LEN];
	    char         passwordvrfy[MAX_MESSAGE_LEN];
            struct
            {
                bn_int ticks;
                bn_int sessionkey;
                bn_int passhash1[5];
            }            temp;
            t_hash       oldpasshash1;
            t_hash       oldpasshash2;
            t_hash       newpasshash1;
	    unsigned int ticks;
	    
            printf("Enter your old and new login information\n");
	    
	    commpos = 0;
	    player[0] = '\0';
	    while ((status = client_get_comm("Username: ",player,MAX_MESSAGE_LEN,&commpos,1,SCREEN_WIDTH))==0);
	    printf("\n");
	    if (status<0)
		continue;
	    
	    commpos = 0;
	    passwordprev[0] = '\0';
	    while ((status = client_get_comm("Old password: ",passwordprev,MAX_MESSAGE_LEN,&commpos,0,SCREEN_WIDTH))==0);
	    printf("\n");
	    if (status<0)
		continue;
	    for (i=0; i<strlen(passwordprev); i++)
		passwordprev[i] = tolower((int)passwordprev[i]);
	    
	    commpos = 0;
	    password[0] = '\0';
	    while ((status = client_get_comm("New password: ",password,MAX_MESSAGE_LEN,&commpos,0,SCREEN_WIDTH))==0);
	    printf("\n");
	    if (status<0)
		continue;
	    for (i=0; i<strlen(password); i++)
		password[i] = tolower((int)password[i]);
	    
	    commpos = 0;
	    passwordvrfy[0] = '\0';
	    while ((status = client_get_comm("Retype new password: ",passwordvrfy,MAX_MESSAGE_LEN,&commpos,0,SCREEN_WIDTH))==0);
	    printf("\n");
	    if (status<0)
		continue;
	    for (i=0; i<strlen(passwordvrfy); i++)
		passwordvrfy[i] = tolower((int)passwordvrfy[i]);
	    
	    if (strcmp(password,passwordvrfy)!=0)
	    {
		printf("New passwords do not match. Try again.\n");
		continue;
	    }
	    
            ticks = 0; /* FIXME: what to use here? */
            bn_int_set(&temp.ticks,ticks);
            bn_int_set(&temp.sessionkey,sessionkey);
            bnet_hash(&oldpasshash1,strlen(passwordprev),passwordprev); /* do the single hash for old */
            hash_to_bnhash((t_hash const *)&oldpasshash1,temp.passhash1); /* avoid warning */
            bnet_hash(&oldpasshash2,sizeof(temp),&temp); /* do the double hash for old */
	    bnet_hash(&newpasshash1,strlen(password),password); /* do the single hash for new */
	    
	    if (!(packet = packet_create(packet_class_normal)))
	    {
		fprintf(stderr,"%s: could not create packet\n",argv[0]);
		if (changed_in)
		    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		return 1;
	    }
	    packet_set_size(packet,sizeof(t_client_changepassreq));
	    packet_set_type(packet,CLIENT_CHANGEPASSREQ);
	    bn_int_set(&packet->u.client_changepassreq.ticks,ticks);
	    bn_int_set(&packet->u.client_changepassreq.sessionkey,sessionkey);
	    hash_to_bnhash((t_hash const *)&oldpasshash2,packet->u.client_changepassreq.oldpassword_hash2); /* avoid warning */
	    hash_to_bnhash((t_hash const *)&newpasshash1,packet->u.client_changepassreq.newpassword_hash1); /* avoid warning */
	    packet_append_string(packet,player);
            client_blocksend_packet(sd,packet);
	    packet_del_ref(packet);
	    
	    do
		if (client_blockrecv_packet(sd,rpacket)<0)
		{
		   fprintf(stderr,"%s: server closed connection\n",argv[0]);
		   if (changed_in)
		       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		   return 1;
		}
	    while (packet_get_type(rpacket)!=SERVER_CHANGEPASSACK);
	    dprintf("Got CHANGEPASSACK\n");
	    if (bn_int_get(rpacket->u.server_changepassack.message)==SERVER_CHANGEPASSACK_MESSAGE_FAIL)
	    {
		printf("Could not change password. Try again.\n");
		continue;
	    }
	    printf("Password changed.\n");
	}
	else
	{
            printf("Enter your login information\n");
	    
	    commpos = 0;
	    player[0] = '\0';
	    while ((status = client_get_comm("Username: ",player,MAX_MESSAGE_LEN,&commpos,1,SCREEN_WIDTH))==0);
	    printf("\n");
	    if (status<0)
		continue;
	    
	    commpos = 0;
	    password[0] = '\0';
	    while ((status = client_get_comm("Password: ",password,MAX_MESSAGE_LEN,&commpos,0,SCREEN_WIDTH))==0);
	    printf("\n");
	    if (status<0)
		continue;
	    for (i=0; i<strlen(password); i++)
		password[i] = tolower((int)password[i]);
	}

	/* now login */
        {
            struct
            {
                bn_int ticks;
                bn_int sessionkey;
                bn_int passhash1[5];
            }            temp;
            t_hash       passhash1;
            t_hash       passhash2;
	    unsigned int ticks;
            
            ticks = 0; /* FIXME: what to use here? */
            bn_int_set(&temp.ticks,ticks);
            bn_int_set(&temp.sessionkey,sessionkey);
            bnet_hash(&passhash1,strlen(password),password); /* do the single hash */
            hash_to_bnhash((t_hash const *)&passhash1,temp.passhash1); /* avoid warning */
            bnet_hash(&passhash2,sizeof(temp),&temp); /* do the double hash */
	    
	    if (!(packet = packet_create(packet_class_normal)))
	    {
		fprintf(stderr,"%s: could not create packet\n",argv[0]);
		if (changed_in)
		    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		return 1;
	    }
	    packet_set_size(packet,sizeof(t_client_loginreq));
	    packet_set_type(packet,CLIENT_LOGINREQ);
	    bn_int_set(&packet->u.client_loginreq.ticks,ticks);
	    bn_int_set(&packet->u.client_loginreq.sessionkey,sessionkey);
	    hash_to_bnhash((t_hash const *)&passhash2,packet->u.client_loginreq.password_hash2); /* avoid warning */
	    packet_append_string(packet,player);
            client_blocksend_packet(sd,packet);
	    packet_del_ref(packet);
	}
	
        do
            if (client_blockrecv_packet(sd,rpacket)<0)
	    {
	       fprintf(stderr,"%s: server closed connection\n",argv[0]);
	       if (changed_in)
	           tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	       return 1;
	    }
        while (packet_get_type(rpacket)!=SERVER_LOGINREPLY);
	if (bn_int_get(rpacket->u.server_loginreply.message)==SERVER_LOGINREPLY_MESSAGE_SUCCESS)
	    break;
	fprintf(stderr,"Login incorrect.\n");
    }
    
    fprintf(stderr,"Logged in.\n");
    printf("----\n");
    
#ifdef CLIENTDEBUG
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_statsreq));
    packet_set_type(packet,CLIENT_STATSREQ);
    bn_int_set(&packet->u.client_statsreq.name_count,1);
    bn_int_set(&packet->u.client_statsreq.key_count,6);
    bn_int_set(&packet->u.client_statsreq.unknown1,CLIENT_STATSREQ_UNKNOWN1);
    packet_append_string(packet,player);
    packet_append_string(packet,"profile\\sex");
    packet_append_string(packet,"profile\\age");
    packet_append_string(packet,"profile\\location");
    packet_append_string(packet,"profile\\description");
    packet_append_string(packet,"BNET\\acct\\username");
    packet_append_string(packet,"BNET\\acct\\userid");
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	   fprintf(stderr,"%s: server closed connection\n",argv[0]);
	   if (changed_in)
	       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	   return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_STATSREPLY);
    
    {
	unsigned int names,keys;
	unsigned int i,j;
	unsigned int strpos;
	char const * temp;
	
	names = bn_int_get(rpacket->u.server_statsreply.name_count);
	keys = bn_int_get(rpacket->u.server_statsreply.key_count);
	
	strpos = sizeof(t_server_statsreply);
	for (i=0; i<names; i++)
	    for (j=0; j<keys; j++)
	    {
		if ((temp = packet_get_str_const(rpacket,strpos,256)))
		{
		    printf("returned[%u][%u]: \"%s\"\n",i,j,temp);
		    strpos += strlen(temp)+1;
		}
		else
		    printf("returned[%u][%u]: none\n",i,j);
	    }
    }
#endif
    
    if (newacct && (strcmp(clienttag,CLIENTTAG_DIABLORTL)==0 ||
		    strcmp(clienttag,CLIENTTAG_DIABLOSHR)==0))
    {
	if (!(packet = packet_create(packet_class_normal)))
	{
	    fprintf(stderr,"%s: could not create packet\n",argv[0]);
	    if (changed_in)
		tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	}
	else
	{
	    packet_set_size(packet,sizeof(t_client_playerinforeq));
	    packet_set_type(packet,CLIENT_PLAYERINFOREQ);
	    packet_append_string(packet,player);
	    packet_append_string(packet,"LTRD 1 2 0 20 25 15 20 100 0");
	    client_blocksend_packet(sd,packet);
	    packet_del_ref(packet);
	}
    }
    
#if 0
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_playerinforeq));
    packet_set_type(packet,CLIENT_PLAYERINFOREQ);
    packet_append_string(packet,player);
    packet_append_string(packet,"");
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    /* wait for playerinforeply */
#endif
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_progident2));
    packet_set_type(packet,CLIENT_PROGIDENT2);
    bn_int_tag_set(&packet->u.client_progident2.clienttag,clienttag);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_joinchannel));
    packet_set_type(packet,CLIENT_JOINCHANNEL);
    bn_int_set(&packet->u.client_joinchannel.channelflag,CLIENT_JOINCHANNEL_GENERIC);
    packet_append_string(packet,channel);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    if (fcntl(sd,F_SETFL,O_NONBLOCK)<0)
	fprintf(stderr,"%s: could not set TCP socket to non-blocking mode (fcntl: %s)\n",argv[0],strerror(errno));
    
    {
	unsigned int i;
	int          munged;
	int          highest_fd;
	fd_set       rfds;
	
	FD_ZERO(&rfds);
	highest_fd = fd_stdin;
	if (sd>highest_fd)
	    highest_fd = sd;
	
	currsize = 0;
	
	munged = 1; /* == need to draw prompt */
	commpos = 0;
	text[0] = '\0';
	
	for (;;)
	{
	    if (munged)
	    {
		printf("] %s", text + ((SCREEN_WIDTH <= 2 + commpos ) ? 3 + commpos - SCREEN_WIDTH: 0));
		fflush(stdout);
		munged = 0;
	    }
	    
	    FD_SET(fd_stdin,&rfds);
	    FD_SET(sd,&rfds);
	    
	    if (select(highest_fd+1,&rfds,NULL,NULL,NULL)<0)
	    {
		if (errno!=EINTR)
		{
		    if (!munged)
		    {
			printf("\r  ");
			for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
			    printf(" ");
			printf("\r  ");
		    }
		    printf("Select failed (select: %s)\n",strerror(errno));
		}
		continue;
	    }
	    
	    if (FD_ISSET(fd_stdin,&rfds)) /* got keyboard data */
	    {
		munged = 0;
		
		switch (client_get_comm("] ",text,MAX_MESSAGE_LEN,&commpos,1,SCREEN_WIDTH))
		{
		case -1: /* cancel */
		    printf("\r  ");
		    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
			printf(" ");
		    printf("\r");
		    munged = 1;
		    commpos = 0;
		    text[0] = '\0';
		    break;
		    
		case 0: /* timeout */
		    break;
		    
		case 1:
		    if (text[0]!='\0')
		    {
			if (text[0]=='/')
			{
			    printf("\r  ");
			    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
				printf(" ");
			    printf("\r");
			}
			else
			{
			    if (useansi)
				ansi_text_color_fore(ansi_text_color_blue);
			    printf("\r<%s>",player);
			    if (useansi)
				ansi_text_reset();
			    printf(" ");
			    str_print_term(stdout,text,0,0);
			    printf("\n");
			}
			munged = 1;
			
			if (!(packet = packet_create(packet_class_normal)))
			{
			    fprintf(stderr,"%s: could not create packet\n",argv[0]);
			    if (changed_in)
				tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
			    return 1;
			}
			packet_set_size(packet,sizeof(t_client_message));
			packet_set_type(packet,CLIENT_MESSAGE);
			packet_append_string(packet,text);
			client_blocksend_packet(sd,packet);
			packet_del_ref(packet);
		    }
		    
		    commpos = 0;
		    text[0] = '\0';
		    break;
		}
	    }
	    
	    if (FD_ISSET(sd,&rfds)) /* got network data */
	    {
		/* rpacket is from server, packet is from client */
		switch (net_recv_packet(sd,rpacket,&currsize))
		{
		case 0: /* nothing */
		    break;
		    
		case 1:
		    switch (packet_get_type(rpacket))
		    {
		    case SERVER_ECHOREQ: /* might as well support it */
			if (packet_get_size(rpacket)<sizeof(t_server_echoreq))
			{
			    if (!munged)
			    {
				printf("\r  ");
				for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
				    printf(" ");
				printf("\r");
				munged = 1;
			    }
			    printf("Got bad SERVER_ECHOREQ packet (expected %u bytes, got %u)\n",sizeof(t_server_echoreq),packet_get_size(rpacket));
			    break;
			}
			
			if (!(packet = packet_create(packet_class_normal)))
			{
			    if (!munged)
			    {
				printf("\r  ");
				for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
				    printf(" ");
				printf("\r");
				munged = 1;
			    }
			    printf("Could not create packet\n");
			    if (changed_in)
				tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
			    return 1;
			}
			packet_set_size(packet,sizeof(t_client_echoreply));
			packet_set_type(packet,CLIENT_ECHOREPLY);
			bn_int_set(&packet->u.client_echoreply.ticks,bn_int_get(rpacket->u.server_echoreq.ticks));
			client_blocksend_packet(sd,packet);
			packet_del_ref(packet);
			
			break;
			
		    case SERVER_PLAYERINFOREPLY: /* hmm. we didn't ask for one... */
			break;
			
		    case SERVER_CHANNELLIST:
			break;
			
		    case SERVER_MESSAGE:
			if (packet_get_size(rpacket)<sizeof(t_server_message))
			{
			    if (!munged)
			    {
				printf("\r  ");
				for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
				    printf(" ");
				printf("\r");
				munged = 1;
			    }
			    printf("Got bad SERVER_MESSAGE packet (expected %u bytes, got %u)",sizeof(t_server_message),packet_get_size(rpacket));
			    break;
			}
			
			{
			    char const * speaker;
			    char const * message;
			    
			    if (!(speaker = packet_get_str_const(rpacket,sizeof(t_server_message),32)))
			    {
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				printf("Got SERVER_MESSAGE packet with bad or missing speaker\n");
				break;
			    }
			    if (!(message = packet_get_str_const(rpacket,sizeof(t_server_message)+strlen(speaker)+1,1024)))
			    {
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				printf("Got SERVER_MESSAGE packet with bad or missing message\n");
				break;
			    }
			    
			    switch (bn_int_get(rpacket->u.server_message.type))
			    {
			    case MT_JOIN:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_green);
				printf("\"");
				str_print_term(stdout,speaker,0,0);
				printf("\" enters\n");
				if (useansi)
				    ansi_text_reset();
				break;
			    case MT_JOINING:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_green);
				printf("Joining channel: \"");
				str_print_term(stdout,message,0,0);
				printf("\"\n");
				if (useansi)
				    ansi_text_reset();
				break;
			    case MT_ADD:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_green);
				printf("\"");
				str_print_term(stdout,speaker,0,0);
				printf("\" is here\n");
				if (useansi)
				    ansi_text_reset();
				break;
			    case MT_USERFLAGS:
				break;
			    case MT_PART:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_green);
				printf("\"");
				str_print_term(stdout,speaker,0,0);
				printf("\" leaves\n");
				if (useansi)
				    ansi_text_reset();
				break;
			    case MT_WHISPER:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_blue);
				printf("<From: ");
				str_print_term(stdout,speaker,0,0);
				printf(">");
				if (useansi)
				    ansi_text_reset();
				printf(" ");
				str_print_term(stdout,message,0,0);
				printf("\n");
				break;
			    case MT_WHISPERACK:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_blue);
				printf("<To: ");
				str_print_term(stdout,speaker,0,0);
				printf(">");
				if (useansi)
				    ansi_text_reset();
				printf(" ");
				str_print_term(stdout,message,0,0);
				printf("\n");
				break;
			    case MT_BROADCAST:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_yellow);
				printf("<Broadcast: ");
				str_print_term(stdout,speaker,0,0);
				printf("> ");
				str_print_term(stdout,message,0,0);
				printf("\n");
				if (useansi)
				    ansi_text_reset();
				break;
			    case MT_ERROR:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_red);
				printf("<Error> ");
				str_print_term(stdout,message,0,0);
				printf("\n");
				if (useansi)
				    ansi_text_reset();
				break;
			    case MT_WARN:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_yellow);
				printf("<Info> ");
				str_print_term(stdout,message,0,0);
				printf("\n");
				if (useansi)
				    ansi_text_reset();
				break;
			    case MT_EMOTE:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_yellow);
				printf("<");
				str_print_term(stdout,speaker,0,0);
				printf(" ");
				str_print_term(stdout,message,0,0);
				printf(">\n");
				if (useansi)
				    ansi_text_reset();
				break;
			    default:
			    case MT_MESSAGE:
				if (!munged)
				{
				    printf("\r  ");
				    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
					printf(" ");
				    printf("\r");
				    munged = 1;
				}
				if (useansi)
				    ansi_text_color_fore(ansi_text_color_yellow);
				printf("<");
				str_print_term(stdout,speaker,0,0);
				printf(">");
				if (useansi)
				    ansi_text_reset();
				printf(" ");
				str_print_term(stdout,message,0,0);
				printf("\n");
			    }
			}
			break;
			
		    default:
			if (!munged)
			{
			    printf("\r  ");
			    for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
				printf(" ");
			    printf("\r");
			    munged = 1;
			}
			printf("Unsupported server packet type 0x%04x\n",packet_get_type(rpacket));
		    }
		    
		    currsize = 0;
		    break;
		    
		case -1: /* error (probably connection closed) */
		default:
		    if (!munged)
		    {
			printf("\r  ");
			for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
			    printf(" ");
			printf("\r");
			munged = 1;
		    }
		    printf("----\nConnection closed by server.\n");
		    if (changed_in)
			tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		    return 0;
		}
	    }
	}
    }
    
    /* not reached */
    return 0;
}
