/*
 * Copyright (C) 1999  Rob Crittenden (rcrit@greyoak.com)
 * Copyright (C) 1999  Ross Combs (rocombs@cs.nmsu.edu)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <math.h>
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include <errno.h>
#include "field_sizes.h"
#include "account.h"
#include "eventlog.h"
#include "prefs.h"
#include "util.h"
#include "game.h"
#include "tag.h"
#include "list.h"
#include "bnettime.h"
#include "ladder.h"


static t_account * * star_ladder_current_highestrated=NULL;
static t_account * * star_ladder_current_mostwins=NULL;
static t_account * * star_ladder_current_mostgames=NULL;
static unsigned int star_ladder_current_len=0;

static t_account * * sexp_ladder_current_highestrated=NULL;
static t_account * * sexp_ladder_current_mostwins=NULL;
static t_account * * sexp_ladder_current_mostgames=NULL;
static unsigned int sexp_ladder_current_len=0;


static t_account * * star_ladder_active_highestrated=NULL;
static t_account * * star_ladder_active_mostwins=NULL;
static t_account * * star_ladder_active_mostgames=NULL;
static unsigned int star_ladder_active_len=0;

static t_account * * sexp_ladder_active_highestrated=NULL;
static t_account * * sexp_ladder_active_mostwins=NULL;
static t_account * * sexp_ladder_active_mostgames=NULL;
static unsigned int sexp_ladder_active_len=0;


static char const * compare_clienttag=NULL;

static int ladder_locate(t_account * const * * ladder, unsigned int * ladder_len, t_ladder_sort lsort, t_ladder_time ltime, char const * clienttag);
static double probability(unsigned int a, unsigned int b);

static double two_player(t_account * * players, char const * clienttag);

static double three_player(t_account * * players, char const * clienttag);

static double four_player(t_account * * players, char const * clienttag);

static double five_player(t_account * * players, char const * clienttag);

static double six_player(t_account * * players, char const * clienttag);
static double six_f1(double a, double b, double c, double d, double e, double f);
static double six_f2(double a, double b, double c, double d, double e, double f);
static double six_f3(double a, double b, double c, double d);

static double seven_player(t_account * * players, char const * clienttag);

static double eight_player(t_account * * players, char const * clienttag);
static double eight_f1(double a, double b, double c, double d, double e, double f, double g);
static double eight_f2(double a, double b, double c, double d, double e, double f, double g);
static double eight_f3(double a, double b, double c, double d, double e);
static double eight_f4(double a, double b, double c, double d);

static int ladder_insert(t_account * account, t_account * * * ladder_highestrated, t_account * * * ladder_mostwins, t_account * * * ladder_mostgames, unsigned int * ladder_len);
static int compare_current_highestrated(void const * a, void const * b);
static int compare_current_mostwins(void const * a, void const * b);
static int compare_current_mostgames(void const * a, void const * b);
static int compare_active_highestrated(void const * a, void const * b);
static int compare_active_mostwins(void const * a, void const * b);
static int compare_active_mostgames(void const * a, void const * b);
static int ladder_current_sort(t_account * * ladder_highestrated, t_account * * ladder_mostwins, t_account * * ladder_mostgames, unsigned int ladder_len);
static int ladder_active_sort(t_account * * ladder_highestrated, t_account * * ladder_mostwins, t_account * * ladder_mostgames, unsigned int ladder_len);
static int ladder_make_active(t_account * const * ladder_highestrated, t_account * const * ladder_mostwins, t_account * const * ladder_mostgames, unsigned int ladder_len, t_account * * * active_ladder_highestrated, t_account * * * active_ladder_mostwins, t_account * * * active_ladder_mostgames, unsigned int * active_ladder_len, char const * clienttag);


/*
 * Set the ladder and length parameters to point to the proper ladder
 * given the sort and time arguments.
 */
static int ladder_locate(t_account * const * * ladder, unsigned int * ladder_len, t_ladder_sort lsort, t_ladder_time ltime, char const * clienttag)
{
    if (!ladder)
    {
	eventlog(eventlog_level_error,"ladder_locate","got NULL ladder");
	return -1;
    }
    if (!ladder_len)
    {
	eventlog(eventlog_level_error,"ladder_locate","got NULL ladder_len");
	return -1;
    }
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"ladder_locate","got bad clienttag");
	return -1;
    }
    
    if (strcmp(clienttag,CLIENTTAG_STARCRAFT)==0)
    {
	switch (ltime)
	{
	case ladder_time_current:
	    switch (lsort)
	    {
	    case ladder_sort_highestrated:
		*ladder     = star_ladder_current_highestrated;
		*ladder_len = star_ladder_current_len;
		break;
	    case ladder_sort_mostwins:
		*ladder     = star_ladder_current_mostwins;
		*ladder_len = star_ladder_current_len;
		break;
	    case ladder_sort_mostgames:
		*ladder     = star_ladder_current_mostgames;
		*ladder_len = star_ladder_current_len;
		break;
	    default:
		eventlog(eventlog_level_error,"ladder_locate","got bad ladder sort %u",(unsigned int)lsort);
		return -1;
	    }
	    break;
	    
	case ladder_time_active:
	    switch (lsort)
	    {
	    case ladder_sort_highestrated:
		*ladder     = star_ladder_active_highestrated;
		*ladder_len = star_ladder_active_len;
		break;
	    case ladder_sort_mostwins:
		*ladder     = star_ladder_active_mostwins;
		*ladder_len = star_ladder_active_len;
		break;
	    case ladder_sort_mostgames:
		*ladder     = star_ladder_active_mostgames;
		*ladder_len = star_ladder_active_len;
		break;
	    default:
		eventlog(eventlog_level_error,"ladder_locate","got bad ladder sort %u",(unsigned int)lsort);
		return -1;
	    }
	    break;
	    
	default:
	    eventlog(eventlog_level_error,"ladder_locate","got bad ladder time %u",(unsigned int)ltime);
	    return -1;
	}
    }
    else if (strcmp(clienttag,CLIENTTAG_BROODWARS)==0)
    {
	switch (ltime)
	{
	case ladder_time_current:
	    switch (lsort)
	    {
	    case ladder_sort_highestrated:
		*ladder     = sexp_ladder_current_highestrated;
		*ladder_len = sexp_ladder_current_len;
		break;
	    case ladder_sort_mostwins:
		*ladder     = sexp_ladder_current_mostwins;
		*ladder_len = sexp_ladder_current_len;
		break;
	    case ladder_sort_mostgames:
		*ladder     = sexp_ladder_current_mostgames;
		*ladder_len = sexp_ladder_current_len;
		break;
	    default:
		eventlog(eventlog_level_error,"ladder_locate","got bad ladder sort %u",(unsigned int)lsort);
		return -1;
	    }
	    break;
	    
	case ladder_time_active:
	    switch (lsort)
	    {
	    case ladder_sort_highestrated:
		*ladder     = sexp_ladder_active_highestrated;
		*ladder_len = sexp_ladder_active_len;
		break;
	    case ladder_sort_mostwins:
		*ladder     = sexp_ladder_active_mostwins;
		*ladder_len = sexp_ladder_active_len;
		break;
	    case ladder_sort_mostgames:
		*ladder     = sexp_ladder_active_mostgames;
		*ladder_len = sexp_ladder_active_len;
		break;
	    default:
		eventlog(eventlog_level_error,"ladder_locate","got bad ladder sort %u",(unsigned int)lsort);
		return -1;
	    }
	    break;
	    
	default:
	    eventlog(eventlog_level_error,"ladder_locate","got bad ladder time %u",(unsigned int)ltime);
	    return -1;
	}
    }
    else
    {
	eventlog(eventlog_level_error,"ladder_locate","got unknown clienttag \"%s\"",clienttag);
	return -1;
    }
    
/*    eventlog(eventlog_level_debug,"ladder_locate","returning *ladder=%p *ladder_len=%u tag=%s ltime=%u lsort=%u",*ladder,*ladder_len,clienttag,ltime,lsort); */
    return 0;
}


/*
 * Compute probability of winning using the Elo system
 *
 * the formula is:
 *
 *  D = rating(payer a) - rating(player b)
 *
 *                    1
 *  Pwin(D) = ------------------
 *                   -(D / 400)
 *             1 + 10
 */
static double probability(unsigned int a, unsigned int b)
{
    double i, j;
    
    i = (((double)a) - ((double)b)) / 400.0;
    
    j = pow(10.0,-i);
    
    return (1.0 / (1.0+j));
}


/*
 *  This is the coefficient k which is meant to enhance the
 *  effect of the Elo system where more experienced players
 *  will gain fewer points when playing against newbies, and
 *  newbies will gain massive points if they win against an
 *  experienced player. It also helps stabilize a player's
 *  rating after they have played 30 games or so.
 *
 *  K=50 for new players
 *  K=30 for players who have played 30 or more ladder games
 *  K=20 for players who have attained a rating of 2400 or higher
 */
static int coefficient(t_account * account, char const * clienttag)
{
    int const total_ladder_games=account_get_ladder_wins(account, clienttag) + 
				 account_get_ladder_losses(account, clienttag) +
				 account_get_ladder_disconnects(account, clienttag);
    
    if (total_ladder_games < 30)
	return 50;
    
    if (account_get_ladder_rating(account,clienttag) < 2400)
	return 30;
    
    return 20;
}


/*
 * The Elo system only handles 2 players, these functions extend
 * the calculation to different numbers of players as if they were
 * in a tournament. It turns out the math for this is really ugly,
 * so we have hardcoded the equations for every number of players.
 */	
static double two_player(t_account * * players, char const * clienttag)
{
    unsigned int a,b;
    double       ab;
    
    a = account_get_ladder_rating(players[0],clienttag);
    b = account_get_ladder_rating(players[1],clienttag);
    
    ab = probability(a,b);
    
    return ab;
}


static double three_player(t_account * * players, char const * clienttag)
{
    unsigned int a,b,c;
    double       ab,ac,bc,cb;
    
    a = account_get_ladder_rating(players[0],clienttag);
    b = account_get_ladder_rating(players[1],clienttag);
    c = account_get_ladder_rating(players[2],clienttag);
    
    ab = probability(a,b);
    ac = probability(a,c);
    bc = probability(b,c);
    cb = probability(c,b);
    
    return ((ab * ac) +
	    (ac * ab) +
	    (bc * ab) +
	    (cb * ac)) / 3;
}
	      

static double four_player(t_account * * players, char const * clienttag)
{
    unsigned int a,b,c,d;
    double       ab,ac,ad,bc,bd,cb,cd,db,dc;
    
    a = account_get_ladder_rating(players[0],clienttag);
    b = account_get_ladder_rating(players[1],clienttag);
    c = account_get_ladder_rating(players[2],clienttag);
    d = account_get_ladder_rating(players[3],clienttag);
    
    ab = probability(a,b);
    ac = probability(a,c);
    ad = probability(a,d);
    bc = probability(b,c);
    bd = probability(b,d);
    cb = probability(c,b);
    cd = probability(c,d);
    db = probability(d,b);
    dc = probability(d,c);
    
    return (ab*cd*ac + ab*dc*ad +
	    ac*bd*ab + ac*db*ad +
	    ad*bc*ab + ad*cb*ac) / 3;
}


static double five_player(t_account * * players, char const * clienttag)
{
    unsigned int a,b,c,d,e;
    
    a = account_get_ladder_rating(players[0],clienttag);
    b = account_get_ladder_rating(players[1],clienttag);
    c = account_get_ladder_rating(players[2],clienttag);
    d = account_get_ladder_rating(players[3],clienttag);
    e = account_get_ladder_rating(players[4],clienttag);

/* FIXME: Insert here calculation for 5 players */

    return 0;
}


static double six_player(t_account * * players, char const * clienttag)
{
    unsigned int a,b,c,d,e,f;
    
    a = account_get_ladder_rating(players[0],clienttag);
    b = account_get_ladder_rating(players[1],clienttag);
    c = account_get_ladder_rating(players[2],clienttag);
    d = account_get_ladder_rating(players[3],clienttag);
    e = account_get_ladder_rating(players[4],clienttag);
    f = account_get_ladder_rating(players[5],clienttag);

/* A B C D
 *  A   C    E F
 *    A       E
 *        A
 */

    return (six_f1(a,b,c,d,e,f)+  /* A is in group of 4 */
          six_f1(a,b,c,e,d,f)+
          six_f1(a,b,e,d,c,f)+
          six_f1(a,e,c,d,b,f)+
          six_f1(a,b,c,f,d,e)+
          six_f1(a,b,f,d,c,e)+
          six_f1(a,f,c,d,b,e)+
          six_f1(a,e,f,b,c,d)+
          six_f1(a,e,f,c,b,d)+
          six_f1(a,e,f,d,b,c)+
          six_f2(a,b,c,d,e,f)+   /* A is in group of 2 */
          six_f2(a,c,b,d,e,f)+
          six_f2(a,d,b,c,e,f)+
          six_f2(a,e,b,c,d,f)+
          six_f2(a,f,b,c,d,e))/45; 
}


/* ABCD = group of 4, EF = group of 2, A must win */

static double six_f1(double a, double b, double c, double d,double e, double f)
{
    double ab,ac,ad,bc,bd,cb,cd,db,dc,ef,fe,ae,af;

    ab = probability(a,b);
    ac = probability(a,c);
    ad = probability(a,d);
    bc = probability(b,c);
    bd = probability(b,d);
    cb = probability(c,b);
    cd = probability(c,d);
    db = probability(d,b);
    dc = probability(d,c);
    ef = probability(e,f);
    fe = probability(f,e);
    ae = probability(a,e);
    af = probability(a,f);

    return (ab*cd*ac + ab*dc*ad +
           ac*bd*ab + ac*db*ad +
           ad*bc*ab + ad*cb*ac)*
              (ef*ae+fe*af);
}


/* AB is group of 2, CDEF is group of 4, A must win */

static double six_f2(double a, double b, double c, double d, double e, double f)
{
    double ab,ac,ad,ae,af;

    ab = probability(a,b);
    ac = probability(a,c);
    ad = probability(a,d);
    ae = probability(a,e);
    af = probability(a,f);

    return (six_f3(c,d,e,f)*ab*ac+
            six_f3(d,c,e,f)*ab*ad+
            six_f3(e,c,d,f)*ab*ae+
            six_f3(f,c,d,e)*ab*af);
}


/* ABCD is group of 4, A win */

static double six_f3(double a, double b, double c, double d)
{
    double ab,ac,ad,bc,bd,cb,cd,db,dc;
    
    ab = probability(a,b);
    ac = probability(a,c);
    ad = probability(a,d);
    bc = probability(b,c);
    bd = probability(b,d);
    cb = probability(c,b);
    cd = probability(c,d);
    db = probability(d,b);
    dc = probability(d,c);
    
    return (ab*cd*ac + ab*dc*ad +
            ac*bd*ab + ac*db*ad +
            ad*bc*ab + ad*cb*ac);
}


static double seven_player(t_account * * players, char const * clienttag)
{
    unsigned int a,b,c,d,e,f,g;
    
    a = account_get_ladder_rating(players[0],clienttag);
    b = account_get_ladder_rating(players[1],clienttag);
    c = account_get_ladder_rating(players[2],clienttag);
    d = account_get_ladder_rating(players[3],clienttag);
    e = account_get_ladder_rating(players[4],clienttag);
    f = account_get_ladder_rating(players[5],clienttag);
    g = account_get_ladder_rating(players[6],clienttag);

/* FIXME: Insert here calculation for seven players */

    return 0;
}


static double eight_player(t_account * * players, char const * clienttag)
{
    unsigned int a,b,c,d,e,f,g,h;
    double       ab,ac,ad,ae,af,ag,ah;
    
    a = account_get_ladder_rating(players[0],clienttag);
    b = account_get_ladder_rating(players[1],clienttag);
    c = account_get_ladder_rating(players[2],clienttag);
    d = account_get_ladder_rating(players[3],clienttag);
    e = account_get_ladder_rating(players[4],clienttag);
    f = account_get_ladder_rating(players[5],clienttag);
    g = account_get_ladder_rating(players[6],clienttag);
    h = account_get_ladder_rating(players[7],clienttag);

    ab = probability(a,b);
    ac = probability(a,c);
    ad = probability(a,d);
    ae = probability(a,e);
    af = probability(a,f);
    ag = probability(a,g);
    ah = probability(a,h);

/* First against A may be one from seven */

    return (eight_f1(a,c,d,e,f,g,h)*ab+
	    eight_f1(a,b,d,e,f,g,h)*ac+
	    eight_f1(a,b,c,e,f,g,h)*ad+
	    eight_f1(a,b,c,d,f,g,h)*ae+
	    eight_f1(a,b,c,d,e,g,h)*af+
	    eight_f1(a,b,c,d,e,f,h)*ag+
	    eight_f1(a,b,c,d,e,f,g)*ah)/315;

}

static double eight_f1(double a, double b, double c, double d, double e, double f, double g)
{
/* The winner of the second group, who'll then play against A, may be one
   from six possible players */
   
    return eight_f2(a,b,c,d,e,f,g)+
           eight_f2(a,c,b,d,e,f,g)+
           eight_f2(a,d,b,c,e,f,g)+
           eight_f2(a,e,b,c,d,f,g)+
           eight_f2(a,f,b,c,d,e,g)+
           eight_f2(a,g,b,c,d,e,f);
}

static double eight_f2(double a, double b, double c, double d, double e, double f, double g)
{
    double ab,bc,bd,be,bf,bg;
    
    ab = probability(a,b);
    bc = probability(b,c);
    bd = probability(b,d);
    be = probability(b,e);
    bf = probability(b,f);
    bg = probability(b,g);

/* There are 5 player who may play against the 3rd. The third (b) will win
   over them and lose against a */
   
    return ab*(eight_f3(a,d,e,f,g)*bc+
               eight_f3(a,c,e,f,g)*bd+
	       eight_f3(a,c,d,f,g)*be+
	       eight_f3(a,c,d,e,g)*bf+
	       eight_f3(a,c,d,e,f)*bg);

}

static double eight_f3(double a, double b, double c, double d, double e)
{
    double ab,ac,ad,ae;
    
    ab = probability(a,b);
    ac = probability(a,c);
    ad = probability(a,d);
    ae = probability(a,e);
    
/* There are 4 possible winners in the second group of 4. 
   They'll have to lose against a */

    return eight_f4(b,c,d,e)*ab+eight_f4(c,b,d,e)*ac+
           eight_f4(d,b,c,e)*ad+eight_f4(e,b,c,d)*ae;
}

/* Basic 4-player tree */ 

static double eight_f4(double a, double b, double c, double d)
{
    double ab,ac,ad,bc,bd,cb,cd,db,dc;
    
    ab = probability(a,b);
    ac = probability(a,c);
    ad = probability(a,d);
    bc = probability(b,c);
    bd = probability(b,d);
    cb = probability(c,b);
    cd = probability(c,d);
    db = probability(d,b);
    dc = probability(d,c);
    
    return (ab*cd*ac + ab*dc*ad +
	    ac*bd*ab + ac*db*ad +
	    ad*bc*ab + ad*cb*ac);
}


/*
 * Insert an account into the proper ladder arrays. This routine
 * does _not_ do the sorting and ranking updates.
 */
static int ladder_insert(t_account * account, t_account * * * ladder_highestrated, t_account * * * ladder_mostwins, t_account * * * ladder_mostgames, unsigned int * ladder_len)
{
    t_account * * temp_ladder_highestrated;
    t_account * * temp_ladder_mostwins;
    t_account * * temp_ladder_mostgames;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"ladder_insert","got NULL account");
	return -1;
    }
    if (!ladder_highestrated || !ladder_mostwins || !ladder_mostgames || !ladder_len)
    {
	eventlog(eventlog_level_error,"ladder_insert","got NULL ladder parameter");
	return -1;
    }
    
    if (*ladder_highestrated) /* some realloc()s are broken, so this gets to be ugly */
    {
	if (!(temp_ladder_highestrated = realloc(*ladder_highestrated,sizeof(t_account *)*(*ladder_len+1))))
	{
	    eventlog(eventlog_level_error,"ladder_insert","could not allocate memory for temp_ladder_highestrated");
	    return -1;
	}
    }
    else
	if (!(temp_ladder_highestrated = malloc(sizeof(t_account *)*(*ladder_len+1))))
	{
	    eventlog(eventlog_level_error,"ladder_insert","could not allocate memory for temp_ladder_highestrated");
	    return -1;
	}
    
    if (*ladder_mostwins)
    {
	if (!(temp_ladder_mostwins = realloc(*ladder_mostwins,sizeof(t_account *)*(*ladder_len+1))))
	{
	    eventlog(eventlog_level_error,"ladder_insert","could not allocate memory for temp_ladder_mostwins");
	    *ladder_highestrated = temp_ladder_highestrated;
	    return -1;
	}
    }
    else
	if (!(temp_ladder_mostwins = malloc(sizeof(t_account *)*(*ladder_len+1))))
	{
	    eventlog(eventlog_level_error,"ladder_insert","could not allocate memory for temp_ladder_mostwins");
	    *ladder_highestrated = temp_ladder_highestrated;
	    return -1;
	}
    
    if (*ladder_mostgames)
    {
	if (!(temp_ladder_mostgames = realloc(*ladder_mostgames,sizeof(t_account *)*(*ladder_len+1))))
	{
	    eventlog(eventlog_level_error,"ladder_insert","could not allocate memory for temp_ladder_mostgames");
	    *ladder_mostwins     = temp_ladder_mostwins;
	    *ladder_highestrated = temp_ladder_highestrated;
	    return -1;
	}
    }
    else
	if (!(temp_ladder_mostgames = malloc(sizeof(t_account *)*(*ladder_len+1))))
	{
	    eventlog(eventlog_level_error,"ladder_insert","could not allocate memory for temp_ladder_mostgames");
	    *ladder_mostwins     = temp_ladder_mostwins;
	    *ladder_highestrated = temp_ladder_highestrated;
	    return -1;
	}
    
    temp_ladder_highestrated[*ladder_len] = account;
    temp_ladder_mostwins[*ladder_len]     = account;
    temp_ladder_mostgames[*ladder_len]    = account;
    
    *ladder_highestrated = temp_ladder_highestrated;
    *ladder_mostwins     = temp_ladder_mostwins;
    *ladder_mostgames    = temp_ladder_mostgames;
    
    (*ladder_len)++;
    
    return 0;
}


static int compare_current_highestrated(void const * a, void const * b)
{
    t_account *  x=*(t_account * const *)a;
    t_account *  y=*(t_account * const *)b;
    unsigned int xrating;
    unsigned int yrating;
    unsigned int xwins;
    unsigned int ywins;
    unsigned int xoldrank;
    unsigned int yoldrank;
    
    if (!x || !y)
    {
	eventlog(eventlog_level_error,"compare_current_highestrated","got NULL account");
	return 0;
    }
    
    xrating = account_get_ladder_rating(x,compare_clienttag);
    yrating = account_get_ladder_rating(y,compare_clienttag);
    if (xrating>yrating)
        return -1;
    if (xrating<yrating)
        return +1;
    
    xwins = account_get_ladder_wins(x,compare_clienttag);
    ywins = account_get_ladder_wins(y,compare_clienttag);
    if (xwins>ywins)
        return -1;
    if (xwins<ywins)
        return +1;
    
    xoldrank = account_get_ladder_rank(x,compare_clienttag);
    yoldrank = account_get_ladder_rank(y,compare_clienttag);
    if (xoldrank<yoldrank)
        return -1;
    if (xoldrank>yoldrank)
        return +1;
    
    return 0;
}


static int compare_current_mostwins(void const * a, void const * b)
{
    t_account *  x=*(t_account * const *)a;
    t_account *  y=*(t_account * const *)b;
    unsigned int xwins;
    unsigned int ywins;
    unsigned int xrating;
    unsigned int yrating;
    unsigned int xoldrank;
    unsigned int yoldrank;
    
    if (!x || !y)
    {
	eventlog(eventlog_level_error,"compare_current_mostwins","got NULL account");
	return 0;
    }
    
    xwins = account_get_ladder_wins(x,compare_clienttag);
    ywins = account_get_ladder_wins(y,compare_clienttag);
    if (xwins>ywins)
        return -1;
    if (xwins<ywins)
        return +1;
    
    xrating = account_get_ladder_rating(x,compare_clienttag);
    yrating = account_get_ladder_rating(y,compare_clienttag);
    if (xrating>yrating)
        return -1;
    if (xrating<yrating)
        return +1;
    
    xoldrank = account_get_ladder_rank(x,compare_clienttag);
    yoldrank = account_get_ladder_rank(y,compare_clienttag);
    if (xoldrank<yoldrank)
        return -1;
    if (xoldrank>yoldrank)
        return +1;
    
    return 0;
}


static int compare_current_mostgames(void const * a, void const * b)
{
    t_account *  x=*(t_account * const *)a;
    t_account *  y=*(t_account * const *)b;
    unsigned int xgames;
    unsigned int ygames;
    unsigned int xrating;
    unsigned int yrating;
    unsigned int xwins;
    unsigned int ywins;
    unsigned int xoldrank;
    unsigned int yoldrank;
    
    if (!x || !y)
    {
	eventlog(eventlog_level_error,"compare_current_mostgames","got NULL account");
	return 0;
    }
    
    xgames = account_get_ladder_wins(x,compare_clienttag)+
	     account_get_ladder_losses(x,compare_clienttag)+
	     account_get_ladder_draws(x,compare_clienttag)+
	     account_get_ladder_disconnects(x,compare_clienttag);
    ygames = account_get_ladder_wins(y,compare_clienttag)+
	     account_get_ladder_losses(y,compare_clienttag)+
	     account_get_ladder_draws(y,compare_clienttag)+
	     account_get_ladder_disconnects(y,compare_clienttag);
    if (xgames>ygames)
        return -1;
    if (xgames<ygames)
        return +1;
    
    xrating = account_get_ladder_rating(x,compare_clienttag);
    yrating = account_get_ladder_rating(y,compare_clienttag);
    if (xrating>yrating)
        return -1;
    if (xrating<yrating)
        return +1;
    
    xwins = account_get_ladder_wins(x,compare_clienttag);
    ywins = account_get_ladder_wins(y,compare_clienttag);
    if (xwins>ywins)
        return -1;
    if (xwins<ywins)
        return +1;
    
    xoldrank = account_get_ladder_rank(x,compare_clienttag);
    yoldrank = account_get_ladder_rank(y,compare_clienttag);
    if (xoldrank<yoldrank)
        return -1;
    if (xoldrank>yoldrank)
        return +1;
    
    return 0;
}


static int compare_active_highestrated(void const * a, void const * b)
{
    t_account *  x=*(t_account * const *)a;
    t_account *  y=*(t_account * const *)b;
    unsigned int xrating;
    unsigned int yrating;
    unsigned int xwins;
    unsigned int ywins;
    unsigned int xoldrank;
    unsigned int yoldrank;
    
    if (!x || !y)
    {
	eventlog(eventlog_level_error,"compare_active_highestrated","got NULL account");
	return 0;
    }
    
    xrating = account_get_ladder_active_rating(x,compare_clienttag);
    yrating = account_get_ladder_active_rating(y,compare_clienttag);
    if (xrating>yrating)
        return -1;
    if (xrating<yrating)
        return +1;
    
    xwins = account_get_ladder_active_wins(x,compare_clienttag);
    ywins = account_get_ladder_active_wins(y,compare_clienttag);
    if (xwins>ywins)
        return -1;
    if (xwins<ywins)
        return +1;
    
    xoldrank = account_get_ladder_active_rank(x,compare_clienttag);
    yoldrank = account_get_ladder_active_rank(y,compare_clienttag);
    if (xoldrank<yoldrank)
        return -1;
    if (xoldrank>yoldrank)
        return +1;
    
    return 0;
}


static int compare_active_mostwins(void const * a, void const * b)
{
    t_account *  x=*(t_account * const *)a;
    t_account *  y=*(t_account * const *)b;
    unsigned int xwins;
    unsigned int ywins;
    unsigned int xrating;
    unsigned int yrating;
    unsigned int xoldrank;
    unsigned int yoldrank;
    
    if (!x || !y)
    {
	eventlog(eventlog_level_error,"compare_active_mostwins","got NULL account");
	return 0;
    }
    
    xwins = account_get_ladder_active_wins(x,compare_clienttag);
    ywins = account_get_ladder_active_wins(y,compare_clienttag);
    if (xwins>ywins)
        return -1;
    if (xwins<ywins)
        return +1;
    
    xrating = account_get_ladder_active_rating(x,compare_clienttag);
    yrating = account_get_ladder_active_rating(y,compare_clienttag);
    if (xrating>yrating)
        return -1;
    if (xrating<yrating)
        return +1;
    
    xoldrank = account_get_ladder_active_rank(x,compare_clienttag);
    yoldrank = account_get_ladder_active_rank(y,compare_clienttag);
    if (xoldrank<yoldrank)
        return -1;
    if (xoldrank>yoldrank)
        return +1;
    
    return 0;
}


static int compare_active_mostgames(void const * a, void const * b)
{
    t_account *  x=*(t_account * const *)a;
    t_account *  y=*(t_account * const *)b;
    unsigned int xgames;
    unsigned int ygames;
    unsigned int xrating;
    unsigned int yrating;
    unsigned int xwins;
    unsigned int ywins;
    unsigned int xoldrank;
    unsigned int yoldrank;
    
    if (!x || !y)
    {
	eventlog(eventlog_level_error,"compare_active_mostgames","got NULL account");
	return 0;
    }
    
    xgames = account_get_ladder_active_wins(x,compare_clienttag)+
	     account_get_ladder_active_losses(x,compare_clienttag)+
	     account_get_ladder_active_draws(x,compare_clienttag)+
	     account_get_ladder_active_disconnects(x,compare_clienttag);
    ygames = account_get_ladder_active_wins(y,compare_clienttag)+
	     account_get_ladder_active_losses(y,compare_clienttag)+
	     account_get_ladder_active_draws(y,compare_clienttag)+
	     account_get_ladder_active_disconnects(y,compare_clienttag);
    if (xgames>ygames)
        return -1;
    if (xgames<ygames)
        return +1;
    
    xrating = account_get_ladder_active_rating(x,compare_clienttag);
    yrating = account_get_ladder_active_rating(y,compare_clienttag);
    if (xrating>yrating)
        return -1;
    if (xrating<yrating)
        return +1;
    
    xwins = account_get_ladder_active_wins(x,compare_clienttag);
    ywins = account_get_ladder_active_wins(y,compare_clienttag);
    if (xwins>ywins)
        return -1;
    if (xwins<ywins)
        return +1;
    
    xoldrank = account_get_ladder_active_rank(x,compare_clienttag);
    yoldrank = account_get_ladder_active_rank(y,compare_clienttag);
    if (xoldrank<yoldrank)
        return -1;
    if (xoldrank>yoldrank)
        return +1;
    
    return 0;
}


/*
 * This routine will actually update the rankings in the accounts
 * as well as the ladder arrays to reflect the proper order.
 */
static int ladder_current_sort(t_account * * ladder_highestrated, t_account * * ladder_mostwins, t_account * * ladder_mostgames, unsigned int ladder_len)
{
    if (ladder_len<1)
	return 0;
    
    qsort(ladder_highestrated,ladder_len,sizeof(t_account *),compare_current_highestrated);
    qsort(ladder_mostwins,ladder_len,sizeof(t_account *),compare_current_mostwins);
    qsort(ladder_mostgames,ladder_len,sizeof(t_account *),compare_current_mostgames);
    
    return 0;
}


/*
 * This routine will actually update the rankings in the accounts
 * as well as the ladder arrays to reflect the proper order.
 */
static int ladder_active_sort(t_account * * ladder_highestrated, t_account * * ladder_mostwins, t_account * * ladder_mostgames, unsigned int ladder_len)
{
    if (ladder_len<1)
	return 0;
    
    qsort(ladder_highestrated,ladder_len,sizeof(t_account *),compare_active_highestrated);
    qsort(ladder_mostwins,ladder_len,sizeof(t_account *),compare_active_mostwins);
    qsort(ladder_mostgames,ladder_len,sizeof(t_account *),compare_active_mostgames);
    
    return 0;
}


/*
 * Make the current ladder statistics the active ones.
 */
static int ladder_make_active(t_account * const * ladder_highestrated, t_account * const * ladder_mostwins, t_account * const * ladder_mostgames, unsigned int ladder_len, t_account * * * active_ladder_highestrated, t_account * * * active_ladder_mostwins, t_account * * * active_ladder_mostgames, unsigned int * active_ladder_len, char const * clienttag)
{
    unsigned int  i;
    t_account * * temp_ladder_highestrated;
    t_account * * temp_ladder_mostwins;
    t_account * * temp_ladder_mostgames;
    t_account *   account;
    char const *  timestr;
    t_bnettime    bt;
    
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"ladder_make_active","got bad clienttag");
	return -1;
    }
    if (!ladder_highestrated || !ladder_mostwins || !ladder_mostgames ||
	!active_ladder_highestrated || !active_ladder_mostwins || !active_ladder_mostgames ||
	!active_ladder_len)
    {
	eventlog(eventlog_level_error,"ladder_make_active","got NULL ladder parameter");
	return -1;
    }
    
    if (ladder_len)
    {
	if (!(temp_ladder_highestrated = malloc(sizeof(t_account *)*ladder_len)))
	{
	    eventlog(eventlog_level_error,"ladder_make_active","unable to allocate memory for temp_ladder_highestrated");
	    return -1;
	}
	if (!(temp_ladder_mostwins = malloc(sizeof(t_account *)*ladder_len)))
	{
	    eventlog(eventlog_level_error,"ladder_make_active","unable to allocate memory for temp_ladder_mostwins");
	    pfree(temp_ladder_highestrated,sizeof(t_account *)*ladder_len);
	    return -1;
	}
	if (!(temp_ladder_mostgames = malloc(sizeof(t_account *)*ladder_len)))
	{
	    eventlog(eventlog_level_error,"ladder_make_active","unable to allocate memory for temp_ladder_mostgames");
	    pfree(temp_ladder_mostwins,sizeof(t_account *)*ladder_len);
	    pfree(temp_ladder_highestrated,sizeof(t_account *)*ladder_len);
	    return -1;
	}
	
	for (i=0; i<ladder_len; i++)
	{
	    temp_ladder_highestrated[i] = ladder_highestrated[i];
	    temp_ladder_mostwins[i]     = ladder_mostwins[i];
	    temp_ladder_mostgames[i]    = ladder_mostgames[i];
	    
	    account = ladder_highestrated[i];
	    
	    account_set_ladder_active_wins(account,clienttag,account_get_ladder_wins(account,clienttag));
	    account_set_ladder_active_losses(account,clienttag,account_get_ladder_losses(account,clienttag));
	    account_set_ladder_active_draws(account,clienttag,account_get_ladder_draws(account,clienttag));
	    account_set_ladder_active_disconnects(account,clienttag,account_get_ladder_disconnects(account,clienttag));
	    account_set_ladder_active_rating(account,clienttag,account_get_ladder_rating(account,clienttag));
	    account_set_ladder_active_rank(account,clienttag,account_get_ladder_rank(account,clienttag));
            if (!(timestr = account_get_ladder_last_time(account,clienttag)))
                timestr = BNETD_LADDER_DEFAULT_TIME;
            bnettime_set_str(&bt,timestr);
	    account_set_ladder_active_last_time(account,clienttag,bt);
	}
    }
    else
    {
	temp_ladder_highestrated = NULL;
	temp_ladder_mostwins     = NULL;
	temp_ladder_mostgames    = NULL;
    }
    
    if (*active_ladder_highestrated)
	pfree(*active_ladder_highestrated,sizeof(t_account *)*(*active_ladder_len));
    if (*active_ladder_mostwins)
	pfree(*active_ladder_mostwins,sizeof(t_account *)*(*active_ladder_len));
    if (*active_ladder_mostgames)
	pfree(*active_ladder_mostgames,sizeof(t_account *)*(*active_ladder_len));
    
    *active_ladder_highestrated = temp_ladder_highestrated;
    *active_ladder_mostwins     = temp_ladder_mostwins;
    *active_ladder_mostgames    = temp_ladder_mostgames;
    *active_ladder_len = ladder_len;
    
    return 0;
}


/*
 * Load the current user accounts with non-zero ladder rankings into
 * the current ladder arrays.
 */
extern int ladder_init(void)
{
    t_list const * const * save;
    t_account *            account;
    unsigned int           i;
    
    for (account=accountlist_get_first(&save); account; account=accountlist_get_next(&save))
    {
	if (account_get_ladder_rating(account,CLIENTTAG_STARCRAFT)>0)
	    ladder_insert(account,
			  &star_ladder_current_highestrated,
			  &star_ladder_current_mostwins,
			  &star_ladder_current_mostgames,
			  &star_ladder_current_len);
	
	if (account_get_ladder_rating(account,CLIENTTAG_BROODWARS)>0)
	    ladder_insert(account,
			  &sexp_ladder_current_highestrated,
			  &sexp_ladder_current_mostwins,
			  &sexp_ladder_current_mostgames,
			  &sexp_ladder_current_len);
	
	if (account_get_ladder_active_rating(account,CLIENTTAG_STARCRAFT)>0)
	    ladder_insert(account,
			  &star_ladder_active_highestrated,
			  &star_ladder_active_mostwins,
			  &star_ladder_active_mostgames,
			  &star_ladder_active_len);
	
	if (account_get_ladder_active_rating(account,CLIENTTAG_BROODWARS)>0)
	    ladder_insert(account,
			  &sexp_ladder_active_highestrated,
			  &sexp_ladder_active_mostwins,
			  &sexp_ladder_active_mostgames,
			  &sexp_ladder_active_len);
    }
    
    compare_clienttag = CLIENTTAG_STARCRAFT;
    ladder_current_sort(star_ladder_current_highestrated,
			star_ladder_current_mostwins,
			star_ladder_current_mostgames,
			star_ladder_current_len);
    compare_clienttag = CLIENTTAG_BROODWARS;
    ladder_current_sort(sexp_ladder_current_highestrated,
			sexp_ladder_current_mostwins,
			sexp_ladder_current_mostgames,
			sexp_ladder_current_len);
    
    compare_clienttag = CLIENTTAG_STARCRAFT;
    ladder_active_sort(star_ladder_active_highestrated,
		       star_ladder_active_mostwins,
		       star_ladder_active_mostgames,
		       star_ladder_active_len);
    compare_clienttag = CLIENTTAG_BROODWARS;
    ladder_active_sort(sexp_ladder_active_highestrated,
		       sexp_ladder_active_mostwins,
		       sexp_ladder_active_mostgames,
		       sexp_ladder_active_len);
    
    /* in case the user files were changed outside of bnetd... */
    for (i=0; i<star_ladder_current_len; i++) /* re-calculate ranks */
	if (account_get_ladder_rank(star_ladder_current_highestrated[i],CLIENTTAG_STARCRAFT)!=i+1)
	    account_set_ladder_rank(star_ladder_current_highestrated[i],i+1,CLIENTTAG_STARCRAFT);
    for (i=0; i<sexp_ladder_current_len; i++) /* re-calculate ranks */
	if (account_get_ladder_rank(sexp_ladder_current_highestrated[i],CLIENTTAG_BROODWARS)!=i+1)
	    account_set_ladder_rank(sexp_ladder_current_highestrated[i],i+1,CLIENTTAG_BROODWARS);
    /* don't change anything for active ladders */
    
    eventlog(eventlog_level_info,"ladder_init","added %u accounts to current \"%s\" ladder",star_ladder_current_len,CLIENTTAG_STARCRAFT);
    eventlog(eventlog_level_info,"ladder_init","added %u accounts to current \"%s\" ladder",sexp_ladder_current_len,CLIENTTAG_BROODWARS);
    
    eventlog(eventlog_level_info,"ladder_init","added %u accounts to active \"%s\" ladder",star_ladder_active_len,CLIENTTAG_STARCRAFT);
    eventlog(eventlog_level_info,"ladder_init","added %u accounts to active \"%s\" ladder",sexp_ladder_active_len,CLIENTTAG_BROODWARS);
    
    return 0;
}


/*
 * Prepare an account for first ladder play if necessary.
 */
extern int ladder_init_account(t_account * account, char const * clienttag)
{
    char const * tname;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"ladder_init_account","got NULL account");
	return -1;
    }
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"ladder_init_account","got bad clienttag");
	return -1;
    }
    
    if (account_get_ladder_rating(account,clienttag)==0)
    {
	if (account_get_ladder_wins(account,clienttag)+
	    account_get_ladder_losses(account,clienttag)>0) /* no ladder games so far... */
	{
	    eventlog(eventlog_level_warn,"ladder_init_account","account for \"%s\" (%s) has %u wins and %u losses but has zero rating",(tname = account_get_name(account)),clienttag,account_get_ladder_wins(account,clienttag),account_get_ladder_losses(account,clienttag));
	    account_unget_name(tname);
	    return -1;
	}
	account_adjust_ladder_rating(account,1000,clienttag); /* start them at 1000 */
	
	if (strcmp(clienttag,CLIENTTAG_STARCRAFT)==0)
	    ladder_insert(account,
			  &star_ladder_current_highestrated,
			  &star_ladder_current_mostwins,
			  &star_ladder_current_mostgames,
			  &star_ladder_current_len);
	else if (strcmp(clienttag,CLIENTTAG_BROODWARS)==0)
	    ladder_insert(account,
			  &sexp_ladder_current_highestrated,
			  &sexp_ladder_current_mostwins,
			  &sexp_ladder_current_mostgames,
			  &sexp_ladder_current_len);
	else
	{
	    eventlog(eventlog_level_error,"ladder_init_account","got unknown clienttag \"%s\"",clienttag);
	    return -1;
	}
	
	eventlog(eventlog_level_info,"ladder_init_account","initialized account for \"%s\" for \"%s\" ladder",(tname = account_get_name(account)),clienttag);
	account_unget_name(tname);
    }
    
    return 0;
}


/*
 * Update player ratings, rankings, etc due to game results.
 */
extern int ladder_update(char const * clienttag, unsigned int count, t_account * * players, t_game_result * results, t_ladder_info * info)
{
    unsigned int curr;
    t_account *  sorted[8];
    unsigned int winners=0;
    unsigned int losers=0;
    unsigned int draws=0;
    
    if (count<2 || count>8)
    {
	eventlog(eventlog_level_error,"ladder_update_ratings","got invalid player count %u",count);
	return -1;
    }
    if (!players)
    {
	eventlog(eventlog_level_error,"ladder_update_ratings","got NULL players");
	return -1;
    }
    if (!results)
    {
	eventlog(eventlog_level_error,"ladder_update_ratings","got NULL results");
	return -1;
    }
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"ladder_update_ratings","got bad clienttag");
	return -1;
    }
    if (!info)
    {
	eventlog(eventlog_level_error,"ladder_update_rating","got NULL info");
	return -1;
    }
    
    for (curr=0; curr<count; curr++)
    {
	if (!players[curr])
	{
	    eventlog(eventlog_level_error,"ladder_update_rating","got NULL player[%u] (of %u)",curr,count);
	    return -1;
	}
	
	switch (results[curr])
	{
	case game_result_win:
	    winners++;
	    break;
	case game_result_loss:
	    losers++;
	    break;
	case game_result_draw:
	    draws++;
	    break;
	case game_result_disconnect: /* FIXME: Hmm. How best to handle this. */
	    break;
	default:
	    eventlog(eventlog_level_error,"ladder_update_ratings","bad results[%u]=%u",curr,(unsigned int)results[curr]);
	    return -1;
	}
    }
    
    if (draws>0)
    {
	if (draws!=count)
	{
	    eventlog(eventlog_level_error,"ladder_update_ratings","some, but not all players had a draw count=%u (winners=%u losers=%u draws=%u)",count,winners,losers,draws);
	    return -1;
	}
	
	return 0; /* no change in case of draw */
    }
    if (winners!=1 || losers<1)
    {
	eventlog(eventlog_level_error,"ladder_update_ratings","missing winner or loser for count=%u (winners=%u losers=%u draws=%u)",count,winners,losers,draws);
	return -1;
    }
    
    for (curr=0; curr<count; curr++)
    {
	double k;
	double prob;
	double delta;
	
	k = coefficient(players[curr],clienttag);
	
	{
	    unsigned int i,j;
	    
	    /* Put the current user into slot 0, others into other slots
	       order is not important for the other players */
	    for (i=0,j=1; i<count; i++)
		if (i==curr) 
		    sorted[0] = players[i];
		else
		    sorted[j++] = players[i];
	}
	
	switch (count)
	{
	case 2:
	    prob = two_player(sorted,clienttag);
            break;
	case 3:
	    prob = three_player(sorted,clienttag);
            break;
	case 4:
	    prob = four_player(sorted,clienttag);
            break;
#if 0
	case 5:
	    prob = five_player(sorted,clienttag);
	    break;
#endif
	case 6:
	    prob = six_player(sorted,clienttag);
	    break;
#if 0
	case 7:
	    prob = seven_player(sorted,clienttag);
	    break;
#endif
	case 8:
	    prob = eight_player(sorted,clienttag);
	    break;
	default:
	    eventlog(eventlog_level_error,"ladder_update_ratings","sorry, unsupported number of ladder %d players",count);
	    return -1;
	}
	
	if (results[curr]==game_result_win)
	    delta = fabs(k * (1.0 - prob)); /* better the chance of winning -> fewer points added */
	else
	    delta = -fabs(k * prob); /* better the chance of winning -> more points subtracted */
	
	eventlog(eventlog_level_debug,"ladder_update_ratings","computed probability=%g, k=%g, deltar=%+g",prob,k,delta);
	
	info[curr].prob      = prob;
	info[curr].k         = (unsigned int)k;
	info[curr].adj       = (int)delta;
	info[curr].oldrating = account_get_ladder_rating(players[curr],clienttag);
	info[curr].oldrank   = account_get_ladder_rank(players[curr],clienttag);
    }
    
    for (curr=0; curr<count; curr++)
	account_adjust_ladder_rating(players[curr],info[curr].adj,clienttag);
    
    compare_clienttag = clienttag;
    if (strcmp(clienttag,CLIENTTAG_STARCRAFT)==0)
    {
	unsigned int i;
	
	ladder_current_sort(star_ladder_current_highestrated,
			    star_ladder_current_mostwins,
			    star_ladder_current_mostgames,
			    star_ladder_current_len);
	for (i=0; i<star_ladder_current_len; i++) /* re-calculate ranks */
	    if (account_get_ladder_rank(star_ladder_current_highestrated[i],clienttag)!=i+1)
		account_set_ladder_rank(star_ladder_current_highestrated[i],i+1,clienttag);
    }
    else if (strcmp(clienttag,CLIENTTAG_BROODWARS)==0)
    {
	unsigned int i;
	
	ladder_current_sort(sexp_ladder_current_highestrated,
			    sexp_ladder_current_mostwins,
			    sexp_ladder_current_mostgames,
			    sexp_ladder_current_len);
	for (i=0; i<sexp_ladder_current_len; i++) /* re-calculate ranks */
	    if (account_get_ladder_rank(sexp_ladder_current_highestrated[i],clienttag)!=i+1)
		account_set_ladder_rank(sexp_ladder_current_highestrated[i],i+1,clienttag);
    }
    else
	eventlog(eventlog_level_error,"ladder_update_ratings","got unknown clienttag \"%s\", did not sort",clienttag);
    
    return 0;
}


extern int ladder_make_all_active(void)
{
    if (ladder_make_active(star_ladder_current_highestrated,
			   star_ladder_current_mostwins,
			   star_ladder_current_mostgames,
			   star_ladder_current_len,
			   &star_ladder_active_highestrated,
			   &star_ladder_active_mostwins,
			   &star_ladder_active_mostgames,
			   &star_ladder_active_len,
			   CLIENTTAG_STARCRAFT)<0)
	return -1;
    if (ladder_make_active(star_ladder_current_highestrated,
			   star_ladder_current_mostwins,
			   star_ladder_current_mostgames,
			   star_ladder_current_len,
			   &star_ladder_active_highestrated,
			   &star_ladder_active_mostwins,
			   &star_ladder_active_mostgames,
			   &star_ladder_active_len,
			   CLIENTTAG_BROODWARS)<0)
	return -1;
    
    return 0;
}


extern int ladder_check_map(char const * mapname, char const * clienttag)
{
    if (!mapname)
    {
	eventlog(eventlog_level_error,"ladder_check_map","got NULL mapname");
	return -1;
    }
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"ladder_check_map","got bad clienttag");
	return -1;
    }
    
    return 0;
}


extern t_account * ladder_get_account_by_rank(unsigned int rank, t_ladder_sort lsort, t_ladder_time ltime, char const * clienttag)
{
    t_account * const * ladder;
    unsigned int        ladder_len;
    
    if (rank<1)
    {
	eventlog(eventlog_level_error,"ladder_get_account_by_rank","got zero rank");
	return NULL;
    }
    
    if (ladder_locate(&ladder,&ladder_len,lsort,ltime,clienttag)<0)
    {
	eventlog(eventlog_level_error,"ladder_get_account_by_rank","could not locate ladder");
	return NULL;
    }
    
    if (rank>ladder_len)
	return NULL;
    return ladder[rank-1];
}


extern unsigned int ladder_get_rank_by_account(t_account * account, t_ladder_sort lsort, t_ladder_time ltime, char const * clienttag)
{
    t_account * const * ladder;
    unsigned int        ladder_len;
    unsigned int        i;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"ladder_get_rank_by_account","got NULL account");
	return 0;
    }
    
    if (ladder_locate(&ladder,&ladder_len,lsort,ltime,clienttag)<0)
    {
	eventlog(eventlog_level_error,"ladder_get_rank_by_account","could not locate ladder");
	return 0;
    }
    
    for (i=0; i<ladder_len; i++)
	if (ladder[i]==account)
	    return i+1;
    
    return 0;
}

