/* example-start selection gettargets.c */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <unistd.h>
#include <getopt.h>

#include <darxite.h>

#define VERSION 	"0.1"
#define OPTION_STRING	"vhc:"
#define AUTHORS		"Ashley Montanaro"

typedef struct _urlList {
    char *Url;
    struct _urlList *Next;
} UrlList;

GtkWidget *Window, *ListBox;
GdkAtom StringAtom;
UrlList *FirstUrl, *LastUrl;
int UrlCount, TimeoutId, RemotePort, DaemonFd;
char RemoteHost[256];

void PrintVersion(void)
{
    printf("dxclip v%s, release %s (%s) by %s\n", VERSION,
           RELEASE_VER, RELEASE_NAME, AUTHORS);
}

void Usage(char *prog_name)
{
    printf("Usage: %s [OPTIONS] command\n"
           "where options are:\n"
           "-v, --version\t\t\tShow version and exit\n"
           "-h, --help\t\t\tShow some usage information\n"
           "-c<host>:<port> --connect<host>\tConnect to <host> on <port>\n",
           prog_name);
    exit(0);
}

char *ReadCmdLine(int argc, char **argv)
{
    char opt;
    int option_index;

    static struct option long_options[] = {
        { "version", 	0, 0, 'v' },
        { "help",		0, 0, 'h' },
        { "connect",	1, 0, 'c' },
        { 0, 			0, 0,  0  }
    };

    opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                      &option_index);
    while (opt != -1)
    {
        switch (opt)
        {
        case 'v':
            PrintVersion();
            exit(0);
            break;
            
        case 'h':
            Usage(argv[0]);
            break;
            
        case 'c':
            if (strchr(optarg, ':'))
            {
                memset(RemoteHost, 0, sizeof(RemoteHost));
                strncpy(RemoteHost, optarg, strchr(optarg, ':') - optarg);
                RemotePort = atoi(strchr(optarg, ':') + 1);
                if ((strlen(RemoteHost) == 0) || (RemotePort <= 0))
                {
                    printf("Invalid host/port to connect to daemon\n");
                    exit(0);
                }
            }
            else
            {
                printf("Syntax: -c <host>:<port>\n");
                exit(0);
            }
            break;
        }
        opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                          &option_index);        
    }
    
    if (optind < argc)
        return argv[optind];
    else
       return NULL;
}

/* Signal handler called when the selections owner returns the data */
void selection_received(GtkWidget *widget, GtkSelectionData *selection_data, 
                        gpointer data)
{
    UrlList *url_item;
    char *url;
    
    // Check to see whether the retrieval succeeded. If it didn't, it's not
    // an error - it just means there wasn't any data in the clipboard.
    if (selection_data->length < 0)
    {
        //printf("Selection retrieval failed\n");
        return;
    }
    
    // This should never happen!
    if (selection_data->type != GDK_SELECTION_TYPE_STRING)
    {
        printf("Primary selection was not returned as a string!\n");
        return;
    }
    
    // Check the last URL first for greater speed when we've got the same
    // URL in the clipboard over and over again.
    if ((LastUrl != NULL) && !strcasecmp(LastUrl->Url, selection_data->data))
    {
        //printf("URL duplicates last one\n");
        return;
    }
    
    // Check whether the URL is already in the list
    url_item = FirstUrl;
    while (url_item)
    {
        if (!strcasecmp(url_item->Url, selection_data->data))
            return;
        url_item = url_item->Next;
    }
    
    //printf("data: \"%s\"\n", selection_data->data);
    url = selection_data->data;
    
    // Is it a valid URL?
    if (strncasecmp(url, "ftp://", 6) && strncasecmp(url, "http://", 7) &&
        strncasecmp(url, "file://", 7))
    {
        return;
    }
    
    // Add it to the linked list
    url_item = malloc(sizeof(UrlList));
    url_item->Url = strdup(url);
    url_item->Next = NULL;
    if (FirstUrl == NULL)
        FirstUrl = url_item;
    if (LastUrl != NULL)
        LastUrl->Next = url_item;
    LastUrl = url_item;
    
    //printf("Appending \"%s\"\n", url);
    gtk_clist_append(GTK_CLIST(ListBox), &url);
    gtk_clist_set_row_data(GTK_CLIST(ListBox), UrlCount, (gpointer)UrlCount);
    UrlCount++;
    
    return;
}

// function called every 1 second
gint get_targets(gpointer data)
{
    //printf("Getting...\n");
    // Try to get the primary selection as a string
    gtk_selection_convert(Window, GDK_SELECTION_PRIMARY, StringAtom,
                          GDK_CURRENT_TIME);
    return 1;
}

// get a url
void get_url(GtkWidget *widget, gpointer data)
{
    char buffer[256];
    UrlList *url_item = FirstUrl;
    int url_count = 0;

    if (GTK_CLIST(ListBox)->selection != NULL)
    {
        //printf("Get URL %d\n", (int)GTK_CLIST(ListBox)->selection->data);
        while (url_item && (url_count <
                            (int)GTK_CLIST(ListBox)->selection->data))
        {
            url_item = url_item->Next;
            url_count++;
        }
        if (url_item)
        {
            //printf("%s\n", url_item->Url);
            sprintf(buffer, "Get %s\n", url_item->Url);
            write(DaemonFd, buffer, strlen(buffer));
        }
    }
}

// the user is dragging something from the clist
void drag_begin(GtkWidget *widget, GdkDragContext *drag_context,
                gpointer data)
{
    printf("Drag begun\n");
}

void clear_list(GtkWidget *widget, gpointer data)
{
    UrlList *url_item = FirstUrl, *last_url;
    
    gtk_clist_clear(GTK_CLIST(ListBox));
    while (url_item)
    {
        last_url = url_item;
        url_item = last_url->Next;
        free(last_url->Url);
        free(last_url);
    }
    FirstUrl = NULL;
    LastUrl = NULL;
    UrlCount = 0;
}

void close_window(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    DX_DisconnectClient(DaemonFd);
    gtk_main_quit();
}

int main(int argc, char *argv[])
{
    GtkWidget *vbox, *hbox, *button;
    char buffer[256];
      
    gtk_init(&argc, &argv);
    ReadCmdLine(argc, argv);
    
    if (strcmp(RemoteHost, "") && (RemotePort > 0))
    {
        sprintf(buffer, "Enter password for daemon on %s: ", RemoteHost);
        DaemonFd = DX_ConnectRemoteClient(RemoteHost, RemotePort,
                                          getpass(buffer), "dxclip");
    }
    else
    {
        DaemonFd = DX_ConnectClient("darxstat-GTK");
    }
    
    if (DaemonFd < 0)
    {
        fprintf(stderr, "Couldn't connect to daemon: %s\n",
                strerror(DX_errno));
        return 1;
    }
    
    StringAtom = gdk_atom_intern("STRING", FALSE);
    
    Window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(Window), "Darxite Clipboard");
    gtk_container_border_width(GTK_CONTAINER(Window), 10);
    gtk_widget_set_usize(Window, 200, 100);
    
    gtk_signal_connect(GTK_OBJECT(Window), "destroy",
                       GTK_SIGNAL_FUNC(close_window), NULL);
    
    vbox = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(Window), vbox);
    gtk_widget_show(vbox);
    
    ListBox = gtk_clist_new(1);
    gtk_signal_connect(GTK_OBJECT(ListBox), "drag_begin",
                       GTK_SIGNAL_FUNC(drag_begin), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), ListBox, TRUE, TRUE, 0);
    gtk_widget_show(ListBox);
    
    hbox = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show(hbox);
    
    button = gtk_button_new_with_label("Get");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(get_url), NULL);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    button = gtk_button_new_with_label("Clear");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(clear_list), NULL);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);
    
    button = gtk_button_new_with_label("Quit");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);
    
    gtk_signal_connect(GTK_OBJECT(Window), "selection_received",
                       GTK_SIGNAL_FUNC(selection_received), NULL);
    
    TimeoutId = gtk_timeout_add(500, get_targets, NULL);
    
    gtk_widget_show(Window);
      
    gtk_main();
      
    return 0;
}
