// File that handles client connections, etc.
// by means of sockets.

#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <netdb.h>
#include <pwd.h>

#include "sockets.h"

// attempt to help libc5 boxes
#ifndef O_ASYNC
#  define O_ASYNC 020000
#endif
#ifndef O_NONBLOCK
#  define O_NONBLOCK 04000
#endif
#ifndef AF_FILE
#  define AF_FILE 1
#endif

int dl_ConnectClient(const char *name)
{
    int rc, fd, size, oldflags;
    struct sockaddr_un addr;
    char buffer[256];

    // Create the socket
    fd = socket(PF_UNIX, SOCK_STREAM, 0);
    if (fd < 0)
    {
        DX_errno = errno;
        return -1;
    }
    
    // Connect the socket to the daemon's socket
    addr.sun_family = AF_FILE;
    sprintf(addr.sun_path, "/tmp/darxsock.%s", getpwuid(getuid())->pw_name);
    size = sizeof(addr.sun_family) + strlen(addr.sun_path) + 1;
    if (connect(fd, (struct sockaddr *)&addr, size) < 0)
    {
        DX_errno = errno;
        close(fd);
        return -1;
    }
    
    // set the socket to be non-blocking (clients using
    // DX_GetResponse() shouldn't need this, but anyway...)
    oldflags = fcntl(fd, F_GETFL, 0);
    if (oldflags == -1)
    {
        DX_errno = errno;
        close(fd);
        return -1;
    }
    oldflags |= O_NONBLOCK;
    fcntl(fd, F_SETFL, oldflags);
    
    sprintf(buffer, "%s\n", name);
    // Tell the daemon our name
    if (write(fd, buffer, strlen(buffer)) < 0)
    {
        DX_errno = errno;
        close(fd);
        return -1;
    }
    rc = DX_GetResponse(fd, buffer, sizeof(buffer), 30);
    if (rc != DX_LIB_OK)
    {
        dl_error(E_TRACE, "Couldn't get name response");
        DX_errno = rc;
        close(fd);
        return -1;
    }
    
    return fd;
}

int dl_ConnectRemoteClient(const char *host, int port, const char *password,
                           const char *name)
{
    int fd, rc, oldflags;
    struct hostent *hostinfo = NULL;
    struct sockaddr_in addr;
    char buffer[256];
    fd_set write_fds;
    struct timeval tv;

    dl_error(E_TRACE, "Connecting to %s, port %d...", host, htons(port));
    // Create the socket
    fd = socket(PF_INET, SOCK_STREAM, 0);
    if (fd < 0)
    {
        DX_errno = errno;
        return -1;
    }

    dl_error(E_TRACE, "Created socket");
    // set the socket to be non-blocking (clients using
    // DX_GetResponse() shouldn't need this, but anyway...)
    oldflags = fcntl(fd, F_GETFL, 0);
    if (oldflags < 0)
    {
        DX_errno = errno;
        close(fd);
        return -1;
    }
    oldflags |= O_NONBLOCK;
    if (fcntl(fd, F_SETFL, oldflags) < 0)
    {
        DX_errno = errno;
        close(fd);
        return -1;
    }

    dl_error(E_TRACE, "About to look up host...");
    hostinfo = gethostbyname(host);
    if (hostinfo == NULL)
    {
        dl_error(E_TRACE, "Unknown host %s", host);
        DX_errno = ENONET;
        close(fd);
        return -1;
    }
    
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    addr.sin_addr = *((struct in_addr *)(hostinfo->h_addr));
    
    dl_error(E_TRACE, "About to connect...");
    rc = connect(fd, (struct sockaddr *)&addr, sizeof(addr));
    // since it's non-blocking, there is a good chance we'll get this error
    if ((rc == -1) && (errno == EINPROGRESS))
    {
        tv.tv_sec = 30;
        tv.tv_usec = 0;
        FD_ZERO(&write_fds);
        FD_SET(fd, &write_fds);
        rc = select(FD_SETSIZE, NULL, &write_fds, NULL, &tv);
        // pretend it's an error for the bit below
        if (rc == 0)
            rc = -1;
    }
    if (rc == -1)
    {
        dl_error(E_WARN, "Couldn't connect to socket: %s", strerror(errno));
        close(fd);
        return -1;
    }

    // Receive "Welcome to Darxite" message
    rc = DX_GetResponse(fd, buffer, sizeof(buffer), 30);
    if (rc != DX_LIB_OK)
    {
        dl_error(E_TRACE, "Couldn't get the daemon's response");
        DX_errno = rc;
        close(fd);
        return -1;
    }
    
    dl_error(E_TRACE, "Connected OK, now to send password...");
    sprintf(buffer, "%s\n", password);
    // Send the daemon the password
    if (write(fd, buffer, strlen(buffer)) < 0)
    {
        DX_errno = errno;
        close(fd);
        return -1;
    }
    rc = DX_GetResponse(fd, buffer, sizeof(buffer), 30);
    if (rc != DX_LIB_OK)
    {
        dl_error(E_TRACE, "Couldn't get password response");
        DX_errno = rc;
        close(fd);
        return -1;
    }
    if (atoi(buffer) == DX_E_PASSWORD)
    {
        dl_error(E_TRACE, "Invalid password");
        DX_errno = EACCES;
        close(fd);
        return -1;
    }
    memset(buffer, 0, sizeof(buffer));
    sprintf(buffer, "%s\n", name);
    // Tell the daemon our name
    if (write(fd, buffer, strlen(buffer)) < 0)
    {
        DX_errno = errno;
        close(fd);
        return -1;
    }
    rc = DX_GetResponse(fd, buffer, sizeof(buffer), 30);
    if (rc != DX_LIB_OK)
    {
        dl_error(E_TRACE, "Couldn't get name response");
        DX_errno = rc;
        close(fd);
        return -1;
    }
    
    dl_error(E_TRACE, "Finished connecting OK");
    return fd;
}
