/*
 * server.c: Things dealing with server connections, etc. 
 *
 * Written By Michael Sandrof
 *
 * Copyright(c) 1990 
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */

#if 0
static	char	rcsid[] = "$Id: server.c,v 1.20 1994/01/17 07:48:35 mrgreen Exp mrgreen $";
#endif

#include "irc.h"
#include "parse.h"

#ifdef ESIX
# include <lan/net_types.h>
#endif /* ESIX */

#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif

#include "server.h"
#include "ircaux.h"
#include "whois.h"
#include "lastlog.h"
#include "exec.h"
#include "window.h"
#include "output.h"
#include "names.h"
#include "hook.h"
#include "notify.h"

static	void	add_to_server_buffer _((int, char *));
static	char *	do_umode (int du_index);
const 	char *	umodes = "abcdefghijklmnopqrstuvwxyz";

/*
 * Don't want to start ircserv by default...
 */
	int	using_server_process = 0;

/* server_list: the list of servers that the user can connect to,etc */
	Server	*server_list = (Server *) 0;

/* number_of_servers: in the server list */
	int	number_of_servers = 0;

extern	WhoisQueue	*WQ_head;
extern	WhoisQueue	*WQ_tail;

	int	primary_server = -1;
	int	from_server = -1;
	int	attempting_to_connect= 0;
	int	never_connected = 1;		/* true until first connection
						 * is made */
	int	connected_to_server = 0;	/* true when connection is
						 * confirmed */
	char	*connect_next_nick;
	int	parsing_server_index = -1;
	int	last_server = -1;


extern	int	dgets_errno;

/*
 * Why this as well as MyHostAddr? Well, if your host is a gateway, this
 * will be set to the address for your host which is nearest to everybody
 * else on IRC in most circumstances. Thus it is most useful for DCC
 * connections telling other clients where you are. MyHostAddr holds the
 * first address returned by the name server, which may or may not be best,
 * but is used in things like talk where the behaviour should mimic the
 * standard programs.
 *
 * None of this applies if you are using ircserv. In this case, local_ip_address
 * is the same as MyHostAddr.
 */
/* extern	struct	in_addr	local_ip_address; */ /* moved to irc.h */

/*
 * close_server: Given an index into the server list, this closes the
 * connection to the corresponding server.  It does no checking on the
 * validity of the index.  It also first sends a "QUIT" to the server being
 * closed 
 */
#ifdef __STDC__
void	close_server (int cs_index, char *message)
#else
void 	close_server(cs_index, message)
	int	cs_index;
	char	*message;
#endif
{
	int	i,
		min,
		max;
	char	buffer[BIG_BUFFER_SIZE + 1];

	if (cs_index == -1)
	{
		min = 0;
		max = number_of_servers;
	}
	else
	{
		min = cs_index;
		max = cs_index + 1;
	}
	for (i = min; i < max; i++)
	{
		if (i == primary_server)
		{
			/* make sure no leftover whois's are out there */
			clean_whois_queue();
			if (waiting)
				waiting = 0;
                        set_waiting_channel(i);
		}
		else
		{
			int old_server = from_server;

			from_server = -1;
			clear_channel_list(i);
			from_server = old_server;
		}
		server_list[i].operator = 0;
		server_list[i].connected = 0;
		server_list[i].buffer = (char *) 0;
                if (server_list[i].write != -1)
                {
                        if (message && *message)
                        {
                                sprintf(buffer, "QUIT :%s\n", message);
                                send(server_list[i].write, buffer, strlen(buffer), 0);
                        }
                        new_close(server_list[i].write);
                        if (server_list[i].write == server_list[i].read)
			{
				FD_CLR(server_list[i].read, &readables);
                                server_list[i].read = -1;
			}
                        server_list[i].write = -1;
                }
                if (server_list[i].read != -1)
                {
                        new_close(server_list[i].read);
			FD_CLR(server_list[i].read, &readables);
                        server_list[i].read = -1;
                }
		if (server_list[i].pid != -1)
		{
#ifdef SCO
			kill((pid_t)server_list[i].pid, SIGKILL);
#else
			kill(server_list[i].pid, SIGKILL);
#endif /* SCO */
			server_list[i].pid = -1;
		}
	}
}

/*
 * set_server_bits: Sets the proper bits in the fd_set structure according to
 * which servers in the server list have currently active read descriptors.  
 */
#ifdef __STDC__
void	set_server_bits (fd_set *rd)
#else
void 	set_server_bits(rd)
	fd_set *rd;
#endif
{
	int	i;

	for (i = 0; i < number_of_servers; i++)
		if (server_list[i].read != -1)
			FD_SET(server_list[i].read, rd);
}

/*
 * do_server: check the given fd_set against the currently open servers in
 * the server list.  If one have information available to be read, it is read
 * and and parsed appropriately.  If an EOF is detected from an open server,
 * one of two things occurs. 1) If the server was the primary server,
 * get_connected() is called to maintain the connection status of the user.
 * 2) If the server wasn't a primary server, connect_to_server() is called to
 * try to keep that connection alive. 
 */
#ifdef __STDC__
void	do_server (fd_set *rd)
#else
void 	do_server(rd)
	fd_set	*rd;
#endif
{
	char	buffer[BIG_BUFFER_SIZE + 1];
	int	des,
		i;
	static	int	times = 0;
	int	old_timeout;

	for (i = 0; i < number_of_servers; i++)
	{
		if ((des = server_list[i].read) != -1 && FD_ISSET(des, rd))
		{
			int	junk;
			char 	*bufptr;
			char	*s;

			last_server = from_server = i;
			old_timeout = dgets_timeout(2);
			s = server_list[from_server].buffer;
			bufptr = buffer;
			if (s && *s)
			{
				int	len = strlen(s);

				strncpy(buffer, s, len);
				bufptr += len;
			}
			junk = dgets(bufptr, BIG_BUFFER_SIZE, des, (char *) 0);
			(void) dgets_timeout(old_timeout);
			switch (junk)
			{
			case -1:
			{
#ifdef PHONE
				say("server %s: dgets timed out.  continuing...(%s)", server_list[i].name, bufptr);
#endif
				add_to_server_buffer(from_server, buffer);
				continue;
				break;
			}
			case 0:
			{
				close_server(i, empty_string);
				say("Connection closed from %s: %s", server_list[i].name,
					(dgets_errno == -1) ? "Remote end closed connection" : sys_errlist[dgets_errno]);

				if (i == primary_server)
				{
					if (server_list[i].eof)
					{
						say("Unable to connect to server %s",
							server_list[i].name);
						if (++i == number_of_servers)
						{
							clean_whois_queue();
                                                        if (do_hook(DISCONNECT_LIST,"Unable to connect to a server"))
        							say("Use /SERVER to connect to a server");
							times = 0;
						}
						else
							get_connected(i);
					}
					else
					{
						if (times++ > 1)
						{
							clean_whois_queue();
                                                        if(do_hook(DISCONNECT_LIST,"No Connection"))
        							say("Use /SERVER to connect to a server");
							times = 0;
						}
						else
							get_connected(i);
					}
				}
				else if (server_list[i].eof)
				{
					say("Connection to server %s lost.",
						server_list[i].name);
					close_server(i, empty_string);
					clean_whois_queue();
					window_check_servers();
				}
				else if (connect_to_server_by_refnum(i, -1))
				{
					say("Connection to server %s lost.",
						server_list[i].name);
					close_server(i, empty_string);
					clean_whois_queue();
					window_check_servers();
				}
				server_list[i].eof = 1;
				break;
			}
			default:
			{
				parsing_server_index = i;
				parse_server(buffer);
				new_free(&server_list[i].buffer);
				parsing_server_index = -1;
				message_from((char *) 0, LOG_CRAP);
				break;
			}
			}
			from_server = primary_server;
		}
	}
}

/*
 * find_in_server_list: given a server name, this tries to match it against
 * names in the server list, returning the index into the list if found, or
 * -1 if not found 
 */
#ifdef __STDC__
extern	int	find_in_server_list (char *server, int port)
#else
extern	int 	find_in_server_list (server, port)
	char	*server;
	int	port;
#endif
{
	int	i,
		len,
		len2;

	len = strlen(server);
	for (i = 0; i < number_of_servers; i++)
	{
		if (port && server_list[i].port && port != server_list[i].port && port != -1)
			continue;

		len2 = strlen(server_list[i].name);
		if (len2 > len)
		{
			if (!my_strnicmp(server, server_list[i].name, len))
				return (i);
		}
		else
		{
			if (!my_strnicmp(server, server_list[i].name, len2))
				return (i);
		}
	}
	return (-1);
}

/*
 * parse_server_index:  given a string, this checks if it's a number, and if
 * so checks it validity as a server index.  Otherwise -1 is returned 
 */
#ifdef __STDC__
int	parse_server_index (char *str)
#else
int 	parse_server_index(str)
	char	*str;
#endif
{
	int	i;

	if (is_number(str))
	{
		i = atoi(str);
		if ((i >= 0) && (i < number_of_servers))
			return (i);
	}
	return (-1);
}

/*
 * This replaces ``get_server_index''.
 */
#ifdef __STDC__
int 	find_server_refnum (char *server, char **rest)
#else
int 	find_server_refnum (server, rest)
	char 	*server,
		**rest;
#endif
{
	int 	refnum;
	int	port = irc_port;
	char 	*cport = NULL, 
		*password = NULL,
		*nick = NULL;

	/*
	 * First of all, check for an existing server refnum
	 */
	if ((refnum = parse_server_index(server)) != -1)
		return refnum;

	/*
	 * Next check to see if its a "server:port:password:nick"
	 */
	else if (index(server, ':'))
		parse_server_info(server, &cport, &password, &nick);

	/*
	 * Next check to see if its "server port password nick"
	 */
	else if (rest && *rest)
	{
		cport = next_arg(*rest, rest);
		password = next_arg(*rest, rest);
		nick = next_arg(*rest, rest);
	}

	if (cport && *cport)
		port = atoi(cport);

	/*
	 * Add to the server list (this will update the port
	 * and password fields).
	 */
	add_to_server_list(server, port, password, nick, 1);
	return from_server;
}


/*
 * add_to_server_list: adds the given server to the server_list.  If the
 * server is already in the server list it is not re-added... however, if the
 * overwrite flag is true, the port and passwords are updated to the values
 * passes.  If the server is not on the list, it is added to the end. In
 * either case, the server is made the current server. 
 */
#ifdef __STDC__
void 	add_to_server_list (char *server, int port, char *password, char *nick, int overwrite)
#else
void 	add_to_server_list (server, port, password, nick, overwrite)
	char	*server;
	int	port;
	char	*password;
	char	*nick;
	int	overwrite;
#endif
{
	if ((from_server = find_in_server_list(server, port)) == -1)
	{
		from_server = number_of_servers++;
		if (server_list)
			server_list = (Server *) new_realloc((char *)server_list,
				number_of_servers * sizeof(Server));
		else
			server_list = (Server *) new_malloc(number_of_servers * sizeof(Server));

		server_list[from_server].name = m_strdup(server);
		server_list[from_server].itsname = (char *) 0;
		server_list[from_server].password = (char *) 0;
		server_list[from_server].away = (char *) 0;
		server_list[from_server].version_string = (char *) 0;
		server_list[from_server].server2_8 = 0;
		server_list[from_server].operator = 0;
		server_list[from_server].read = -1;
		server_list[from_server].write = -1;
		server_list[from_server].pid = -1;
		server_list[from_server].version = 0;
		server_list[from_server].whois = 0;
		server_list[from_server].flags = 0;
		server_list[from_server].nickname = (char *) 0;
		server_list[from_server].connected = 0;
		server_list[from_server].eof = 0;
		server_list[from_server].motd = 1;
		server_list[from_server].port = port;
		server_list[from_server].WQ_head = (WhoisQueue *) 0;
		server_list[from_server].WQ_tail = (WhoisQueue *) 0;
		server_list[from_server].whois_stuff.nick = (char *) 0;
		server_list[from_server].whois_stuff.user = (char *) 0;
		server_list[from_server].whois_stuff.host = (char *) 0;
		server_list[from_server].whois_stuff.channel = (char *) 0;
		server_list[from_server].whois_stuff.channels = (char *) 0;
		server_list[from_server].whois_stuff.name = (char *) 0;
		server_list[from_server].whois_stuff.server = (char *) 0;
		server_list[from_server].whois_stuff.server_stuff = (char *) 0;
		server_list[from_server].whois_stuff.away = (char *) 0;
		server_list[from_server].whois_stuff.oper = 0;
		server_list[from_server].whois_stuff.chop = 0;
		server_list[from_server].whois_stuff.not_on = 0;
		server_list[from_server].buffer = (char *) 0;
		server_list[from_server].local_addr.s_addr = 0;
		if (password && *password)
			malloc_strcpy(&(server_list[from_server].password), password);
		if (nick && *nick)
			malloc_strcpy(&(server_list[from_server].nickname), nick);
		make_notify_list(from_server);
		do_umode(from_server);
	}
	else
	{
		if (overwrite)
		{
			server_list[from_server].port = port;
			if (password || !server_list[from_server].password)
			{
				if (password && *password)
					malloc_strcpy(&(server_list[from_server].password), password);
				else
					new_free(&(server_list[from_server].password));
			}
			if (nick || !server_list[from_server].nickname)
			{
				if (nick && *nick)
					malloc_strcpy(&(server_list[from_server].nickname), nick);
				else
					new_free(&(server_list[from_server].nickname));
			}
		}
		if (strlen(server) > strlen(server_list[from_server].name))
			malloc_strcpy(&(server_list[from_server].name), server);
	}
}

#ifdef __STDC__
static 	void 	remove_from_server_list (int i)
#else
static	void 	remove_from_server_list(i)
	int	i;
#endif
{
	int	old_server = from_server,
		flag = 1;
	Window	*tmp;

	if (i < 0 || i >= number_of_servers)
		return;

	from_server = i;
	clean_whois_queue();
	from_server = old_server;

	if (server_list[i].name)
		new_free(&server_list[i].name);
	if (server_list[i].itsname)
		new_free(&server_list[i].itsname);
	if (server_list[i].password)
		new_free(&server_list[i].password);
	if (server_list[i].away)
		new_free(&server_list[i].away);
	if (server_list[i].version_string)
		new_free(&server_list[i].version_string);
	if (server_list[i].nickname)
		new_free(&server_list[i].nickname);
	if (server_list[i].whois_stuff.nick)
		new_free(&server_list[i].whois_stuff.nick);
	if (server_list[i].whois_stuff.user)
		new_free(&server_list[i].whois_stuff.user);
	if (server_list[i].whois_stuff.host)
		new_free(&server_list[i].whois_stuff.host);
	if (server_list[i].whois_stuff.channel)
		new_free(&server_list[i].whois_stuff.channel);
	if (server_list[i].whois_stuff.channels)
		new_free(&server_list[i].whois_stuff.channels);
	if (server_list[i].whois_stuff.name)
		new_free(&server_list[i].whois_stuff.name);
	if (server_list[i].whois_stuff.server)
		new_free(&server_list[i].whois_stuff.server);
	if (server_list[i].whois_stuff.server_stuff)
		new_free(&server_list[i].whois_stuff.server_stuff);

	/* 
	 * this should save a coredump.  If number_of_servers drops
	 * down to zero, then trying to do a realloc ends up being
	 * a free, and accessing that is a no-no.
	 */
	if (number_of_servers == 1)
	{
		say("Sorry, the server list is empty and I just dont know what to do.");
		irc_exit(1, NULL);
	}

	bcopy((char *) &server_list[i + 1], (char *) &server_list[i], 
		(number_of_servers - i) * sizeof(Server));
	server_list = (Server *) new_realloc((char *)server_list,
		--number_of_servers * sizeof(Server));

	/* update all he structs with server in them */
	channel_server_delete(i);
	exec_server_delete(i);
        if (i < primary_server)
                --primary_server;
        if (i < from_server)
                --from_server;
	while ((tmp = traverse_all_windows(&flag)) != NULL)
		if (tmp->server > i)
			tmp->server--;
}



/*
 * parse_server_info:  This parses a single string of the form
 * "server:portnum:password:nickname".  It the points port to the portnum
 * portion and password to the password portion.  This chews up the original
 * string, so * upon return, name will only point the the name.  If portnum
 * or password are missing or empty,  their respective returned value will
 * point to null. 
 */
#ifdef __STDC__
void	parse_server_info (char *name, char **port, char **password, char **nick)
#else
void 	parse_server_info (name, port, password, nick)
	char	*name,
		**port,
		**password,
		**nick;
#endif
{
	char *ptr;

	*port = *password = *nick = NULL;
	if ((ptr = (char *) index(name, ':')) != NULL)
	{
		*(ptr++) = (char) 0;
		if (strlen(ptr) == 0)
			*port = (char *) 0;
		else
		{
			*port = ptr;
			if ((ptr = (char *) index(ptr, ':')) != NULL)
			{
				*(ptr++) = (char) 0;
				if (strlen(ptr) == 0)
					*password = '\0';
				else
				{
					*password = ptr;
					if ((ptr = (char *) index(ptr, ':'))
							!= NULL)
					{
						*(ptr++) = '\0';
						if (!strlen(ptr))
							*nick = NULL;
						else
							*nick = ptr;
					}
				}
			}
		}
	}
}

/*
 * build_server_list: given a whitespace separated list of server names this
 * builds a list of those servers using add_to_server_list().  Since
 * add_to_server_list() is used to added each server specification, this can
 * be called many many times to add more servers to the server list.  Each
 * element in the server list case have one of the following forms: 
 *
 * servername 
 *
 * servername:port 
 *
 * servername:port:password 
 *
 * servername::password 
 *
 * Note also that this routine mucks around with the server string passed to it,
 * so make sure this is ok 
 */
#ifdef __STDC__
void	build_server_list (char *servers)
#else
void 	build_server_list(servers)
	char	*servers;
#endif
{
	char	*host,
		*rest,
		*password = (char *) 0,
		*port = (char *) 0,
		*nick = (char *) 0;
	int	port_num;

	if (servers == (char *) 0)
		return;

	while (servers)
	{
		if ((rest = (char *) index(servers, '\n')) != NULL)
			*rest++ = '\0';
		while ((host = next_arg(servers, &servers)) != NULL)
		{
			parse_server_info(host, &port, &password, &nick);
                        if (port && *port)
                        {
                                port_num = atoi(port);
                                if (!port_num)
                                        port_num = irc_port;
                        }
			else
				port_num = irc_port;

			add_to_server_list(host, port_num, password, nick, 0);
		}
		servers = rest;
	}
}

/*
 * connect_to_server_direct: handles the tcp connection to a server.  If
 * successful, the user is disconnected from any previously connected server,
 * the new server is added to the server list, and the user is registered on
 * the new server.  If connection to the server is not successful,  the
 * reason for failure is displayed and the previous server connection is
 * resumed uniterrupted. 
 *
 * This version of connect_to_server() connects directly to a server 
 */
#ifdef __STDC__
static	int	connect_to_server_direct (char *server_name, int port)
#else
static	int 	connect_to_server_direct(server_name, port)
	char	*server_name;
	int	port;
#endif
{
	int		new_des;
struct sockaddr_in 	localaddr;
	int		address_len;
	unsigned short	this_sucks;

	using_server_process = 0;
	oper_command = 0;
	errno = 0;
	bzero(&localaddr, sizeof(localaddr));
	this_sucks = (unsigned short)port;

#ifdef HAVE_SYS_UN_H
	if (*server_name == '/')
		new_des = connect_by_number(server_name, &this_sucks, SERVICE_CLIENT, PROTOCOL_TCP);

	else
#endif /* HAVE_SYS_UN_H */
		new_des = connect_by_number(server_name, &this_sucks, SERVICE_CLIENT, PROTOCOL_TCP);

	port = this_sucks;
	if (new_des < 0)
	{
		if (x_debug)
			say("new_des is %d", new_des);
		say("Unable to connect to port %d of server %s: %s", port,
				server_name, errno ? sys_errlist[errno] :
				"unknown host");
		if ((from_server != -1)&& (server_list[from_server].read != -1))
			say("Connection to server %s resumed...", server_list[from_server].name);
		return (-1);
	}

#ifdef HAVE_SYS_UN_H
	if (*server_name != '/')
#endif /* HAVE_SYS_UN_H */
	{
		address_len = sizeof(struct sockaddr_in);
		getsockname(new_des, (struct sockaddr *) &localaddr, &address_len);
		local_ip_address.s_addr = localaddr.sin_addr.s_addr;
	}

	update_all_status();
	add_to_server_list(server_name, port, (char *) 0, (char *) 0, 1);
	if (port)
	{
		server_list[from_server].read = new_des;
		server_list[from_server].write = new_des;
	}
	else
		server_list[from_server].read = new_des;

	FD_SET(new_des, &readables);
        server_list[from_server].local_addr.s_addr = localaddr.sin_addr.s_addr;
	server_list[from_server].operator = 0;
	return (0);
}

/*
 * connect_to_server_process: handles the tcp connection to a server.  If
 * successful, the user is disconnected from any previously connected server,
 * the new server is added to the server list, and the user is registered on
 * the new server.  If connection to the server is not successful,  the
 * reason for failure is displayed and the previous server connection is
 * resumed uniterrupted. 
 *
 * This version of connect_to_server() uses the ircserv process to talk to a
 * server 
 */
#ifdef __STDC__
static	int 	connect_to_server_process (char *server_name, int port)
#else
static	int 	connect_to_server_process(server_name, port)
	char	*server_name;
	int	port;
#endif
{
	int	socket_des[2],
		pid,
		c;
	char	*path,
		*name = (char *) 0,
		*s;
	int	old_timeout;
	char	buffer[BIG_BUFFER_SIZE + 1];

	path = IRCSERV_PATH;
	if (!path || !*path)
		return (connect_to_server_direct(server_name, port));

	if ((s = rindex(path, '/')) != NULL)
		name = m_strdup(s+1);
	if (!name)
		name = m_strdup(path);

	using_server_process = 1;
	oper_command = 0;
	socket_des[0] = socket_des[1] = -1;
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, socket_des) < 0)
	{
		say("socketpair: Couldn't start new process: %s", sys_errlist[errno]);
		new_free((char **)&name);
		return (connect_to_server_direct(server_name, port));
	}
	switch (pid = fork())
	{
		case -1:
		{
			say("fork: Couldn't start new process: %s\n", sys_errlist[errno]);
			close(socket_des[0]);
			close(socket_des[1]);
			new_free((char **)&name);
			return connect_to_server_direct(server_name, port);
		}

		/* Child,  Close des 0 which is the parent side. */
		case 0:
		{
			char *message = "!!!IRCSERV!! execl: Couldnt start new process";

			dup2(socket_des[1], 0);
			close(socket_des[0]);
			close(socket_des[1]);
			strcpy(buffer, ltoa(port));
			setuid(getuid());
			execl(path, name, server_name, buffer, NULL);

			write(0, message, strlen(message));
			_exit(1);
		}

		/* Parent.   Close des 1, which is the child side. */
		default:
		{
			new_free((char **)&name);
			close(socket_des[1]);
			break;
		}
	}

	old_timeout = dgets_timeout(3);
	c = dgets(buffer, BIG_BUFFER_SIZE, socket_des[0], NULL);
	(void) dgets_timeout(old_timeout);
	if (!c || (c = atoi(buffer)))
	{
		if (c == -5)
			return (connect_to_server_direct(server_name, port));
		else
		{
			char *ptr;

			if ((ptr = (char *) index(buffer, ' ')) != NULL)
			{
				ptr++;
				if (atoi(ptr) > 0)
					say("Unable to connect to port %d of server %s: %s", port, server_name, sys_errlist[atoi(ptr)]);
				else
					say("Unable to connect to port %d of server %s: Unknown host", port, server_name);
			}
			else
				say("Unable to connect to port %d of server %s: Unknown host", port, server_name);

			if ((from_server != -1) && (server_list[from_server].read != -1))
				say("Connection to server %s resumed...", server_list[from_server].name);

			close(socket_des[0]);
			return (-1);
		}
	}
	update_all_status();
	add_to_server_list(server_name, port, (char *) 0, (char *) 0, 1);
	server_list[from_server].read = socket_des[0];
	server_list[from_server].write = socket_des[0];
	server_list[from_server].pid = pid;
	server_list[from_server].operator = 0;
	FD_SET(socket_des[0], &readables);
	return (0);
}



#ifdef __STDC__
int 	connect_to_server_by_refnum (int refnum, int c_server)
#else
int	connect_to_server_by_refnum (refnum, c_server)
	int 	refnum, 
		c_server;
#endif
{
	char *sname;
	int sport;
	int conn;

	if (refnum == -1)
	{
		say ("Connecting to refnum -1.  That makes no sense.");
		return -1;		/* XXXX */
	}

	attempting_to_connect++;
	sname = server_list[refnum].name;
	sport = server_list[refnum].port;

	if (server_list[refnum].read == -1)
	{
		if (sport == -1)
			sport = irc_port;

		from_server = refnum;
		say("Connecting to port %d of server %s", sport, sname);

		if (using_server_process)
			conn = connect_to_server_process(sname, sport);
		else
			conn = connect_to_server_direct(sname, sport);

		if (conn)
		{
			attempting_to_connect--;
			return -1;
		}

		if ((c_server != -1) && (c_server != from_server))
			close_server(c_server, "changing servers");

		if (connect_next_nick && *connect_next_nick)
		{
			malloc_strcpy(&(server_list[from_server].nickname), connect_next_nick);
			new_free(&connect_next_nick);
		}

		from_server = refnum;

		if (server_list[from_server].password)
			send_to_server("PASS %s", server_list[from_server].password);

		if (server_list[from_server].nickname == (char *) 0)
			malloc_strcpy(&(server_list[from_server].nickname), nickname);
		send_to_server("NICK %s", server_list[from_server].nickname);

		send_to_server("USER %s %s %s :%s", username,
			(send_umode && *send_umode) ? send_umode : hostname,
			server_list[from_server].name, realname);
	}
	else
	{
		say("Connected to port %d of server %s", sport, sname);
		from_server = refnum;
	}

	message_from((char *) 0, LOG_CRAP);
	update_all_status();
	return 0;
}

#if 0
/*
 * connect_to_server: Given a name and portnumber, this will attempt to
 * connect to that server using either a direct connection or process
 * connection, depending on the value of using_server_process.  If connection
 * is successful, the proper NICK, USER, and PASS commands are sent to the
 * server.  If the c_server parameter is not -1, then the server with that
 * index will be closed upon successful connection here. Also, if connection
 * is successful, the attempting_to_connect variable is incremented.  This is
 * checked in the notice.c routines to make sure that connection was truely
 * successful (and not closed immediately by the server). 
 */
#ifdef __STDC__
int 	connect_to_server (char *server_name, int port, int c_server)
#else
int 	connect_to_server (server_name, port, c_server)
	char	*server_name;
	int	port;
	int	c_server;
#endif
{
	int	cts_index;

	cts_index = find_in_server_list(server_name, port);
	return connect_to_server_by_refnum(cts_index, c_server);
}
#endif

/*
 * get_connected: This function connects the primary server for IRCII.  It
 * attempts to connect to the given server.  If this isn't possible, it
 * traverses the server list trying to keep the user connected at all cost.  
 */
#ifdef __STDC__
void	get_connected (int server)
#else
void 	get_connected(server)
	int	server;
#endif
{
	int	s,
		ret = -1;

	if (server_list)
	{
		if (server == number_of_servers)
			server = 0;
		else if (server < 0)
			server = number_of_servers - 1;

		s = server;
		if (connect_to_server_by_refnum(server, primary_server))
		{
			while (server_list[server].read == -1)
			{
				server++;
				if (server == number_of_servers)
					server = 0;
				if (server == s)
				{
				    clean_whois_queue();
                                    if (do_hook(DISCONNECT_LIST,"unable to connect to a server"))
        				    say("Use /SERVER to connect to a server");
				    break;
				}
				from_server = server;
				ret = connect_to_server_by_refnum(server, primary_server);
			}
			if (!ret)
				from_server = server;
			else
				from_server = -1;
		}
		change_server_channels(primary_server, from_server);
		set_window_server(-1, from_server, 1);
	}
	else
	{
		clean_whois_queue();
                if (do_hook(DISCONNECT_LIST,"No Server List"))
        		say("Use /SERVER to connect to a server");
	}
}

#ifdef SERVERS_FILE
/*
 * read_server_file: reads hostname:portnum:password server information from
 * a file and adds this stuff to the server list.  See build_server_list()/ 
 */
int read_server_file (void)
{
	FILE *fp;
	char format[11];
	char *file_path = (char *) 0;
	char	buffer[BIG_BUFFER_SIZE + 1];

	malloc_strcpy(&file_path, irc_lib);
	malloc_strcat(&file_path, SERVERS_FILE);
	sprintf(format, "%%%ds", BIG_BUFFER_SIZE);
	fp = fopen(file_path, "r");
	new_free(&file_path);
	if ((FILE *) 0 != fp)
	{
		while (fscanf(fp, format, buffer) != EOF)
			build_server_list(buffer);
		fclose(fp);
		return (0);
	}
	return (1);
}
#endif

/* display_server_list: just guess what this does */
void display_server_list (void)
{
	int	i;

	if (server_list)
	{
		if (from_server != -1)
			say("Current server: %s %d",
					server_list[from_server].name,
					server_list[from_server].port);
		else
			say("Current server: <None>");

		if (primary_server != -1)
			say("Primary server: %s %d",
				server_list[primary_server].name,
				server_list[primary_server].port);
		else
			say("Primary server: <None>");

		say("Server list:");
		for (i = 0; i < number_of_servers; i++)
		{
			if (!server_list[i].nickname)
			{
				if (server_list[i].read == -1)
					say("\t%d) %s %d", i,
						server_list[i].name,
						server_list[i].port);
				else
					say("\t%d) %s %d", i,
						server_list[i].name,
						server_list[i].port);
			}
			else
			{
				if (server_list[i].read == -1)
					say("\t%d) %s %d (was %s)", i,
						server_list[i].name,
						server_list[i].port,
						server_list[i].nickname);
				else
					say("\t%d) %s %d (%s)", i,
						server_list[i].name,
						server_list[i].port,
						server_list[i].nickname);
			}
		}
	}
	else
		say("The server list is empty");
}

#ifdef __STDC__
void	MarkAllAway (char *command, char *message, char *subargs)
#else
void 	MarkAllAway (command, message, subargs)
	char	*command;
	char	*message;
	char	*subargs;
#endif
{
	int	old_server;

	old_server = from_server;
	for (from_server=0; from_server<number_of_servers; from_server++)
	{
		if (server_list[from_server].connected)
			send_to_server("%s :%s", command, message);
	}
	from_server = old_server;
}


/*
 * set_server_password: this sets the password for the server with the given
 * index.  If password is null, the password for the given server is returned 
 */
#ifdef __STDC__
char	*set_server_password (int ssp_index, char *password)
#else
char	*set_server_password (ssp_index, password)
	int	ssp_index;
	char	*password;
#endif
{

	if (server_list)
	{
		if (password)
			malloc_strcpy(&(server_list[ssp_index].password), password);
		return (server_list[ssp_index].password);
	}
	else
		return ((char *) 0);
}

/* server_list_size: returns the number of servers in the server list */
int server_list_size (void)
{
	return (number_of_servers);
}

/*
 * server: the /SERVER command. Read the SERVER help page about 
 */
#ifdef __STDC__
void	server (char *command, char *args, char *subargs)
#else
void 	server (command, args, subargs)
	char	*command,
		*args;
	char	*subargs;
#endif
{
	char	*server = NULL;
	int	i;

	if ((server = next_arg(args, &args)) != NULL)
	{
		/*
		 * Delete an existing server
		 */
		if (!my_strnicmp(server, "-DELETE", strlen(server)))
		{
			if ((server = next_arg(args, &args)) != NULL)
			{
				if ((i = parse_server_index(server)) == -1)
				{
					if ((i = find_in_server_list(server, 0)) == -1)
					{
						say("No such server in list");
						return;
					}
				}
				if (server_list[i].connected)
				{
					say("Can not delete server that is already open");
					return;
				}
				remove_from_server_list(i);
			}
			else
				say("Need server number for -DELETE");
		}

		/*
		 * Add a server, but dont connect
		 */
		else if (!my_strnicmp(server, "-ADD", strlen(server)))
		{
			if ((server = next_arg(args, &args)))
				(void) find_server_refnum(server, &args);
			else
				say("Need server info for -ADD");
		}

		/*
		 * The difference between /server +foo.bar.com  and
		 * /window server foo.bar.com is that this can support
		 * doing a server number.  That makes it a tad bit more
		 * difficult to parse, too. :P  They do the same thing,
		 * though.
		 */
		else if (*server == '+')
		{
			if (*++server)
			{
				i = find_server_refnum(server, &args);
				if (!connect_to_server_by_refnum(i, -1))
				{
					set_window_server(0, i, 0);
					set_level_by_refnum(0, LOG_ALL);
				}
			}
			else
				get_connected(primary_server + 1);
		}

		/*
		 * You can only detach a server using its refnum here.
		 */
		else if (*server == '-')
		{
			if (*++server)
			{
				i = find_server_refnum(server, &args);
				if (i == primary_server)
				{
				    say("You can't close your primary server!");
				    return;
				}
				close_server(i, "closing server");
				window_check_servers();
			}
			else
				get_connected(primary_server - 1);
		}

		/*
		 * Just a naked /server with no flags
		 */
		else
		{
			i = find_server_refnum(server, &args);
			if (!connect_to_server_by_refnum(i, primary_server))
			{
				change_server_channels(primary_server, from_server);
				if (primary_server > -1 && from_server != primary_server &&
				    !server_list[from_server].away && server_list[primary_server].away)
					malloc_strcpy(&server_list[from_server].away, server_list[primary_server].away);
				set_window_server(-1, from_server, 1);
			}
		}
	}
	else
		display_server_list();
}

/*
 * flush_server: eats all output from server, until there is at least a
 * second delay between bits of servers crap... useful to abort a /links. 
 */
void flush_server (void)
{
	fd_set rd;
	struct timeval timeout;
	int	flushing = 1;
	int	des;
	int	old_timeout;
	char	buffer[BIG_BUFFER_SIZE + 1];

	if ((des = server_list[from_server].read) == -1)
		return;
	timeout.tv_usec = 0;
	timeout.tv_sec = 1;
	old_timeout = dgets_timeout(1);
	while (flushing)
	{
		FD_ZERO(&rd);
		FD_SET(des, &rd);
		switch (new_select(&rd, (fd_set *) 0, &timeout))
		{
			case -1:
			case 0:
				flushing = 0;
				break;
			default:
				if (FD_ISSET(des, &rd))
				{
					if (!dgets(buffer, BIG_BUFFER_SIZE, des, NULL))
						flushing = 0;
				}
				break;
		}
	}
	/* make sure we've read a full line from server */
	FD_ZERO(&rd);
	FD_SET(des, &rd);
	if (new_select(&rd, (fd_set *) 0, &timeout) > 0)
		dgets(buffer, BIG_BUFFER_SIZE, des, (char *) 0);
	(void)dgets_timeout(old_timeout);
}


#ifdef __STDC__
static char *do_umode (int du_index)
#else
static char *do_umode (du_index)
int du_index;
#endif
{
	char *c = server_list[du_index].umode;
	int flags = server_list[du_index].flags;
	int i;

	for (i = 0; umodes[i]; i++)
	{
		if (flags & USER_MODES << i)
			*c++ = umodes[i];
	}

	*c = '\0';
	return server_list[du_index].umode;
}

#ifdef __STDC__
char *get_umode (int gu_index)
#else
char *get_umode (gu_index)
int gu_index;
#endif
{
	if (gu_index == -1)
		gu_index = primary_server;
	else if (gu_index >= number_of_servers)
		return empty_string;

	return server_list[gu_index].umode;
}

#ifdef __STDC__
void	set_server_flag (int ssf_index, int flag, int value)
#else
void 	set_server_flag (ssf_index, flag, value)
	int	ssf_index;
	int	flag;
	int	value;
#endif
{
	if (ssf_index == -1)
		ssf_index = primary_server;
	else if (ssf_index >= number_of_servers)
		return;

	if (value)
		server_list[ssf_index].flags |= flag;
	else
		server_list[ssf_index].flags &= ~flag;

	do_umode(ssf_index);
}

#ifdef __STDC__
int	get_server_flag (int gsf_index, int value)
#else
int 	get_server_flag (gsf_index, value)
	int	gsf_index;
	int	value;
#endif
{
	if (gsf_index == -1)
		gsf_index = primary_server;
	else if (gsf_index >= number_of_servers)
		return 0;

	return server_list[gsf_index].flags & value;
}

/*
 * set_server_version: Sets the server version for the given server type.  A
 * zero version means pre 2.6, a one version means 2.6 aso. (look server.h
 * for typedef)
 */
#ifdef __STDC__
void	set_server_version (int ssv_index, int version)
#else
void 	set_server_version (ssv_index, version)
	int	ssv_index;
	int	version;
#endif
{
	if (ssv_index == -1)
		ssv_index = primary_server;
	else if (ssv_index >= number_of_servers)
		return;
	server_list[ssv_index].version = version;
}

/*
 * get_server_version: returns the server version value for the given server
 * index 
 */
#ifdef __STDC__
int	get_server_version (int gsv_index)
#else
int 	get_server_version (gsv_index)
	int	gsv_index;
#endif
{
	if (gsv_index == -1)
		gsv_index = primary_server;
	else if (gsv_index >= number_of_servers)
		return 0;

	return (server_list[gsv_index].version);
}

/* get_server_name: returns the name for the given server index */
#ifdef __STDC__
char	*get_server_name (int gsn_index)
#else
char	*get_server_name (gsn_index)
	int	gsn_index;
#endif
{
	if (gsn_index == -1)
		gsn_index = primary_server;
	else if (gsn_index >= number_of_servers)
		return empty_string;

	return (server_list[gsn_index].name);
}

/* get_server_itsname: returns the server's idea of its name */
#ifdef __STDC__
char	*get_server_itsname (int gsi_index)
#else
char	*get_server_itsname (gsi_index)
	int	gsi_index;
#endif
{
	if (gsi_index==-1)
		gsi_index=primary_server;
	else if (gsi_index >= number_of_servers)
		return empty_string;

	/* better check gsi_index for -1 here CDE */
	if (gsi_index == -1)
		return empty_string;

	if (server_list[gsi_index].itsname)
		return server_list[gsi_index].itsname;
	else
		return server_list[gsi_index].name;
}

#ifdef __STDC__
void	set_server_itsname (int ssi_index, char *name)
#else
void	set_server_itsname (ssi_index, name)
	int	ssi_index;
	char	*name;
#endif
{
	if (ssi_index==-1)
		ssi_index=primary_server;
	else if (ssi_index >= number_of_servers)
		return;

	malloc_strcpy(&server_list[ssi_index].itsname, name);
}

/*
 * is_server_open: Returns true if the given server index represents a server
 * with a live connection, returns false otherwise 
 */
#ifdef __STDC__
int	is_server_open (int iso_index)
#else
int 	is_server_open (iso_index)
	int	iso_index;
#endif
{
	if (iso_index < 0 || iso_index >= number_of_servers) 
		return (0);
	return (server_list[iso_index].read != -1);
}

/*
 * is_server_connected: returns true if the given server is connected.  This
 * means that both the tcp connection is open and the user is properly
 * registered 
 */
#ifdef __STDC__
int	is_server_connected (int isc_index)
#else
int 	is_server_connected (isc_index)
	int	isc_index;
#endif
{
	if (isc_index >= 0 && isc_index < number_of_servers)
		return (server_list[isc_index].connected);
	return 0;
}

/* get_server_port: Returns the connection port for the given server index */
#ifdef __STDC__
int	get_server_port (int gsp_index)
#else
int 	get_server_port (gsp_index)
	int	gsp_index;
#endif
{
	if (gsp_index == -1)
		gsp_index = primary_server;
	else if (gsp_index >= number_of_servers)
		return 0;

	return (server_list[gsp_index].port);
}

/*
 * get_server_nickname: returns the current nickname for the given server
 * index 
 */
#ifdef __STDC__
char	*get_server_nickname (int gsn_index)
#else
char	*get_server_nickname (gsn_index)
	int	gsn_index;
#endif
{
	if (gsn_index >= number_of_servers)
		return empty_string;
	else if (gsn_index != -1 && server_list[gsn_index].nickname)
		return (server_list[gsn_index].nickname);
	else
		return (nickname);
}



/* get_server_qhead - get the head of the whois queue */
#ifdef __STDC__
WhoisQueue *get_server_qhead (int gsq_index)
#else
WhoisQueue *get_server_qhead (gsq_index)
	int	gsq_index;
#endif
{
	if (gsq_index == -1)
		return WQ_head;
	else if (gsq_index >= number_of_servers)
		return NULL;

	return server_list[gsq_index].WQ_head;
}

/* get_server_whois_stuff */
#ifdef __STDC__
WhoisStuff *get_server_whois_stuff (int gsw_index)
#else
WhoisStuff *get_server_whois_stuff (gsw_index)
	int	gsw_index;
#endif
{
	if (gsw_index == -1)
		gsw_index=primary_server;
	else if (gsw_index >= number_of_servers)
		return NULL;

	return &server_list[gsw_index].whois_stuff;
}

/* get_server_qtail - get the tail of the whois queue */
#ifdef __STDC__
WhoisQueue *get_server_qtail (int gsq_index)
#else
WhoisQueue *get_server_qtail (gsq_index)
	int	gsq_index;
#endif
{
	if (gsq_index == -1)
		return WQ_tail;
	else if (gsq_index < number_of_servers)
		return server_list[gsq_index].WQ_tail;

	return NULL;
}

/* set_server_qhead - set the head of the whois queue */
#ifdef __STDC__
void 	set_server_qhead (int ssq_index, WhoisQueue *value)
#else
void 	set_server_qhead (ssq_index, value)
	int	ssq_index;
	WhoisQueue *value;
#endif
{
	if (ssq_index == -1)
		WQ_head = value;
	else if (ssq_index < number_of_servers)
		server_list[ssq_index].WQ_head=value;

	return;
}

/* set_server_qtail - set the tail of the whois queue */
#ifdef __STDC__
void	set_server_qtail (int ssq_index, WhoisQueue *value)
#else
void 	set_server_qtail (ssq_index, value)
	int	ssq_index;
	WhoisQueue *value;
#endif
{
	if (ssq_index == -1)
		WQ_tail = value;
	else if (ssq_index < number_of_servers)
		server_list[ssq_index].WQ_tail=value;

	return;
}

/* 
 * set_server2_8 - set the server as a 2.8 server 
 * This is used if we get a 001 numeric so that we dont bite on
 * the "kludge" that ircd has for older clients
 */
#ifdef __STDC__
void 	set_server2_8 (int ss2_index, int value)
#else
void 	set_server2_8 (ss2_index, value)
int 	ss2_index;
int 	value;
#endif
{
	if (ss2_index < number_of_servers)
		server_list[ss2_index].server2_8 = value;
	return;
}

/* get_server2_8 - get the server as a 2.8 server */
#ifdef __STDC__
int 	get_server2_8 (int gs2_index)
#else
int 	get_server2_8 (gs2_index)
int 	gs2_index;
#endif
{
	if (gs2_index == -1)
		gs2_index = primary_server;
	else if (gs2_index >= number_of_servers)
		return 0;
	return (server_list[gs2_index].server2_8);
}
	
/*
 * get_server_operator: returns true if the user has op privs on the server,
 * false otherwise 
 */
#ifdef __STDC__
int	get_server_operator (int gso_index)
#else
int 	get_server_operator (gso_index)
	int	gso_index;
#endif
{
	if ((gso_index < 0) || (gso_index >= number_of_servers))
		return 0;
	return (server_list[gso_index].operator);
}

/*
 * set_server_operator: If flag is non-zero, marks the user as having op
 * privs on the given server.  
 */
#ifdef __STDC__
void	set_server_operator (int sso_index, int flag)
#else
void 	set_server_operator (sso_index, flag)
	int	sso_index;
	int	flag;
#endif
{
	if (sso_index < 0 || sso_index >= number_of_servers)
		return;
	server_list[sso_index].operator = flag;
	do_umode(sso_index);
}

/*
 * set_server_nickname: sets the nickname for the given server to nickname.
 * This nickname is then used for all future connections to that server
 * (unless changed with NICK while connected to the server 
 */
#ifdef __STDC__
void 	set_server_nickname (int ssn_index, char *nick)
#else
void 	set_server_nickname (ssn_index, nick)
	int	ssn_index;
	char	*nick;
#endif
{
	if (ssn_index != -1 && ssn_index < number_of_servers)
	{
		malloc_strcpy(&(server_list[ssn_index].nickname), nick);
		if (ssn_index == primary_server)
			strmcpy(nickname,nick,
#ifdef ALLOW_LONG_NICKNAMES
						LONG_NICKNAME_LEN
#else
						NICKNAME_LEN
#endif
								);
	}
	update_all_status();
}

#ifdef __STDC__
void 	set_server_motd (int ssm_index, int flag)
#else
void 	set_server_motd (ssm_index,flag)
	int	ssm_index;
	int	flag;
#endif
{
	if (ssm_index != -1 && ssm_index < number_of_servers)
		server_list[ssm_index].motd = flag;
}

#ifdef __STDC__
int 	get_server_motd (int gsm_index)
#else
int 	get_server_motd (gsm_index)
	int	gsm_index;
#endif
{
	if (gsm_index != -1 && gsm_index < number_of_servers)
		return(server_list[gsm_index].motd);
	return (0);
}

#ifdef __STDC__
void 	server_is_connected (int sic_index, int value)
#else
void 	server_is_connected (sic_index, value)
	int	sic_index,
		value;
#endif
{
	if (sic_index < 0 || sic_index >= number_of_servers)
		return;

	server_list[sic_index].connected = value;
	if (value)
		server_list[sic_index].eof = 0;
}

/* send_to_server: sends the given info the the server */
#if defined(__STDC__) && defined(HAVE_STDARG_H)
void 	send_to_server (const char *format, ...)
#else
void 	send_to_server (format, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10)
const 	char	*format;
	char	*arg1, *arg2, *arg3, *arg4, *arg5,
		*arg6, *arg7, *arg8, *arg9, *arg10;
#endif
{
	char	buffer[BIG_BUFFER_SIZE + 1];	/* make this buffer *much*
						 * bigger than needed */
	char	*buf = buffer;
	int	len,
		des;
	int	server;

	if ((server = from_server) == -1)
		server = primary_server;

	if (server != -1 && ((des = server_list[server].write) != -1) && format)
	{
#if defined(__STDC__) && defined(HAVE_STDARG_H)
		va_list args;
		va_start(args, format);
		vsprintf(buf, format, args);
		va_end(args);
#else
		sprintf(buf, format, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10);
#endif

		server_list[server].sent = 1;
		len = strlen(buffer);
		if (len > (IRCD_BUFFER_SIZE - 2))
			buffer[IRCD_BUFFER_SIZE - 2] = (char) 0;
		strmcat(buffer, "\n", IRCD_BUFFER_SIZE);
		if (x_debug)
			yell("[%d] -> [%s]", des, buffer);
		if (do_hook(SEND_TO_SERVER_LIST, "%d %d %s", server, des, buffer))
			send(des, buffer, strlen(buffer), 0);
	}
	else
        {
                if (do_hook(DISCONNECT_LIST,"No Connection"))
			say("You are not connected to a server, use /SERVER to connect.");
        }
}

/*
 * close_all_server: Used when creating new screens to close all the open
 * server connections in the child process...
 */
extern	void close_all_server (void)
{
	int	i;

	for (i = 0; i < number_of_servers; i++)
	{
		if (server_list[i].read != -1)
			new_close(server_list[i].read);
		if (server_list[i].write != -1)
			new_close(server_list[i].write);
	}
}

extern	char *create_server_list (void)
{
	int	i;
	char	*value = (char *) 0;
	char buffer[BIG_BUFFER_SIZE + 1];

	*buffer = '\0';
	for (i = 0; i < number_of_servers; i++)
	{
		if (server_list[i].read != -1)
		{
			if (server_list[i].itsname)
			{
				strcat(buffer, server_list[i].itsname);
				strcat(buffer, space);
			}
			else
				yell("Warning: server_list[%d].itsname is null and it shouldnt be", i);
		}
	}
	malloc_strcpy(&value, buffer);

	return value;
}

#ifdef __STDC__
static 	void add_to_server_buffer (int server, char *buf)
#else
static	void add_to_server_buffer (server, buf)
	int	server;
	char	*buf;
#endif
{
	if (buf && *buf && server >= 0 && server < number_of_servers)
	{
		if (server_list[server].buffer)
			malloc_strcat(&server_list[server].buffer, buf);
		else
			malloc_strcpy(&server_list[server].buffer, buf);
	}
}


#ifdef __STDC__
void 	disconnectcmd (char *command, char *args, char *subargs)
#else
void 	disconnectcmd (command, args, subargs)
	char	*command,
		*args,
		*subargs;
#endif
{
	char	*server;
	char	*message;
	int	i;

	if ((server = next_arg(args, &args)) != NULL)
	{
		i = parse_server_index(server);
		if (-1 == i)
		{
			say("No such server!");
			return;
		}
	}
	else
		i = get_window_server(0);

        /*
         * XXX - this is a major kludge.  i should never equal -1 at
         * this point.  we only do this because something has gotten
         * *really* confused at this point.  .mrg.
	 *
	 * Like something so obvious as already being disconnected?
         */
        if (i == -1)
        {
		if (connected_to_server)
		{
			for (i = 0; i < number_of_servers; i++)
			{
				clear_channel_list(i);
				server_list[i].eof = -1;
				new_close(server_list[i].read);
				new_close(server_list[i].write);
			}
		}
		goto done;
	}
 
	if (!args || !*args)
		message = "Disconnecting";
	else
		message = args;

	if (-1 == server_list[i].write)
	{
		say("That server isn't connected!");
		return;
	}
	say("Disconnecting from server %s", server_list[i].itsname);
	clear_channel_list(i);
	close_server(i, message);
	server_list[i].eof = 1;
done:
	clean_whois_queue();
	window_check_servers();
	if (!connected_to_server)
		say("You are not connected to a server, Use /SERVER to connect.");
} 
