/*****************************************************************************/
/*  rfc2068.c - General purpose routines for the HTTP protocol               */
/*  Copyright (C) 1999 Brian Masney <masneyb@seul.org>                       */
/*                                                                           */
/*  This library is free software; you can redistribute it and/or            */
/*  modify it under the terms of the GNU Library General Public              */
/*  License as published by the Free Software Foundation; either             */
/*  version 2 of the License, or (at your option) any later version.         */
/*                                                                           */
/*  This library is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU        */
/*  Library General Public License for more details.                         */
/*                                                                           */
/*  You should have received a copy of the GNU Library General Public        */
/*  License along with this library; if not, write to the Free Software      */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "protocols.h"

static int rfc2068_connect (gftp_request *request);
static void rfc2068_disconnect (gftp_request *request);
static long rfc2068_get_file (gftp_request *request, const char *filename, long startsize);
static int rfc2068_end_transfer (gftp_request *request);
static int rfc2068_list_files (gftp_request *request);
static int rfc2068_get_next_file (gftp_request *request, gftp_file *fle, FILE *fd);
static int rfc2068_chdir (gftp_request *request, const char *directory);
static unsigned long rfc2068_send_command (gftp_request *request, const char *command);
static unsigned long rfc2068_read_response (gftp_request *request);
static int parse_html_line (char *tempstr, gftp_file *fle);
static char *base64_encode (char *str);

void rfc2068_init (gftp_request *request) {
   g_return_if_fail (request != NULL);
   
   request->init = rfc2068_init;
   request->connect = rfc2068_connect;
   request->disconnect = rfc2068_disconnect;
   request->get_file = rfc2068_get_file;
   request->put_file = NULL;
   request->end_transfer = rfc2068_end_transfer;
   request->list_files = rfc2068_list_files;
   request->get_next_file = rfc2068_get_next_file;
   request->set_data_type = NULL;
   request->get_file_size = NULL;
   request->chdir = rfc2068_chdir;
   request->rmdir = NULL;
   request->rmfile = NULL;
   request->mkdir = NULL;
   request->rename = NULL;
   request->chmod = NULL;
   request->site = NULL;
   request->url_prefix = "http";
   request->protocol_name = "HTTP";
   request->supports_resume = 0;
   request->unsafe_symlinks = 1;
   request->add_onek = 1;
}
/*****************************************************************************/
static int rfc2068_connect (gftp_request *request) {
   struct sockaddr_in remote_address;
   int curhost, sock, port;
   struct servent *service;
   char *connect_host;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->hostname != NULL, -2);
   
   if (request->datafd != NULL) return (0);
   if ((request->use_proxy = gftp_need_proxy (request)) < 0) {
      return (-1);
   }
   else if (request->use_proxy == 1) request->host = NULL;
   
   if ((sock = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
      if (request->logging) {
         request->logging_function (gftp_logging_error, request->user_data, _("Failed to create a socket: %s\n"),
      		g_strerror (errno));
      }
      return (-1);
   }

   memset (&remote_address, 0, sizeof (remote_address));
   remote_address.sin_family = AF_INET;
   
   port = htons (request->use_proxy ? request->proxy_port : request->port);
   connect_host = request->use_proxy ? request->proxy_hostname : request->hostname;

   if (ntohs (port) == 0) {
      if ((service = getservbyname ("http", "tcp")) == NULL) {
         request->port = 80;
         port = htons (80);
      }
      else {
         request->port = ntohs (service->s_port);
         port = service->s_port;
      }
   }
   remote_address.sin_port = port;

   if (request->host == NULL) {
      if (request->logging) {
         request->logging_function (gftp_logging_misc, request->user_data, _("Looking up %s...\n"), connect_host);
      }

      if ((request->host = gethostbyname (connect_host)) == NULL) {
         if (request->logging) {
            request->logging_function (gftp_logging_error, request->user_data, _("Cannot look up hostname %s: %s\n"), 
            	connect_host, g_strerror (errno));
         }
         close (sock);
         return (-1);
      }
   }

   for (curhost = 0; request->host->h_addr_list[curhost] != NULL; curhost++) {
      memcpy (&remote_address.sin_addr, request->host->h_addr_list[curhost], request->host->h_length);
      if (request->logging) {
         request->logging_function (gftp_logging_misc, request->user_data, 
         	_("Trying %s:%d...\n"), request->host->h_name, ntohs (port));
      }
      if (connect (sock, (struct sockaddr *) &remote_address, sizeof (remote_address)) == -1 && request->logging) {
         request->logging_function (gftp_logging_error, request->user_data, _("Cannot connect to %s: %s\n"), 
         	request->host->h_name, g_strerror (errno));
      }
      else break;
   }

   if (request->host->h_addr_list[curhost] == NULL) {
      close (sock);
      return (-1);
   }

   if (!request->use_proxy) {
      g_free (request->hostname);
      request->hostname = g_malloc (strlen (request->host->h_name) + 1);
      strcpy (request->hostname, request->host->h_name);
      connect_host = request->hostname;
   }
   
   if (request->logging) {
      request->logging_function (gftp_logging_misc, request->user_data, 
      	_("Connected to %s:%d\n"), connect_host, ntohs (port));
   }
   
   if (!request->directory) {
      request->directory = g_malloc (2);
      strcpy (request->directory, "/");
   }
   
   request->datafd = fdopen (sock, "r+");
   request->datafd_unbuffered = sock;
   return (0);
}
/*****************************************************************************/
static void rfc2068_disconnect (gftp_request *request) {
   g_return_if_fail (request != NULL);
   
   if (request->datafd != NULL) {
      if (request->logging) {
         request->logging_function (gftp_logging_misc, request->user_data,
         	_("Disconnecting from site %s\n"), request->hostname);
      }
      fclose (request->datafd);
      request->datafd = NULL;
   }
   request->host = NULL;
}
/*****************************************************************************/
static long rfc2068_get_file (gftp_request *request, const char *filename, long startsize) {
   unsigned long size;
   char *tempstr;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (filename != NULL, -2);
   
   if (request->datafd == NULL && !rfc2068_connect (request)) return (-2);

   if (request->use_proxy && (request->username == NULL || *request->username == '\0' 
   	|| strcmp (request->username, "anonymous") == 0)) {
      tempstr = g_strconcat ("GET ", request->proxy_config, "://", request->hostname, "/", filename, " HTTP/1.1\n", NULL);
   }
   else if (request->use_proxy) {
      tempstr = g_strconcat ("GET ", request->proxy_config, "://", request->username, ":", request->password, 
      	"@", request->hostname, "/", filename, " HTTP/1.1\n", NULL);
   }
   else {
      tempstr = g_strconcat ("GET ", filename, " HTTP/1.1\n", NULL);
   }

   size = rfc2068_send_command (request, tempstr);
   g_free (tempstr);

   if (request->logging) {
      request->logging_function (gftp_logging_send, request->user_data, 
      	_("Retrieving file: %s...\n"), filename);
   }
   return (size);
}
/*****************************************************************************/
static int rfc2068_end_transfer (gftp_request *request) {
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->datafd != NULL, -2);
   
   fclose (request->datafd);
   request->datafd = NULL;
   request->max_bytes = 0;
   if (request->logging) {
      request->logging_function (gftp_logging_misc, request->user_data,
      	_("Finished retrieving directory listing\n"));
   }
   return (0);
}
/*****************************************************************************/
static int rfc2068_list_files (gftp_request *request) {
   char *tempstr;
   
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->datafd == NULL && !rfc2068_connect (request)) return (-2);

   if (request->use_proxy && (request->username == NULL || *request->username == '\0' 
   	|| strcmp (request->username, "anonymous") == 0)) {
      tempstr = g_strconcat ("GET ", request->proxy_config, "://", request->hostname, "/", request->directory, "/ HTTP/1.1\n", NULL);
   }
   else if (request->use_proxy) {
      tempstr = g_strconcat ("GET ", request->proxy_config, "://", request->username, ":", request->password, 
      	"@", request->hostname, "/", request->directory, "/ HTTP/1.1\n", NULL);
   }
   else {
      tempstr = g_strconcat ("GET ", request->directory, "/ HTTP/1.1\n", NULL);
   }
   rfc2068_send_command (request, tempstr);
   g_free (tempstr);

   if (request->logging) {
      request->logging_function (gftp_logging_send, request->user_data, 
      	_("Retrieving directory listing...\n"));
   }
   request->read_bytes = 0;
   if (strlen (request->last_ftp_response) > 9) {
      return (strncmp (request->last_ftp_response + 9, "200", 3) == 0 ? 0 : -2);
   }
   else return (-2);
}
/*****************************************************************************/
static int rfc2068_get_next_file (gftp_request *request, gftp_file *fle, FILE *fd) {
   char tempstr[255];

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (fle != NULL, -2);

   if (request->last_dir_entry) {
      g_free (request->last_dir_entry);
      request->last_dir_entry = NULL;
   }
   do {
      if (!fgets (tempstr, sizeof (tempstr), fd)) {
         memset (fle, 0, sizeof (gftp_file));
         return (-2);
      }
      request->read_bytes += strlen (tempstr);
      if (parse_html_line (tempstr, fle) == 0) {
         gftp_file_destroy (fle);
      }
      else break;
      
      if (request->max_bytes != 0 && request->read_bytes == request->max_bytes) {
         break;
      }
   } while (1);

   request->last_dir_entry = g_malloc (strlen (tempstr) + 1);
   strcpy (request->last_dir_entry, tempstr);
   return (feof (fd) ? 0 : strlen (request->last_dir_entry));
}
/*****************************************************************************/
static int rfc2068_chdir (gftp_request *request, const char *directory) {
   return (0);
}
/*****************************************************************************/
static unsigned long rfc2068_send_command (gftp_request *request, const char *command) {
   char *tempstr, *str;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (command != NULL, -2);
   g_return_val_if_fail (request->datafd != NULL, -2);
   
   if (request->logging) {
      request->logging_function (gftp_logging_send, request->user_data, command);
   }
   fprintf (request->datafd, "%sUser-Agent: gFTP\n", command);
   if (ferror (request->datafd) != 0) return (-2);

   if (!request->use_proxy) {
      fprintf (request->datafd, "Host: %s\n", request->hostname);
      if (ferror (request->datafd) != 0) return (-2);
   }

   if (request->use_proxy && request->proxy_username != NULL && *request->proxy_username != '\0') {
      tempstr = g_strconcat (request->proxy_username, ":", request->proxy_password, NULL);
      str = base64_encode (tempstr);
      g_free (tempstr);

      fprintf (request->datafd, "Proxy-authorization: Basic %s\n", str);
      g_free (str);
      if (ferror (request->datafd) != 0) return (-2);
   }
   
   if (!request->use_proxy && request->username == NULL && strcmp (request->username, "anonymous") == 0 && *request->username == '\0') {
      tempstr = g_strconcat (request->username, ":", request->password, NULL);
      str = base64_encode (tempstr);
      g_free (tempstr);

      fprintf (request->datafd, "Authorization: Basic %s\n", str);
      g_free (str);
      if (ferror (request->datafd) != 0) return (-2);
   }
   
   fprintf (request->datafd, "\n");
   if (ferror (request->datafd) != 0) return (-2);
   return (rfc2068_read_response (request));
}
/*****************************************************************************/
static unsigned long rfc2068_read_response (gftp_request *request) {
   char tempstr[255];
   
   request->max_bytes = 0;
   if (!fgets (tempstr, sizeof (tempstr), request->datafd)) return (0);
   if (request->last_ftp_response ) g_free (request->last_ftp_response);
   request->last_ftp_response = g_malloc (strlen (tempstr) + 1);
   strcpy (request->last_ftp_response, tempstr);

   if (request->logging) {
      request->logging_function (gftp_logging_recv, request->user_data, tempstr);
   }

   do {
      /* Read rest of proxy header */
      if (!fgets (tempstr, sizeof (tempstr), request->datafd)) return (0);
      if (strncmp (tempstr, "Content-Length:", 15) == 0) {
         request->max_bytes = strtol (tempstr + 16, NULL, 10);
      }
   } while (*tempstr != '\n' && *tempstr != '\r');
   return (request->max_bytes);
}
/*****************************************************************************/
static int parse_html_line (char *tempstr, gftp_file *fle) {
   char *stpos, *pos, month[4];
   struct tm t;
   
   memset (fle, 0, sizeof (gftp_file));

   if ((pos = strstr (tempstr, "<A HREF=")) == NULL) {
      return (0);
   }

   /* Find the filename */
   while (*pos != '"' && *pos != '\0') pos++;
   if (*pos == '\0') return (0);
   pos++;
   
   for (stpos = pos; *pos != '"' && *pos != '\0'; pos++);
   if (*pos == '\0') return (0);
   *pos = '\0';

   /* Copy file attributes. Just about the only thing we can get is whether it
      is a directory or not */
   fle->attribs = g_malloc (11);
   strcpy (fle->attribs, "----------");
   if (*(pos-1) == '/') {
      *(pos-1) = '\0';
      *fle->attribs = 'd';
   }

   /* Copy filename */
   if (strchr (stpos, '/') != NULL || strncmp (stpos, "mailto:", 7) == 0) {
      return (0);
   }
   fle->file = g_malloc (strlen (stpos) + 1);
   strcpy (fle->file, stpos);
   if (*(pos-1) == '\0') *(pos-1) = '/';
   *pos = '"';
   pos++;

   /* Skip whitespace and html tags after file and before date */
   stpos = pos;
   if ((pos = strstr (stpos, "</A>")) == NULL) {
      if ((pos = strstr (stpos, "</a>")) == NULL) {
         return (0);
      }
   }
   pos += 4;

   while (*pos == ' ' || *pos == '.' || *pos == '<') {
      if (*pos == '<') {
         if (strncmp (pos, "<A ", 3) == 0 || strncmp (pos, "<a ", 3) == 0) {
            stpos = pos;
            if ((pos = strstr (stpos, "</A>")) == NULL) {
               if ((pos = strstr (stpos, "</a>")) == NULL) {
                  return (0);
               }
            }
            pos += 4;
         }
         else {
            while (*pos != '>' && *pos != '\0') pos++;
            if (*pos == '\0') return (0);
         }
      }
      pos++;
   }

   /* Get the size */
   if ((stpos = strchr (pos, 'k')) == NULL) return (1); /* Return successfully
                                                           since we got the file */
   while (*(stpos-1) != ' ' && *(stpos-1) != '\t' && stpos > tempstr) stpos--;
   fle->size = strtol (stpos, NULL, 10) * 1024;
   
   /* Now get the date */
   memset (&t, 0, sizeof (t));
   memset (month, 0, sizeof (month));
   if (strchr (pos, ':') != NULL) {
      if (*pos == '[') pos++;
      sscanf (pos, "%02d-%3s-%04d %02d:%02d", &t.tm_mday, month, &t.tm_year, 
      		&t.tm_hour, &t.tm_min);
      while (*pos != ' ' && *pos != '\0') pos++;
      if (*pos == '\0') return (1);
      while (*pos == ' ') pos++;
      while (*pos != ' ' && *pos != '\0') pos++;
      if (*pos == '\0') return (1);
      while (*pos == ' ') pos++;
   }
   else {
      pos++;
      strncpy (month, pos, 3);
      pos += 3;
      
      while (*pos == ' ') pos++;
      t.tm_mday = strtol (pos, NULL, 10);
      while (*pos != ' ' && *pos != '\0') pos++;

      while (*pos == ' ') pos++;
      t.tm_year = strtol (pos, NULL, 10);
      while (*pos != ' ' && *pos != '\0') pos++;
   }
   fle->datetime = mktime (&t);
   return (1);
}
/*****************************************************************************/
static char *base64_encode (char *str) {

/* The standard to Base64 encoding can be found in RFC2045 */

   char *newstr, *newpos, *fillpos, *pos;
   unsigned char table[63], encode[3];
   int i, num;
   
   for (i=0; i<26; i++) {
      table[i] = 'A' + i;
      table[i + 26] = 'a' + i;
   }
   
   for (i=0; i<10; i++) {
      table[i + 52] = '0' + i;
   }
   
   table[62] = '+';
   table[63] = '-';
 
   num = strlen (str) / 3;
   if (strlen (str) % 3 > 0) num++;
   newstr = g_malloc (num * 4 + 1);
   newstr[num * 4] = '\0';
   newpos = newstr;
   
   pos = str;
   while (*pos != '\0') {
      memset (encode, 0, sizeof (encode));
      for (i=0; i<3 && *pos != '\0'; i++) encode[i] = *pos++;
      
      fillpos = newpos;
      *newpos++ = table[encode[0] >> 2];
      *newpos++ = table[(encode[0] & 3) << 4 | encode[1] >> 4];
      *newpos++ = table[(encode[1] & 0xF) << 2 | encode[2] >> 6];
      *newpos++ = table[encode[2] & 0x3F];
      while (i < 3) fillpos[++i] = '=';
   }
   return (newstr);
}
/*****************************************************************************/
