/* gnome-napster, a GNOME Napster client.
 * Copyright (C) 1999 Evan Martin <eeyem@u.washington.edu>
 */

/* network */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
/* printf */
#include <stdio.h>
#include <stdlib.h>
/* itoa */
#include <unistd.h>
/* isprint */
#include <ctype.h>
/* strerror */
#include <string.h>
#include <errno.h>
/* va_start */
#include <stdarg.h>

#include "lnap.h"

int nap_socknap = -1, nap_socklocal = -1;
char* nap_conntypes[] = {
	"Unknown", "14.4 Modem", "28.8 Modem", "33.6 Modem", "57.6 Modem", 
	"ISDN 64K", "ISDN 128K", "Cable", "DSL", "T1", "T3 or greater", NULL
};

static statusfunc napstatusfunc = NULL;
static void *napstatusdata;
static char statustext[1024];

#define errorreport(x) {sprintf(statustext, "%s: %s", x, strerror(errno));nap_status(statustext);}

const static struct {
	int msg;
	char *text;
} msglookup[] = {
	{ NM_ERROR, "error" },
	{ NM_LOGINRESP, "login response" },
	{ NM_SEARCH_RESULT, "search result" },
	{ NM_FILEINFO_REQUEST, "file info request" },
	{ NM_FILEINFO_RESULT, "file info" },
	{ NM_FILECOUNT, "file count" },
	{ NM_SYSMSG, NULL, /*"system message"*/ },
	{ -1, 0 }
};

void nap_setstatusfunc(statusfunc s, void* d) {
	napstatusfunc = s;
	napstatusdata = d;
}

void nap_getstatusfunc(statusfunc *s, void** d) {
	*s = napstatusfunc;
	*d = napstatusdata;
}

inline void nap_status(char *text) {
	if (napstatusfunc) (*napstatusfunc)(text, napstatusdata);
}

int nap_str_to_sockaddr_in(char *text, struct sockaddr_in *host) {
	struct hostent *he;
	char *colon;
	char *scan = strdup(text);
	host->sin_family = AF_INET;

	colon = strchr(scan, ':');
	if (colon == NULL) {
		nap_status("No port in server address?");
		free(scan);
		return -1;
	}
	*colon = 0;
	
	/* this reportedly doesn't work on solaris.  comments? 
	if (!inet_aton(scan, &host->sin_addr)) {
		nap_status("Bad server address.");
		return -1;
	} */
	if ((he = gethostbyname(scan)) == NULL) {
		sprintf(statustext, "gethostbyname: %s", hstrerror(h_errno));
		endhostent();
		nap_status(statustext);
		free(scan);
		return -1;
	}
	endhostent();
	memcpy(&host->sin_addr, he->h_addr_list[0], he->h_length);

	/* inet address is already in host order, yay! */

	colon++;
	host->sin_port = htons(atoi(colon));
	free(scan);
	return 0;
}

int nap_getbesthost(char *hostbuf, char *serv) {
	int sock, len;
	struct sockaddr_in s;
	char *newline;
	struct hostent *he;

	nap_status("Looking up redirector...");
	if ((he = gethostbyname(serv)) == NULL) {
		sprintf(statustext, "gethostbyname: %s", hstrerror(h_errno));
		endhostent();
		nap_status(statustext);
		return -1;
	}
	endhostent();

	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		errorreport("sock");
		return -1;
	}

	s.sin_family = AF_INET;
	s.sin_port = htons(NAP_SEARCHPORT);
	s.sin_addr.s_addr = *((unsigned long*)he->h_addr_list[0]);
	nap_status(inet_ntoa(s.sin_addr));

	nap_status("Connecting to redirector...");
	if (connect(sock, (struct sockaddr *)&s, sizeof(struct sockaddr_in)) < 0) {
		errorreport("connect");
		return -1;
	}

	if ((len = read(sock, hostbuf, 256)) < 0) {
		errorreport("read");
		return -1;
	}

	hostbuf[len] = 0;
	if ((newline = strchr(hostbuf, '\n')) != NULL) 
		*newline = 0;

	close(sock);
	
	nap_status("Napster server redirect:");
	nap_status(hostbuf);
	/* why is there a "wait" response, and a "busy" response!? 
	 * I'll assume that "wait" means it'll be unbusy soon, while
	 * "busy" means it's busy for a longer time... not a very
	 * good assumtion, though.  */
	if (strncmp(hostbuf, "wait", 4) == 0) {
		nap_status("Server too busy.  Try again in a few seconds.");
		return -1;
	} else if (strncmp(hostbuf, "busy", 4) == 0) {
		nap_status("Server too busy.  Try again later.");
		return -1;
	}
	
	return 0;
}


int nap_bindlocal(int port) {
	struct sockaddr_in bindaddr;
	int optval = 1;

	if (nap_socklocal != -1) return 0;

	if ((nap_socklocal = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		errorreport("socket");
		return -1;
	}

	if (setsockopt(nap_socklocal, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval)) < 0) {
		errorreport("setsockopt");
		return -1;
	}
	memset(&bindaddr, 0, sizeof(struct sockaddr_in));
	bindaddr.sin_family = AF_INET;
	bindaddr.sin_port = htons(port);
	bindaddr.sin_addr.s_addr = 0;
	if (bind(nap_socklocal, (struct sockaddr*)&bindaddr, sizeof(struct sockaddr_in)) < 0) {
		errorreport("bind");
		return -1;
	}

	if (listen(nap_socklocal, 5) < 0) {
		errorreport("bind");
		return -1;
	}
	return 0;
}

int nap_connect(struct sockaddr_in *host) {
	if (nap_socknap != -1) return 0;

	if ((nap_socknap = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		errorreport("socket");
		return -1;
	}

	if (connect(nap_socknap, (struct sockaddr *)host, sizeof(struct sockaddr_in)) < 0) {
		errorreport("connect");
		return -1;
	}
	return 0;
}

int nap_sendlogin(char *user, char *pass, int port, enap_conntype conntype) {
	if (nap_sendpacketf(nap_socknap, NM_LOGIN, "%s %s %d %s %d",
			user, pass, port, NAP_VER, conntype) < 0) 
		return -1;
	return 0;
}

int nap_sendnewuser(char *user) {
	if (nap_sendpacketf(nap_socknap, NM_NEWUSER, "%s", user) < 0) return -1;
	return 0;
}

int nap_sendnewuserlogin(char *user, char *pass, int port, enap_conntype conntype,
		char *email) {
	if (nap_sendpacketf(nap_socknap, NM_NEWUSERLOGIN, "%s %s %d %s %d %s", user, pass, 
				port, NAP_VER, conntype, email) < 0) return -1;
	return 0;
}


#if 0
int nap_login(, int create_user) {
	int len;
	char buf[512];
	snap_packet *pack;

	if (create_user) {
		nap_status("Creating user...");
		len = sprintf(buf, "%s", user);
		if (nap_sendpacket(NM_CREATEUSER, len, buf, nap_socknap) < 0) return -1;
	}

	nap_status("Sending login...");

	nap_status("Waiting for login response...");
	if ((pack = nap_readpacket(nap_socknap)) == 0) return -1;
	/* (pack->msg == NM_LOGINRESP) is a successful login */
	if (pack->msg == NM_ERROR) { /* failed login? */
		nap_status(pack->txt); 
		return -1;
	}
	free(pack);

	nap_status("Logged in.");
	return 0;
}
#endif

void nap_logout(void) {
	close(nap_socknap);
	close(nap_socklocal);
	nap_socknap = nap_socklocal = -1;
}

void nap_dumppacket(snap_packet *p) {
	int i;
	char *text = NULL;
	char buf[8192];
	char *b = buf;

	for (i = 0; msglookup[i].msg != -1; i++) {
		if (msglookup[i].msg == p->msg) {
			text = msglookup[i].text;
			break;
		}
	}
	if (p->len == 0) {
		b += sprintf(b, "<no content>");
	} else {
		b += snprintf(b, 8191, "[%d]%s - ", p->msg, (text ? text : ""));
	}
	snprintf(b, 8191-strlen(buf), p->txt);
	/*for (i = 0; i < p->len; i++) {
		if (isprint(p->txt[i])) {
			b += sprintf(b, "%c", p->txt[i]);
		} else {
			b += sprintf(b, "[%3d]", p->txt[i]);
		}
	}*/
	nap_status(buf);
}

/* this is an implementation of the MSG_WAITALL flag, basically. */
static int recv_waitall(int s, void *buf, size_t len, int flags) {
	int result;
	size_t bytes = 0;
	while (bytes < len) {
		if ((result = recv(s, buf+bytes, len-bytes, flags)) < 0) {
			return result;
		}
		bytes += result;
	}
	return bytes;
}

snap_packet* nap_readpacket(int sock) {
	snap_packet *p = NULL;
	unsigned short plen, nlen, nmsg;

	if (recv_waitall(sock, &nlen, 2, 0) < 0) {
		errorreport("recv1"); return NULL;
	}
	plen = nap_naptohs(nlen);

	/* new packet: allocate space for header, data, and a terminating NULL */
	if ((p = malloc(sizeof(snap_packet) + plen + 1)) == NULL) {
		errorreport("malloc"); return NULL;
	}
	p->len = plen;

	if (recv_waitall(sock, &nmsg, 2, 0) < 0) {
		errorreport("recv2"); return NULL;
	}
	p->msg = nap_naptohs(nmsg);

	/* packet data to recv() */
	if (recv_waitall(sock, p->txt, p->len, 0) < 0) {
		errorreport("recv3"); return NULL;
	}
	
	/* packet complete */
	p->txt[p->len] = 0; /* null terminate */
	return p;
}

static int recv_nowait(int sock, void* buf, int len, int* ofs) {
	int l = 0;
	if (*ofs < len) {
		if ((l = recv(sock, buf+*ofs, len-*ofs, 0)) < 0) {
			errorreport("recv"); return -1;
		}
		*ofs += l;
		if (*ofs < len) return -1;
	}
	return 0;
}

/* FIXME this is sort of a hack, but oh well. 
 * this function won't block as long as there's *some* data available on the socket.
 * it reads all of the available data, and returns a malloc'd snap_packet only when
 * it has read a full packet. */
snap_packet* nap_readpacket_nonblock(int sock) {
	static snap_packet* p = NULL;
	static int lenbytes = 0, msgbytes = 0, databytes = 0;
	static short plen = 0, pmsg = 0;

	if (lenbytes < 2) {
		if (recv_nowait(sock, &plen, 2, &lenbytes) < 0) return NULL;
		plen = nap_naptohs(plen);
	}

	if (msgbytes == 0) {
		/* new packet: allocate space for header, data, and a terminating NULL */
		if ((p = malloc(sizeof(snap_packet) + plen + 1)) == NULL) {
			errorreport("malloc"); return NULL;
		}
		p->len = plen;
	}

	if (msgbytes < 2) {
		if (recv_nowait(sock, &pmsg, 2, &msgbytes) < 0) return NULL;
		p->msg = nap_naptohs(pmsg);
	}

	if (databytes < plen) {
		if (recv_nowait(sock, p->txt, plen, &databytes) < 0) return NULL;
	}

	/* if it gets here, the packet is complete. reset statics, return packet. */
	p->txt[plen] = 0;
	lenbytes = msgbytes = databytes = 0;
	return p;
}

int nap_sendpacketf(int sock, short int pmsg, char* txt, ...) {
	char buf[2048]; /* FIXME how large? */
	va_list ap;
	int len, ret = 0;

	va_start(ap, txt);
	len = vsnprintf(buf, 2048, txt, ap);
	if (len > 2040) printf("lnap warning: almost overflowing sendpacketf!\n");
	ret = nap_sendpacket(sock, pmsg, len, buf);
	va_end(ap);
	return ret;
}

int nap_sendpacket(int sock, short int pmsg, short int plen, char* txt) {
	char *buf;
	int len;
	snap_packet *p;
	
	if ((buf = malloc(4+plen+1)) == NULL) {
		errorreport("malloc"); return -1;
	}
	p = (snap_packet*)buf;
	p->len = nap_htonaps(plen);
	p->msg = nap_htonaps(pmsg);
	strcpy(p->txt, txt);

	if ((len = send(sock, buf, plen+4, 0)) < 0) {
		errorreport("send");
		return -1;
	}
	free(buf);

	return 0;
}

enap_conntype nap_str_to_conntype(char *ct) {
	int i;
	for (i = 0; nap_conntypes[i] != NULL; i++) {
		if (strcmp(nap_conntypes[i], ct) == 0) return i;
	}
	return 0;
}


