/* gnome-napster, a GNOME Napster client.
 * Copyright (C) 1999 Evan Martin <eeyem@u.washington.edu>
 */
#include <gtk/gtk.h>
#include <gnome.h>
#include <pthread.h>
#include "lnap.h"
#include "util.h"
#include "login.h"
#include "ui.h"
#include "gconfig.h"

#include "../pixmaps/napster.xpm"

#define CONNECT_step 3

static GtkWidget *inputdlg = NULL;
static GtkWidget *conndlg, *label, *progress, *anim; 

static pthread_t thread = 0;
G_LOCK_DEFINE_STATIC(threadmutex);

static int pmax, relog_tag, relog_delay;
static gboolean newuser=FALSE, autoretry=FALSE;

slogininfo g_logininfo = {
	FALSE, NULL, NULL, NULL
};

static int login_proc(gpointer d);
static int newuser_proc(gpointer d);
static gint retry_cb(gpointer d);

/* first, generic connection stuff */
static void connect_status(char *text, gpointer d) {
	G_LOCK(threadmutex);
	gdk_threads_enter();
	if (conndlg && label) {
		gtk_label_set_text(GTK_LABEL(label), text);
		gtk_progress_set_value(GTK_PROGRESS(progress), 
			gtk_progress_get_value(GTK_PROGRESS(progress))+1);
		/*printf("value is %f\n", 
			gtk_progress_get_value(GTK_PROGRESS(progress)));*/
	}
	printcn(text);
	gdk_threads_leave();
	G_UNLOCK(threadmutex);
}

static void init_connect_dlg() {
	gtk_window_set_title(GTK_WINDOW(conndlg), "Connecting to server...");
	gnome_animator_start(GNOME_ANIMATOR(anim));
	gtk_progress_configure(GTK_PROGRESS(progress), 1, 0, pmax);
	relog_tag = 0;
	relog_delay = gnome_config_get_int(CONFIG_PREFIX "/Login/RetryDelay=30");
}

static void* connect_proc(void* d) {
	statusfunc oldstatus; void *olddata;
	int loginok;
	struct sockaddr_in addr;
	char *hostlookup = NULL;
	char hostbuf[256];

	nap_getstatusfunc(&oldstatus, &olddata);
	nap_setstatusfunc(connect_status, NULL);

for (loginok = FALSE; loginok != TRUE; loginok = TRUE) { /* run once */
	if (g_logininfo.serv == NULL) {
		connect_status("Getting best host...", NULL);
		hostlookup = gnome_config_get_string(CONFIG_PREFIX "/Network/Redirector");
		if (nap_getbesthost(hostbuf, 
					hostlookup ? hostlookup : NAP_SEARCHHOST_DEF) < 0) break;
		if (g_logininfo.serv) g_free(g_logininfo.serv);
		g_logininfo.serv = g_strdup(hostbuf);
	} 

	if (nap_str_to_sockaddr_in(g_logininfo.serv, &addr) < 0) break;

	connect_status("Connecting to server...", NULL);
	if (nap_connect(&addr) < 0) break;

	if (newuser) {
		if (newuser_proc(NULL) < 0) break;
	} else {
		if (login_proc(NULL) < 0) break;
	}
}
	if (hostlookup) g_free(hostlookup);

	/* no matter what, be sure to restore status function, etc. */
	nap_setstatusfunc(oldstatus, olddata);

	FUNCPRINT("locking.");
	G_LOCK(threadmutex);
	FUNCPRINT("locked.");
	thread = 0;
	G_UNLOCK(threadmutex);
	FUNCPRINT("unlocking.");

	if (loginok) {
		connect_status("Connected.", NULL);
	} else {
		nap_logout();
		gdk_beep();
		T( gnome_animator_stop(GNOME_ANIMATOR(anim)); )
		if (autoretry) {
			/* start a relog */
			gdk_threads_enter();
			gtk_window_set_title(GTK_WINDOW(conndlg), "Delaying before retry...");
			gtk_progress_set_value(GTK_PROGRESS(progress), 0);
			gtk_progress_configure(GTK_PROGRESS(progress), 0, 0, relog_delay+1);
			gdk_threads_leave();
			relog_tag = gtk_timeout_add(1000, retry_cb, NULL);
			retry_cb(NULL);
		} 
	}

	FUNCPRINT("ending");
	return NULL;
}

static int conndlgclose_cb(GtkWidget *w, gpointer d) {
	G_LOCK(threadmutex);
	if (relog_tag != 0) gtk_timeout_remove(relog_tag);
	relog_tag = 0;
	gnome_animator_stop(GNOME_ANIMATOR(anim));
	if (thread != 0) {
		/* thread still running? stop it */
		printf("cancelling thread...\n");
		printcn("Cancelled.");
		pthread_cancel(thread);
		thread = 0;
	}
	G_UNLOCK(threadmutex);
	return FALSE; /* close */
}

static gint retry_cb(gpointer d) {
	char buf[100];
	/*FUNCPRINT("delay=%d", relog_delay);*/
	G_LOCK(threadmutex);
	if (relog_delay > 0) {
		sprintf(buf, "Trying again in %d second%s...", 
				relog_delay, (relog_delay == 1) ? "" : "s");
		T( gtk_label_set_text(GTK_LABEL(label), buf);
		gtk_progress_set_value(GTK_PROGRESS(progress), 
				gtk_progress_get_value(GTK_PROGRESS(progress))+1); )
		relog_delay--;
		G_UNLOCK(threadmutex);
		return TRUE; /* continue running */
	} else {
		/* retry login */
		T( init_connect_dlg(); )
		pthread_create(&thread, NULL, &connect_proc, NULL);
		G_UNLOCK(threadmutex);
		return FALSE; /* stop running */
	}
}

static void connect_begin() {
	GtkWidget *hbox, *vbox;
	gchar *animfile;

	conndlg = gnome_dialog_new(
		newuser ? "Creating new user..." : "Logging in...",
		GNOME_STOCK_BUTTON_OK,
		GNOME_STOCK_BUTTON_CANCEL,
		NULL);

	hbox = gtk_hbox_new(FALSE, 10);
		anim = gnome_animator_new_with_size(91, 68);
		gnome_animator_set_loop_type(GNOME_ANIMATOR(anim), 
				GNOME_ANIMATOR_LOOP_RESTART);
		if ((animfile = pixmap_file("gnome-napster-animate.png")) == NULL) {
			g_warning("gnome-napster-animate.png not found!");
		} else {
			gnome_animator_append_frames_from_file(GNOME_ANIMATOR(anim), 
				animfile, 0, 0,
				150, 91);
			g_free(animfile);
		}
	gtk_box_pack_start(GTK_BOX(hbox), anim, FALSE, FALSE, 0);
		vbox = gtk_vbox_new(FALSE, 10);
			label = gtk_label_new("Logging in to Napster:");
		gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
			label = gtk_label_new("");
		gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
			progress = gtk_progress_bar_new();
		gtk_box_pack_start(GTK_BOX(vbox), progress, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(conndlg)->vbox), hbox, FALSE, FALSE, 0);

	gnome_dialog_set_sensitive(GNOME_DIALOG(conndlg), 0, FALSE); /* ok is disabled */
	gnome_dialog_set_parent(GNOME_DIALOG(conndlg), GTK_WINDOW(app));
	gnome_dialog_set_close(GNOME_DIALOG(conndlg), TRUE); 
		/* clicking a button closes the dlg */
	gtk_signal_connect(GTK_OBJECT(conndlg), "close",
			GTK_SIGNAL_FUNC(conndlgclose_cb), NULL);

	/* set initial values */
	init_connect_dlg();
	/* show it first, so thread won't be upset */
	gtk_widget_show_all(conndlg); 
	while (gtk_events_pending())
		gtk_main_iteration();

	pthread_create(&thread, NULL, &connect_proc, NULL);
}

static void connect_enable_ok() {
	/* make cancel insensitive, and close sensitive */
	gnome_dialog_set_sensitive(GNOME_DIALOG(conndlg), 1, FALSE);
	gnome_dialog_set_sensitive(GNOME_DIALOG(conndlg), 0, TRUE);
	/* and "OK" default */
	gnome_dialog_set_default(GNOME_DIALOG(conndlg), 0);
}


/* and now the dialog */

void login_relog_cb(GtkWidget *w, gpointer d) {
	if (g_logininfo.connected) login_loginout_cb(w, d); /* this call will log out */
	login_loginout_cb(w, d); /* and this will log in */
}

static void manualserver_cb(GtkWidget *w, gpointer d) {
	gtk_widget_set_sensitive(GTK_WIDGET(d), 
			gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)));
}

void login_loginout_cb(GtkWidget *w, gpointer d) {
	GtkWidget *mainbox, *vbox, *hbox, *euser, *cnewuser, *epass;
	GtkWidget *pix;
	GtkWidget *cserver, *cautoretry, *cmanualserver;
	int i;
	gchar *val, *serv;

	G_LOCK(threadmutex);
	if (thread != 0) {
		FUNCPRINT("called while already connecting!");
		return; /* if thread is running, return */
	}
	G_UNLOCK(threadmutex);

	if (g_logininfo.connected) { 
		nap_logout();
		g_logininfo.connected = FALSE;
		ui_loginstatus();
		return;
	}

	/* login: construct a login dialog box */
	inputdlg = gnome_dialog_new("Napster Login",
		GNOME_STOCK_BUTTON_OK,
		GNOME_STOCK_BUTTON_CANCEL,
		NULL);
	gtk_window_set_default_size(GTK_WINDOW(inputdlg), 500, 0);
	
	mainbox = gtk_hbox_new(FALSE, 10);
	pix = gnome_pixmap_new_from_xpm_d(napster_xpm);
	gtk_box_pack_start(GTK_BOX(mainbox), pix, FALSE, FALSE, 0);

	vbox = gtk_vbox_new(FALSE, 0);
		euser = gtk_entry_new();
		if ((val = gnome_config_get_string(CONFIG_PREFIX "/Login/UserName")) != NULL) {
			gtk_entry_set_text(GTK_ENTRY(euser), val);
			g_free(val);
		}
	gtk_box_pack_start(GTK_BOX(vbox), 
			hbox_new_with_label("User Name:", euser, TRUE), 
			TRUE, FALSE, 0);

		epass = gtk_entry_new();
		if ((val = gnome_config_private_get_string(CONFIG_PREFIX 
						"/Login/Password")) != NULL) {
			gtk_entry_set_text(GTK_ENTRY(epass), val);
			g_free(val);
		}
		gtk_entry_set_visibility(GTK_ENTRY(epass), FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), 
			hbox_new_with_label("Password:", epass, TRUE), 
			TRUE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(mainbox), vbox, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(inputdlg)->vbox), mainbox, FALSE, FALSE, 0);

		hbox = gtk_hbox_new(FALSE, 10);
			cautoretry = gtk_check_button_new_with_label("Retry login until successful");
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cautoretry), 
					gnome_config_get_bool(CONFIG_PREFIX "/Login/AutoRetry=false"));

		gtk_box_pack_start(GTK_BOX(hbox), cautoretry, FALSE, FALSE, 0);
			cnewuser = gtk_check_button_new_with_label("New User");
		gtk_box_pack_start(GTK_BOX(hbox), cnewuser, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(inputdlg)->vbox), hbox, FALSE, FALSE, 0);
	
		hbox = gtk_hbox_new(FALSE, 0);
		cmanualserver = gtk_check_button_new_with_label("Manually specify server:");
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cmanualserver), FALSE);
		gtk_box_pack_start(GTK_BOX(hbox), cmanualserver, FALSE, FALSE, 0);

		cserver = gnome_entry_new("server");
		gnome_entry_load_history(GNOME_ENTRY(cserver));

		gtk_signal_connect(GTK_OBJECT(cmanualserver), "toggled",
				GTK_SIGNAL_FUNC(manualserver_cb), cserver);
		manualserver_cb(cmanualserver, cserver);
		gtk_box_pack_start(GTK_BOX(hbox), cserver, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(inputdlg)->vbox), hbox,
			FALSE, FALSE, 0);


	gtk_widget_show_all(GNOME_DIALOG(inputdlg)->vbox);

	gnome_dialog_set_parent(GNOME_DIALOG(inputdlg), GTK_WINDOW(app));
	gnome_dialog_set_default(GNOME_DIALOG(inputdlg), 0); /* OK is default */
	gnome_dialog_close_hides(GNOME_DIALOG(inputdlg), TRUE);

	i = gnome_dialog_run_and_close(GNOME_DIALOG(inputdlg));
	if (i != 0) { /* didn't click ok */
		gtk_widget_destroy(inputdlg);
		return; /* stop login */
	}

	if (g_logininfo.user) g_free(g_logininfo.user);
	g_logininfo.user = g_strdup(gtk_entry_get_text(GTK_ENTRY(euser)));
	if (g_logininfo.pass) g_free(g_logininfo.pass);
	g_logininfo.pass = g_strdup(gtk_entry_get_text(GTK_ENTRY(epass)));
	serv = gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(cserver))));
	newuser = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cnewuser));
	autoretry = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cautoretry));

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cmanualserver))) {
		pmax = CONNECT_step+2+1;
		if (g_logininfo.serv) g_free(g_logininfo.serv);
		g_logininfo.serv = g_strdup(serv); 
		gnome_entry_prepend_history(GNOME_ENTRY(cserver), TRUE, serv);
	} else {
		/* autochoose server */
		pmax = NAP_getbesthost_step+1+CONNECT_step+2+1;
		if (g_logininfo.serv) g_free(g_logininfo.serv);
		g_logininfo.serv = NULL;
	}

	gnome_entry_save_history(GNOME_ENTRY(cserver));

	connect_begin();
}

static int login_proc(gpointer d) {
	snap_packet *p;

	/* gtk_widget_destroy(d); FIXME why can't i destroy the dialog? */
	connect_status("Sending login information...", NULL);
	nap_sendlogin(g_logininfo.user, g_logininfo.pass, 
			gnome_config_get_int(CONFIG_PREFIX "/Login/DataPort=6699"),
			gnome_config_get_int(CONFIG_PREFIX "/Login/ConnectionType=0"));
	connect_status("Waiting for login response...", NULL);

	p = nap_readpacket(nap_socknap);
	g_return_val_if_fail(p != NULL, -1);

	T( gnome_animator_stop(GNOME_ANIMATOR(anim)); )
	if (p->msg == NM_LOGINRESP) {
		FUNCPRINT("email=%s", p->txt);
		g_logininfo.connected = TRUE;
		connect_status("Login successful.", NULL);
		T( connect_enable_ok(); ui_loginstatus(); )
	} else if (p->msg == NM_ERROR || p->msg == NM_USERDOESNTEXIST) {
		FUNCPRINT("login error=%s", p->txt);
		connect_status(p->txt, NULL);
		nap_logout();
		free(p);
		return -1;
	} else {
		FUNCPRINT("unexpected_msg=%d,%s", p->msg, p->txt);
		free(p);
		return -1;
	}
	free(p);
	return 0;
}

static int newuser_proc(gpointer d) {
	snap_packet *p;

	FUNCPRINT("d=%p", d);
	connect_status("Sending new user request...", NULL);
	nap_sendnewuser(g_logininfo.user);
	connect_status("Waiting for response...", NULL);

	p = nap_readpacket(nap_socknap);
	g_return_val_if_fail(p != NULL, -1);
	if (p->msg == NM_NEWUSER_SUCCESS) {
		connect_status("New user successfully created.", NULL);
		T( connect_enable_ok(); )
	} else if (p->msg == NM_NEWUSER_EXISTS) {
		connect_status(p->txt, NULL);
		free(p);
		return -1;
	} else if (p->msg == NM_NEWUSER_INVALIDNICK) {
		connect_status(p->txt, NULL);
		free(p);
		return -1;
	}
	free(p);
	if (login_proc(NULL) < 0) return -1;
	return 0;
}
