/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998,1999 Bert Vermeulen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <gtk/gtk.h>

#include <lber.h>
#include <ldap.h>

#include <string.h>

#include "common.h"
#include "errorchain.h"
#include "configfile.h"
#include "query.h"
#include "detail.h"
#include "util.h"
#include "debug.h"
#include "browse.h"

#include "line.xpm"
#include "textarea.xpm"

extern GtkWidget *mainwin;
extern struct gq_config config;
extern GString *cur_dn;

void view_entry(struct resultset *set)
{
     LDAP *ld;
     LDAPMessage *res, *e;
     BerElement *ptr;

     GtkWidget *dnwin, *vbox1, *vbox2, *hbox1, *label, *okbutton;
     GtkWidget *scwin, *table;

     struct ldapserver *server;
     int detail_context, msg, row, i;
     char *attr, **vals;

     detail_context = error_new_context("Error getting entry");

     dnwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_widget_set_usize(dnwin, 400, 310);
     gtk_window_set_title(GTK_WINDOW(dnwin), set->dn);
     vbox1 = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox1), 5);
     gtk_widget_show(vbox1);
     gtk_container_add(GTK_CONTAINER(dnwin), vbox1);

     scwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_container_border_width(GTK_CONTAINER(scwin), 5);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(scwin);
     gtk_box_pack_start(GTK_BOX(vbox1), scwin, TRUE, TRUE, 5);
     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scwin), vbox2);

     /* table will self-expand */
     table = gtk_table_new(2, 2, FALSE);
     gtk_container_border_width(GTK_CONTAINER(table), 5);
     gtk_table_set_row_spacings(GTK_TABLE(table), 1);
     gtk_table_set_col_spacings(GTK_TABLE(table), 10);
     gtk_widget_show(table);
     gtk_box_pack_start(GTK_BOX(vbox2), table, FALSE, TRUE, 0);

     /* dn in first row */
     label = gtk_label_new("dn");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
     label = gtk_label_new(set->dn);
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);

     row = 1;

     /* query server for full entry */
     server = server_by_name(config.cur_servername);
     if(!server) {
	  error_push(detail_context, "Oops! No server found!");
	  error_flush(detail_context);
	  return;
     }

     set_busycursor();

     if( (ld = open_connection(server) ) == NULL) {
	  set_normalcursor();
	  statusbar_msg("");
	  return;
     }

     msg = ldap_search_s(ld, set->dn, LDAP_SCOPE_BASE,
			 "objectclass=*", NULL, 0, &res);
     if(msg != LDAP_SUCCESS) {
	  set_normalcursor();
	  statusbar_msg("");
	  error_push(detail_context, ldap_err2string(msg));
	  error_flush(detail_context);
	  return;
     }

     e = ldap_first_entry(ld, res);
     if(e != NULL) {
	  for(attr = ldap_first_attribute(ld, e, &ptr); attr != NULL;
	      attr = ldap_next_attribute(ld, e, ptr)) {

	       vals = ldap_get_values(ld, e, attr);
	       if(vals) {
		    for(i = 0; vals[i] != NULL; i++) {
			 if(i == 0) {
			      /* first value: show attribute in first column */
			      label = gtk_label_new(attr);
			      gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
			      gtk_widget_show(label);
			      gtk_table_attach_defaults(GTK_TABLE(table), label,
							0, 1, row, row + 1);
			 }
			 label = gtk_label_new(vals[i]);
			 gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
			 gtk_widget_show(label);
			 gtk_table_attach_defaults(GTK_TABLE(table), label,
						   1, 2, row, row + 1);
			 row++;
		    }

	       }

	  }

     }
     else {
	  /* this entry doesn't seem to have any objectclasses in it */
	  label = gtk_label_new(NO_RESULTS_MSG);
	  gtk_widget_show(label);
	  gtk_box_pack_end(GTK_BOX(vbox2), label, FALSE, FALSE, 20);
     }

     ldap_msgfree(res);

     set_normalcursor();
     statusbar_msg("");

     hbox1 = gtk_hbox_new(TRUE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_end(GTK_BOX(vbox1), hbox1, FALSE, TRUE, 10);

     okbutton = gtk_button_new_with_label("  OK  ");
     gtk_widget_show(okbutton);
     gtk_box_pack_start(GTK_BOX(hbox1), okbutton, FALSE, FALSE, 0);
     gtk_signal_connect_object(GTK_OBJECT(okbutton), "clicked",
			       GTK_SIGNAL_FUNC(gtk_widget_destroy),
			       GTK_OBJECT(dnwin));
     gtk_signal_connect_object(GTK_OBJECT(dnwin), "key_press_event",
			       GTK_SIGNAL_FUNC(close_on_esc),
			       (gpointer) dnwin);
     GTK_WIDGET_SET_FLAGS(okbutton, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(okbutton);

     error_flush(detail_context);

     gtk_widget_show(dnwin);

     close_connection(server, FALSE);

}


/*
 * callback from "Edit"
 */
void edit_entry_values(struct resultset *set)
{
     GtkWidget *edit_window, *vbox;
     struct ldapserver *server;

     edit_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title(GTK_WINDOW(edit_window), set->dn);
     gtk_widget_set_usize(edit_window, 500, 310);
     gtk_widget_show(edit_window);
     gtk_signal_connect_object(GTK_OBJECT(edit_window), "key_press_event",
			       GTK_SIGNAL_FUNC(close_on_esc),
			       (gpointer) edit_window);

     vbox = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox), 5);
     gtk_container_add(GTK_CONTAINER(edit_window), vbox);
     gtk_object_set_data(GTK_OBJECT(vbox), "dn", set->dn);
     gtk_object_set_data(GTK_OBJECT(vbox), "is a popup", "yes");
     gtk_object_set_data(GTK_OBJECT(vbox), "parent window", edit_window);
     gtk_widget_show(vbox);

     if( (server = server_by_name(config.cur_servername)) == NULL)
	  return;

     set_busycursor();

     edit_entry_table(vbox, server, set->dn, TRUE);

     set_normalcursor();

}


/*
 * callback from "Use as template"
 */
void edit_entry_novalues(struct resultset *set)
{
     GtkWidget *edit_window, *vbox;
     struct ldapserver *server;

     edit_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title(GTK_WINDOW(edit_window), "New entry");
     gtk_widget_set_usize(edit_window, 500, 310);
     gtk_widget_show(edit_window);
     gtk_signal_connect_object(GTK_OBJECT(edit_window), "key_press_event",
			       GTK_SIGNAL_FUNC(close_on_esc),
			       (gpointer) edit_window);

     vbox = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox), 5);
     gtk_container_add(GTK_CONTAINER(edit_window), vbox);
     gtk_object_set_data(GTK_OBJECT(vbox), "dn", NULL);
     gtk_object_set_data(GTK_OBJECT(vbox), "is a popup", "yes");
     gtk_object_set_data(GTK_OBJECT(vbox), "parent window", edit_window);
     gtk_widget_show(vbox);

     if( (server = server_by_name(config.cur_servername)) == NULL)
	  return;

     set_busycursor();

     edit_entry_table(vbox, server, set->dn, FALSE);

     set_normalcursor();

}



/*
 * pack edit window for entry into target_vbox
 */
void edit_entry_table(GtkWidget *target_vbox, struct ldapserver *server,
		      char *dn, int show_values)
{
     LDAP *ld;
     LDAPMessage *res, *e;
     BerElement *ptr;
     gpointer is_popup;
     GdkPixmap *icon;
     GdkBitmap *icon_mask;
     GtkWidget *target_window, *vbox2, *hbox1, *hbox2, *label;
     GtkWidget *button, *linebutton, *textareabutton, *arrowbutton;
     GtkWidget *scwin, *table, *inputbox;
     GtkWidget *pixmap;
     int detail_context, msg, row, i, is_objectclass;
     char *attr, **vals, **oldrdn;
     char message[MAX_DN_LEN + 32], newdn[MAX_DN_LEN];

     detail_context = error_new_context("Error getting entry");
     is_popup = gtk_object_get_data(GTK_OBJECT(target_vbox), "is a popup");
     target_window = gtk_object_get_data(GTK_OBJECT(target_vbox),
					 "parent window");

     hbox1 = gtk_hbox_new(FALSE, 5);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(target_vbox), hbox1, FALSE, FALSE, 0);

     /* line button */
     linebutton = gtk_button_new();
     gtk_object_set_data(GTK_OBJECT(linebutton), "transform", "make line");
     GTK_WIDGET_UNSET_FLAGS(linebutton, GTK_CAN_FOCUS);
     gtk_signal_connect(GTK_OBJECT(linebutton), "clicked",
			GTK_SIGNAL_FUNC(rebuild_edit_table),
			(gpointer) target_vbox);

     icon = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(target_vbox)->window,
					 &icon_mask,
					 &target_vbox->style->white,
					 line_xpm);
     pixmap = gtk_pixmap_new(icon, icon_mask);
     gtk_widget_show(pixmap);
     gtk_container_add(GTK_CONTAINER(linebutton), pixmap);
     gtk_widget_show(linebutton);
     gtk_box_pack_start(GTK_BOX(hbox1), linebutton, FALSE, FALSE, 5);

     /* textarea button */
     textareabutton = gtk_button_new();
     gtk_object_set_data(GTK_OBJECT(textareabutton), "transform", "make textarea");
     GTK_WIDGET_UNSET_FLAGS(textareabutton, GTK_CAN_FOCUS);
     gtk_signal_connect(GTK_OBJECT(textareabutton), "clicked",
			GTK_SIGNAL_FUNC(rebuild_edit_table),
			(gpointer) target_vbox);
     icon = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(target_vbox)->window,
					 &icon_mask,
					 &target_vbox->style->white,
					 textarea_xpm);
     pixmap = gtk_pixmap_new(icon, icon_mask);
     gtk_widget_show(pixmap);
     gtk_container_add(GTK_CONTAINER(textareabutton), pixmap);
     gtk_widget_show(textareabutton);
     gtk_box_pack_start(GTK_BOX(hbox1), textareabutton, FALSE, FALSE, 0);

     /* scrolled window with vbox2 inside */
     scwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_container_border_width(GTK_CONTAINER(scwin), 5);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(scwin);
     gtk_box_pack_start(GTK_BOX(target_vbox), scwin, TRUE, TRUE, 5);
     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scwin), vbox2);

     /* table inside vbox2, will self-expand */
     table = gtk_table_new(3, 2, FALSE);

     /* I know this looks stupid, but there's a really good reason for it :-) */
     gtk_object_set_data(GTK_OBJECT(target_vbox), "table", table);
     gtk_object_set_data(GTK_OBJECT(table), "target_vbox", target_vbox);

     gtk_object_set_data(GTK_OBJECT(table), "window", target_window);
     gtk_object_set_data(GTK_OBJECT(table), "vbox", vbox2);
     gtk_container_border_width(GTK_CONTAINER(table), 5);
     gtk_table_set_row_spacings(GTK_TABLE(table), 1);
     gtk_table_set_col_spacings(GTK_TABLE(table), 10);
     gtk_widget_show(table);
     gtk_box_pack_start(GTK_BOX(vbox2), table, FALSE, TRUE, 0);

     /* dn in first row */
     label = gtk_label_new("dn");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 0, 0, 0, 0);
     inputbox = gtk_entry_new();
     gtk_table_attach(GTK_TABLE(table), inputbox, 1, 2, 1, 2,
		      GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
     gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
			       GTK_SIGNAL_FUNC(edit_entry_ok_callback),
			       GTK_OBJECT(target_vbox));
     if(!is_popup && show_values)
	  gtk_signal_connect(GTK_OBJECT(inputbox), "key_press_event",
			     GTK_SIGNAL_FUNC(keypress_inputbox),
			     (gpointer) target_vbox);
     gtk_widget_show(inputbox);
     if(show_values)
	  gtk_entry_set_text(GTK_ENTRY(inputbox), dn);
     else {
	  /* use old dn minus rdn for the new entry */
	  oldrdn = ldap_explode_dn(dn, 0);
	  newdn[0] = 0;
	  for(i = 1; oldrdn[i]; i++) {
	       strcat(newdn, ", ");
	       strcat(newdn,oldrdn[i]);
	  }
	  ldap_value_free(oldrdn);
	  gtk_entry_set_text(GTK_ENTRY(inputbox), newdn);

	  /* focus on the new DN, since it must be changed */
	  gtk_widget_grab_focus(inputbox);
     }

     /* store server, so selecting a new server while editing
	doesn't screw things up */
     gtk_object_set_data(GTK_OBJECT(target_vbox), "servername", server->name);

     if( (ld = open_connection(server) ) == NULL)
	  return;

     /* base search on this entry */
     sprintf(message, "base search on %s", dn);
     statusbar_msg(message);
     msg = ldap_search_s(ld, dn, LDAP_SCOPE_BASE,
			 "objectclass=*", NULL, 0, &res);
     if(msg != LDAP_SUCCESS) {
	  error_push(detail_context, ldap_err2string(msg));
	  error_flush(detail_context);
	  return;
     }

     row = 2;
     e = ldap_first_entry(ld, res);
     if(e != NULL) {
	  for(attr = ldap_first_attribute(ld, e, &ptr); attr != NULL;
	      attr = ldap_next_attribute(ld, e, ptr)) {

	       /* objectclass gets special treatment: always shown */
	       is_objectclass = !strcasecmp(attr, "objectclass");

	       vals = ldap_get_values(ld, e, attr);
	       if(vals) {
		    for(i = 0; vals[i] != NULL &&
			     (!(i && show_values == 0) || is_objectclass); i++) {
			 if(i == 0) {
			      /* first value: show attribute in first column */
			      label = gtk_label_new(attr);
			      gtk_misc_set_alignment(GTK_MISC(label), 0.0, .5);
			      gtk_widget_show(label);
			      gtk_table_attach(GTK_TABLE(table), label,
					       0, 1, row, row + 1,
					       0, GTK_FILL|GTK_EXPAND, 0, 0);
			 }

			 /* input box in second column */
			 /* if value has LFs in it, render as textbox */
			 if(show_values && (strchr(vals[i], '\n') || strlen(vals[i]) > 45)) {
			      inputbox = gtk_text_new(NULL, NULL);
			      gtk_table_attach(GTK_TABLE(table), inputbox,
					       1, 2, row, row + 1,
					       GTK_FILL|GTK_EXPAND,
					       GTK_FILL|GTK_EXPAND, 0, 0);
			      gtk_widget_show(inputbox);

			      if(show_values || !strcasecmp(attr, "objectclass")) {
				   gtk_text_freeze(GTK_TEXT(inputbox));
				   gtk_text_set_editable(GTK_TEXT(inputbox), TRUE);
				   gtk_text_set_point(GTK_TEXT(inputbox), 0);
				   gtk_text_insert(GTK_TEXT(inputbox), NULL, NULL,
						   NULL, vals[i], -1);
				   gtk_text_thaw(GTK_TEXT(inputbox));
			      }
			 }
			 else {
			      inputbox = gtk_entry_new();
			      if(show_values || !strcasecmp(attr, "objectclass"))
				   gtk_entry_set_text(GTK_ENTRY(inputbox), vals[i]);
			      gtk_table_attach(GTK_TABLE(table), inputbox,
					       1, 2, row, row + 1,
					       GTK_FILL|GTK_EXPAND,
					       GTK_FILL|GTK_EXPAND, 0, 0);
			      gtk_widget_show(inputbox);
			 }

			 /* editing existing entry: first non-DN field gets focus */
			 if(show_values && row == 2)
			      gtk_widget_grab_focus(inputbox);

			 gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
						   GTK_SIGNAL_FUNC(edit_entry_ok_callback),
						   GTK_OBJECT(target_vbox));
			 if(!is_popup && show_values)
			      gtk_signal_connect(GTK_OBJECT(inputbox), "key_press_event",
						 GTK_SIGNAL_FUNC(keypress_inputbox),
						 (gpointer) target_vbox);

			 row++;
		    }

		    /* arrow in third column */
		    arrowbutton = gq_new_arrowbutton(target_vbox);
		    gtk_table_attach(GTK_TABLE(table), arrowbutton,
				     2, 3, row - 1, row,
				     0, GTK_EXPAND, 5, 0);

	       }

	  }

     }
     else {
	  /* message -- entry doesn't have any objectclasses in it */
	  label = gtk_label_new(NO_RESULTS_MSG);
	  gtk_widget_show(label);
	  gtk_box_pack_end(GTK_BOX(vbox2), label, FALSE, FALSE, 20);
     }

     ldap_msgfree(res);

     hbox2 = gtk_hbox_new(TRUE, 0);
     gtk_widget_show(hbox2);
     gtk_box_pack_end(GTK_BOX(target_vbox), hbox2, FALSE, TRUE, 10);

     if(is_popup) {
	  button = gtk_button_new_with_label("  OK  ");
	  gtk_widget_show(button);
	  gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
	  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				    GTK_SIGNAL_FUNC(edit_entry_ok_callback),
				    GTK_OBJECT(target_vbox));
	  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	  gtk_widget_grab_default(button);

	  button = gtk_button_new_with_label("  Cancel  ");
	  gtk_widget_show(button);
	  gtk_box_pack_end(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
	  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				    GTK_SIGNAL_FUNC(gtk_widget_destroy),
				    GTK_OBJECT(target_window));
     }
     else {
	  if(show_values) {
	       /* edit entry in browse mode: apply data but don't destroy parent */
	       button = gtk_button_new_with_label("  Apply  ");
	       gtk_widget_set_sensitive(button, FALSE);
	  }
	  else {
	       /* use as template in browse mode */
	       button = gtk_button_new_with_label("  OK  ");
	  }

	  gtk_object_set_data(GTK_OBJECT(target_vbox), "apply button", button);
	  gtk_widget_show(button);
	  gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
	  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				    GTK_SIGNAL_FUNC(edit_entry_ok_callback),
				    GTK_OBJECT(target_vbox));
	  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	  gtk_widget_grab_default(button);

	  /* reload original entry */
	  button = gtk_button_new_with_label("  Refresh  ");
	  if(!show_values)
	       /* refreshing a new entry doesn't make any sense... */
	       gtk_widget_set_sensitive(button, FALSE);
	  gtk_widget_show(button);
	  gtk_box_pack_end(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
	  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				    GTK_SIGNAL_FUNC(refresh_right_pane),
				    NULL);
     }

     statusbar_msg("");
     error_flush(detail_context);

     close_connection(server, FALSE);

}


GtkWidget *gq_new_arrowbutton(GtkWidget *widget)
{
     GtkWidget *newabutton, *arrow;

     newabutton = gtk_button_new();
     gtk_object_set_data(GTK_OBJECT(newabutton), "transform", "add new row");
     GTK_WIDGET_UNSET_FLAGS(newabutton, GTK_CAN_FOCUS);
     gtk_signal_connect(GTK_OBJECT(newabutton), "clicked",
			GTK_SIGNAL_FUNC(rebuild_edit_table),
			(gpointer) widget);
     arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_OUT);
     gtk_widget_show(arrow);
     gtk_container_add(GTK_CONTAINER(newabutton), arrow);
     gtk_widget_show(newabutton);

     return(newabutton);
}


gboolean keypress_inputbox(GtkWidget *inputbox, GdkEventKey *event, GtkWidget *target_vbox)
{
     GtkWidget *apply_button;

     if(!inputbox || !event)
	  return(TRUE);

     if(event->type == GDK_KEY_PRESS && strlen(event->string)) {
	  apply_button = gtk_object_get_data(GTK_OBJECT(target_vbox), "apply button");
	  gtk_widget_set_sensitive(apply_button, TRUE);
     }

     return(FALSE);
}


GtkWidget *widget_by_coord(GList *children, int x, int y)
{
     GtkTableChild *child;

     while(children) {
	  child = children->data;
	  if(child->left_attach == x && child->top_attach == y)
	       return(child->widget);

	  children = children->next;
     }

     return(NULL);
}


void value_from_widget(GtkWidget *inputbox, char **value)
{

     *value = NULL;
     if(!inputbox)
	  return;

     if(GTK_IS_ENTRY(inputbox))
	  *value = gtk_entry_get_text(GTK_ENTRY(inputbox));
     else if(GTK_IS_TEXT(inputbox))
	  *value = gtk_editable_get_chars(GTK_EDITABLE(inputbox), 0, -1);

}


void edit_entry_ok_callback(GtkWidget *target_vbox)
{
     LDAP *ld;
     LDAPMod *mods[MAX_NUM_ATTRIBUTES + 1], *curmod;
     GList *children;
     GNode *node;
     GtkWidget *target_window, *table, *label, *inputbox, *apply_button;
     struct ldapserver *server;
     struct node_entry *entry;
     int i, y, got_inputbox, change_rdn, is_popup;
     int msg, replace_context;
     int cur_ldapmod, numvals_per_attr[MAX_VALS_PER_ATTR + 1];
     char *server_name, *dn, *value, *newdn, **rdn, **newrdn;
     char errstring[256], message[MAX_DN_LEN + 32];
     change_rdn = 0;

     replace_context = error_new_context("Error replacing entry");

     is_popup = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(target_vbox), "is a popup"));
     server_name = gtk_object_get_data(GTK_OBJECT(target_vbox), "servername");
     if( (server = server_by_name(server_name)) == NULL) {
	  error_push(replace_context, "server disappeared!");
	  error_flush(replace_context);
	  return;
     }

     set_busycursor();

     ld = open_connection(server);
     if(!server || !ld) {
	  set_normalcursor();
	  error_flush(replace_context);
	  return;
     }

     table = gtk_object_get_data(GTK_OBJECT(target_vbox), "table");
     children = GTK_TABLE(table)->children;
     dn = gtk_object_get_data(GTK_OBJECT(target_vbox), "dn");

     if(dn) {
	  /* editing an existing entry */

	  /* if dn changed, modrdn it */
	  value_from_widget(widget_by_coord(children, 1, 1), &value);
	  if(value && strcasecmp(dn, value)) {
	       change_rdn = 1;
	       rdn = ldap_explode_dn(dn, 0);
	       newrdn = ldap_explode_dn(value, 0);

	       /* make sure only the first component changed,
		  since anything else would be a move */
	       /* which would be nice to have, actually */
	       for(i = 1; rdn[i]; i++) {
		    if(!rdn[i] || !newrdn[i] || strcasecmp(rdn[i], newrdn[i])) {
			 sprintf(errstring, "Can only change the RDN of this entry (%s)",
				 rdn[0]);
			 error_push(replace_context, errstring);
			 change_rdn = 0;
			 break;
		    }
	       }

	       if(change_rdn) {

		    sprintf(message, "modifying RDN to %s", newrdn[0]);
		    statusbar_msg(message);
		    /* this isn't in RFC 1823... */
		    msg = ldap_modrdn2_s(ld, dn, newrdn[0], 1);
		    if(msg == LDAP_SUCCESS) {
			 newdn = value;
		    }
		    else {
			 sprintf(errstring, "ldap_modrdn2_s(): %s", ldap_err2string(msg));
			 error_push(replace_context, errstring);

			 /* modrdn failed, but we can still submit any other changes.
			    Just make sure they're done on the old DN then... */
			 change_rdn = 0;
		    }

	       }

	       ldap_value_free(newrdn);
	       ldap_value_free(rdn);

	  }

     }
     else {
	  /* adding a new entry */

	  inputbox = widget_by_coord(children, 1, 1);
	  value_from_widget(inputbox, &newdn);
     }



     /* prepare LDAPMod for modifications */
     cur_ldapmod = 0;
     got_inputbox = 1;
     curmod = NULL;
     for(y = 2; got_inputbox; y++) {
	  label = widget_by_coord(children, 0, y);
	  if(label && cur_ldapmod < MAX_NUM_ATTRIBUTES) {
	       /* terminate previous attribute's values array, if any */
	       if(curmod && curmod->mod_values)
		    curmod->mod_values[numvals_per_attr[cur_ldapmod - 1]] = NULL;

	       /* new attribute */
	       curmod = MALLOC(sizeof(LDAPMod), "LDAPMod");
	       curmod->mod_op = LDAP_MOD_REPLACE;
	       curmod->mod_type = GTK_LABEL(label)->label;
	       curmod->mod_values = MALLOC(sizeof(char *) * MAX_VALS_PER_ATTR,
					   "LDAPMod.mod_values");
	       curmod->mod_values[0] = NULL;
	       numvals_per_attr[cur_ldapmod] = 0;
	       mods[cur_ldapmod] = curmod;
	       cur_ldapmod++;
	  }

	  inputbox = widget_by_coord(children, 1, y);
	  if(inputbox && (curmod || y == 1)) {

	       value_from_widget(inputbox, &value);
	       if(!value) {
		    sprintf(errstring, "unknown field type on row %d", y);
		    error_push(replace_context, errstring);
		    value = "";
	       }

	       if( (strlen(value) >= 0) && numvals_per_attr[cur_ldapmod - 1] < MAX_VALS_PER_ATTR)
		    curmod->mod_values[numvals_per_attr[cur_ldapmod - 1]++] = value;

	  }
	  else
	       got_inputbox = 0;

	  if(numvals_per_attr[cur_ldapmod - 1] == 0) {
	       /* attributes without values need special treatment here... */

	       if(dn) {
		    /* editing an existing entry: change mod_op to LDAP_MOD_DELETE */
		    FREE(curmod->mod_values, "LDAPMod.mod_values");
		    curmod->mod_values = NULL;
		    curmod->mod_op = LDAP_MOD_DELETE;
	       }
	       else {
		    /* adding a new entry: don't submit this attribute at all */
		    FREE(curmod->mod_values, "LDAPMod.mod_values");
		    FREE(curmod, "LDAPMod");
		    curmod = NULL;
		    cur_ldapmod--;
	       }

	  }

     }

     /* terminate pointer array and last attribute's values array */
     mods[cur_ldapmod] = NULL;
     if(curmod && curmod->mod_values && cur_ldapmod)
	  curmod->mod_values[numvals_per_attr[cur_ldapmod - 1]] = NULL;


     if(dn) {
	  /* submit modifications */

	  /* if RDN changed, submit modifications to the new DN */
	  if(change_rdn) {
	       sprintf(message, "modifying %s", newdn);
	       statusbar_msg(message);
	       msg = ldap_modify_s(ld, newdn, mods);

	       if(!is_popup) {
		    /* change entry in node tree */
		    node = node_by_cur_dn();
		    if( (entry = (struct node_entry *) node->data)) {
			 free(entry->dn);
			 entry->dn = g_strdup(newdn);

			 /* refresh button needs this */
			 g_string_assign(cur_dn, entry->dn);
			 gtk_object_set_data(GTK_OBJECT(target_vbox), "dn", newdn);
		    }
	       }
	  }
	  else {
	       sprintf(message, "modifying %s", dn);
	       statusbar_msg(message);
	       msg = ldap_modify_s(ld, dn, mods);
	  }

	  if(msg != LDAP_SUCCESS) {
	       sprintf(errstring, "ldap_modify_s(): %s", ldap_err2string(msg));
	       error_push(replace_context, errstring);
	  }
	  else {
	       sprintf(message, "modified %s", change_rdn ? newdn : dn);
	       statusbar_msg(message);
	  }

     }
     else {
	  /* submit new entry */

	  sprintf(message, "adding %s", newdn);
	  statusbar_msg(message);
	  msg = ldap_add_s(ld, newdn, mods);

	  if(msg != LDAP_SUCCESS) {
	       sprintf(errstring, "ldap_add_s(): %s", ldap_err2string(msg));
	       error_push(replace_context, errstring);
	  }
	  else {
	       sprintf(message, "added %s", newdn);
	       statusbar_msg(message);
	  }

     }

     /* free LDAPMod stuff */
     for(i = 0; i < cur_ldapmod; i++) {
	  if(mods[i]) {
	       if(mods[i]->mod_values)
		    FREE(mods[i]->mod_values, "LDAPMod.mod_values");
	       FREE(mods[i], "LDAPMod");
	  }
     }

     set_normalcursor();
     error_flush(replace_context);
     close_connection(server, FALSE);

     /* destroy edit popup */
     if(is_popup) {
	  target_window = gtk_object_get_data(GTK_OBJECT(target_vbox), "parent window");
	  gtk_widget_destroy(target_window);
     }
     else if(dn) {
	  /* browser mode, entry was edited. */
	  apply_button = gtk_object_get_data(GTK_OBJECT(target_vbox), "apply button");
	  gtk_widget_set_sensitive(apply_button, FALSE);

     }

}


void rebuild_edit_table(GtkWidget *button, GtkWidget *window)
{
     GtkWidget *vbox, *widget, *inputbox, *label, *newabutton;
     GtkTable *oldtable, *target_vbox;
     GtkWidget *newtable, *focusbox;
     GtkTableChild *child;
     GList *children;
     gchar *oldinput;
     char *transform;
     int i, xtype, row, x, y, ny;

     struct transform_types types[] = {
	  { ADD_NEW_ROW,       "add new row" },
	  { MAKE_LINE,         "make line" },
	  { MAKE_TEXTAREA,     "make textarea" },
	  { 0,                 "" }
     };


     row = i = 0;
     widget = focusbox = NULL;
     oldtable = gtk_object_get_data(GTK_OBJECT(window), "table"); 
     target_vbox = gtk_object_get_data(GTK_OBJECT(oldtable), "target_vbox");
     vbox = gtk_object_get_data(GTK_OBJECT(oldtable), "vbox");
     children = GTK_TABLE(oldtable)->children;
     transform = gtk_object_get_data(GTK_OBJECT(button), "transform");

     while(types[i].num && strcmp(transform, types[i].string))
	  i++;
     xtype = types[i].num;

     if(!xtype) {
	  /* this shouldn't happen at all */
	  error_popup("Boom", "rebuild_edit_table(): transform type not found");
	  return;
     }

     /* when adding a new row, look for the arrowbutton
      * when changing an entrybox to line/textarea, look for focus */
     while(children) {
	  child = children->data;
	  widget = child->widget;
	  if(xtype == ADD_NEW_ROW) {
	       if(widget == button)
		    row = child->top_attach;
	  }
	  else {
	       /* MAKE_LINE or MAKE_TEXTAREA */
	       if(GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(widget)))
		    row = child->top_attach;
	  }
	  if(row)
	       break;
	  children = children->next;
     }

     if(row == 0)
	  /* row not set -- only happens when doing a line/textarea update,
	     and none of the entry boxes had focus */
	  return;

     if( (xtype == MAKE_LINE && GTK_IS_ENTRY(widget)) ||
	 (xtype == MAKE_TEXTAREA && GTK_IS_TEXT(widget)) )
	  /* entrybox with focus is already of the requested type */
	  return;

     newtable = gtk_table_new(3, 2, FALSE);
     gtk_widget_show(newtable);
     gtk_object_set_data(GTK_OBJECT(window), "table", newtable);
     gtk_object_set_data(GTK_OBJECT(newtable), "target_vbox", target_vbox);
     gtk_object_set_data(GTK_OBJECT(newtable), "window", window);
     gtk_object_set_data(GTK_OBJECT(newtable), "vbox", vbox);
     gtk_container_border_width(GTK_CONTAINER(newtable), 5);
     gtk_table_set_row_spacings(GTK_TABLE(newtable), 1);
     gtk_table_set_col_spacings(GTK_TABLE(newtable), 10);
     gtk_box_pack_start(GTK_BOX(vbox), newtable, FALSE, TRUE, 0);

     children = GTK_TABLE(oldtable)->children;
     for(y = 1, ny = 1; y <= oldtable->nrows; y++, ny++) {

	  /* ADD_NEW_ROW mode: have we reached the row below pressed arrowbutton? */
	  if(xtype == ADD_NEW_ROW && y == row + 1) {
	       /* initialize the new row -- entrybox and arrowbutton only */
	       widget = widget_by_coord(children, 1, row);
	       if(GTK_IS_ENTRY(widget))
		    inputbox = gtk_entry_new();
	       else if(GTK_IS_TEXT(widget))
		    inputbox = gtk_text_new(NULL, NULL);
	       else break;
	       gtk_widget_show(inputbox);

	       gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
					 GTK_SIGNAL_FUNC(edit_entry_ok_callback),
					 GTK_OBJECT(window));
	       gtk_signal_connect(GTK_OBJECT(inputbox), "key_press_event",
				  GTK_SIGNAL_FUNC(keypress_inputbox),
				  (gpointer) target_vbox);
	       gtk_table_attach(GTK_TABLE(newtable), inputbox, 1, 2, ny, ny + 1, 
				GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0,0);
	       newabutton = gq_new_arrowbutton(window);
	       gtk_table_attach(GTK_TABLE(newtable), newabutton,
				2, 3, ny, ny + 1,
				0, GTK_EXPAND, 5, 0);

	       /* everything below gets moved down one row from now on */
	       ny++;
	  }

	  for(x = 0; x < 3; x++) {
	       widget = widget_by_coord(children, x, y);
	       switch(x) {
	       case 0:
		    /* label */
		    if(widget) {
			 label = gtk_label_new(GTK_LABEL(widget)->label);
			 gtk_widget_show(label);
			 gtk_table_attach(GTK_TABLE(newtable), label,
					  0, 1, ny, ny + 1,
					  0, GTK_FILL|GTK_EXPAND, 0, 0);
		    }
		    break;

	       case 1:
		    /* entrybox */
		    if(widget) {
			 if(GTK_IS_ENTRY(widget)) {
			      /* widget is an entry box */
			      oldinput = gtk_entry_get_text(GTK_ENTRY(widget));
			 }
			 else {
			      /* widget is a text box */
			      oldinput = gtk_editable_get_chars(GTK_EDITABLE(widget), 0, -1);
			 }

			 if( (y == row && xtype == MAKE_LINE) ||
			     (GTK_IS_ENTRY(widget) && xtype == ADD_NEW_ROW) ||
			     (y != row && GTK_IS_ENTRY(widget)) ) {
			      /* entry box */
			      inputbox = gtk_entry_new();
			      gtk_widget_show(inputbox);
			      gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
							GTK_SIGNAL_FUNC(edit_entry_ok_callback),
							GTK_OBJECT(window));
			      gtk_signal_connect(GTK_OBJECT(inputbox), "key_press_event",
						 GTK_SIGNAL_FUNC(keypress_inputbox),
						 (gpointer) target_vbox);
			      gtk_entry_set_text(GTK_ENTRY(inputbox), oldinput);

			      gtk_table_attach(GTK_TABLE(newtable), inputbox,
					       1, 2, ny, ny + 1,
					       GTK_FILL|GTK_EXPAND,
					       GTK_FILL|GTK_EXPAND, 0,0);
			 }
			 else {
			      /* text box */
			      inputbox = gtk_text_new(NULL, NULL);

			      /* need to do this here, or gtk_text_insert() bombs out :-( */
			      gtk_table_attach(GTK_TABLE(newtable), inputbox,
					       1, 2, ny, ny + 1,
					       GTK_FILL|GTK_EXPAND,
					       GTK_FILL|GTK_EXPAND, 0,0);		

			      gtk_text_freeze(GTK_TEXT(inputbox));
			      gtk_text_set_editable(GTK_TEXT(inputbox), TRUE);
			      gtk_text_set_point(GTK_TEXT(inputbox), 0);
			      gtk_text_insert(GTK_TEXT(inputbox), NULL, NULL,
					      NULL, oldinput, -1);
			      gtk_text_thaw(GTK_TEXT(inputbox));
			      gtk_widget_show(inputbox);
			 }

			 if(GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(widget)))
			      focusbox = inputbox;

		    }
		    break;

	       case 2:
		    /* arrowbutton */
		    /* draw arrowbutton unless this is ADD_NEW_ROW mode and
		       we're at the row that the selected arrowbutton was on */
		    if(widget && (xtype != ADD_NEW_ROW || y != row) ) {
			 newabutton = gq_new_arrowbutton(window);
			 gtk_table_attach(GTK_TABLE(newtable), newabutton,
					  2, 3, ny, ny + 1,
					  0, GTK_EXPAND, 5, 0);
		    }
		    break;
	       }

	  }

     }

     if(focusbox)
	  gtk_widget_grab_focus(focusbox);
     gtk_widget_destroy(GTK_WIDGET(oldtable));

}

