/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998,1999 Bert Vermeulen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <lber.h>
#include <ldap.h>

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <pwd.h>

#include <config.h>

#include "common.h"
#include "util.h"
#include "query.h"
#include "configfile.h"
#include "errorchain.h"

extern GtkWidget *mainwin;
extern GtkWidget *statusbar;
extern struct gq_config config;


/*
 * open connection to LDAP server, and store connection for caching
 */
LDAP *open_connection(struct ldapserver *server)
{
     LDAP *ld;
     int open_context, msg;
     char message[128];
     char *binddn, *bindpw;

     /* reuse previous connection if available */
     if(server->connection)
	  return(server->connection);

     sprintf(message, "connecting to %s", server->ldaphost);
     if(server->ldapport != 389)
	  sprintf(message + strlen(message), " port %d", server->ldapport);
     statusbar_msg(message);

     ld = NULL;
     open_context = error_new_context("Error connecting to server");

     if(server) {
	  ld = ldap_open(server->ldaphost, server->ldapport);
	  if(!ld) {
	       sprintf(message, "Failed to open connection to %s:%d",
		       server->ldaphost, server->ldapport);
	       error_push(open_context, message);
	  }
	  else {
	       if(server->binddn[0]) {
		    binddn = server->binddn;
		    bindpw = server->bindpw;
	       }
	       else {
		    binddn = NULL;
		    bindpw = NULL;
	       }

	       msg = LDAP_SUCCESS;
	       if(server->bindtype == BINDTYPE_KERBEROS) {
		    #ifdef HAVE_KERBEROS
		    msg = ldap_bind_s(ld, binddn, bindpw, LDAP_AUTH_KRBV4);
		    #else
		    error_push(open_context, "GQ was compiled without Kerberos support.\n"
			       "Run 'configure --help' for more information\n");
		    statusbar_msg("");
		    ldap_unbind(ld);
		    ld = NULL;
		    #endif
	       }
	       else
		    msg = ldap_simple_bind_s(ld, binddn, bindpw);

	       if( msg != LDAP_SUCCESS) {
		    error_push(open_context, ldap_err2string(msg));
		    statusbar_msg("");
		    /* might as well clean this up */
		    ldap_unbind(ld);
                    ld = NULL;
	       }
	       else
		    /* always store connection handle, regardless of connection
		       caching -- call close_connection() after each operation
		       to do the caching thing or not */
		    server->connection = ld;

	  }

     }

     error_flush(open_context);

     return(ld);
}


/*
 * called after every set of LDAP operations. This preserves the connection
 * if caching is enabled for the server. Set always to TRUE if you want to
 * close the connection regardless of caching.
 */
void close_connection(struct ldapserver *server, int always)
{

     if(server->connection && (!server->cacheconn || always)) {
	  ldap_unbind(server->connection);
	  server->connection = NULL;
     }

}


/*
 * delete entry
 */
void delete_entry(struct ldapserver *server, char *dn)
{
     LDAP *ld;
     int msg;
     char message[MAX_DN_LEN + 128];

     /* FIXME confirm-mod check here */

     set_busycursor();

     if( (ld = open_connection(server) ) == NULL) {
	  set_normalcursor();
	  return;
     }

     sprintf(message, "deleting %s", dn);
     statusbar_msg(message);

     msg = ldap_delete_s(ld, dn);
     if(msg != LDAP_SUCCESS)
	  error_popup("Error deleting entry", ldap_err2string(msg));

     sprintf(message, "deleted %s", dn);
     statusbar_msg(message);

     set_normalcursor();
     close_connection(server, FALSE);

}


/*
 * display hourglass cursor on mainwin
 */
void set_busycursor(void)
{
     GdkCursor *busycursor;

     busycursor = gdk_cursor_new(GDK_WATCH);
     gdk_window_set_cursor(mainwin->window, busycursor);
     gdk_cursor_destroy(busycursor);

}


/*
 * set mainwin cursor to default
 */
void set_normalcursor(void)
{

     gdk_window_set_cursor(mainwin->window, NULL);

}


void make_message(char *buffer, int cnt, char *singular, char *plural, char *suffix)
{

     switch(cnt) {
     case 0:
	  sprintf(buffer, "no %s %s", plural, suffix);
	  break;
     case 1:
	  sprintf(buffer, "1 %s %s", singular, suffix);
	  break;
     default:
	  sprintf(buffer, "%d %s %s", cnt, plural, suffix);
	  break;
     }

}


/*
 * callback for key_press_event on a widget, destroys obj if key was esc
 */
int close_on_esc(GtkWidget *widget, GdkEventKey *event, gpointer obj)
{

     if(event && event->type == GDK_KEY_PRESS && event->keyval == GDK_Escape) {
	  gtk_widget_destroy(GTK_WIDGET(obj));
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
     }

     return(TRUE);
}


/*
 * callback for key_press_event on a widget, calls func if key was esc
 */
int func_on_esc(GtkWidget *widget, GdkEventKey *event, GtkWidget *window)
{
     void (*func)(GtkWidget *);

     if(event && event->type == GDK_KEY_PRESS && event->keyval == GDK_Escape) {
	  func = gtk_object_get_data(GTK_OBJECT(window), "close_func");
	  func(widget);
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
     }

     return(TRUE);
}


int tokenize(struct tokenlist *list, char *keyword)
{
     int i;

     for(i = 0; strlen(list[i].keyword); i++)
	  if(!strcasecmp(list[i].keyword, keyword))
	       return(list[i].token);

     return(0);
}


char *detokenize(struct tokenlist *list, int token)
{
     int i;

     for(i = 0; strlen(list[i].keyword); i++)
	  if(list[i].token == token)
	       return(list[i].keyword);

     return(list[0].keyword);
}


/*
 * return pointer to username (must be freed)
 */
char *get_username(void)
{
     struct passwd *pwent;
     char *username;

     username = NULL;
     pwent = getpwuid(getuid());
     if(pwent && pwent->pw_name)
	  username = strdup(pwent->pw_name);
     endpwent();

     return(username);
}


/*
 * display message in main window's statusbar, and flushes the
 * GTK event queue
 */
void statusbar_msg(char *message)
{
     static guint context, msgid = 0;

     if(!context)
	  context = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar),
						 "mainwin");
     if(msgid)
	  gtk_statusbar_remove(GTK_STATUSBAR(statusbar), context, msgid);

     gtk_statusbar_push(GTK_STATUSBAR(statusbar), context, message);

     /* make sure statusbar gets updated right away */
     while(gtk_events_pending())
	  gtk_main_iteration();

}


/*
 * return pointer to (struct ldapserver *) matching name
 */
struct ldapserver *server_by_name(char *name)
{
     struct ldapserver *server;

     if(name == NULL)
	  return(NULL);

     server = config.ldapservers;
     while(server) {
	  if(!strcmp(server->name, name))
	       return(server);
	  server = server->next;
     }

     error_popup("Error finding server", "Where did it go?");

     return(NULL);
}


/*
 * check if entry has a subtree
 */
int is_leaf_entry(struct ldapserver *server, char *dn)
{
     LDAP *ld;
     LDAPMessage *res;
     GString *gmessage;
     int msg, is_leaf;

     is_leaf = 0;

     set_busycursor();

     if( (ld = open_connection(server) ) == NULL) {
	  set_normalcursor();
	  return(-1);
     }

     gmessage = g_string_sized_new(128);
     g_string_sprintf(gmessage, "checking subtree for %s", dn);
     statusbar_msg(gmessage->str);
     g_string_free(gmessage, TRUE);

     msg = ldap_search(ld, dn, LDAP_SCOPE_ONELEVEL, "(objectclass=*)",
		       NULL, 0);
     if(msg != -1) {
	  if( (ldap_result(ld, msg, 0, NULL, &res) != LDAP_RES_SEARCH_ENTRY))
	       is_leaf = 1;
	  ldap_msgfree(res);
     }

     close_connection(server, FALSE);
     set_normalcursor();
     statusbar_msg("");

     return(is_leaf);
}
