#include "hx_types.h"
#include "hxlib.h"
#include "hx.h"
#include "dhargs.h"
#include "screen.h"
#include "term.h"
#include <netinet/in.h>
#include <sys/socket.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include "list.h"
#include <time.h>
#include "xmalloc.h"

struct xfer {
	struct xfer *next;
	int nr, type;
	u_int32_t size, data_pos, rsrc_pos;
	char *local, *remote;
	time_t time;
	pid_t pid;
};

void htlc_snd_file_list (const char *path);
void htlc_snd_file_get (const char *path, u_int32_t data_size, u_int32_t rsrc_size);
void htlc_snd_file_put (const char *path, u_int32_t data_size, u_int32_t rsrc_size);
void htlc_snd_file_mkdir (const char *path);
void htlc_snd_file_delete (const char *path);
void rcv_file_get (struct xfer *xf);
void rcv_file_put (struct xfer *xf);
void rcv_file_list (void);

#define XFER_GET 0
#define XFER_PUT 1

struct xfer *xfer_list = 0;
int xfer_nr = 0;

void xfer_del (struct xfer *xf);
void xfer_do (struct xfer *xf);
void xfer_new (int type, const char *local, const char *remote);
struct xfer *xfer_lookup_pid (pid_t pid);
struct xfer *xfer_lookup_nr (int nr);
int cmd_xfers (void);
int cmd_xfkill (int argc, char *const *argv);
int cmd_get (int argc, char *const *argv);
int cmd_put (int argc, char *const *argv);
int cmd_dirchar (int argc, char *const *argv);
int cmd_ls (int argc, char *const *argv);
int cmd_pwd (void);
int cmd_lpwd (void);
int cmd_cd (int argc, char *const *argv);
int cmd_lcd (int argc, char *const *argv);
int cmd_mkdir (int argc, char *const *argv);
int cmd_rm (int argc, char *const *argv);

void
xfer_del (struct xfer *xf)
{
	LISTREM(struct xfer *, xfer_list, xf)
	if (!xfer_list)
		xfer_nr = 0;
	if (xf->pid)
		kill(xf->pid, SIGKILL);
	xfree(xf->local);
	xfree(xf->remote);
	xfree(xf);
}

void
xfer_do (struct xfer *xf)
{
	struct stat sb, rsb;
	char rsrcpath[2048];

	if (stat(xf->local, &sb))
		sb.st_size = 0;
	sprintf(rsrcpath, "%.2000s.rsrc", xf->local);
	if (stat(rsrcpath, &rsb))
		rsb.st_size = 0;

	if (xf->type == XFER_GET) {
		xf->data_pos = sb.st_size;
		xf->rsrc_pos = rsb.st_size;
		task_new(hx_trans, (task_fn_t)rcv_file_get, (task_fn_t)xfer_del, xf);
		htlc_snd_file_get(xf->remote, sb.st_size, rsb.st_size);
	} else {
		task_new(hx_trans, (task_fn_t)rcv_file_put, (task_fn_t)xfer_del, xf);
		htlc_snd_file_put(xf->remote, sb.st_size, rsb.st_size);
	}
}

void
xfer_new (int type, const char *local, const char *remote)
{
	struct xfer *xf;

	xf = xmalloc(sizeof *xf);
	memset(xf, 0, sizeof *xf);
	xf->nr = xfer_nr++;
	xf->type = type;
	xf->local = xstrdup(local);
	xf->remote = xstrdup(remote);
	if (!xfer_list)
		xfer_do(xf);
	LISTADD(struct xfer *, xfer_list, xf)
}

struct xfer *
xfer_lookup_nr (int nr)
{
	register struct xfer *xfp;

	for (xfp = xfer_list; xfp; xfp = xfp->next)
		if (xfp->nr == nr)
			return xfp;

	return 0;
}

int
cmd_xfers (void)
{
	register struct xfer *xfp;
	struct stat sb;
	register time_t t;

	term_mode_underline();
	curscr_printf("\n #  type    pid  speed  local  remote");
	term_mode_clear();
	t = time(0);
	for (xfp = xfer_list; xfp; xfp = xfp->next) {
		stat(xfp->local, &sb);
		curscr_printf("\n[%d]  %s  %5lu  %lu/%lu (%lu bytes/sec)  '%s'  '%s'",
				xfp->nr, xfp->type == XFER_GET ? "get" : "put", (unsigned long)xfp->pid,
				(unsigned long)sb.st_size, (unsigned long)xfp->size,
				(unsigned long)(sb.st_size / (t - xfp->time)),
				xfp->local, xfp->remote);
	}

	return 0;
}

int
cmd_xfkill (int argc, char *const *argv)
{
	register int i;
	register struct xfer *xf;

	if (argc < 2) {
		curscr_printf("\nusage: %s {all|xfer#}", argv[0]);
		return 0;
	}
	if (!strcmp(argv[1], "all")) {
		register struct xfer *next;
		for (xf = xfer_list; xf; xf = next) {
			next = xf->next;
			xfer_del(xf);
		}
		return 0;
	}
	for (i = 1; i < argc; i++) {
		if ((xf = xfer_lookup_nr(atoi(argv[i]))))
			xfer_del(xf);
	}

	return 0;
}

extern struct sockaddr_in hx_sockaddr;

static char dir_char = '/', remote_cwd[4096] = "/";

static const char *
dirchar_basename (const char *str)
{
	register int len;

	if (!str)
		return 0;
	len = strlen(str);
	while (len) {
		if (str[len] == dir_char)
			return &(str[len + 1]);
		len--;
	}

	return str;
}

int
cmd_get (int argc, char *const *argv)
{
	register int i;

	if (argc < 2) {
		curscr_printf("\nusage: %s [-aR] [-z <remote> <local>] file1 [file2...]", argv[0]);
		curscr_printf("\nwarning: -aRz not implemented yet");
		return 0;
	}
	for (i = 1; i < argc; i++) {
		if (argv[i][0] != dir_char) {
			char buf[sizeof remote_cwd];

			snprintf(buf, sizeof buf, "%s%c%s", remote_cwd, dir_char, argv[i]);
			xfer_new(XFER_GET, dirchar_basename(argv[i]), buf);
		} else
			xfer_new(XFER_GET, dirchar_basename(argv[i]), argv[i]);
	}

	return 0;
}

int
cmd_put (int argc, char *const *argv)
{
	register int i;

	if (argc < 2) {
		curscr_printf("\nusage: %s [-aR] [-z <local> <remote>] file1 [file2...]", argv[0]);
		curscr_printf("\nwarning: -aRz not implemented yet");
		return 0;
	}
	for (i = 1; i < argc; i++) {
		if (argv[i][0] != dir_char) {
			char buf[sizeof remote_cwd];

			snprintf(buf, sizeof buf, "%s%c%s", remote_cwd, dir_char, argv[i]);
			xfer_new(XFER_PUT, dirchar_basename(argv[i]), buf);
		} else
			xfer_new(XFER_PUT, dirchar_basename(argv[i]), argv[i]);
	}

	return 0;
}

int
cmd_dirchar (int argc, char *const *argv)
{
	register char c = dir_char;

	if (argc < 2 || argv[1][0] == c) {
		curscr_printf("\n <\xa5> dirchar is %c", c);
	} else {
		register char *p = remote_cwd;

		dir_char = argv[1][0];
		while (*p) {
			if (*p == c)
				*p = dir_char;
			p++;
		}
	}

	return (int)dir_char;
}

void
rcv_file_list (void)
{
	struct hx_filelist_hdr *fh;
	char buf[4096];
	register u_int16_t i, bpos;
	u_int32_t fnlen, fsize;

	dh_start(&(hx_buf[SIZEOF_HX_HDR]), hx_pos - SIZEOF_HX_HDR)
		L16NTOH(i, &dh->type);
		if (i != HTLS_DATA_FILE_LIST)
			continue;
		fh = (struct hx_filelist_hdr *)dh;
		L32NTOH(fnlen, &fh->fnlen);
		bpos = 0;
		for (i = 0; i < fnlen; i++) {
			if (fh->fname[i] > 127 || fh->fname[i] < 32) {
				buf[bpos++] = '\\';
				switch (fh->fname[i]) {
					case '\r':
						buf[bpos++] = 'r';
						break;
					case '\n':
						buf[bpos++] = 'n';
						break;
					case '\t':
						buf[bpos++] = 't';
						break;
					default:
						sprintf(&(buf[bpos]), "x%2.2x", fh->fname[i]);
						bpos += 3;
				}
			} else
				buf[bpos++] = fh->fname[i];
		}
		buf[bpos] = 0;
		L32NTOH(fsize, &fh->fsize);
		curscr_printf("\n%4.4s/%4.4s %10u %s",
			fh->fcreator, fh->ftype, fsize, buf);
	dh_end()
}

int
cmd_ls (int argc, char *const *argv)
{
	int i;

	if (!argv[1]) {
		task_new(hx_trans, (task_fn_t)rcv_file_list, 0, 0);
		htlc_snd_file_list(remote_cwd);
		return 0;
	}
	for (i = 1; i < argc; i++) {
		task_new(hx_trans, (task_fn_t)rcv_file_list, 0, 0);
		if (argv[i][0] == dir_char)
			htlc_snd_file_list(argv[i]);
		else {
			char buf[sizeof remote_cwd * 2];

			snprintf(buf, sizeof buf, "%s%c%s", remote_cwd, dir_char, argv[i]);
			htlc_snd_file_list(buf);
		}
	}

	return 0;
}

int
cmd_cd (int argc, char *const *argv)
{
	if (argc < 2) {
		curscr_printf("\nusage: %s <directory>", argv[0]);
		return 0;
	}
	if (argv[1][0] == dir_char) {
		strncpy(remote_cwd, argv[1], sizeof remote_cwd - 1);
		remote_cwd[sizeof remote_cwd - 1] = 0;
	} else {
		char buf[sizeof remote_cwd];

		snprintf(buf, sizeof buf, "%s%c%s", remote_cwd, dir_char, argv[1]);
		strcpy(remote_cwd, buf);
	}
	curscr_printf("\n <\xa5> remote cwd now %s", remote_cwd);

	return 0;
}

int
cmd_pwd (void)
{
	curscr_printf("\n <\xa5> %s", remote_cwd);

	return 0;
}

int
cmd_lcd (int argc, char *const *argv)
{
	if (argc < 2) {
		curscr_printf("\nusage: %s <local directory>", argv[0]);
		return 0;
	} else { /* XXX */
		char buf[1024];
		expand_tilde(buf, argv[1]);
		if (chdir(buf))
			curscr_printf("\n <\xa5> chdir(%s): %s", buf, strerror(errno));
	}

	return 0;
}

int
cmd_lpwd (void)
{
	char buf[4096];

	if (!getcwd(buf, sizeof buf - 1))
		curscr_printf("\n <\xa5> getcwd: %s", strerror(errno));
	else {
		buf[sizeof buf - 1] = 0;
		curscr_printf("\n <\xa5> %s", buf);
	}

	return 0;
}

struct x_fhdr {
	u_int16_t enc;
	u_int8_t len, name[1];
};

struct hx_data_hdr *path_to_hldir (const char *path, int is_file);

struct hx_data_hdr *
path_to_hldir (const char *path, int is_file)
{
	struct hx_data_hdr *dh;
	struct x_fhdr *fh;
	register char const *p, *p2;
	u_int16_t pos = 2, dc = 0;
	register u_int8_t nlen;

	dh = xmalloc(SIZEOF_HX_DATA_HDR + 2);
	p = path;
	while ((p2 = strchr(p, dir_char))) {
		if (!(p2 - p)) {
			p++;
			continue;
		}
		nlen = (u_int8_t)(p2 - p);
		pos += 3 + nlen;
		dh = xrealloc(dh, SIZEOF_HX_DATA_HDR + pos);
		fh = (struct x_fhdr *)(&(dh->data[pos - (3 + nlen)]));
		memset(&fh->enc, 0, 2);
		fh->len = nlen;
		memcpy(fh->name, p, nlen);
		dc++;
		p = &(p2[1]);
	}
	if (!is_file && *p) {
		nlen = (u_int8_t)strlen(p);
		pos += 3 + nlen;
		dh = xrealloc(dh, SIZEOF_HX_DATA_HDR + pos);
		fh = (struct x_fhdr *)(&(dh->data[pos - (3 + nlen)]));
		memset(&fh->enc, 0, 2);
		fh->len = nlen;
		memcpy(fh->name, p, nlen);
		dc++;
	}
	dh->type = htons(HTLC_DATA_DIR);
	dh->len = htons(pos);
	*((u_int16_t *)dh->data) = htons(dc);

	return dh;
}

void
htlc_snd_file_list (const char *path)
{
	u_int8_t buf[SIZEOF_HX_HDR];
	struct hx_hdr *h = (struct hx_hdr *)buf;
	struct hx_data_hdr *dh;

	h->type = htonl(HTLC_HDR_FILE_LIST);
	h->trans = htonl(hx_trans++);
	h->flag = 0;
	if (!path || !path[0] || (path[0] == dir_char && !path[1])) {
		h->len = h->len2 = htonl(2);
		h->hc = 0;
		send(hx_sock, buf, SIZEOF_HX_HDR, 0);
		return;
	}
	dh = path_to_hldir(path, 0);
	h->len = h->len2 = htonl(2 + SIZEOF_HX_DATA_HDR + ntohs(dh->len));
	h->hc = htons(1);
	send(hx_sock, buf, SIZEOF_HX_HDR, 0);
	send(hx_sock, dh, SIZEOF_HX_DATA_HDR + ntohs(dh->len), 0);
	xfree(dh);
}

void
htlc_snd_file_get (const char *path, u_int32_t data_size, u_int32_t rsrc_size)
{
	char buf[SIZEOF_HX_HDR + SIZEOF_HX_DATA_HDR + 74];
	struct hx_hdr *h = (struct hx_hdr *)buf;
	struct hx_data_hdr *dh = (struct hx_data_hdr *)(&(buf[SIZEOF_HX_HDR]));
	u_int16_t nlen;
	char const *p;

	h->type = htonl(HTLC_HDR_FILE_GET);
	h->trans = htonl(hx_trans++);
	h->flag = 0;
	if (!path || !path[0] || !(p = dirchar_basename(path)))
		return;
	dh->type = htons(HTLC_DATA_FILE);
	dh->len = htons((nlen = strlen(p)));
	if (p != path) {
		dh = path_to_hldir(path, 1);
		h->len = h->len2 = htonl(74 + 2 + (SIZEOF_HX_DATA_HDR * 3) + ntohs(dh->len) + nlen);
		h->hc = htons(3);
		send(hx_sock, buf, 26, 0);
		send(hx_sock, p, nlen, 0);
		send(hx_sock, dh, SIZEOF_HX_DATA_HDR + ntohs(dh->len), 0);
		xfree(dh);
	} else {
		h->len = h->len2 = htonl(74 + 2 + (SIZEOF_HX_DATA_HDR * 2) + nlen);
		h->hc = htons(2);
		send(hx_sock, buf, 26, 0);
		send(hx_sock, p, nlen, 0);
	}
	dh = (struct hx_data_hdr *)buf;
	dh->type = htons(HTLC_DATA_RFLT);
	dh->len = htons(74);
	memcpy(&(buf[4]),
		"\x52\x46\x4c\x54\0\x01\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
		"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\x02\x44"
		"\x41\x54\x41\0\0\0\0\0\0\0\0\0\0\0\0\x4d\x41\x43\x52"
		"\0\0\0\0\0\0\0\0\0\0\0\0",
	74);
	S32HTON(data_size, &buf[50]);
	S32HTON(rsrc_size, &buf[66]);
	send(hx_sock, buf, 78, 0);
}

void
htlc_snd_file_put (const char *path, u_int32_t data_size, u_int32_t rsrc_size)
{
	char buf[SIZEOF_HX_HDR + SIZEOF_HX_DATA_HDR + 74];
	struct hx_hdr *h = (struct hx_hdr *)buf;
	struct hx_data_hdr *dh = (struct hx_data_hdr *)(&(buf[SIZEOF_HX_HDR]));
	u_int16_t nlen;
	char const *p;

	h->type = htonl(HTLC_HDR_FILE_PUT);
	h->trans = htonl(hx_trans++);
	h->flag = 0;
	if (!path || !path[0] || !(p = dirchar_basename(path)))
		return;
	dh->type = htons(HTLC_DATA_FILE);
	dh->len = htons((nlen = strlen(p)));
	if (p != path) {
		dh = path_to_hldir(path, 1);
		h->len = h->len2 = htonl(74 + 2 + (SIZEOF_HX_DATA_HDR * 3) + ntohs(dh->len) + nlen);
		h->hc = htons(3);
		send(hx_sock, buf, 26, 0);
		send(hx_sock, p, nlen, 0);
		send(hx_sock, dh, SIZEOF_HX_DATA_HDR + ntohs(dh->len), 0);
		xfree(dh);
	} else {
		h->len = h->len2 = htonl(74 + 2 + (SIZEOF_HX_DATA_HDR * 2) + nlen);
		h->hc = htons(2);
		send(hx_sock, buf, 26, 0);
		send(hx_sock, p, nlen, 0);
	}
	dh = (struct hx_data_hdr *)buf;
	dh->type = htons(HTLC_DATA_RFLT);
	dh->len = htons(74);
	memcpy(&(buf[4]),
		"\0\xcb\0\x4a\x52\x46\x4c\x54\0\x01\0\0\0\0\0\0\0\0"
		"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
		"\0\0\x02\x44\x41\x54\x41\0\0\0\0\0\0\0\0\0\0\0\0\x4d"
		"\x41\x43\x52\0\0\0\0\0\0\0\0\0\0\0\0",
	74);
	S32HTON(data_size, &buf[50]);
	S32HTON(rsrc_size, &buf[66]);
	send(hx_sock, buf, 78, 0);
}

struct xfer *
xfer_lookup_pid (pid_t pid)
{
	register struct xfer *xfp;

	for (xfp = xfer_list; xfp; xfp = xfp->next)
		if (xfp->pid == pid)
			return xfp;

	return 0;
}

#ifndef RETSIGTYPE
#define RETSIGTYPE void
#endif

RETSIGTYPE sig_chld (int st);

RETSIGTYPE
sig_chld (int st)
{
	pid_t pid;
	struct xfer *xf, *next;

	pid = waitpid(0, &st, WNOHANG);
	curscr_printf(
		!WEXITSTATUS(st) ? "\n <\xa5> sig_chld: process %u exited successfully" :
		"\n <\xa5> sig_chld: process %u exited with errno %d (%s)",
		pid, WEXITSTATUS(st), strerror(WEXITSTATUS(st)));

	if ((xf = xfer_lookup_pid(pid))) {
		time_t t = time(0) - xf->time;

		if (!t)
			t = 1;
		if (xf->type == XFER_GET) {
			curscr_printf(
				"\n <\xa5> [%d] %lu bytes received in %lu seconds (%lu bytes/sec)",
				xf->nr, (unsigned long)xf->size, (unsigned long)t,
				(unsigned long)(xf->size / t));
		} else {
			curscr_printf(
				"\n <\xa5> [%d] %lu bytes sent in %lu seconds (%lu bytes/sec)",
				xf->nr, (unsigned long)xf->size, (unsigned long)t,
				(unsigned long)(xf->size / t));
		}
		next = xf->next;
		xfer_del(xf);
		if (next)
			xfer_do(next);
	}
}

void install_child_handler (void);
void ignore_signals (void);

void
install_child_handler (void)
{
	struct sigaction act;

	memset(&act, 0, sizeof act);
	act.sa_handler = sig_chld;
	sigaction(SIGCHLD, &act, 0);
}

void
ignore_signals (void)
{
	struct sigaction act;

	memset(&act, 0, sizeof act);
	act.sa_handler = SIG_IGN;
	sigaction(SIGINT, &act, 0);
	sigaction(SIGTSTP, &act, 0);
	sigaction(SIGCONT, &act, 0);
	sigaction(SIGWINCH, &act, 0);
}

void rd_wr (int rd_fd, int wr_fd, u_int32_t data_len);

void
rd_wr (int rd_fd, int wr_fd, u_int32_t data_len)
{
	register int r;
	register u_int32_t pos, len;
#ifdef __MWERKS__
	u_int8_t buf[16384];
#else
	u_int8_t buf[0xffff];
#endif

	while (data_len) {
		if ((len = read(rd_fd, buf, sizeof buf < data_len ? sizeof buf : data_len)) < 1)
			_exit(errno);
		pos = 0;
		while (len) {
			if ((r = write(wr_fd, &(buf[pos]), len)) < 1)
				_exit(errno);
			pos += r;
			len -= r;
		}
		data_len -= pos;
	}
}

void
rcv_file_get (struct xfer *xf)
{
	u_int32_t ref = 0, size = 0;

	dh_start(&(hx_buf[SIZEOF_HX_HDR]), hx_pos - SIZEOF_HX_HDR)
		switch (ntohs(dh->type)) {
			case HTLS_DATA_HTXF_SIZE:
				dh_getint(size);
				break;
			case HTLS_DATA_HTXF_REF:
				dh_getint(ref);
				break;
		}
	dh_end()
	if (!size || !ref)
		return;
	xf->size = size;
	curscr_printf("\n <\xa5> get: local: %s; remote: %s; %lu bytes",
		xf->local, xf->remote, (unsigned long)size);
	xf->time = time(0);
	switch ((xf->pid = fork())) {
		case -1:
			curscr_printf("\nrcv_file_get: fork: %s", strerror(errno));
			xfer_del(xf);
			return;
		case 0:
			ignore_signals();
		{
			int s, fd;
			struct sockaddr_in addr;
			struct htxf_hdr xfh;

			if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
				_exit(errno);
			memcpy(&addr, &hx_sockaddr, sizeof addr);
			addr.sin_port = htons(ntohs(addr.sin_port) + 1);
			if (connect(s, (struct sockaddr *)&addr, sizeof addr))
				_exit(errno);
			xfh.magic = htonl(HTXF_MAGIC_INT);
			xfh.ref = htonl(ref);
			xfh.type = xfh.len = 0;
			if (write(s, &xfh, SIZEOF_HTXF_HDR) != SIZEOF_HTXF_HDR)
				_exit(errno);
			{
#ifdef __MWERKS__
				char buf[16384 + 16];
#else
				char buf[0xffff + 16];
#endif
				register int r;
				register u_int32_t pos = 0, len = 40;

				while (len) {
					if ((r = read(s, &(buf[pos]), len)) < 1)
						_exit(errno);
					pos += r;
					len -= r;
				}
				pos = 0;
				L16NTOH(len, &buf[38]);
				len += 16;
				while (len) {
					if ((r = read(s, &(buf[pos]), len)) < 1)
						_exit(errno);
					pos += r;
					len -= r;
				}
				L32NTOH(len, &buf[pos - 4]);
				if (!len)
					goto rsrc;
				if ((fd = open(xf->local, O_CREAT|O_WRONLY, S_IRUSR|S_IWUSR)) < 0)
					_exit(errno);
				if (xf->data_pos)
					lseek(fd, xf->data_pos, SEEK_SET);
				rd_wr(s, fd, len);
				fsync(fd);
				close(fd);
rsrc:
				pos = 0;
				len = 16;
				while (len) {
					if ((r = read(s, &(buf[pos]), len)) < 1)
						_exit(errno);
					pos += r;
					len -= r;
				}
				L32NTOH(len, &buf[12]);
				if (!len)
					goto done;
				sprintf(buf, "%s.rsrc", xf->local);
				if ((fd = open(buf, O_CREAT|O_WRONLY, S_IRUSR|S_IWUSR)) < 0)
					_exit(errno);
				if (xf->rsrc_pos)
					lseek(fd, xf->rsrc_pos, SEEK_SET);
				rd_wr(s, fd, len);
				fsync(fd);
				close(fd);
done:
				close(s);
			}
			_exit(0);
		}
		default:
			install_child_handler();
	}
}

void
rcv_file_put (struct xfer *xf)
{
	u_int32_t ref = 0;
	struct stat sb, rsb;
	char rsrcpath[2048];

	dh_start(&(hx_buf[SIZEOF_HX_HDR]), hx_pos - SIZEOF_HX_HDR)
		switch (ntohs(dh->type)) {
			case HTLS_DATA_HTXF_REF:
				dh_getint(ref);
				break;
		}
	dh_end()
	if (!ref)
		return;
	if (stat(xf->local, &sb))
		sb.st_size = 0;
	sprintf(rsrcpath, "%.2000s.rsrc", xf->local);
	if (stat(rsrcpath, &rsb))
		rsb.st_size = 0;
	xf->size = 133 + sb.st_size + rsb.st_size;
	curscr_printf("\n <\xa5> put: local: %s; remote: %s; %lu bytes",
		xf->local, xf->remote, (unsigned long)xf->size);
	xf->time = time(0);
	switch ((xf->pid = fork())) {
		case -1:
			curscr_printf("\nrcv_file_put: fork: %s", strerror(errno));
			xfer_del(xf);
			return;
		case 0:
			ignore_signals();
		{
			int s, fd;
			struct sockaddr_in addr;
			struct htxf_hdr xfh;

			if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
				_exit(errno);
			memcpy(&addr, &hx_sockaddr, sizeof addr);
			addr.sin_port = htons(ntohs(addr.sin_port) + 1);
			if (connect(s, (struct sockaddr *)&addr, sizeof addr))
				_exit(errno);
			xfh.magic = htonl(HTXF_MAGIC_INT);
			xfh.ref = htonl(ref);
			xfh.type = htonl(133 + sb.st_size + rsb.st_size);
			xfh.len = 0;
			if (write(s, &xfh, SIZEOF_HTXF_HDR) != SIZEOF_HTXF_HDR)
				_exit(errno);
			{
				char buf[133];

				memcpy(buf,
					"FILP\0\x01\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
					"\0\0\x03INFO\0\0\0\0\0\0\0\0\0\0\0MAMAC"
					"BINAhDmp"
					"\0\0\0\0\0\0\x01\0\0\0\0\0\0\0\0\0\0\0\0\0"
					"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\x07"
					"\0\0\0\0\0\0\0\x07\0\0\0\0\0\0\0\0\0\0\x03"
					"hxd\0\0DATA\0\0\0\0\0\0\0\0",
				129);
				S32HTON(sb.st_size, &buf[129]);
				if (write(s, buf, 133) != 133)
					_exit(errno);
				if (!sb.st_size)
					goto rsrc;
				if ((fd = open(xf->local, O_RDONLY)) < 0)
					_exit(errno);
				if (xf->data_pos)
					lseek(fd, xf->data_pos, SEEK_SET);
				rd_wr(fd, s, sb.st_size);
				close(fd);
rsrc:
				memcpy(buf, "MACR\0\0\0\0\0\0\0\0", 12);
				S32HTON(rsb.st_size, &buf[12]);
				if (write(s, buf, 16) != 16)
					_exit(errno);
				if (!rsb.st_size)
					goto done;
				if ((fd = open(rsrcpath, O_RDONLY)) < 0)
					_exit(errno);
				if (xf->rsrc_pos)
					lseek(fd, xf->rsrc_pos, SEEK_SET);
				rd_wr(fd, s, rsb.st_size);
				close(fd);
done:
				close(s);
			}
			_exit(0);
		}
		default:
			install_child_handler();
	}
}

#if 0
void
htlc_snd_file_getinfo ()
{
}

void
htlc_snd_file_setinfo ()
{
}

void
htlc_snd_file_move (const char *oldpath, const char *newpath)
{
}
#endif

void
htlc_snd_file_mkdir (const char *path)
{
	char buf[SIZEOF_HX_HDR + SIZEOF_HX_DATA_HDR];
	struct hx_hdr *h = (struct hx_hdr *)buf;
	struct hx_data_hdr *dh = (struct hx_data_hdr *)(&(buf[SIZEOF_HX_HDR]));
	u_int16_t nlen;
	char const *p;

	h->type = htonl(HTLC_HDR_FILE_MKDIR);
	h->trans = htonl(hx_trans++);
	h->flag = 0;
	if (!path || !path[0] || !(p = dirchar_basename(path)))
		return;
	dh->type = htons(HTLC_DATA_FILE);
	dh->len = htons((nlen = strlen(p)));
	if (p != path) {
		dh = path_to_hldir(path, 1);
		h->len = h->len2 = htonl(2 + (SIZEOF_HX_DATA_HDR * 2) + ntohs(dh->len) + nlen);
		h->hc = htons(2);
		send(hx_sock, buf, 26, 0);
		send(hx_sock, p, nlen, 0);
		send(hx_sock, dh, SIZEOF_HX_DATA_HDR + ntohs(dh->len), 0);
		xfree(dh);
	} else {
		h->len = h->len2 = htonl(2 + SIZEOF_HX_DATA_HDR + nlen);
		h->hc = htons(1);
		send(hx_sock, buf, 26, 0);
		send(hx_sock, p, nlen, 0);
	}
}

void
htlc_snd_file_delete (const char *path)
{
	char buf[SIZEOF_HX_HDR + SIZEOF_HX_DATA_HDR];
	struct hx_hdr *h = (struct hx_hdr *)buf;
	struct hx_data_hdr *dh = (struct hx_data_hdr *)(&(buf[SIZEOF_HX_HDR]));
	u_int16_t nlen;
	char const *p;

	h->type = htonl(HTLC_HDR_FILE_DELETE);
	h->trans = htonl(hx_trans++);
	h->flag = 0;
	if (!path || !path[0] || !(p = dirchar_basename(path)))
		return;
	dh->type = htons(HTLC_DATA_FILE);
	dh->len = htons((nlen = strlen(p)));
	if (p != path) {
		dh = path_to_hldir(path, 1);
		h->len = h->len2 = htonl(2 + (SIZEOF_HX_DATA_HDR * 2) + ntohs(dh->len) + nlen);
		h->hc = htons(2);
		send(hx_sock, buf, 26, 0);
		send(hx_sock, p, nlen, 0);
		send(hx_sock, dh, SIZEOF_HX_DATA_HDR + ntohs(dh->len), 0);
		xfree(dh);
	} else {
		h->len = h->len2 = htonl(2 + SIZEOF_HX_DATA_HDR + nlen);
		h->hc = htons(1);
		send(hx_sock, buf, 26, 0);
		send(hx_sock, p, nlen, 0);
	}
}

int
cmd_mkdir (int argc, char *const *argv)
{
	register int i;

	if (argc < 2) {
		curscr_printf("\nusage: %s file1 [file2...]", argv[0]);
		return 0;
	}
	for (i = 1; i < argc; i++) {
		task_new(hx_trans, 0, 0, 0);
		if (argv[i][0] != dir_char) {
			char buf[sizeof remote_cwd];

			snprintf(buf, sizeof buf, "%s%c%s", remote_cwd, dir_char, argv[i]);
			htlc_snd_file_mkdir(buf);
		} else
			htlc_snd_file_mkdir(argv[i]);
	}

	return 0;
}

int
cmd_rm (int argc, char *const *argv)
{
	register int i;

	if (argc < 2) {
		curscr_printf("\nusage: %s file1 [file2...]", argv[0]);
		return 0;
	}
	for (i = 1; i < argc; i++) {
		task_new(hx_trans, 0, 0, 0);
		if (argv[i][0] != dir_char) {
			char buf[sizeof remote_cwd];

			snprintf(buf, sizeof buf, "%s%c%s", remote_cwd, dir_char, argv[i]);
			htlc_snd_file_delete(buf);
		} else
			htlc_snd_file_delete(argv[i]);
	}

	return 0;
}
