/*
 dialog-notifylist.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static GtkWidget *clist = NULL;

static void clist_insert_notify(GtkWidget *clist, LIST_REC *notify, gint row)
{
    SERVER_REC *server;
    gchar *titles[5];

    g_return_if_fail(clist != NULL);
    g_return_if_fail(notify != NULL);

    server = notifylist_ison(notify->key, "");

    titles[0] = server == NULL ? _("No") : _("Yes");
    titles[1] = notify->key;
    titles[2] = server == NULL || server->ircnet == NULL ? "" : server->ircnet;
    titles[3] = "";
    titles[4] = "";

    if (row == -1)
        row = gtk_clist_append(GTK_CLIST(clist), titles);
    else
        row = gtk_clist_insert(GTK_CLIST(clist), row, titles);
    gtk_clist_set_row_data(GTK_CLIST(clist), row, notify);
}

static gint clist_remove_notify(GtkWidget *clist, LIST_REC *notify)
{
    gint row;

    row = gtk_clist_find_row_from_data(GTK_CLIST(clist), notify);
    if (row != -1) gtk_clist_remove(GTK_CLIST(clist), row);
    return row;
}

static void sig_query(GtkWidget *clist)
{
    SERVER_REC *server;
    LIST_REC *rec;
    gint selection;

    g_return_if_fail(clist != NULL);
    if (servers == NULL) return;

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1) return;

    rec = gtk_clist_get_row_data(GTK_CLIST(clist), selection);
    server = notifylist_ison(rec->key, rec->value);
    if (server != NULL)
        signal_emit("command query", 3, rec->key, server, cur_channel);
}

static void sig_send_msg_ok(GtkWidget *dialog)
{
    GtkWidget *clist, *entry;
    SERVER_REC *server;
    LIST_REC *rec;
    gint selection;
    gchar *str, *text;

    g_return_if_fail(dialog != NULL);

    clist = gtk_object_get_data(GTK_OBJECT(dialog), "data");
    entry = gtk_object_get_data(GTK_OBJECT(dialog), "entry");
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));

    if (*text != '\0' && servers != NULL && selection != -1)
    {
        rec = gtk_clist_get_row_data(GTK_CLIST(clist), selection);
        server = notifylist_ison(rec->key, rec->value);
        if (server != NULL)
        {
            str = g_strdup_printf("%s %s", rec->key, text);
            signal_emit("command msg", 3, str, server, cur_channel);
            g_free(str);

        }
    }

    gtk_widget_destroy(dialog);
}

static void sig_send_msg(GtkWidget *clist)
{
    g_return_if_fail(clist != NULL);
    if (servers == NULL) return;

    gui_entry_dialog(_("Message: "), NULL, NULL, sig_send_msg_ok, clist);
}

static void sig_select_row(GtkWidget *clist, gint row)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(row));
}

static void sig_unselect_row(GtkWidget *clist)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
}

static void sig_destroy(GtkWidget **dialog)
{
    g_return_if_fail(dialog != NULL);
    g_return_if_fail(*dialog != NULL);

    gtk_widget_destroy(*dialog);
    *dialog = NULL;
    clist = NULL;
}

static gboolean sig_notifylist_new(LIST_REC *rec)
{
    g_return_val_if_fail(rec != NULL, FALSE);

    if (clist != NULL)
        clist_insert_notify(clist, rec, -1);

    return TRUE;
}

static gboolean sig_notifylist_remove(LIST_REC *rec)
{
    g_return_val_if_fail(rec != NULL, FALSE);

    if (clist != NULL)
        clist_remove_notify(clist, rec);

    return TRUE;
}

static gboolean sig_notifylist_join(SERVER_REC *server, gchar *nick, gchar *user, gchar *host, gchar *realname)
{
    LIST_REC *rec;
    gint row;

    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(nick != NULL, FALSE);

    if (clist != NULL)
    {
        rec = notifylist_find(nick);
        row = gtk_clist_find_row_from_data(GTK_CLIST(clist), rec);
        if (row != -1)
        {
            gchar *str;

            str = g_strdup_printf("%s@%s", user, host);
            gtk_clist_set_text(GTK_CLIST(clist), row, 0, _("Yes"));
	    gtk_clist_set_text(GTK_CLIST(clist), row, 2, server->ircnet == NULL ? "" : server->ircnet);
            gtk_clist_set_text(GTK_CLIST(clist), row, 3, str);
            gtk_clist_set_text(GTK_CLIST(clist), row, 4, realname);
            g_free(str);
        }
    }

    /* create dialog */
    if (setup_get_bool("toggle_notifylist_popups"))
        gui_dialog(DIALOG_OK, _("%s (%s@%s) (%s) has joined IRC"), nick, user, host, realname);

    return TRUE;
}

static gboolean sig_notifylist_part(SERVER_REC *server, gchar *nick)
{
    LIST_REC *rec;
    gint row;

    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(nick != NULL, FALSE);

    if (clist != NULL)
    {
        rec = notifylist_find(nick);
        row = gtk_clist_find_row_from_data(GTK_CLIST(clist), rec);
        if (row != -1)
        {
            gtk_clist_set_text(GTK_CLIST(clist), row, 0, _("No"));
            gtk_clist_set_text(GTK_CLIST(clist), row, 2, "");
            gtk_clist_set_text(GTK_CLIST(clist), row, 3, "");
            gtk_clist_set_text(GTK_CLIST(clist), row, 4, "");
        }
    }

    return TRUE;
}

static void clist_click_column(GtkCList *clist, gint column)
{
    if (column != clist->sort_column)
	gtk_clist_set_sort_column(clist, column);
    else
    {
	if (clist->sort_type == GTK_SORT_ASCENDING)
	    clist->sort_type = GTK_SORT_DESCENDING;
	else
	    clist->sort_type = GTK_SORT_ASCENDING;
    }

    gtk_clist_sort(clist);
}

void dialog_notifylist(void)
{
    static GtkWidget *dialog = NULL;
    GtkWidget *scrollbox;
    gchar *titles[5];
    GList *tmp;

    if (dialog != NULL)
    {
        /* dialog already open */
        gdk_window_raise(dialog->window);
        return;
    }

    titles[0] = _("Joined"); titles[1] = _("Nick"); titles[2] = _("IRC net");
    titles[3] = _("Host"); titles[4] = _("Real Name");

    dialog = gnome_dialog_new(PACKAGE, _("Query"), _("Send msg"), GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_widget_set_usize(dialog, -1, 250);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "destroy",
                              GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &dialog);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    /* Create notify clist widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrollbox, TRUE, TRUE, 0);

    clist = gtk_clist_new_with_titles(5, titles);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 4, TRUE);
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);
    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
                       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "click_column",
		       GTK_SIGNAL_FUNC(clist_click_column), NULL);

    gtk_clist_freeze(GTK_CLIST(clist));
    gtk_clist_clear(GTK_CLIST(clist));
    for (tmp = g_list_first(notifies); tmp != NULL; tmp = tmp->next)
        clist_insert_notify(clist, tmp->data, -1);
    gtk_clist_thaw(GTK_CLIST(clist));

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_query), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(sig_send_msg), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 2, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    /* Make columns resizeable */
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 2, TRUE);
    gtk_widget_show_all(dialog);
}

void dialog_notifylist_init(void)
{
    signal_add("notifylist joined", (SIGNAL_FUNC) sig_notifylist_join);
    signal_add("notifylist left", (SIGNAL_FUNC) sig_notifylist_part);
    signal_add("notifylist new", (SIGNAL_FUNC) sig_notifylist_new);
    signal_add("notifylist remove", (SIGNAL_FUNC) sig_notifylist_remove);
}

void dialog_notifylist_deinit(void)
{
    signal_remove("notifylist joined", (SIGNAL_FUNC) sig_notifylist_join);
    signal_remove("notifylist left", (SIGNAL_FUNC) sig_notifylist_part);
    signal_remove("notifylist new", (SIGNAL_FUNC) sig_notifylist_new);
    signal_remove("notifylist remove", (SIGNAL_FUNC) sig_notifylist_remove);
}
