/*
 gui-statusbar-items.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

/* how often to redraw lagging time */
#define LAG_REFRESH_TIME 10
/* If we haven't been able to check lag for this long, "(??)" is added after
   the lag */
#define MAX_LAG_UNKNOWN_TIME 30

/* clock */
static gint clock_tag, clock_timetag;
static time_t clock_last;

/* nick */
static gint nick_tag;

/* channel */
static gint channel_tag;

/* activity */
static gint activity_tag;
static GList *activity_list;

/* more */
static gint more_tag;

/* lag */
static gint lag_tag, lag_timetag;
static time_t lag_last_draw;

/* redraw clock */
static void statusbar_clock(gint xpos, gint ypos, gint size)
{
    struct tm *tm;
    gchar str[5];

    clock_last = time(NULL);
    tm = localtime(&clock_last);

    sprintf(str, "%02d:%02d", tm->tm_hour, tm->tm_min);

    move(ypos, xpos);
    set_color((1 << 4)+3); addch('[');
    set_color((1 << 4)+15); addstr(str);
    set_color((1 << 4)+3); addch(']');

    screen_refresh();
}

/* check if we need to redraw clock.. */
static gint statusbar_clock_timeout(void)
{
    struct tm *tm;
    time_t t;
    gint min;

    tm = localtime(&clock_last);
    min = tm->tm_min;

    t = time(NULL);
    tm = localtime(&t);

    if (tm->tm_min != min)
    {
        /* minute changed, redraw! */
        gui_statusbar_redraw(clock_tag);
    }
    return 1;
}

/* redraw nick */
static void statusbar_nick(gint xpos, gint ypos, gint size)
{
    SERVER_REC *server;
    NICK_REC *nickrec;
    gint size_needed;
    gint umode_size;
    gchar nick[10];

    server = cur_channel == NULL ? NULL : cur_channel->server;

    umode_size = server == NULL || server->usermode == NULL ? 0 : strlen(server->usermode)+3;

    /* nick */
    if (server == NULL || server->nick == NULL)
    {
        nick[0] = '\0';
        nickrec = NULL;
    }
    else
    {
        strncpy(nick, server->nick, 9);
        nick[9] = '\0';
        nickrec = nicklist_find(cur_channel, server->nick);
    }

    size_needed = 2 + strlen(nick) + umode_size +
        (server != NULL && server->usermode_away ? 7 : 0) +
        (nickrec != NULL && (nickrec->op || nickrec->voice) ? 1 : 0); /* @ + */

    if (size != size_needed)
    {
        /* we need more (or less..) space! */
        gui_statusbar_resize(nick_tag, size_needed);
        return;
    }

    /* size ok, draw the nick */
    move(ypos, xpos);

    set_color((1 << 4)+3); addch('[');
    if (nickrec != NULL && (nickrec->op || nickrec->voice))
    {
        set_color((1 << 4)+15); addch(nickrec->op ? '@' : '+');
    }
    set_color((1 << 4)+7); addstr(nick);
    if (umode_size)
    {
        set_color((1 << 4)+15); addch('(');
        set_color((1 << 4)+3); addch('+');
        set_color((1 << 4)+7); addstr(server->usermode);
        set_color((1 << 4)+15); addch(')');
        if (server->usermode_away)
        {
            set_color((1 << 4)+7); addstr(" (");
            set_color((1 << 4)+10); addstr("zZzZ");
            set_color((1 << 4)+7); addch(')');
        }
    }
    set_color((1 << 4)+3); addch(']');
    screen_refresh();
}

static gboolean sig_statusbar_nick_redraw(void)
{
    gui_statusbar_redraw(nick_tag);
    return TRUE;
}

/* redraw channel */
static void statusbar_channel(gint xpos, gint ypos, gint size)
{
    SERVER_REC *server;
    gchar channame[21], *window, *mode;
    gint size_needed;
    gint mode_size;

    server = cur_channel == NULL || cur_channel->server == NULL ? NULL :
	cur_channel->server;

    window = cur_channel == NULL ? "0" :
	g_strdup_printf("%d", g_list_index(windows, CHANNEL_PARENT(cur_channel))+1);
    if (cur_channel == NULL || cur_channel->type == CHANNEL_TYPE_EMPTY)
    {
	/* display server tag */
        channame[0] = '\0';
        mode = NULL;
	mode_size = 0;

	size_needed = 3 + strlen(window) + (server == NULL ? 0 : strlen(server->tag));
    }
    else
    {
	/* display channel + mode */
        strncpy(channame, cur_channel->name, 20); channame[20] = '\0';

        mode = channel_get_mode(cur_channel);
        mode_size = strlen(mode);
        if (mode_size > 0) mode_size += 3; /* (+) */

	size_needed = 3 + strlen(window) + strlen(channame) + mode_size;
    }

    if (size != size_needed)
    {
        /* we need more (or less..) space! */
        gui_statusbar_resize(channel_tag, size_needed);
        if (mode != NULL) g_free(mode);
	if (cur_channel != NULL) g_free(window);
        return;
    }

    move(ypos, xpos);
    set_color((1 << 4)+3); addch('[');

    /* window number */
    set_color((1 << 4)+7); addstr(window);
    set_color((1 << 4)+3); addch(':');

    if (channame[0] == '\0' && server != NULL)
    {
	/* server tag */
	set_color((1 << 4)+7); addstr(server->tag);
    }
    else if (channame[0] != '\0')
    {
	/* channel + mode */
	set_color((1 << 4)+7); addstr(channame);
	if (mode_size)
	{
	    set_color((1 << 4)+15); addch('(');
	    set_color((1 << 4)+3); addch('+');
	    set_color((1 << 4)+7); addstr(mode);
	    set_color((1 << 4)+15); addch(')');
	}
    }
    set_color((1 << 4)+3); addch(']');
    screen_refresh();

    if (mode != NULL) g_free(mode);
    if (cur_channel != NULL) g_free(window);
}

static gboolean sig_statusbar_channel_redraw(void)
{
    gui_statusbar_redraw(channel_tag);
    return TRUE;
}

/* redraw activity */
static void statusbar_activity(gint xpos, gint ypos, gint size)
{
    WINDOW_REC *window;
    GList *tmp;
    gchar *str;
    gint size_needed;

    size_needed = 0;
    for (tmp = g_list_first(activity_list); tmp != NULL; tmp = tmp->next)
    {
	if (tmp != activity_list)
	    size_needed++;

	str = g_strdup_printf("%d", g_list_index(windows, tmp->data)+1);
        size_needed += strlen(str);
	g_free(str);
    }

    if (size_needed != 0)
	size_needed += 7;

    if (size != size_needed)
    {
        /* we need more (or less..) space! */
        gui_statusbar_resize(activity_tag, size_needed);
        return;
    }

    if (size == 0)
        return;

    move(ypos, xpos);
    set_color((1 << 4)+3); addch('[');
    set_color((1 << 4)+7); addstr("Act: ");

    for (tmp = activity_list; tmp != NULL; tmp = tmp->next)
    {
	window = tmp->data;

	if (tmp != activity_list)
	{
	    set_color((1 << 4)+3);
	    addch(',');
	}

	str = g_strdup_printf("%d", g_list_index(windows, tmp->data)+1);
	switch (window->new_data)
	{
	    case NEWDATA_TEXT:
		set_color((1 << 4)+7);
		break;
	    case NEWDATA_MSG:
		set_color((1 << 4)+15);
		break;
	    case NEWDATA_MSG_FORYOU:
		set_color((1 << 4)+13);
		break;
	}
	addstr(str);
	g_free(str);
    }

    set_color((1 << 4)+3); addch(']');
    screen_refresh();
}

static gboolean sig_statusbar_activity_hilight(WINDOW_REC *window, gpointer oldlevel)
{
    gint pos, inspos;
    GList *tmp;

    g_return_val_if_fail(window != NULL, FALSE);

    if (setup_get_bool("toggle_actlist_moves"))
    {
	/* Move the window to the first in the activity list */
	if (g_list_find(activity_list, window) != NULL)
	    activity_list = g_list_remove(activity_list, window);
	if (window->new_data != 0)
	    activity_list = g_list_prepend(activity_list, window);
	gui_statusbar_redraw(activity_tag);
	return TRUE;
    }

    if (g_list_find(activity_list, window) != NULL)
    {
	/* already in activity list */
	if (window->new_data == 0)
	{
	    /* remove from activity list */
	    activity_list = g_list_remove(activity_list, window);
	    gui_statusbar_redraw(activity_tag);
	}
	else if (window->new_data != GPOINTER_TO_INT(oldlevel))
	{
	    /* different level as last time, just redraw it. */
	    gui_statusbar_redraw(activity_tag);
	}
        return TRUE;
    }

    /* add window to activity list .. */
    pos = g_list_index(windows, window);

    inspos = 0;
    for (tmp = activity_list; tmp != NULL; tmp = tmp->next, inspos++)
    {
	if (pos < g_list_index(windows, tmp->data))
	{
	    activity_list = g_list_insert(activity_list, window, inspos);
	    break;
	}
    }
    if (tmp == NULL)
	activity_list = g_list_append(activity_list, window);

    //if (((WINDOW_REC *) activity_list->data)->new_data == NEWDATA_MSG_FORYOU)
    //if (activity_list->data == window)

    gui_statusbar_redraw(activity_tag);
    return TRUE;
}

static gboolean sig_statusbar_activity_window_destroyed(WINDOW_REC *window)
{
    g_return_val_if_fail(window != NULL, FALSE);

    if (g_list_find(activity_list, window) != NULL)
    {
        activity_list = g_list_remove(activity_list, window);
        gui_statusbar_redraw(activity_tag);
    }

    return TRUE;
}

/* redraw -- more -- */
static void statusbar_more(gint xpos, gint ypos, gint size)
{
    if (size != 10) return;

    move(ypos, xpos);
    set_color((1 << 4)+3); addstr("-- more --");
}

static gboolean sig_statusbar_more_check_remove(CHANNEL_REC *channel)
{
    WINDOW_REC *window;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    if (WINDOW_GUI(window)->parent->active != window)
        return TRUE;

    if (more_tag != -1 && window->lines-WINDOW_GUI(window)->starty <= LINES-1)
    {
        gui_statusbar_remove(more_tag);
        more_tag = -1;
    }

    return TRUE;
}

static gboolean sig_statusbar_more_check(CHANNEL_REC *channel)
{
    WINDOW_REC *window;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    if (WINDOW_GUI(window)->parent->active != window)
        return TRUE;

    if (window->lines-WINDOW_GUI(window)->starty > LINES-1)
    {
        if (more_tag == -1)
            more_tag = gui_statusbar_allocate(10, FALSE, statusbar_more);
    }
    else if (more_tag != -1)
    {
        gui_statusbar_remove(more_tag);
        more_tag = -1;
    }

    return TRUE;
}

static void statusbar_lag(gint xpos, gint ypos, gint size)
{
    SERVER_REC *server;
    GString *str;
    gint size_needed;
    time_t now;

    now = time(NULL);
    str = g_string_new(NULL);

    server = cur_channel == NULL ? NULL : cur_channel->server;
    if (server == NULL || server->lag_last_check == 0)
	size_needed = 0;
    else
    {
	if (server->lag_sent == 0 || now-server->lag_sent < 5)
	{
	    g_string_sprintf(str, "%d.%02d", server->lag/1000, (server->lag % 1000)/10);
	    if (now-server->lag_last_check > MAX_LAG_UNKNOWN_TIME)
		g_string_append(str, "(??) ");
	}
	else
	{
	    /* big lag, still waiting .. */
	    g_string_sprintf(str, "%ld (??)", now-server->lag_sent);
	}
	size_needed = str->len+7;
    }

    if (size != size_needed)
    {
        /* we need more (or less..) space! */
        gui_statusbar_resize(lag_tag, size_needed);
        g_string_free(str, TRUE);
        return;
    }

    if (size != 0)
    {
	lag_last_draw = now;
	move(ypos, xpos);
	set_color((1 << 4)+3); addch('[');
	set_color((1 << 4)+7); addstr("Lag: ");

	set_color((1 << 4)+15); addstr(str->str);
	set_color((1 << 4)+3); addch(']');
    }
    g_string_free(str, TRUE);
}

static gboolean sig_statusbar_lag_redraw(void)
{
    gui_statusbar_redraw(lag_tag);
    return TRUE;
}

static gint statusbar_lag_timeout(void)
{
    /* refresh statusbar every 10 seconds */
    if (time(NULL)-lag_last_draw < LAG_REFRESH_TIME)
	return 1;

    gui_statusbar_redraw(lag_tag);
    return 1;
}

void gui_statusbar_items_init(void)
{
    /* clock */
    clock_tag = gui_statusbar_allocate(7, FALSE, statusbar_clock);
    clock_timetag = gui_timeout_add(1000, (GUITimeoutFunction) statusbar_clock_timeout, NULL);

    /* nick */
    nick_tag = gui_statusbar_allocate(2, FALSE, statusbar_nick);
    signal_add("server connected", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_add("channel wholist", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_add("channel changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_add("nick mode changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_add("user mode changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_add("server nick changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_add("channel server changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);

    /* channel */
    channel_tag = gui_statusbar_allocate(2, FALSE, statusbar_channel);
    signal_add("channel changed", (SIGNAL_FUNC) sig_statusbar_channel_redraw);
    signal_add("channel mode changed", (SIGNAL_FUNC) sig_statusbar_channel_redraw);
    signal_add("channel server changed", (SIGNAL_FUNC) sig_statusbar_channel_redraw);

    /* activity */
    activity_list = NULL;
    activity_tag = gui_statusbar_allocate(0, FALSE, statusbar_activity);
    signal_add("gui window activity", (SIGNAL_FUNC) sig_statusbar_activity_hilight);
    signal_add("window destroyed", (SIGNAL_FUNC) sig_statusbar_activity_window_destroyed);

    /* more */
    more_tag = -1;
    signal_add("gui page scrolled", (SIGNAL_FUNC) sig_statusbar_more_check_remove);
    signal_add("channel changed", (SIGNAL_FUNC) sig_statusbar_more_check);
    signal_add("gui print text", (SIGNAL_FUNC) sig_statusbar_more_check);

    /* lag */
    lag_tag = gui_statusbar_allocate(0, FALSE, statusbar_lag);
    lag_timetag = gui_timeout_add(1000*LAG_REFRESH_TIME, (GUITimeoutFunction) statusbar_lag_timeout, NULL);
    signal_add("server lag", (SIGNAL_FUNC) sig_statusbar_lag_redraw);
    signal_add("channel server changed", (SIGNAL_FUNC) sig_statusbar_lag_redraw);
}

void gui_statusbar_items_deinit(void)
{
    /* clock */
    gui_statusbar_remove(clock_tag);

    /* nick */
    gui_statusbar_remove(nick_tag);
    gui_timeout_remove(clock_timetag);
    signal_remove("server connected", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_remove("channel wholist", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_remove("channel changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_remove("nick mode changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_remove("user mode changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_remove("server nick changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);
    signal_remove("channel server changed", (SIGNAL_FUNC) sig_statusbar_nick_redraw);

    /* channel */
    gui_statusbar_remove(channel_tag);
    signal_remove("channel changed", (SIGNAL_FUNC) sig_statusbar_channel_redraw);
    signal_remove("channel mode changed", (SIGNAL_FUNC) sig_statusbar_channel_redraw);

    /* activity */
    gui_statusbar_remove(activity_tag);
    signal_remove("gui window activity", (SIGNAL_FUNC) sig_statusbar_activity_hilight);
    signal_remove("window destroyed", (SIGNAL_FUNC) sig_statusbar_activity_window_destroyed);
    g_list_free(activity_list);

    /* more */
    if (more_tag != -1) gui_statusbar_remove(more_tag);
    signal_remove("gui page scrolled", (SIGNAL_FUNC) sig_statusbar_more_check_remove);
    signal_remove("channel changed", (SIGNAL_FUNC) sig_statusbar_more_check);
    signal_remove("gui print text", (SIGNAL_FUNC) sig_statusbar_more_check);

    /* lag */
    gui_statusbar_remove(lag_tag);
    gui_timeout_remove(lag_timetag);
    signal_remove("server lag", (SIGNAL_FUNC) sig_statusbar_lag_redraw);
    signal_remove("channel server changed", (SIGNAL_FUNC) sig_statusbar_lag_redraw);
}
