/*
 server-idle.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

typedef struct
{
    gchar *event;
    gchar *signal;
    gint argpos;
}
REDIRECT_IDLE_REC;

typedef struct
{
    gchar *cmd;
    gchar *arg;
    gint tag;

    gint last;
    GList *redirects;
}
SERVER_IDLE_REC;

static gint idle_tag, idlepos;

/* Add new idle command to queue */
static SERVER_IDLE_REC *server_idle_create(SERVER_REC *server, gchar *cmd, gchar *arg, gint last, va_list args)
{
    REDIRECT_IDLE_REC *rrec;
    SERVER_IDLE_REC *rec;
    gchar *event;

    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(cmd != NULL, FALSE);

    rec = g_new0(SERVER_IDLE_REC, 1);

    rec->tag = ++idlepos;
    rec->arg = arg == NULL ? NULL : g_strdup(arg);
    rec->cmd = g_strdup(cmd);
    rec->last = last;

    while ((event = va_arg(args, gchar *)) != NULL)
    {
	rrec = g_new(REDIRECT_IDLE_REC, 1);
	rec->redirects = g_list_append(rec->redirects, rrec);

        rrec->event = g_strdup(event);
        rrec->signal = g_strdup(va_arg(args, gchar *));
	rrec->argpos = va_arg(args, gint);
    }

    return rec;
}

static SERVER_IDLE_REC *server_idle_find_rec(SERVER_REC *server, gint tag)
{
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    for (tmp = g_list_first(server->idles); tmp != NULL; tmp = tmp->next)
    {
	SERVER_IDLE_REC *rec = tmp->data;

	if (rec->tag == tag)
	    return rec;
    }

    return NULL;
}

/* Add new idle command to queue */
gint server_idle_add(SERVER_REC *server, gchar *cmd, gchar *arg, gint last, ...)
{
    va_list args;
    SERVER_IDLE_REC *rec;

    g_return_val_if_fail(server != NULL, -1);

    va_start(args, last);
    rec = server_idle_create(server, cmd, arg, last, args);
    server->idles = g_list_append(server->idles, rec);
    va_end(args);
    return rec->tag;
}

/* Add new idle command to first of queue */
gint server_idle_add_first(SERVER_REC *server, gchar *cmd, gchar *arg, gint last, ...)
{
    va_list args;
    SERVER_IDLE_REC *rec;

    g_return_val_if_fail(server != NULL, -1);

    va_start(args, last);
    rec = server_idle_create(server, cmd, arg, last, args);
    server->idles = g_list_prepend(server->idles, rec);
    va_end(args);
    return rec->tag;
}

/* Add new idle command to specified position of queue */
gint server_idle_insert(SERVER_REC *server, gchar *cmd, gchar *arg, gint tag, gint last, ...)
{
    va_list args;
    SERVER_IDLE_REC *rec;
    gint pos;

    g_return_val_if_fail(server != NULL, -1);

    va_start(args, last);

    /* find the position of tag in idle list */
    rec = server_idle_find_rec(server, tag);
    pos = g_list_index(server->idles, rec);

    rec = server_idle_create(server, cmd, arg, last, args);
    if (pos < 0)
	server->idles = g_list_append(server->idles, rec);
    else
	server->idles = g_list_insert(server->idles, rec, pos);
    va_end(args);
    return rec->tag;
}

static void server_idle_destroy(SERVER_REC *server, SERVER_IDLE_REC *rec)
{
    GList *tmp;

    g_return_if_fail(server != NULL);

    server->idles = g_list_remove(server->idles, rec);

    for (tmp = g_list_first(rec->redirects); tmp != NULL; tmp = tmp->next)
    {
	REDIRECT_IDLE_REC *rec = tmp->data;

	g_free(rec->event);
        g_free(rec->signal);
	g_free(rec);
    }
    g_list_free(rec->redirects);

    if (rec->arg != NULL) g_free(rec->arg);
    g_free(rec->cmd);
    g_free(rec);
}

/* Check if record is still in queue */
gboolean server_idle_find(SERVER_REC *server, gint tag)
{
    return server_idle_find_rec(server, tag) != NULL;
}

/* Remove record from idle queue */
gboolean server_idle_remove(SERVER_REC *server, gint tag)
{
    SERVER_IDLE_REC *rec;

    g_return_val_if_fail(server != NULL, FALSE);

    rec = server_idle_find_rec(server, tag);
    if (rec == NULL)
	return FALSE;

    server_idle_destroy(server, rec);
    return TRUE;
}

/* Execute next idle command */
static void server_idle_next(SERVER_REC *server)
{
    SERVER_IDLE_REC *rec;
    GList *tmp;
    gint group;

    g_return_if_fail(server != NULL);

    if (server->idles == NULL) return;
    rec = server->idles->data;

    /* Send command */
    irc_send_cmd(server, rec->cmd);

    /* Add server redirections */
    group = 0;
    for (tmp = rec->redirects; tmp != NULL; tmp = tmp->next)
    {
	REDIRECT_IDLE_REC *rrec = tmp->data;

	group = server_redirect_single_event(server, rec->arg, rec->last > 0,
					     group, rrec->event, rrec->signal, rrec->argpos);
	if (rec->last > 0) rec->last--;
    }

    server_idle_destroy(server, rec);
}

static gboolean sig_disconnected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    while (server->idles != NULL)
	server_idle_destroy(server, server->idles->data);
    return TRUE;
}

static gint sig_idle_timeout(void)
{
    GList *tmp;

    /* Scan through every server */
    for (tmp = g_list_first(servers); tmp != NULL; tmp = tmp->next)
    {
	SERVER_REC *rec = tmp->data;

	if (rec->idles != NULL && rec->cmdcount == 0)
	{
	    /* We're idling and we have idle commands to run! */
	    server_idle_next(rec);
	}
    }
    return TRUE;
}

void servers_idle_init(void)
{
    idlepos = 0;
    idle_tag = gui_timeout_add(1000, (GUITimeoutFunction) sig_idle_timeout, NULL);

    signal_add("server disconnected", (SIGNAL_FUNC) sig_disconnected);
}

void servers_idle_deinit(void)
{
    gui_timeout_remove(idle_tag);
    signal_remove("server disconnected", (SIGNAL_FUNC) sig_disconnected);
}
