/*
 ui-commands.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

typedef struct
{
    gchar *ban;
    gint timeleft;
}
KNOCKOUT_REC;

static gint knockout_tag;

static gboolean cmd_kickban(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *params, *nick;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 1, &nick);
    if (*nick == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    signal_emit("command ban", 3, nick, server, channel);
    signal_emit("command kick", 3, data, server, channel);
    g_free(params);

    return TRUE;
}

/* timeout function: knockout */
static void knockout_timeout_server(SERVER_REC *server)
{
    GList *tmp, *next;
    time_t t;

    g_return_if_fail(server != NULL);

    t = server->knockout_lastcheck == 0 ? 0 :
	time(NULL)-server->knockout_lastcheck;
    server->knockout_lastcheck = time(NULL);

    for (tmp = g_list_first(server->knockoutlist); tmp != NULL; tmp = next)
    {
        KNOCKOUT_REC *rec;

        next = tmp->next;
        rec = (KNOCKOUT_REC *) tmp->data;
        if (rec->timeleft > t)
            rec->timeleft -= t;
        else
        {
            /* timeout, unban. */
            signal_emit("command unban", 3, rec->ban, server, cur_channel);
            server->knockoutlist = g_list_remove(server->knockoutlist, rec);
            g_free(rec->ban);
            g_free(rec);
        }
    }
}

static gint knockout_timeout(void)
{
    g_list_foreach(servers, (GFunc) knockout_timeout_server, NULL);
    return 1;
}

static gboolean cmd_knockout(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    KNOCKOUT_REC *rec;
    gchar *params, *nick, *reason, *ban, *timeoutstr, *ptr;
    gint timeleft;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(channel != NULL, FALSE);
    if (server == NULL) cmd_return_error(CMDERR_NOT_CONNECTED);

    /* first argument is number? */
    for (ptr = data; *ptr != '\0' && !isspace((gint) *ptr); ptr++)
        if (!isdigit((gint) *ptr)) break;

    if (!isspace((gint) *ptr) && *ptr != '\0')
    {
        timeoutstr = NULL;
        params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &nick, &reason);
    }
    else
    {
        /* yep, first argument is the timeout */
        params = cmd_get_params(data, 3 | PARAM_FLAG_GETREST, &timeoutstr, &nick, &reason);
    }
    if (timeoutstr == NULL || sscanf(timeoutstr, "%d", &timeleft) != 1)
        timeleft = setup_get_int("knockout_time");
    if (*nick == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    signal_emit("command ban", 3, nick, server, channel);
    ptr = g_strdup_printf("%s %s", nick, reason);
    signal_emit("command kick", 3, ptr, server, channel);
    g_free(ptr);

    /* create knockout record */
    rec = g_new(KNOCKOUT_REC, 1);
    rec->timeleft = timeleft;
    ban = ban_get_mask(channel, nick);
    rec->ban = g_strdup_printf("%s %s", channel->name, ban);
    g_free(ban);

    server->knockoutlist = g_list_append(server->knockoutlist, rec);

    g_free(params);
    return TRUE;
}

static gboolean server_disconnected(SERVER_REC *server)
{
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    for (tmp = g_list_first(server->knockoutlist); tmp != NULL; tmp = tmp->next)
    {
        KNOCKOUT_REC *rec = tmp->data;

        g_free(rec->ban);
        g_free(rec);
    }
    g_list_free(server->knockoutlist);

    return TRUE;
}

void extra_commands_init(void)
{
    knockout_tag = gui_timeout_add(KNOCKOUT_TIMECHECK, (GUITimeoutFunction) knockout_timeout, NULL);
    command_bind("kickban", NULL, (SIGNAL_FUNC) cmd_kickban);
    command_bind("knockout", NULL, (SIGNAL_FUNC) cmd_knockout);
    signal_add("server disconnected", (SIGNAL_FUNC) server_disconnected);
}

void extra_commands_deinit(void)
{
    gui_timeout_remove(knockout_tag);
    command_unbind("kickban", (SIGNAL_FUNC) cmd_kickban);
    command_unbind("knockout", (SIGNAL_FUNC) cmd_knockout);
    signal_remove("server disconnected", (SIGNAL_FUNC) server_disconnected);
}
