/*
 ui-ctcp.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gboolean ctcp_print(gchar *pre, gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    gchar *str;

    g_return_val_if_fail(data != NULL, FALSE);

    str = g_strconcat(pre, " ", data, NULL);
    printformat(server, ischannel(*target) ? target : nick, MSGLEVEL_CTCPS,
                IRCTXT_CTCP_REQUESTED, nick, addr, str, target);
    g_free(str);

    return TRUE;
}

static gboolean ctcp_default_msg(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    return ctcp_print("unknown CTCP", data, server, nick, addr, target);
}

static gboolean ctcp_ping_msg(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    return ctcp_print("CTCP PING", data, server, nick, addr, target);
}

static gboolean ctcp_version_msg(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    return ctcp_print("CTCP VERSION", data, server, nick, addr, target);
}

static gboolean ctcp_action_msg(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    CHANNEL_REC *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    if (ignore_check(server, nick, addr, MSGLEVEL_ACTIONS) ||
	(ischannel(*target) && ignore_check(server, target, NULL, MSGLEVEL_ACTIONS)))
    {
	/* ignored */
    }
    else if (ischannel(*target))
    {
        /* channel action */
        channel = channel_find(server, target);

        if (channel != NULL && CHANNEL_PARENT(channel)->active == channel)
        {
            /* message to active channel in window */
            printformat(server, target, MSGLEVEL_ACTIONS,
                        IRCTXT_ACTION_PUBLIC, nick, data);
        }
        else
        {
            /* message to not existing/active channel */
            printformat(server, target, MSGLEVEL_ACTIONS,
                        IRCTXT_ACTION_PUBLIC_CHANNEL, nick, target, data);
        }
    }
    else
    {
        /* private action */
        if (setup_get_bool("toggle_autocreate_query") && channel_find(server, nick) == NULL)
	    channel = channel_create(server, nick, CHANNEL_TYPE_QUERY);
	else
	    channel = channel_find(server, nick);

	if (setup_get_bool("toggle_privmsg_beeps"))
	    ui_beep();

	printformat(server, nick, MSGLEVEL_ACTIONS,
		    channel == NULL ? IRCTXT_ACTION_PRIVATE : IRCTXT_ACTION_PRIVATE_QUERY, nick, addr == NULL ? "" : addr, data);
    }

    return TRUE;
}

static gboolean ctcp_default_reply(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    gchar *ptr;

    g_return_val_if_fail(data != NULL, FALSE);

    data = g_strdup(data);
    ptr = strchr(data, ' ');
    if (ptr != NULL) *ptr++ = '\0'; else ptr = "";

    printformat(server, ischannel(*target) ? target : nick, MSGLEVEL_CTCPS,
                IRCTXT_CTCP_REPLY, data, nick, ptr);
    g_free(data);

    return TRUE;
}

static gboolean ctcp_ping_reply(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    struct timeval tv;
    glong secs, usecs;

    g_return_val_if_fail(data != NULL, FALSE);

    if (sscanf(data, "%ld %ld", &secs, &usecs) == 2 && gettimeofday(&tv, NULL) == 0)
    {
        secs = tv.tv_sec - secs;
        usecs = tv.tv_usec - usecs;
        if (usecs < 0)
        {
            usecs += 1000000;
            secs--;
        }

        printformat(server, ischannel(*target) ? target : nick, MSGLEVEL_CTCPS,
                    IRCTXT_CTCP_PING_REPLY, nick, secs, usecs/1000);
    }

    return TRUE;
}

void ui_ctcp_init(void)
{
    signal_add("default ctcp msg", (SIGNAL_FUNC) ctcp_default_msg);
    signal_add("ctcp msg ping", (SIGNAL_FUNC) ctcp_ping_msg);
    signal_add("ctcp msg version", (SIGNAL_FUNC) ctcp_version_msg);
    signal_add("ctcp msg action", (SIGNAL_FUNC) ctcp_action_msg);
    signal_add("default ctcp reply", (SIGNAL_FUNC) ctcp_default_reply);
    signal_add("ctcp reply ping", (SIGNAL_FUNC) ctcp_ping_reply);
}

void ui_ctcp_deinit(void)
{
    signal_remove("default ctcp msg", (SIGNAL_FUNC) ctcp_default_msg);
    signal_remove("ctcp msg ping", (SIGNAL_FUNC) ctcp_ping_msg);
    signal_remove("ctcp msg version", (SIGNAL_FUNC) ctcp_version_msg);
    signal_remove("ctcp msg action", (SIGNAL_FUNC) ctcp_action_msg);
    signal_remove("default ctcp reply", (SIGNAL_FUNC) ctcp_default_reply);
    signal_remove("ctcp reply ping", (SIGNAL_FUNC) ctcp_ping_reply);
}
