/*****************************************************************************
 * UNIX Command Line Masqdialer Client v2.8                                  *
 * July 12, 1999                                                            *
 * Copyright (c) 1998, 1999 Kevin Lindsay and Charles P. Wright	             *
 *                                                                           *
 * Patches:  Brad Garcia <bgarcia@telerama.com>                              *
 *              - Fixed a bug where the client was not always quiting from   *
 *                server properly.                                           *
 *                                                                           *
 * For use with the C Masqdialer server by Charles P. Wright                 *
 *                                                                           *
 * Report any Bugs to: cpwright@villagenet.com                               *
 *****************************************************************************/

/* 
   Copyright (C) 1998,1999  Kevin Lindsay and Charles P. Wright

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation.
 */


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <sys/time.h>
#include <unistd.h>
#include <ctype.h>

/* Globals */
#define VERSION "Unix Command Line Masqdialer Client v2.7\nCopyright (c) 1998, 1999 Kevin Lindsay and Charles P. Wright\n"
#define BUFSIZE 4096

#ifndef DEFAULTPORT
#define DEFAULTPORT 224
#endif

#define false 0
#define true 1

char *progname;
int VERBOSE = 0;
int port = 0;

/* Usage */
void usage()
{
	printf("\n%s\n"
	       "usage: %s [-p port] <-dv> -h hostname[:port] -c connectin -U user -P password\n"
	       "usage: %s [-p port] <-Llsku> -h hostname[:port] -U user -P password\n"
	       "usage: %s -V\n\n"
	       "OPTIONS\n\n"
	"   -h hostname[:port]  - Specify hostname with optional port.\n"
	       "   -p port             - Specify port. Overrides -h port option.\n"
	       "   -c connection       - Specify connection name.\n"
	    "   -U user             - Specify Username for connection.\n"
	  "   -P password         - Specify Password for connection.\n\n"
	       "   -d                  - Dial specified connection.\n"
	       "   -l                  - List available connections.\n"
	       "   -L                  - Lock the modem.\n"
	"   -s                  - Request status of current connection.\n"
	       "   -k                  - Kill current connectin.\n\n"
	       "   -u                  - Unlock the modem.\n\n"
	       "   -v                  - Verbose Mode.\n"
	       "   -V                  - Display Version.\n\n"
	       "", VERSION, progname, progname, progname);
	exit(0);
}

void getline(int sock, char *buffer, int size)
{
	int bytes, grow;
	char *fullbuf;
	char tempbuf[BUFSIZE];

	fullbuf = malloc(1);
	fullbuf[0] = '\0';
	grow = 1;

	while ((bytes = recv(sock, tempbuf, 1, 0)) > 0)
	{
		grow += bytes;
		fullbuf = realloc(fullbuf, grow);

		strncat(fullbuf, tempbuf, bytes);

		fullbuf[grow - 1] = 0;

		if (fullbuf[strlen(fullbuf) - 1] == '\n')
		{
			break;
		}
	}

	strncpy(buffer, fullbuf, size);
#ifdef DEBUG	
	printf("recieved: %s", buffer);
#endif
}

void putline(int sock, char *writedata)
{
	char temp[BUFSIZE];

	strncpy(temp, writedata, BUFSIZE);	
	strncat(temp, "\r\n", BUFSIZE);
	send(sock, temp, strlen(temp), 0);

#ifdef DEBUG	
	printf("sent: %s\n", writedata);
#endif
}

/* Get READY from server */
void getready(int sock)
{
	char buffer[BUFSIZE];

	do
	{
		getline(sock, buffer, BUFSIZE);
	}
	while (!strstr(buffer, "READY"));
}

/* Get Host information - Parse PORT if necessary */
char *get_host(char *str)
{
	char *ptr1;
	char *hoststr;

	if ((ptr1 = strchr(str, ':')) == NULL)
	{
		hoststr = malloc(strlen(str) + 1);
		hoststr[0] = '\0';
		strcat(hoststr, str);
	}
	else
	{
		hoststr = malloc((ptr1 - str) + 1);
		hoststr[0] = '\0';
		strncat(hoststr, str, ptr1 - str);

		
		if (!port)
		{
			ptr1++;
			port = atoi(ptr1);
		}
	}

	return (hoststr);
}

/* Print Dial */
int print_dial(int sock)
{
	int res = 0;
	
	char buffer[BUFSIZE];

	printf("\nDialing ... \n");

	getline(sock, buffer, BUFSIZE);
	if (!strncmp(buffer, "BEGIN\n", BUFSIZE))
	{
		getline(sock, buffer, BUFSIZE);

		while (!strstr(buffer, "END"))
		{
			if (VERBOSE)
			{
				printf("%s", buffer);
			}
			getline(sock, buffer, BUFSIZE);
		}
		printf("\n");
	}
	else
	{
		printf("%s\n", buffer);
	}

	getready(sock);
	return (res);
}

/* Print List */
int print_list(int sock)
{
	int res = 0;

	char buffer[BUFSIZE];

	printf("\nDialup Connections Available\n");
	printf("----------------------------\n");
	
	getline(sock, buffer, BUFSIZE);
	if (!strncmp(buffer, "BEGIN\n", BUFSIZE))
	{
		getline(sock, buffer, BUFSIZE);

		while (!strstr(buffer, "END"))
		{
			printf("%s", buffer);
			getline(sock, buffer, BUFSIZE);
		}
	}
	printf("\n");

	getready(sock);

	return (res);
}

/* Print Kill */
int print_kill(int sock)
{
	int res = 0;
	char retval[BUFSIZE];

	getline(sock, retval, BUFSIZE);

	if (!strncmp(retval, "SUCCESS\n", BUFSIZE))
	{
		printf("\nConnection terminated!\n\n");
	}
	else
	{
		printf("\n%s\n\n", retval);
	}

	getready(sock);

	return (res);
}

/* Print Lock */
int print_lock(int sock)
{
	int res = 0;
	char retval[BUFSIZE];

	getline(sock, retval, BUFSIZE);
	getready(sock);

	return (res);
}

/* Print Connection Status */
int print_stat(int sock)
{
	int res = 0;
	char statline[BUFSIZE];

	char *tmpspeed;
	char *tmpname;

	char *tok;

	time_t contime;
	time_t curtime;
	long timediff;


	getline(sock, statline, BUFSIZE);
	getready(sock);

	printf("\nConnection Status\n");
	printf("-----------------\n\n");

	if (!strcmp(statline, "DOWN\n"))
	{
		printf("No connections!\n\n");
		return (res);
	}

	if (statline[0] != '\0')
	{
		tmpname = strchr(statline, ':') + 1;
		*strchr(statline, ':') = '\0';

		tmpspeed = strchr(tmpname, ':') + 1;
		*strchr(tmpname, ':') = '\0';
	
		printf("  Connection: %s\n", tmpname);
		printf("      Status: %s\n", statline);
		printf("       Speed: %s\n", tmpspeed);
	}

	putline(sock, "CTIME");
	getline(sock, statline, BUFSIZE);
	getready(sock);
	curtime = (time_t) atol(statline);
	
	putline(sock, "TIME");
	getline(sock, statline, BUFSIZE);
	getready(sock);
	contime = (time_t) atol(statline);

	timediff = (long) (curtime - contime);
	printf("    Duration: %ld Seconds\n\n", timediff);

	putline(sock, "NETLOAD");
	getline(sock, statline, BUFSIZE);
	if (!strcmp(statline, "BEGIN\n"))
	{
		getline(sock, statline, BUFSIZE);
		do
		{	
			if (!strncmp(statline, "INPUT: ", 7))
			{
				tok = strtok(statline, " ");

				tok = strtok(NULL, " ");
				if (tok)
					printf("    Bytes In: %s\n", tok);

				tok = strtok(NULL, " ");
				if (tok)
					printf("  Packets In: %s", tok);
			}
		
			if (!strncmp(statline, "OUTPUT: ", 8))
			{
				tok = strtok(statline, " ");

				tok = strtok(NULL, " ");
				if (tok)
					printf("   Bytes Out: %s\n", tok);

				tok = strtok(NULL, " ");
				if (tok)
					printf(" Packets Out: %s", tok);
			}

			getline(sock, statline, BUFSIZE);
		}
		while(strcmp(statline, "END\n"));
	}
	getready(sock);

	printf ("\n\n");

	return (res);
}

/* Authenticate the user */

int authenticate(int sock, char *user, char *pass)
{
	int res;
	char buffer[BUFSIZE];

	if (user == NULL || pass == NULL)
	{
		res = true;
	}
	else
	{
		snprintf(buffer, BUFSIZE, "USER:%s", user);
		putline(sock, buffer);
		getready(sock);

		snprintf(buffer, BUFSIZE, "PASS:%s", pass);
		putline(sock, buffer);
		getready(sock);

		putline(sock, "AUTH");
		getline(sock, buffer, BUFSIZE);

		if (!strncmp(buffer, "NOT AUTHORIZED", BUFSIZE))
		{
			res = false;
		}
		else
		{
			res = true;

			if (VERBOSE && res)
			{
				fprintf(stdout, "\nAuthentication Successful!\n");
			}
		}

		getready(sock);
	}

	return res;
}

/* Process the proper command */
int do_command(char *hoststr, char *command, char *connection, char *user, char *pass)
{
	int sock;
	struct sockaddr_in address;
	struct in_addr inaddr;
	struct hostent *host;
	int res = 0;
	char buffer[BUFSIZE];
	char temp[BUFSIZE];

	char mycommand[1024];

#ifdef SOLARIS
	in_addr_t tempaddr;
#endif
	strncpy(mycommand, command, 1024);

#ifdef SOLARIS

	tempaddr = inet_addr(hoststr);
	memcpy(&inaddr, &tempaddr, sizeof(struct in_addr));
	if (tempaddr != -1)
	{
		host = gethostbyaddr((char *) &inaddr, sizeof(inaddr), AF_INET);
	}
	else
	{
		host = gethostbyname(hoststr);
	}
#else
/*	This is a nice clean way, but Solaris chokes on it */
	if (inet_aton(hoststr, &inaddr))
	{
		host = gethostbyaddr((char *) &inaddr, sizeof(inaddr), AF_INET);
	}
	else
	{
		host = gethostbyname(hoststr);
	}
#endif

	if (!host)
	{
		fprintf(stderr, "do_command(): Could not find IP address for %s\n", hoststr);
		exit(1);
	}

	if (VERBOSE)
	{
		printf("\nConnecting to server: %s\n", hoststr);
	}

	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		fprintf(stderr, "do_command(): socket: (%d) %s\n", errno, strerror(errno));
		exit(1);
	}

	address.sin_family = AF_INET;
	address.sin_port = htons(port);

	memcpy(&address.sin_addr, host->h_addr_list[0], sizeof(address.sin_addr));

	errno = 0;

	if (connect(sock, (struct sockaddr *) &address, sizeof(address)))
	{
		fprintf(stderr, "do_command(): connect: (%d) %s\n", errno, strerror(errno));
		exit(1);
	}

	if (errno != 0)
	{
		fprintf(stderr, "do_command(): connect: (%d) %s\n", errno, strerror(errno));
	}

	getline(sock, buffer, BUFSIZE);

	if (strcmp(buffer, "READY\n") != 0)
	{
		fprintf(stderr, "do_command(): connect: remote host not a masqdialer server\n");
		exit(1);
	}

	if (VERBOSE)
		printf("\nConnection Established...\n");

	if (!authenticate(sock, user, pass))
	{
		fprintf(stderr, "\n%s: Authentication failed!\n", progname);
		exit(1);
	}

	if (!strcmp(mycommand, "DIAL"))
	{
		strncat(mycommand, ":", 1024);
		if (!strncmp(connection, "default", BUFSIZE))
		{
			putline(sock, "LIST");
			getline(sock, temp, BUFSIZE);
			getline(sock, temp, BUFSIZE);
			connection = strdup(temp);
			if (VERBOSE)
				printf ("\nUsing default connection %s\n", connection);
			getready(sock);	
		}
		strncat(mycommand, connection, 1024);
	}

	putline(sock, mycommand);

	if (!strcmp(mycommand, "LIST"))
		print_list(sock);
	else if (!strcmp(mycommand, "STAT"))
		print_stat(sock);
	else if (!strcmp(mycommand, "KILL"))
		print_kill(sock);
	else if (strstr(mycommand, "DIAL"))
		print_dial(sock);
	else if (strstr(mycommand, "LOCK"))
		print_lock(sock);

	putline(sock, "QUIT");

	return (res);
}

/* Main Function */
int main(int argc, char **argv)
{
	char *p;
	extern char *optarg;
	extern int optind, opterr, optopt;
	int ch;
	int res = 0;
	char *host = NULL;
	char *connection = NULL;
	char *command = NULL;
	char *user = NULL;
	char *pass = NULL;

	p = strrchr(argv[0], '/');
	progname = strdup(p ? p + 1 : argv[0]);

	if (argc < 2)
		usage();

	while ((ch = getopt(argc, argv, "dkslvLuVU:P:p:h:c:")) != EOF)
	{
		switch (ch)
		{
		case 'v':
			VERBOSE = 1;
			printf("%s\n", VERSION);
			break;
		case 'V':
			printf("%s", VERSION);
			return (0);
			break;
		case 'd':
			if (!command)
				command = "DIAL";
			break;
		case 'k':
			if (!command)
				command = "KILL";
			break;
		case 's':
			if (!command)
				command = "STAT";
			break;
		case 'l':
			if (!command)
				command = "LIST";
			break;
		case 'L':
			if (!command)
				command = "LOCK:KILL";
			break;
		case 'u':
			if (!command)
				command = "UNLOCK:KILL";
			break;
		case 'p':
			port = atoi(optarg);
			break;
		case 'h':
			host = get_host(optarg);
			break;
		case 'c':
			connection = strdup(optarg);
			break;
		case 'U':
			user = strdup(optarg);
			break;
		case 'P':
			pass = strdup(optarg);
			break;
		default:
			usage();
			break;
		}
	}

	if (!host)
		host = "localhost";

	if (!port)
		port = DEFAULTPORT;

	if (!command)
	{
		fprintf(stderr, "%s: No action specified!\n", progname);
		exit(1);
	}

	if (!strcmp(command, "DIAL") && !connection)
		connection = "default";

	if (!res)
	{
		do_command(host, command, connection, user, pass);
	}

	return (0);
}
