/*
  $Id: cfvar.c,v 1.6 1996/08/20 21:55:53 luik Exp $

  cfvar.c - omirrd config file variable (macro) handling.
  Based on lookup.c of rdist of the BSD 4.3-Net/2 release.
  UCB copyrights: see below.
  Modified by Andreas Luik, <luik@pharao.s.bawue.de>.

*/

/*
 * Copyright (c) 1983 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: cfvar.c,v 1.6 1996/08/20 21:55:53 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#if defined(SCCSID) && !defined(lint)
static char sccsid[] UNUSED__ = "@(#)lookup.c	5.5 (Berkeley) 6/1/90";
#endif /* not lint */

#include <stdio.h>
#include <stdlib.h>
#include "libomirr.h"
#include "error.h"
#include "cfnl.h"
#include "cfvar.h"

#define HASHSIZE 1021

typedef struct CfVarEnt_ *CfVarEnt;

typedef struct CfVarEnt_ {
    int replace;		/* replaces other value */
    char *name;
    CfNameList value;
    CfVarEnt next;
} CfVarEntRec;

static CfVarEnt hashtab[HASHSIZE];

typedef enum ActionTypes_ { LOOKUP_VAR, SET_VAR, REPLACE_VAR } ActionTypes;


/* cfDoVar - process variable name lookup (action = LOOKUP_VAR) or
   variable name setting. Called from `cfLookupVar', `cfSetVar' and
   `cfReplaceVar'. See there for detailed description.  */
static CfNameList cfDoVar(char *name, ActionTypes action, CfNameList value)
{
    unsigned int n;
    char *cp;
    CfVarEnt var;

    n = 0;
    for (cp = name; *cp; )
	n += *cp++;
    n %= HASHSIZE;

    for (var = hashtab[n]; var != NULL; var = var->next) {
	if (strcmp(name, var->name))
	    continue;
	if ((action == SET_VAR && var->replace == 0) || action == REPLACE_VAR) {
	    warning("variable `%s' redefined\n", name);
	    return CFVAR_ERROR;	/* redefined */
	}
	return(var->value);
    }

    if (action == LOOKUP_VAR) {
	warning("variable `%s' undefined\n", name);
	return CFVAR_ERROR;	/* undefined */
    }

    var = (CfVarEnt) xmalloc(sizeof(CfVarEntRec));
    var->next = hashtab[n];
    hashtab[n] = var;
    var->replace = (action == REPLACE_VAR);
    var->name = name;
    var->value = value;

    return(value);
}


/* cfLookupVar - returns value of variable `name' (which can be NULL
   if variable is set to an empty namelist), or error if name is not
   defined.  */
CfNameList cfLookupVar(char *name)
{
    return cfDoVar(name, LOOKUP_VAR, NULL);
}


/* cfSetVar - set variable `name' to `value'. Setting is ignored if
   variable `name' is already set. Returns an error only if the
   previous setting was done with `cfSetVar', not if done with
   `cfReplaceVar'.  Returns (new) value of variable otherwise.  */
CfNameList cfSetVar(char *name, CfNameList value)
{
    return cfDoVar(name, SET_VAR, value);
}


/* cfReplaceVar - set variable `name' to `value'. Return error if name
   is already defined.  Returns (new) value of variable otherwise.  */
CfNameList cfReplaceVar(char *name, CfNameList value)
{
    return cfDoVar(name, REPLACE_VAR, value);
}


/* cfDefineVar - define a variable from a command line argument. This
   sets the replace flag in the entry because definitions from the
   command line override assignment from the config file.  */
CfNameList cfDefineVar(char *name)
{
    char *cp, *s;
    CfNameList nl;
    CfNameList value;

    name = xstrdup(name);
    cp = strchr(name, '=');
    if (cp == NULL)
	value = NULL;
    else if (cp[1] == '\0') {
	*cp = '\0';
	value = NULL;
    } else if (cp[1] != '(') {
	*cp++ = '\0';
	value = cfAllocNameListEnt(xstrdup(cp));
    } else {
	value = nl = NULL;
	*cp++ = '\0';
	do {
	    cp++;
	} while (*cp == ' ' || *cp == '\t');
	for (s = cp; ; s++) {
	    switch (*s) {
	      case ')':
		*s = '\0';
	      case '\0':
		break;
	      case ' ':
	      case '\t':
		*s++ = '\0';
		while (*s == ' ' || *s == '\t')
		    s++;
		if (*s == ')')
		    *s = '\0';
		break;
	      default:
		continue;
	    }
	    if (nl == NULL)
		value = nl = cfAllocNameListEnt(xstrdup(cp));
	    else
		nl = cfAppendNameListEnt(nl, xstrdup(cp));
	    if (*s == '\0')
		break;
	    cp = s;
	}
    }
    return cfReplaceVar(name, value);
}


/* cfUnsetVars - unsets and frees all variables. If `replace' is 0,
   variables with the replace flag set are kept, if `replace' is 1,
   these are removed as well.  */
void cfUnsetVars(int replace)
{
    int n;
    CfVarEnt var;
    CfVarEnt prev;
    CfVarEnt next;

    for (n = 0; n < HASHSIZE; n++) {
	for (prev = NULL, var = hashtab[n]; var != NULL; var = next) {
	    next = var->next;
	    if (replace || var->replace == 0) {	/* need to remove it */
		free(var->name);
		cfFreeNameList(var->value);
		free(var);
		if (prev == NULL)
		    hashtab[n] = next;
		else
		    prev->next = next;
	    }
	    else prev = var;
	}
    }
}




/* cfExpandVar - expand first variable in first element of namelist
   `list'.  If nothing to expand, returns `list'. If expansion fails,
   returns next element in list and the first element is
   free'ed. Otherwise, if expansion succeeds, it returns new list with
   expanded elements. In this latter case, the next pointer of the
   last element of the new list points to the second element of the
   original list and the first element is free'ed.  */
CfNameList cfExpandVar(CfNameList list)
{
    char *cp;			/* points to start of variable name */
    char *end;			/* points to first char after variable ref */
    char saved_char = '\0';	/* saved char pointed to by `end' */
    CfNameList nl;		/* temporary work list */
    CfNameList el;		/* expanded list */
    CfNameList newnl = NULL;	/* new list built from el and returned */
    CfNameList last_newnl = NULL; /* last element of new list */

    if ((cp = strchr(list->n_name, '$')) == NULL)
	return (list);

    if (*++cp == '{') {
	++cp;
	if ((end = strchr(cp, '}')))
	    *end = '\0';
	else warning("syntax error, missing `}'\n"); 
    }
    else {			/* single character variable name */
	end = cp + 1;
	saved_char = *end;
	*end = '\0';
    }

    el = cfLookupVar(cp);	/* expanded list */

    if (end) {			/* restore character at end of variable name */
	if (saved_char)
	    *end = saved_char;
	else
	    *end++ = '}';
    }

    cp = strchr(list->n_name, '$');
    *cp = '\0';

    if (el != CFVAR_ERROR) {
	/* Concatenate list->n_name (up to cp), el->n_name, end.  */
	for (nl = el; nl; nl = nl->n_next) {
	    char *newname;
	    int len = (strlen(list->n_name) + strlen(nl->n_name)
		       + (end ? strlen(end) : 0));
	    newname = xmalloc(len + 1);
	    sprintf(newname, "%s%s%s",
		    list->n_name, nl->n_name, (end ? end : ""));

	    if (newnl == NULL)
		last_newnl = newnl = cfAllocNameListEnt(newname);
	    else
		last_newnl = cfAppendNameListEnt(last_newnl, newname);
	}
	if (el == NULL) {	/* variable expanded to nothing */
	    char *newname;
	    int len = strlen(list->n_name) + (end ? strlen(end) : 0);
	    if (len) {
		newname = xmalloc(len + 1);
		sprintf(newname, "%s%s", list->n_name, (end ? end : ""));

		if (newnl == NULL)
		    last_newnl = newnl = cfAllocNameListEnt(newname);
		else
		    last_newnl = cfAppendNameListEnt(last_newnl, newname);
	    }
	}
    }

    *cp = '$';

    if (!newnl)
	newnl = list->n_next;
    else if (last_newnl)	/* should always be true */
	last_newnl->n_next = list->n_next;

    cfFreeNameListEnt(list);

    return (newnl);
}


/* cfExpandVars - expand all variables in namelist `list'.  Returns
   new list, expanded entries in old list are free'ed.  */
CfNameList cfExpandVars(CfNameList list)
{
    CfNameList prev;
    CfNameList nl;
    CfNameList newnl;

    prev = NULL;
    nl = list;

    while (nl) {
	newnl = cfExpandVar(nl);
	if (newnl != nl) {	/* expanded, try again */
	    if (prev)
		prev->n_next = newnl;
	    else
		list = newnl;
	    nl = newnl;
	}
	else if (nl) {		/* nothing expanded, try next */
	    prev = nl;
	    nl = nl->n_next;
	}
    }
    return list;
}

