#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <termios.h>
#include <signal.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <sys/time.h>

#include <config.h>

#ifdef	HAVE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include <support.h>
#include <xcio.h>
#include <msgcat.h>

#include "str_smsg.h"

#define	FILE_KEYBIND	"/keybind"
#define	FILE_CATCAP	"/catcap"
#define	DEFAULT_PROMPT	"ppxp> "

static Keymap ttyKeymap;
static Keymap stdKeymap;

int ppxpFd, ttyMode=0;
static char ttyPrompt[sizeof(DEFAULT_PROMPT)];
static fd_set orgRfds;
static struct termios newTio, oldTio;
static bool_t tioBackuped;

void
TtyEcho(bool_t sw)
{
    static bool_t old;

    if (old == sw) return;
    old = sw;
    if (sw) newTio.c_lflag |= ECHO;
    else newTio.c_lflag &= ~ECHO;
    tcsetattr(0, TCSAFLUSH, &newTio);
}

static void
TtyUpdate(unsigned char *pppinfo)
{
/*    if (pppInfo.l_stat != pppinfo->l_stat) ttyMode = LSTAT_TTY;*/
    memcpy(&pppInfo, pppinfo, sizeof(pppInfo));
    if (pppinfo && !(pppInfo.l_stat & ttyMode)) {
	if (pppInfo.l_stat & LSTAT_CHAT)
	    memcpy(ttyPrompt, "chat", 4);
	else if (pppInfo.l_stat & LSTAT_DIAL)
	    memcpy(ttyPrompt, "dial", 4);
	else if (pppInfo.l_stat & LSTAT_TTY)
	    memcpy(ttyPrompt, "term", 4);
	else {
	    ttyPrompt[0] = (pppInfo.phase >= PS_ESTABLISH) ? 'P': 'p';
	    ttyPrompt[1] = (pppInfo.l_stat & LSTAT_PPP) ? 'P': 'p';
	    ttyPrompt[2] = (pppInfo.phase > PS_ESTABLISH) ?
		(pppInfo.phase > PS_AUTHENTICATE) ? 'X': '+': 'x';
	    ttyPrompt[3] = (pppInfo.n_stat) ? 'P': 'p';
	}
	rl_expand_prompt(ttyPrompt);
    } else rl_expand_prompt("");
    rl_refresh_line(0, 0);
}

static int
TtyOut(char *buf, int len)
{
    write(0, buf, len);
    return(0);
}

void
XcDo(struct xcio_s *xc)
{
    int n, argc;
    char *argv[10], *msg;

    switch(xc->type & XCIO_MASK) {
    case XCIO_UP_INFO:
	TtyUpdate(xc->buf);
	break;
    case XCIO_S_OUT:
	TtyOut(xc->buf, xc->len);
	break;
    case XCIO_MESSAGE:
	printf("\r");
	if ((msg = SysMsgGet(xc->buf[0])) != NULL) {
	    if (xc->len > 2)
		printf(msg, &xc->buf[1]);
	    else
		printf("%s", msg);
	    printf("\n");
	}
	break;
    }
}

void
SetTerminal(bool_t set)
{
    if (tioBackuped) tcsetattr(0, TCSAFLUSH, set ? &newTio: &oldTio);
}

static void
restore(int sig)
{
    SetTerminal(FALSE);
    putchar('\r');
    exit(0);
}

static int
RunCommand(int argc, char *argv[])
{
    struct xcio_s xc;
    xcmd_t xcmd;
    int id, ret, n;

    ret = RunBuiltinCommand(argc, argv);
    if (ret < 0 && ((xcmd = PPxPCommandType(argv[0])) < XCMD_MAX)) {
	if ((xcmd == XCMD_CONNECT || xcmd == XCMD_TERMINAL)) {
	    ttyMode = LSTAT_TTY;
	}
	id = PPxPCommand(ppxpFd, xcmd, argc - 1, &argv[1]);
	if (xcmd == XCMD_QUIT || xcmd == XCMD_BYE) restore(0);
	if (id >= 0) {
	    while ((n = PPxPRead(ppxpFd, id, &xc)) >= 0) if (n > 0) {
		if (xc.type == XCIO_RETURN) {
		    ret = xc.buf[0];
		    break;
		}
		XcDo(&xc);
	    }
	}
    }
    return(ret);
}

static int
RunCommandLine(char *line)
{
    int n, ret=-1;
    int argc=0;
    char *argv[128], *p, *buf, *home;

    buf = line;
    if ((p = strpbrk(line, "\n\r")) != NULL) *p = '\0';
    while (*buf) {
	while (*buf && strchr("\t ", *buf)) buf ++;
	if (*buf) {
	    if ((p = strpbrk(buf, "\t ")) != NULL) *p = '\0';
	    argv[argc ++] = strdup(buf);
	    if (!p) break;
	    buf = p + 1;
	}
    }
    if (argc <= 0) return(-1);
    argv[argc] = NULL;

    buf = line;
    *buf = '\0';
    home = getenv("HOME");
    for (n = 0; n < argc;) {
	strcat(buf, argv[n]);
	if (home && *argv[n] == '~' && *(argv[n] + 1) == '/') {
	    p = argv[n];
	    argv[n] = Malloc(strlen(p) + strlen(home));
	    strcpy(argv[n], home);
	    strcat(argv[n], p + 1);
	    free(p);
	}
	n ++;
	if (n < argc) strcat(buf, " ");
    }
    add_history(buf);
#if 1
    ret = RunCommand(argc, argv);
#else
    ret = RunBuiltinCommand(argc, argv);
    if (ret < 0) {
	struct xcio_s xc;
	xcmd_t xcmd;
	int id;

	if ((xcmd = PPxPCommandType(argv[0])) < XCMD_MAX) {
	    if ((xcmd == XCMD_CONNECT || xcmd == XCMD_TERMINAL)) {
		ttyMode = LSTAT_TTY;
	    }
	    id = PPxPCommand(ppxpFd, xcmd, argc - 1, &argv[1]);
	    if (!(xcmd == XCMD_QUIT || xcmd == XCMD_BYE) && id >= 0) {
		while ((n = PPxPRead(ppxpFd, id, &xc)) >= 0) if (n > 0) {
		    if (xc.type == XCIO_RETURN) {
			ret = xc.buf[0];
			break;
		    }
		    XcDo(&xc);
		}
	    }
	}
    }
#endif
    if (ret < 0) Exec(argc, argv, buf);
    return(ret);
}

static int
hook_poll()
{
    static char buf[XCBUFSIZ];
    static Keymap map;
    struct xcio_s xc;
    fd_set rfds;
    Function *func;
    int n, i;

    do {
	rfds = orgRfds;
	while ((n = PPxPRead(ppxpFd, XID_ANY, &xc)) > 0) XcDo(&xc);
	if (n < 0) restore(0);
	n = select(ppxpFd + 1, &rfds, NULL, NULL, NULL);
    } while (n <= 0 && errno == EINTR);
    if ((pppInfo.l_stat & ttyMode) && FD_ISSET(0, &rfds)) {
	if ((n = read(0, buf, sizeof(buf))) > 0) {
	    for (i = 0; i < n; i ++) {
		int key=buf[i];

		if (!map) map = ttyKeymap;
		func = map[key].function;
		switch(map[key].type) {
		case ISKMAP:
		    map = (Keymap)((long)map[key].function);
		    break;
		case ISFUNC:
		    if (map[key].function) {
			key = (*map[key].function)(0, 0);
		    } else key = 0;
		    map = NULL;
		    if (key) n = 0;
		}
	    }
	}
	if (n > 0) {
	    xc.len = n;
	    xc.type = XCIO_S_IN;
	    memcpy(xc.buf, buf, xc.len);
	    xc.xid = 1;
	    XcioWrite(ppxpFd, &xc);
	}
    }
    return(0);
}

static int
EscConsole()
{
    ttyMode = 0;
    PPxPUpdateRequest(ppxpFd);
    TtyEcho(TRUE);
}

static int
EscActivate()
{
    printf("activate\n");
    PPxPRequest(ppxpFd, XCIO_ACTIVATE);
}

static int
EscHelp()
{
/*
    Keymap km;

    km = rl_get_keymap_by_name("esc-console");
    km = rl_get_keymap_by_name("esc-activate");
    km = rl_get_keymap_by_name("esc-help");
*/
    printf("help\n");
}

static void
MakeDirectories()
{
    char *path, *sub[]={"conf", "chat", "log", "rc"};
    unsigned n;
    int fd;

    path = Malloc(strlen(usrPPxP) + 20);

    if (mkdir(usrPPxP, 0700) < 0 && errno != EEXIST) return;
    for (n = 0; n < sizeof(sub)/sizeof(sub[0]); n ++) {
	sprintf(path, "%s/%s", usrPPxP, sub[n]);
	mkdir(path, 0700);
    }
    sprintf(path, "%s/passwd", usrPPxP);
    fd = open(path, O_CREAT, 0600);
    /* ... */
    close(fd);
    free(path);
}

static int
LoadRcFiles()
{
    FILE *fp;
    char *kbf, *tlf, *lang, *nlang, *term;

    DirNameInit(getuid());

    tlf = Malloc(strlen(sysPPxP) + sizeof(FILE_CATCAP) + 1);
    strcpy(tlf, sysPPxP);
    strcat(tlf, FILE_CATCAP);

    term = getenv("TERM");
    lang = getenv("LANG");
    nlang = NULL;
    if (lang && term && (fp = fopen(tlf, "r")) != NULL) {
	char *p, buf[256];
	int len;

	while (fgets(buf, sizeof(buf), fp)) {
	    if ((p = strchr(buf, ':')) != NULL) {
		*p = '\0';
		len = p - buf;
		p ++;
		if (!strncmp(buf, lang, len)) {
		    if (strstr(p, term)) {
			nlang = lang;
			break;
		    }
		}
	    }
	}
	fclose(fp);
    }
    Free(tlf);
    if (!nlang) setenv("LANG", "C", 1);	/* change */

    MsgInit("ppxp", strMsg);

    /* if (!nlang) setenv("LANG", lang, 1); /* restore */

#ifdef	HAVE_READLINE
    stdKeymap = rl_get_keymap();
    ttyKeymap = rl_make_bare_keymap();
    rl_set_keymap(ttyKeymap);

    rl_add_funmap_entry("esc-console", EscConsole);
    rl_add_funmap_entry("esc-activate", EscActivate);
    rl_add_funmap_entry("esc-help", EscHelp);

    kbf = Malloc(strlen(sysPPxP) + sizeof(FILE_KEYBIND) + 1);
    strcpy(kbf, sysPPxP);
    strcat(kbf, FILE_KEYBIND);
    rl_read_init_file(kbf);
    Free(kbf);

    if (usrPPxP) {
	MakeDirectories();
	kbf = Malloc(strlen(usrPPxP) + sizeof(FILE_KEYBIND) + 1);
	strcpy(kbf, usrPPxP);
	strcat(kbf, FILE_KEYBIND);
	rl_read_init_file(kbf);
	Free(kbf);
    }

    rl_set_keymap(stdKeymap);
#else
    if (usrPPxP) MakeDirectories();
#endif
    return(0);
}

int
main(int argc, char *argv[])
{
    char *line, *path, *p;
    struct xcio_s xc;
    u_int8_t xid;

    if ((p = strrchr(argv[0], '/')) != NULL) p ++;
    else p = argv[0];
    if (!strcmp(p, "plast")) return(PlastMain(argc, argv));

    LoadRcFiles();

    if ((ppxpFd = PPxPSetup(&argc, argv)) < 0) exit(-1);

    if (argc > 1 && !strcmp(argv[1], "-C")) {
	if (argc > 2) RunCommand(argc - 2, &argv[2]);
	restore(0);
    }

    xid = PPxPEnvRequestv(ppxpFd, "version", NULL);
    printf("PPxP version %s\n", PPxPEnvGet(ppxpFd, xid));
    xid = PPxPEnvRequestv(ppxpFd, "IF.DEV", NULL);
    printf("interface: %s\n", PPxPEnvGet(ppxpFd, xid));

    PPxPUpdateRequest(ppxpFd);
    strcpy(ttyPrompt, DEFAULT_PROMPT);

    FD_ZERO(&orgRfds);
    FD_SET(0, &orgRfds);
    FD_SET(ppxpFd, &orgRfds);
    rl_event_hook = hook_poll;

    if (tcgetattr(0, &oldTio)) return(-1);
    tioBackuped = TRUE;
    newTio = oldTio;

    newTio.c_lflag &= ~(ISIG|ICANON);
    newTio.c_iflag &= ~(INLCR|ICRNL|ISTRIP|IXON|BRKINT);
    newTio.c_cc[VMIN] = 1;
    newTio.c_cc[VTIME] = 0;
    newTio.c_cc[VQUIT] = 0;
    TtyEcho(TRUE);
    InitSignals();
    signal(SIGPIPE, restore);

/*
    xc.xid = 100;
    xc.type = XCIO_LIST;
    xc.len = 0;
    XcioWrite(ppxpFd, &xc);
*/

    while (1) {
	if (!(pppInfo.l_stat & ttyMode)) {
	    WaitJob();

	    if ((line = readline(ttyPrompt)) == NULL) {
		line = Strdup("BYE BYE !!");
		printf("%s\n", line);
	    }
	    RunCommandLine(line);
	    Free(line);
	} else {
	    TtyEcho(FALSE);
	    hook_poll();
	}
    }
}
