#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>

#include <config.h>
#include <support.h>

#include <frame.h>
#include <fcs.h>
#include <dev/device.h>
#include <ccp.h>
#include <log.h>

/*
 * pred.c -- Test program for Dave Rand's rendition of the
 * predictor algorithm
 * Updated by: iand@labtam.labtam.oz.au (Ian Donaldson)
 * Updated by: Carsten Bormann <cabo@cs.tu-berlin.de>
 * Original  : Dave Rand <dlr@bungi.com>/<dave_rand@novell.com>
 */

/* The following hash code is the heart of the algorithm:
 * It builds a sliding hash sum of the previous 3-and-a-bit
 * characters which will be used to index the guess table.
 * A better hash function would result in additional compression,
 * at the expense of time.
 */

#define HASH(h, x) h = (h << 4) ^ (x)

#define	GUSSIZ	65536

static u_int16_t rHash, sHash;
static u_char *sGuessTable;
static u_char *rGuessTable;
static u_char *dstBuf, *srcBuf, *dstHead, *srcHead;

static inline u_int16_t
compress(source, dest, len)
unsigned char *source, *dest;
int len;
{
    int i, bitmask;
    unsigned char *flagdest, flags, *orgdest;

    orgdest = dest;
    while (len) {
	flagdest = dest++; flags = 0;	    /* All guess wrong initially */
	for (bitmask=1, i=0; i < 8 && len; i++, bitmask <<= 1) {
	    if (sGuessTable[sHash] == *source) {
		/* Guess was right - don't output */
		flags |= bitmask;
	    } else {
		/* Guess wrong, output char */
		sGuessTable[sHash] = *source;
		*dest++ = *source;
	    }
	    HASH(sHash, *source++);len--;
	}
	*flagdest = flags;
    }
    return((u_int16_t)(dest - orgdest));
}

static inline int
decompress(source, dest, len)
unsigned char *source, *dest;
int len;
{
    unsigned int i, bitmask;
    unsigned char flags, *orgdest;

    orgdest = dest;
    while (len) {
	flags = *source++;
	len--;
	for (i=0, bitmask = 1; i < 8; i++, bitmask <<= 1) {
	    if (flags & bitmask) {
		/* Guess correct */
		*dest = rGuessTable[rHash];
	    } else {
		if (!len) break;
		/* Guess wrong */
		rGuessTable[rHash] = *source;
		/* Read from source */
		*dest = *source++;
		len--;
	    }
	    HASH(rHash, *dest++);
	}
    }
    return((int)(dest - orgdest));
}

void
PredInit(int direction)
{
    CompAllocBuffer(&srcHead, &dstHead);
    srcBuf = srcHead + sizeof(u_int16_t);
    dstBuf = dstHead + sizeof(u_int16_t);
    if (direction & COMP_INITC) {
	if (!sGuessTable) sGuessTable = Malloc(GUSSIZ);
	memset(sGuessTable, 0, GUSSIZ);
	sHash = 0;
    }
    if (direction & COMP_INITUC) {
	if (!rGuessTable) rGuessTable = Malloc(GUSSIZ);
	memset(rGuessTable, 0, GUSSIZ);
	rHash = 0;
    }
}

int
Pred1Compress(u_char *buf, int *lenp, u_int16_t nbo_proto)
{
    int n;
    u_int16_t len, i;
    u_int16_t fcs=INITIAL_FCS;
    extern int devFd;
    u_char *sp, *dp;

    n = *lenp;
    n += sizeof(u_int16_t);
    sp = srcHead;
    dp = dstHead;
    *dp ++ = *sp ++ = n >> 8;
    *dp ++ = *sp ++ = n & 0xFF;
    /* dp == dstBuf, sp == srcBuf */
    memcpy(sp, &nbo_proto, sizeof(u_int16_t));
    sp += sizeof(u_int16_t);
    fcs = AddToFcs(fcs, srcHead[0]);
    fcs = AddToFcs(fcs, srcHead[1]);
    fcs = AddToFcs(fcs, srcHead[2]);
    fcs = AddToFcs(fcs, srcHead[3]);
    for (i = 0; i < n - sizeof(u_int16_t); i ++, sp ++) {
	*sp = buf[i];
	fcs = AddToFcs(fcs, *sp);
    }
    fcs = ~fcs;
    if ((len = compress(srcBuf, dstBuf, n)) < n) {
	dstHead[0] |= 0x80;
	dp += len;
    } else {
	memcpy(dp, srcBuf, n);
	dp += n;
    }
    *dp ++ = fcs & 0xFF;
    *dp ++ = fcs >> 8;
    *lenp = (int)(dp - dstHead);
    return(DevWrite(devFd, dstHead, *lenp, NBO_PROTO_COMP));
}

u_char *
Pred1Uncompress(u_char *buf, int *lenp, u_int16_t *nbo_protop)
{
    unsigned int nlen, n;
    u_char *sp, *dp;
    u_int16_t fcs=INITIAL_FCS;

    sp = buf;
    fcs = AddToFcs(fcs, *sp & 0x7f);
    nlen = *sp ++ << 8;
    fcs = AddToFcs(fcs, *sp);
    nlen += *sp ++;
    if (*buf & 0x80) {
	unsigned int ulen;

	/* compressed size = received size - length field - fcs */
	n = (*lenp) - (sizeof(u_int16_t) + sizeof(u_int16_t));

	nlen &= 0x7FFF;
	ulen = decompress(sp, dstBuf, n);
	if (nlen != ulen) {
	    Logf(LOG_ERROR, "predictor1: wrong size (%d!=%d)\n",
		 nlen, ulen);
	    return(NULL);
	}
	sp += n;
	dp = dstBuf;
	for (n = 0; n < nlen; n ++) fcs = AddToFcs(fcs, dstBuf[n]);
    } else {
	dp = sp;
	n = nlen;
	while (n > 0) {
	    if (rGuessTable[rHash] != *sp)
		rGuessTable[rHash] = *sp;
	    fcs = AddToFcs(fcs, *sp);
	    HASH(rHash, *sp ++);
	    n --;
	}
    }
    fcs = AddToFcs(fcs, *sp ++);
    fcs = AddToFcs(fcs, *sp);
    memcpy(nbo_protop, dp, sizeof(u_int16_t));
/*
printf("read:%x\n", fcs);
FrameDump("Pred1Read", dstBuf, nlen);
*/
    if (fcs == GOOD_FCS) {
	*lenp = nlen - sizeof(u_int16_t);
	return(dp + sizeof(u_int16_t));
    } else {
	Logf(LOG_ERROR, "predictor1: bad FCS(%#x)\n", fcs);
	return(NULL);
    }
}

#if 0
#define SIZ1 8192

static void
compress_file(f) FILE *f; {
    char bufp[SIZ1];
    char bufc[SIZ1/8*9+9];
    int len1, len2;
    while ((len1 = fread(bufp, 1, SIZ1, f)) > 0) {
	len2 = compress((unsigned char *)bufp,
			(unsigned char *)bufc, len1);
	(void) fwrite(bufc, 1, len2, stdout);
    }
}

static void
decompress_file(f) FILE *f; {
    char bufp[SIZ1+9];
    char bufc[SIZ1*9+9];
    int len1, len2, len3;

    len1 = 0;
    while ((len3 = fread(bufp+len1, 1, SIZ1, f)) > 0) {
	len1 += len3;
	len3 = len1;
	len2 = decompress((unsigned char *)bufp,
			  (unsigned char *)bufc, &len1, 0);
	(void) fwrite(bufc, 1, len2, stdout);
	(void) memcpy(bufp, bufp+len3-len1, len1);
    }
    len2 = decompress((unsigned char *)bufp,
		      (unsigned char *)bufc, &len1, 1);
    (void) fwrite(bufc, 1, len2, stdout);
}

int
main(ac, av)
int ac;
char** av;
{
    char **p = av+1;
    int dflag = 0;

    for (; --ac > 0; p++) {
	if (!strcmp(*p, "-d"))
	    dflag = 1;
	else if (!strcmp(*p, "-"))
	    (dflag?decompress_file:compress_file)(stdin);
	else {
	    FILE *f = fopen(*p, "r");
	    if (!f) {
		perror(*p);
		exit(1);
	    }
	    (dflag?decompress_file:compress_file)(f);
	    (void) fclose(f);
	}
    }
    return(0);
}
#endif
