#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include <config.h>
#include <support.h>

#include "option.h"
#include "log.h"
#include "chat.h"
#include "env.h"
#include "console.h"
#include "auth.h"
#include "dev/device.h"

extern char *EnvToStr();

static struct chatcmd_s {
    const char *name;
    bool_t todo;
} cmdTable[]={
    {"abort", FALSE},
    {"retry", FALSE},
    {"done", FALSE},
    {"name", TRUE},
    {"passwd", TRUE},
    {NULL, FALSE},
};

struct chatent_s {
    enum {CENT_STR, CENT_OUT, CENT_CMD, CENT_LBL} type;
    char *string[3];
    int timeout;
    int command;
};

static struct chat_s {
    struct chat_s *next;
    struct chatent_s event;
#define	e_t event.type
#define	e_str0 event.string[0]
#define	e_out event.timeout
#define	e_hide event.hide
    struct chatent_s action;
#define	a_t action.type
#define	a_str0 action.string[0]
#define	a_str1 action.string[1]
#define	a_str2 action.string[2]
#define	a_out action.timeout
#define	a_cmd action.command
} *chatHead, *chatP;

#define	CHATBUFSIZ	40
#define	NEVENTLIST	40

static char chatBuf[CHATBUFSIZ+1];
static char lineBuf[256];
static int errorN, labelN, cbufLen;
static struct chat_s *errorP[NEVENTLIST];
static struct chat_s *labelP[NEVENTLIST];
static struct chat_s *timeoutP;

static char *
ChatStr(char *in, bool_t secure)
{
    char *dst, ch;
    bool_t quote=FALSE;
    extern char *ExtractEnvs();
    static char tmpbuf[256];

    in = ExtractEnvs(in, secure ? 0: ENV_HIDE);
    dst = tmpbuf;
    while (*in) {
	ch = *in ++;
	if (quote == TRUE) {
	    if (ch == '"') quote = FALSE;
	    else *dst ++ = ch;
	    continue;
	}
	if (ch == '"') {
	    quote = TRUE;
	    continue;
#if 0
	}
	if (ch == '$') {
	    int len;

	    if ((p = strpbrk(in, "\"$")) == NULL) len = strlen(in);
	    else len = p - in;
	    strncpy(lineBuf, in, len);
	    lineBuf[len] = '\0';
	    if ((p = EnvToStr(lineBuf, secure ? 0: ENV_HIDE)) != NULL) {
		strcpy(dst, p);
		dst += strlen(dst);
	    }
	    in += len;
#endif
	} else *dst ++ = ch;
    }
    *dst = '\0';
    return(tmpbuf);
}

static int
ChatFileProc(char *line, int secure, void *argp)
{
    static struct chat_s *csp0;
    struct chat_s *csp;
    char *p, *ep, *ap;

    if (!line) {
	csp0 = NULL;
	return(0);
    }
    csp = TCALLOC(struct chat_s);
    if (csp0) csp0->next = csp;
    else chatHead = csp;
    csp0 = csp;
    if ((p = strpbrk(line, "\r\n")) != NULL) *p = '\0';
    ep = line;
    p = strpbrk(ep, ":\"");
    if (p && *p == '"') {
	if ((p = strchr(p + 1, '"')) == NULL) {
	    Logf(LOG_ERROR, "%s: '\"' expected\n", line);
	    return(0);
	}
	p ++;
    } else p = ep;
    if ((p = strchr(p, ':')) != NULL) {
	*p = '\0';
	ap = p + 1;
    } else ap = NULL;
    if (!ep || !*ep || *ep == '"' || *ep == '$') {
	csp->e_t = CENT_STR;
	if (ep && *ep && *(ep + 1) && (p = ChatStr(ep, TRUE)) != NULL) {
	    csp->e_str0 = Strdup(p);
	} else csp->e_str0 = NULL;
    } else if (*ep == '@') {
	csp->e_t = CENT_LBL;
	csp->e_str0 = Strdup(ep + 1);
    } else {
	csp->e_out = atoi(ep);
	csp->e_t = CENT_OUT;
    }
    if (!ap || *ap == '"' || !*ap || *ap == '$') {
	csp->a_t = CENT_STR;
	csp->a_str0 = (!ap || !*ap || !*(ap + 1)) ? NULL: Strdup(ap);
    } else if (*ap == '@') {
	csp->a_t = CENT_LBL;
	csp->a_str0 = Strdup(ap + 1);
    } else {
	int i;

	csp->a_t = CENT_CMD;
	for (i = 0; cmdTable[i].name; i ++)
	    if (!strcmp(cmdTable[i].name, ap)) break;
	csp->a_cmd = i;
    }
    return(0);
}

static int
LoadChat(char *file)
{
    FILE *fp;
    int secure=0;
    struct chat_s *csp, *csp0;

    csp = chatHead;
    while (csp) {
	csp0 = csp->next;
	if (csp->e_t == CENT_STR || csp->e_t == CENT_LBL) {
	    if (csp->e_str0) Free(csp->e_str0);
	}
	if (csp->a_t == CENT_STR || csp->a_t == CENT_LBL) {
	    if (csp->a_str0) Free(csp->a_str0);
	    if (csp->a_str1) Free(csp->a_str1);
	    if (csp->a_str2) Free(csp->a_str2);
	}
	Free(csp);
	csp = csp0;
    }
    ChatFileProc(NULL, 0, NULL);	/* initialize */
    fp = OpenFile(file, "chat", &secure);
    return(fp ? LoadFile(fp, ChatFileProc, secure, NULL): -1);
}

static int
ChatInterpret()
{
    while (chatP) {
	if (chatP->e_t == CENT_LBL) labelP[labelN ++] = chatP;
	else if (chatP->e_t == CENT_OUT) timeoutP = chatP;
	else if (chatP->a_t == CENT_CMD
		 && !cmdTable[chatP->a_cmd].todo) {
	    errorP[errorN ++] = chatP;
	} else if (chatP->e_t == CENT_STR) {
	    if (chatP->e_str0) {
		if (ISLOG(LOG_CHAT))
		    Logf(LOG_CHAT, "waiting: %s\n",
			 ChatStr(chatP->e_str0, FALSE));
		return(CHAT_NEXT);
	    } else {
/*		ChatErrorF("waiting: ""\n");*/
		return(CHAT_NOP);
	    }
	}
	chatP = chatP->next;
    }
    return(CHAT_DONE);
}

static int
ChatAction(struct chat_s *csp)
{
    char *astr;
    extern int devFd;

    switch(csp->a_t) {
    case CENT_STR:
	if (csp->a_str0) {
	    sprintf(lineBuf, "%s\r", ChatStr(csp->a_str0, TRUE));
	    DevWrite(devFd, lineBuf, strlen(lineBuf), 0);
	}
	if (ISLOG(LOG_CHAT)) {
	    char *p;

/*	    if (ISLOG(LOG_SECRET)) ConsoleOutf("%s\n", lineBuf);*/
	    if (csp->a_str0) p = ChatStr(csp->a_str0, FALSE);
	    else p = NULL;
	    Logf(LOG_CHAT, "action: %s\n", p ? p: "");
	}
	break;
    case CENT_CMD:
	switch (csp->a_cmd) {
	case CHAT_NAME:
	    astr = AuthName();
	    sprintf(lineBuf, "%s\r", astr);
	    if (ISLOG(LOG_CHAT)) {
		Logf(LOG_CHAT, "event: %s -> %s\n", csp->e_str0,
		     ISLOG(LOG_PRIVATE) ? astr: "<name>");
	    }
	    DevWrite(devFd, lineBuf, strlen(lineBuf), 0);
	    break;
	case CHAT_PASSWD:
	    AuthDecode(lineBuf, 100);
	    if (ISLOG(LOG_CHAT)) {
		Logf(LOG_CHAT, "event: %s -> %s\n", csp->e_str0,
		     ISLOG(LOG_SECRET) ? lineBuf: "<passwd>");
	    }
	    strcat(lineBuf, "\r");
	    DevWrite(devFd, lineBuf, strlen(lineBuf), 0);
	    break;
	default:
	    if (ISLOG(LOG_CHAT)) {
		Logf(LOG_CHAT, "event: %s -> %s\n",
		     csp->e_str0, cmdTable[csp->a_cmd].name);
	    }
	    return(csp->a_cmd);
	}
	break;
    default:
    }
    return(CHAT_NEXT);
}

int
ChatStart(char *file)
{
    if (!file || !strcmp(file, "none") || (LoadChat(file) < 0)) {
	return(CHAT_DONE);
    }
/*    ChatShow();*/
    chatP = chatHead;
    cbufLen = 0;
    labelN = errorN = 0;
    timeoutP = labelP[labelN] = errorP[errorN] = NULL;
    while (ChatInterpret() == CHAT_NOP) {
	ChatAction(chatP);
	chatP = chatP->next;
    }
    return(CHAT_NEXT);
}

int
ChatIn(u_char c)
{
    int i;
    static char *waiting;

/*    if (ISLOG(LOG_CHAT) && ISLOG(LOG_SECRET)) ConsoleOutf("%c", c);*/
    chatBuf[cbufLen ++] = c;
    if (strchr("\r\n", c)) cbufLen = 0;
    if (cbufLen > CHATBUFSIZ) {
	memcpy(chatBuf, &chatBuf[1], CHATBUFSIZ);
	cbufLen = CHATBUFSIZ;
    }
    chatBuf[cbufLen] = '\0';
    if (!waiting) waiting = ChatStr(chatP->e_str0, TRUE);
    if (strstr(chatBuf, waiting)) {
	ChatAction(chatP);
	chatP = chatP->next;
	ChatInterpret();
	waiting = NULL;
	cbufLen = 0;
    } else for (i = 0; i < errorN; i ++) {
	if (strstr(chatBuf, errorP[i]->e_str0))
	    return(ChatAction(errorP[i]));
    }
    return(chatP ? CHAT_NEXT: CHAT_DONE);
}
