#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>

#include <config.h>
#include <support.h>

#include "log.h"
#include "option.h"
#include "timer.h"
#include "fsm.h"
#include "cps.h"
#include "frame.h"

static struct {
    const char *name;
    bool_t reply;	/* reply: check received id */
} codeInfo[]={
    {"Configure-Request", FALSE},
    {"Configure-Ack", TRUE},
    {"Configure-Nak", TRUE},
    {"Configure-Reject", TRUE},
    {"Terminate-Request", FALSE},
    {"Terminate-Ack", TRUE},
    {"Code-Reject", TRUE},
    {"Protocol-Reject", FALSE},
    {"Echo-Request", FALSE},
    {"Echo-Reply", TRUE},
    {"Discard-Request", FALSE},
    {"Identification", FALSE},
    {"Time-Remaining", FALSE},
    {"Reset-Request", FALSE},
    {"Reset-Ack", TRUE},
    {"Reserved", FALSE}
};

#define	NUM_CODEINFO	sizeof(codeInfo)/sizeof(codeInfo[0])

static u_char cpsBuf[BUFSIZ];

static struct cpreg_s {
    struct cpreg_s *next;
    struct fsmreg_s *fsm;
} *cpHead;

void
CpEnqueueFrame(struct fsmreg_s *fsm, cpcode_t code,
	       void *cpreg, u_int8_t id)
{
    struct cpframeheader_s *cfp;
    struct cpstate_s *cpst;
    struct cpcodeop_s *cop;
    u_char *buf;
    u_int16_t len=0;
    extern int devFd;

    if (devFd < 0) {
	/* down or finish ? */
	if (fsm->tld) fsm->tld(fsm);
	return;
    }
    cpst = (struct cpstate_s *)fsm->priv;
    buf = cpsBuf;
    cfp = (struct cpframeheader_s *)buf;
    cfp->code = code;
    cfp->id = id;
    cop = &cpst->cop[code - 1];

    if (ISLOG(LOG_CP))
	Logf(LOG_CP, "%s: Send-%s id=%d\n",
	     fsm->name, codeInfo[cfp->code -1].name, cfp->id);
    len = sizeof(struct cpframeheader_s);
    if (cop->encode)
	len += cop->encode(buf + sizeof(struct cpframeheader_s), cpreg);
/*    else if (ISLOG(LOG_CP)) Logf(LOG_CP, " Not implemented\n");*/
    cfp->nbo_len = htons(len);
    FrameEnqueue(buf, len, cpst->nbo_proto, PRI_CP);
}

static void
CpScj(struct cpstate_s *cpst, u_char *buf, int len)
{
    struct cpframeheader_s *cfp;

    cfp = (struct cpframeheader_s *)cpsBuf;
    cfp->code = CPCODE_CODE_REJ;
    cfp->id = cpst->r_id;
    memcpy(cpsBuf + sizeof(struct cpframeheader_s), buf, len);
    len += sizeof(struct cpframeheader_s);
    cfp->nbo_len = htons(len);
    if (ISLOG(LOG_CP))
	Logf(LOG_CP, ": Send-%s id=%d\n",
	     codeInfo[cfp->code -1].name, cfp->id);
    FrameEnqueue(cpsBuf, len, cpst->nbo_proto, PRI_CP);
}

static void
CpScr(struct fsmreg_s *fsm)
{
    struct cpstate_s *cpst;

    cpst = (struct cpstate_s *)fsm->priv;
    cpst->e_order = cpst->l_order;
    B256_CPY(&cpst->e_list, &cpst->l_list);
    cpst->l_id ++;
    CpEnqueueFrame(fsm, CPCODE_CONF_REQ, cpst->l_reg, cpst->l_id);
}

static void
CpSca(struct fsmreg_s *fsm)
{
    struct cpstate_s *cpst;

    cpst = (struct cpstate_s *)fsm->priv;
    cpst->e_order = cpst->r_order;
    B256_CPY(&cpst->e_list, &cpst->r_list);
    CpEnqueueFrame(fsm, CPCODE_CONF_ACK, cpst->r_reg, cpst->r_id);
}

static void
CpScn(struct fsmreg_s *fsm)
{
    struct cpstate_s *cpst;

    cpst = (struct cpstate_s *)fsm->priv;
    cpst->e_order = cpst->r_order;
    B256_CPY(&cpst->e_list, &cpst->r_list);
    CpEnqueueFrame(fsm, fsm->rej_rcr ? CPCODE_CONF_REJ: CPCODE_CONF_NAK,
		   cpst->r_reg, cpst->r_id);
/*    CpEnqueueFrame(fsm, CPCODE_CONF_REJ, cpst->r_reg, cpst->r_id);*/
}

static void
CpStr(struct fsmreg_s *fsm)
{
    struct cpstate_s *cpst;

    cpst = (struct cpstate_s *)fsm->priv;
    cpst->l_id ++;
    CpEnqueueFrame(fsm, CPCODE_TERM_REQ, NULL, cpst->l_id);
}

static void
CpSta(struct fsmreg_s *fsm)
{
    struct cpstate_s *cpst;

    cpst = (struct cpstate_s *)fsm->priv;
    CpEnqueueFrame(fsm, CPCODE_TERM_ACK, NULL, cpst->r_id);
}

inline static struct cptypeop_s *
Type2Op(cptype_t type, struct cptypeop_s *tops, int max)
{
    int i;

    for (i = 0; i < max; i ++) {
	if (tops[i].type == type) {
	    return(&tops[i]);
	}
    }
    return(NULL);
}

int
CpEncodeReg(struct cpstate_s *cpst, u_char *buf, bool_t rej)
{
    struct cpdataheader_s *chp;
    struct cptypeop_s *top;
    int i=0;
    u_char *data, *head;
    cptype_t type;

    head = buf;
/*    cpst->r_list = 0;*/
    while ((type = cpst->e_order[i++]) < ENDOF_LIST) {
	if (!B256_ISSET(&cpst->e_list, type)) continue;
	chp = (struct cpdataheader_s *)buf;
	data = buf + sizeof(struct cpdataheader_s);
	chp->type = type;
	top = Type2Op(type, cpst->top, cpst->num_type);
	chp->len = sizeof(struct cpdataheader_s);
	if (top) {
	    if (ISLOG(LOG_CP)) Logf(LOG_CP, " %s: ", top->name);
	    if (top->coder) {
		chp->len += top->coder(data, 0);
	    } else {
		chp->len += top->size;
		if (top->size && top->entry)
		    memcpy(data, top->entry, top->size);
		else if (rej && cpst->r_bp && cpst->r_bp[i - 1]) {
		    struct cpdataheader_s *dp;
		    dp = (struct cpdataheader_s *)cpst->r_bp[i - 1];
		    memcpy(buf, cpst->r_bp[i - 1], dp->len);
		    Logf(LOG_CP, "(%d)", dp->len);
		}
		if (ISLOG(LOG_CP)) {
		    if (top->size && top->tostr)
			Logf(LOG_CP, "%s", top->tostr(top->entry));
		    Logf(LOG_CP, "\n");
		}
	    }
	    if (cpst->r_bp && cpst->r_bp[i - 1]) {
		Free(cpst->r_bp[i - 1]);
		cpst->r_bp[i - 1] = NULL;
	    }
	} else if (ISLOG(LOG_CP)) Logf(LOG_CP, " %d\n", type);
	buf += chp->len;
    }
    return(buf - head);
}

int
CpDecodeReg(struct cpstate_s *cpst, u_char *buf, int n)
{
    struct cpdataheader_s *chp;
    struct cptypeop_s *top;
    int i=0, len;
    u_char *data;

    B256_ZERO(&cpst->r_list);
    while (n > 0) {
	chp = (struct cpdataheader_s *)buf;
	data = buf + sizeof(struct cpdataheader_s);
	top = Type2Op(chp->type, cpst->top, cpst->num_type);
	B256_SET(&cpst->r_list, chp->type);
	if (cpst->r_bp) {
	    if (cpst->r_bp[i]) Free(cpst->r_bp[i]);
	    cpst->r_bp[i] = Malloc(chp->len);
	    memcpy(cpst->r_bp[i], buf, chp->len);
	}
	cpst->r_order[i ++] = chp->type;
	if (top) {
	    if (ISLOG(LOG_CP)) Logf(LOG_CP, " %s: ", top->name);
	    if (top->coder) {
		/* special decoder */
		top->coder(data, chp->len);
	    } else {
		len = (top->size < chp->len) ? top->size: chp->len;
		if (!top->entry) {
		    /* not supported */
		} else if (!len) {
		    /* boolean type */
		    *(bool_t *)top->entry = TRUE;
		}
		if (len && top->entry) memcpy(top->entry, data, len);
		if (ISLOG(LOG_CP)) {
		    if (top->size && top->tostr)
			Logf(LOG_CP, "%s", top->tostr(top->entry));
		    Logf(LOG_CP, "\n");
		}
	    }
	} else {
	    if (ISLOG(LOG_CP)) {
		int k;
		for (k = 0; k < chp->len; k ++)
		    Logf(LOG_CP, " %d", buf[k]);
		Logf(LOG_CP, "\n");
	    }
	    /* unknown-type -> Send Configure Reject */
#if 0
	    CpScj(cpst, buf, chp->len);
	    return(-1);
#endif
	}
	buf += chp->len;
	n -= chp->len;
    }
    cpst->r_order[i] = ENDOF_LIST;
    return(0);
}

time_t
CpDecodeFrame(u_char *buf, int len, long arg)
{
    struct cpframeheader_s *cfp;
    struct cpcodeop_s *cop;
    struct cpstate_s *cpst=NULL;
    struct cpreg_s *cprs;
    u_int16_t hbo_len;

    cprs = cpHead;
    while (cprs) {
	cpst = (struct cpstate_s *)cprs->fsm->priv;
	if (cpst->nbo_proto == arg) break;
	cprs = cprs->next;
    }
    cfp = (struct cpframeheader_s *)buf;
    if (!cprs || cfp->code >= NUM_CODEINFO) {
	if (ISLOG(LOG_CP)) Logf(LOG_CP, "%x: decoder not found\n", arg);
	/* cfp->code >= NUM_CODEINFO then should Scj ? */
	return(0);
    }
    if (ISLOG(LOG_CP))
	Logf(LOG_CP, "%s: Received-%s id=%d",
	     cprs->fsm->name, codeInfo[cfp->code -1].name, cfp->id);
    if (codeInfo[cfp->code -1].reply) {
	if (cpst->l_id != cfp->id) {
	    if (ISLOG(LOG_CP))
		Logf(LOG_CP, "!=%d:", cpst->l_id);
	    if (pppOpt.id_check) {
		Logf(LOG_CP, " discard\n");
		return(0);
	    }
	}
    } else {
	cpst->r_id = cfp->id;
    }
    if (ISLOG(LOG_CP)) Logf(LOG_CP, "\n");
    cop = &cpst->cop[cfp->code - 1];
    hbo_len = ntohs(cfp->nbo_len);
    len = (len < hbo_len) ? len: hbo_len;
    if (cop->fsmevent)
	cop->fsmevent(buf + sizeof(struct cpframeheader_s),
		      len - sizeof(struct cpframeheader_s));
    else if (ISLOG(LOG_CP)) Logf(LOG_CP, " Not implemented\n");
    return(0);
}

/* Event functions */

int
CpRtr(struct fsmreg_s *fsm, u_char *buf, int n)
{
    FsmRTR(fsm);
    return(0);
}

int
CpRta(struct fsmreg_s *fsm, u_char *buf, int n)
{
    FsmRTA(fsm);
    return(0);
}

void
RegisterCp(struct cpstate_s *cpst, struct fsmreg_s *fsm)
{
    struct cpreg_s *cprs;

    cprs = TCALLOC(struct cpreg_s);
    cprs->fsm = fsm;
    cprs->next = cpHead;
    cpHead = cprs;
    fsm->priv = cpst;
    fsm->scr = CpScr;
    fsm->sca = CpSca;
    fsm->scn = CpScn;
    fsm->str = CpStr;
    fsm->sta = CpSta;
/*
    cpst->cop[CPCODE_CONF_REQ-1].encode = CpEncodeReg;
    cpst->cop[CPCODE_CONF_ACK-1].encode = CpEncodeReg;
*/
}
