/*****************************************************************************
* ppipemon.c	PPP Monitor.
*
* Author:	Nenad Corbic <ncorbic@sangoma.com>		
*
* Copyright:	(c) 1995-1999 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* Oct 18, 1999  Nenad Corbic    Added new features for 2.1.0 release version
* Aug 13, 1998	Jaspreet Singh	Improved line tracing code
* Dec 19, 1997	Jaspreet Singh	Added 'irq timeouts' stat in 
*				'READ_COMMS_ERR_STATISTICS'.
* Nov 24, 1997  Jaspreet Singh  Added new stats for driver statistics
* Oct 20, 1997	Jaspreet Singh	Added new cblock.commands for Driver statistics and
*				Router Up time.
* Jul 28, 1997	Jaspreet Singh	Added a new cblock.command for running line trace 
*				displaying RAW data. 
* Jun 24, 1997  Jaspreet Singh	S508/FT1 test cblock.commands
* Apr 25, 1997	Farhan Thawar	Initial version based on ppipemon for WinNT.
*****************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <string.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <linux/wanpipe.h>
#include <linux/sdla_ppp.h>

#define TIMEOUT 1
#define TRUE 1
#define FALSE 0
#define MDATALEN 2024

/*
 * Structures and Types
 */
#pragma pack(1)
typedef struct {
	wp_mgmt_t	wp_mgmt PACKED;
	cblock_t	cblock  PACKED;
        unsigned char data[MDATALEN] PACKED;
} CBLOCK;
#pragma pack()


/* Structures for tracing */
   
/* The one frame trace */

typedef struct {
	unsigned char status;	/* equals to 0x00 if the frame is incoming, 
				   0x01 if outgoing.  For the first frame of the
 				   bunch of frames sent to user the bits 2-7 of
				   this byte tells number of frames passed, and
				   bit 1 tell if there are more frames already 
			 	   available on the board */
	unsigned char passed_length;	/* 0 if no data passed. 1 - if real 
				           length bytes passed
					 */
	unsigned short  real_length;	/* real(original) frame length */
	unsigned short  time_stamp;	/* the frame's time stamp */
	unsigned char data[1];		/* first byte of data */

} FRAME_DATA, *PFRAME_DATA;


/* Prototypes */
void ResetCB(CBLOCK *c1);
int MakeConnection( void );
int ObtainConfiguration( void );
unsigned char DoCommand( CBLOCK *cb );
void SaveCB( CBLOCK *dest, CBLOCK *src );
void init( int argv, char *argc[]);
unsigned long decode_bps( unsigned char bps );
char * get_ip_addr(char *if_name);
void banner(char *);

/* global for now */
int sock;
struct sockaddr_in soin;
CBLOCK cb, cbconfig;
char codeversion[10];
unsigned int frame_count;
int is_508;
int raw_data = FALSE;

/* for S508/FT1 test cblock.commands */
static int fail;
unsigned char par_port_A_byte, par_port_B_byte;
int off_counter, green_counter, red_counter;
int loop_counter;


/* defines for now */
char ipaddress[16];
char cmd[5];
int udp_port = 9000;

void ResetCB(CBLOCK *c1)
{
	memset((void *)&c1->cblock.opp_flag, 0, sizeof(CBLOCK)-16);
}

int MakeConnection( void ) 
{
	sock = socket( PF_INET, SOCK_DGRAM, 0 );

	if( sock < 0 ) {
		printf("Error: Unable to open socket!\n");
		return( FALSE );
	} /* if */

	soin.sin_family = AF_INET;
	soin.sin_addr.s_addr = inet_addr(ipaddress);
     	
	if(soin.sin_addr.s_addr < 0){
                printf("Error: Invalid IP address!\n");
                return( FALSE );
        }
	
	soin.sin_port = htons((u_short)udp_port);

	if( !connect( sock, (struct sockaddr *)&soin, sizeof(soin)) == 0 ) {
		printf("Error: Cannot make connection!\nMake sure the IP address is correct\n");
		return( FALSE );
	} /* if */

	if( !ObtainConfiguration() ) {
		printf("Error: Unable to obtain PPP information.\nMake sure the IP and UDP port are correct.\n");
		close( sock );
		return( FALSE );
	} /* if */   

	return( TRUE );
}; /* MakeConnection */

int ObtainConfiguration( void ) 
{
	unsigned char x;
   
	x = 0;
	cb.cblock.command = PPP_READ_CONFIG;
	cb.cblock.length = 0;

	while (DoCommand(&cb) != 0 && ++x < 4) {
		if (cb.cblock.result == 0xaa) {
			printf("Error: Command timeout occurred"); 
			return(FALSE);
		}

		if (cb.cblock.result == 0xCC ) return(FALSE);
	}

	if (x >= 4) return(FALSE);

	if( cb.cblock.length == 0x73 ) {
		is_508 = TRUE;
	} else {
		is_508 = FALSE;
	} 
   
	strcpy(codeversion, "?.??");
   
	cb.cblock.command = PPP_READ_CODE_VERSION;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if (cb.cblock.result == 0) {
		cb.data[cb.cblock.length] = 0;
		strcpy(codeversion, cb.data);
	}

	return(TRUE);
}; /* ObtainConfiguration */

unsigned char DoCommand( CBLOCK *cb ) 
{
	static unsigned char id = 0;
	fd_set ready;
	struct timeval to;
	int x;

	for( x = 0; x < 4; x += 1 ) {
		cb->cblock.opp_flag = 0;
		cb->wp_mgmt.request_reply = 0x01;
		cb->wp_mgmt.id = id;
		/* 0xAA is our special return code indicating packet timeout */
		cb->cblock.result = 0xaa;	

		send(sock, cb->wp_mgmt.signature, 32+cb->cblock.length,0);
		
		FD_ZERO(&ready);
		FD_SET(sock,&ready);
		to.tv_sec = 5;
		to.tv_usec = 0;

		if(select(sock + 1,&ready, NULL, NULL, &to)) {
			recv(sock, cb->wp_mgmt.signature, 32+MDATALEN,0);
			break;
		} else {
			printf("Error: No Data received from the connected socket.\n"); 
		} /* if */
	} //for

	/* make sure the id is correct (returning results from a previous
	   call may be disastrous if not recognized)
	   also make sure it is a reply
	*/

	if (cb->wp_mgmt.id != id || cb->wp_mgmt.request_reply != 0x02) 
		cb->cblock.result = 0xbb;
	id++;
	if (cb->cblock.result == 0xCC) {
		printf("Error: Code is not running on the board!");
		exit(1);
	}

	return(cb->cblock.result);
   
}; /* DoCommand */


void init( int argc, char *argv[])
{
	int i, i_cnt=0, u_cnt=0, c_cnt=0, if_found=0;
	char * ip_addr;
	struct in_addr *ip_str=NULL;
	cb.wp_mgmt.id = 0;

	for (i=0;i<argc;i++){
		if (!strcmp(argv[i],"-i")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid IP or Interface Name, Type ppipemon <cr> for help\n\n");
				exit(0);
			}

			strcpy(ipaddress,argv[i+1]);
			if (inet_aton(ipaddress,ip_str) != 0 ){
			}else{
				if_found=1;
			}
			i_cnt=1;
		}else if (!strcmp(argv[i],"-u")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid UDP PORT, Type ppipemon <cr> for help\n\n");
				exit(0);
			}

		 	if(isdigit(argv[i+1][0])){
				udp_port = atoi(argv[i+1]);
			}else{
				printf("ERROR: Invalid UDP Port, Type ppipemon <cr> for help\n\n");
				exit(0);
			}
			u_cnt=1;
		}else if (!strcmp(argv[i],"-c")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid Command, Type ppipemon <cr> for help\n\n");
				exit(0);
			}

			strcpy(cmd,argv[i+1]);
			c_cnt=1;
		}
	}

	if (!i_cnt){
		printf("ERROR: No IP address or Interface Name, Type ppipemon <cr> for help\n\n");
		exit(0);
	}
	if (!u_cnt){
		printf("ERROR: No UDP Port, Type ppipemon <cr> for help\n\n");
		exit(0);
	}
	if (!c_cnt){
		printf("ERROR: No Command, Type ppipemon <cr> for help\n\n");
		exit(0);
	}

	if (if_found){
		ip_addr = get_ip_addr(ipaddress);
		strcpy(ipaddress,ip_addr);
	}

	/* signature for UDP Management */
	strcpy( cb.wp_mgmt.signature, "PTPIPEAB");
};

char * get_ip_addr(char *if_name)
{

	int skfd;
	struct sockaddr_in *sin;
	struct ifreq ifr;

    	if ((skfd = socket(AF_INET, SOCK_STREAM, IPPROTO_IP)) < 0) {
		perror("socket");
		exit(1);
    	}

	strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));

        if (ioctl(skfd, SIOCGIFDSTADDR , &ifr) < 0) {
		fprintf(stderr, "Error: Unknown interface: %s\n",if_name);
		exit(0);
    	}

	sin = (struct sockaddr_in *)&ifr.ifr_ifru.ifru_dstaddr;
	sin->sin_family = AF_INET;
	sin->sin_port = 0;

	close (skfd);

	return (inet_ntoa(sin->sin_addr));

} 

unsigned long decode_bps( unsigned char bps ) 
{
	unsigned long number;
 
	switch( bps ) {
		case 0x00:
			number = 0;
			break;
		case 0x01:
			number = 1200;
			break;
		case 0x02:
			number = 2400;
			break;
		case 0x03:
			number = 4800;
			break;
		case 0x04:
			number = 9600;
			break;
		case 0x05:
			number = 19200;
			break;
		case 0x06:
			number = 38400;
			break;
		case 0x07:
			number = 45000;
			break;
		case 0x08:
			number = 56000;
			break;
		case 0x09:
			number = 64000;
			break;
		case 0x0A:
			number = 74000;
			break;
		case 0x0B:
			number = 112000;
			break;
		case 0x0C:
			number = 128000;
			break;
		case 0x0D:
			number = 156000;
			break;
		default:
			number = 0;
			break;
	} /* switch */

	return number;
}; /* decode_bps */

void error( void ) 
{

	printf("Error: Command failed!\n");

}; /* error */

void general_conf( void ) 
{
	unsigned long baud;
	unsigned short s_number, mtu, mru;
	unsigned char misc;
 
	cb.cblock.command = PPP_READ_CONFIG;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		if( is_508 == TRUE ) {
			banner("GENERAL CONFIGURATION: 508 Board");
			memcpy(&baud,&cb.data[0],4);
			memcpy(&s_number,&cb.data[4],2);
			printf("Transmit buffer allocation in percent: %d\n",
				s_number);
			misc = cb.data[6];
			printf("Use alternate adapter frequency: ");
			( misc & 0x10 ) ? printf("Yes\n") : printf("No\n");
			printf("Set interface type to RS-232: ");
			( misc & 0x20 ) ? printf("Yes\n") : printf("No\n");
			memcpy(&mtu, &cb.data[8],2);
			memcpy(&mru, &cb.data[10],2);
		} else {
			banner("GENERAL CONFIGURATION: 502 Board");
			baud = decode_bps(cb.data[0]);
			misc = cb.data[3];
			printf("Discard transmit-aborted frames: ");
			( misc & 0x01 ) ? printf("Yes\n") : printf("No\n");
			memcpy(&mtu, &cb.data[5],2);
			memcpy(&mru, &cb.data[7],2);
		} 

		printf("Baud rate in bps: %lu\n",baud);
		printf("Update transmit statistics( user data ): ");
		( misc & 0x02 ) ? printf("Yes\n") : printf("No\n");
		printf("Update receive statistics( user data ): ");
		( misc & 0x04 ) ? printf("Yes\n") : printf("No\n");
		printf("Timestamp received packets: ");
		( misc & 0x08 ) ? printf("Yes\n") : printf("No\n");
		printf("Maximum Receive/Transmit Unit(MRU/MTU): %d\n",mtu);
		printf("Minimum remote MRU required on connection: %d\n",mtu);
	} else {
		error();
	} 
}; /* general_conf */

void timers( void ) 
{
	int i;
	unsigned short tmp;
   
	cb.cblock.command = PPP_READ_CONFIG;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("PPP TIMERS");

		if( is_508 == TRUE ) {
			i = 12;
		} else {
			i = 9;
		} 

		memcpy(&tmp,&cb.data[i],2);
		printf("Restart timer: %d\n",tmp);
		memcpy(&tmp,&cb.data[i+2],2);
		printf("Authentication restart timer: %d\n",tmp);
		memcpy(&tmp,&cb.data[i+4],2);
		printf("Authentication wait timer: %d\n",tmp);
		memcpy(&tmp,&cb.data[i+6],2);
		printf("DCD/CTS failure detection timer: %d\n",tmp);
		memcpy(&tmp,&cb.data[i+8],2);
		printf("Drop DTR duration timer: %d\n",tmp);
		memcpy(&tmp,&cb.data[i+10],2);
		printf("Connection attempt timeout: %d\n",tmp);
		memcpy(&tmp,&cb.data[i+12],2);
		printf("Max-Configure counter: %d\n",tmp);
		memcpy(&tmp,&cb.data[i+14],2);
		printf("Max-Terminate counter: %d\n",tmp);
		memcpy(&tmp,&cb.data[i+16],2);
		printf("Max-Failure counter: %d\n",tmp);
		memcpy(&tmp,&cb.data[i+18],2);
		printf("Max-Authenticate counter: %d\n",tmp);

	} else {
		error();
	} 
}; /* timers */

void authent( void ) 
{
	unsigned char tmp;
   
	cb.cblock.command = PPP_READ_CONFIG;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {
	
		banner("PPP AUTHENTICATION");

		if( is_508 == TRUE ) {
			tmp = cb.data[32];
		} else {
			tmp = cb.data[29];
		} 
	
		if( tmp & 0x01 ) {
		     printf("Allow the use of PAP for inbound/outbound: Yes\n");
			if( tmp & 0x80 ) {
				printf("Using inbound authentication.\n");
			} else {
				printf("Using outbound authentication.\n");
			} //if
		} else {
			printf("Allow the use of PAP for inbound/outbound: No\n"				);
		} //if

		if( tmp & 0x02 ) {
	 	    printf("Allow the use of CHAP for inbound/outbound: Yes\n");
			if( tmp & 0x80 ) {
				printf("Using inbound authentication.\n");
			} else {
				printf("Using outbound authentication.\n");
			} //if
		} else {
		     printf("Allow the use of CHAP for inbound/outbound: No\n");
		} //if

	} else {
		error();
	} //if
}; /* authent */

void ip_config( void ) 
{
	int i;
   
	cb.cblock.command = PPP_READ_CONFIG;
	cb.cblock.length = 0;

	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("PPP IP CONFIGURATION");

		if( is_508 == TRUE ) {
			i = 33;
		} else {
			i = 30;
		} //if
		printf("Enable the use of IP: ");
		( cb.data[i] & 0x80 ) ? printf("Yes\n") : printf("No\n");
		printf("Notify remote of locally-configure address: ");
		( cb.data[i] & 0x01 ) ? printf("Yes\n") : printf("No\n");
		printf("Local IP address( 0.0.0.0 = request ): %d.%d.%d.%d\n",
			cb.data[i+1],cb.data[i+2],cb.data[i+3],cb.data[i+4]);
		printf("Request remote to provide local address: ");
		( cb.data[i] & 0x02 ) ? printf("Yes\n") : printf("No\n");
		printf("Provide remote with pre-configured address: ");
		( cb.data[i] & 0x04 ) ? printf("Yes\n") : printf("No\n");
		printf("Remote IP address: %d.%d.%d.%d.\n",cb.data[i+5],
			cb.data[i+6],cb.data[i+7],cb.data[i+8]);
		printf("Require that remote provide an address: ");
		( cb.data[i] & 0x08 ) ? printf("Yes\n") : printf("No\n");
	} else {
		error();
	} 
}; /* ip_config */

void ipx_config( void )
{
	int i;
   
	cb.cblock.command = PPP_READ_CONFIG;
	cb.cblock.length = 0;

	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {
	
		banner("PPP IPX CONFIGURATION");

		if( is_508 == TRUE ) {
			i = 42;
		} else {
			i = 39;
		} //if

		printf("Enable the use of IPX: ");
		( cb.data[i] & 0x80 ) ? printf("Yes\n") : printf("No\n");
		printf("Include network number in Config-Request: ");
		( cb.data[i] & 0x01 ) ? printf("Yes\n") : printf("No\n");
		printf("Network number( 00000000 = request ): %02X %02X %02X %02X\n",cb.data[i+1],cb.data[i+2],cb.data[i+3],cb.data[i+4]);
		printf("Include local node # in Config-Request: ");
		( cb.data[i] & 0x02 ) ? printf("Yes\n") : printf("No\n");
		printf("Local node number( 000000000000 = request ): %02X %02X %02X %02X %02X %02X\n",cb.data[i+5],cb.data[i+6],cb.data[i+7],cb.data[i+8],cb.data[i+9],cb.data[i+10]);
		printf("Force remote to accept remote node number: ");
		( cb.data[i] & 0x04 ) ? printf("Yes\n") : printf("No\n");
		printf("Remote node number: %02X %02X %02X %02X %02X %02X\n",
			cb.data[i+11],cb.data[i+12],cb.data[i+13],cb.data[i+14],
			cb.data[i+15],cb.data[i+16]);
		printf("Include config-complete in Config-Request: ");
		( cb.data[i] & 0x40 ) ? printf("Yes\n") : printf("No\n");

		if( cb.data[i] & 0x20 ) {
			printf("Routing protocol: Request default( normally RIP/SAP )\n");
		} else if( cb.data[i] & 0x18 ){
			printf("Routing protocol: Use either RIP/SAP or NLSP\n");
		} else if( cb.data[i] & 0x08 ){
			printf("Routing protocol: Use RIP/SAP only\n");
		} else if( cb.data[i] & 0x10 ){
			printf("Routing protocol: Use NLSP only\n");
		} else {
			printf("Routing protocol: No routing protocol\n");
		} //if
		printf("Local router name( max. 47 characters ): %s\n",
			&cb.data[i+17]);
	} else {
		error();
	} //if
}; /* ipx_config */

void set_state( unsigned char tmp ) 
{
	switch( tmp ) {
		case 0:
			printf("Initial\n");
			break;
		case 1:
			printf("Starting\n");
			break;
		case 2:
			printf("Closed\n");
			break;
		case 3:
			printf("Stopped\n");
			break;
		case 4:
			printf("Closing\n");
			break;
		case 5:
			printf("Stopping\n");
			break;
		case 6:
			printf("Request Sent\n");
			break;
		case 7:
			printf("Ack Received\n");
			break;
		case 8:
			printf("Ack Sent\n");
			break;
		case 9:
			printf("Opened\n");
			break;
		default:
			printf("Unknown\n");
			break;
	} //switch

}; /* set_state */

void state( void ) 
{
	cb.cblock.command = PPIPE_GET_IBA_DATA;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("PPP LINK STATE");

		switch( cb.data[2] ) {
			case 0:
				printf("PPP Phase: Link Dead\n");
				break;
			case 1:
				printf("PPP Phase: Establishment( LCP )\n");
				break;
			case 2:
				printf("PPP Phase: Authentication\n");
				break;
			case 3:
				printf("PPP Phase: Network Layer\n");
				break;
			case 4:
				printf("PPP Phase: Link Termination\n");
				break;
			default:
				printf("PPP Phase: Unknown\n");
				break;
		} 

		printf("LCP State: ");
		set_state( cb.data[1] );
		printf("IPCP State: ");
		set_state( cb.data[3] );
		printf("IPXCP State: ");
		set_state( cb.data[4] );
		switch( cb.data[5] ) {
			case 0:
				printf("PAP State: Initial( Inactive )\n");
				break;
			case 1:
				printf("PAP State: Failed\n");
				break;
			case 2:
				printf("PAP State: Request Sent\n");
				break;
			case 3:
				printf("PAP State: Waiting\n");
				break;
			case 4:
				printf("PAP State: Opened( Success )\n");
				break;
			default:
				printf("PAP State: Unknown\n");
				break;
		} 

		switch( cb.data[5] ) {
			case 0:
				printf("CHAP State: Initial( Inactive )\n");
				break;
			case 1:
				printf("CHAP State: Failed\n");
				break;
			case 2:
				printf("CHAP State: Challenge Sent\n");
				break;
			case 3:
				printf("CHAP State: Waiting for Challenge\n");
				break;
			case 4:
				printf("CHAP State: Response Sent\n");
				break;
			case 5:
				printf("CHAP State: Opened( Success )\n");
				break;
			default:
				printf("CHAP State: Unknown\n");
				break;
		} 
	} else {
		error();
	} 
}; /* state */

void negot( void ) {
	unsigned short mru;
   
	cb.cblock.command = PPP_GET_CONNECTION_INFO;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("PPP NEGOTIATIONS");

		memcpy(&mru,&cb.data[0],2);
		printf("Remote Maximum Receive Unit: %d\n",mru);
		printf("Negotiated IP options: %02X",cb.data[2]);
		( cb.data[2] & 0x80 ) ? printf("( IP Enabled )\n") : 
					printf("( IP Disabled )\n");
		printf("Local IP address: %d.%d.%d.%d\n",cb.data[3],cb.data[4],
			cb.data[5],cb.data[6]);
		printf("Remote IP address: %d.%d.%d.%d\n",cb.data[7],cb.data[8],
			cb.data[9],cb.data[10]);
		printf("Negotiated IPX options: %02X",cb.data[11]);
		( cb.data[11] & 0x80 ) ? printf("( IPX Enabled )\n") : 
					printf("( IPX Disabled )\n");
		printf("IPX network number: %02X %02X %02X %02X\n",cb.data[12],
			cb.data[13],cb.data[14],cb.data[15]);
		printf("Local IPX node number: %02X %02X %02X %02X %02X %02X\n",
			cb.data[16],cb.data[17],cb.data[18],cb.data[19],
			cb.data[20],cb.data[21]);
		printf("Remote IPX node number: %02X %02X %02X %02X %02X %02X\n"
			,cb.data[22],cb.data[23],cb.data[24],cb.data[25],
			cb.data[26],cb.data[27]);
		printf("Remote IPX router name: %s\n",&cb.data[28]);
		
		switch( cb.data[76] ) {
			case 0:
				printf("Authentication status: No inbound authentication negotiated\n");
				break;
			case 1:
				printf("Authentication status: PeerID valid, Password Incorrect\n");
				break;
			case 2:
				printf("Authentication status: PeerID was invalid\n");
				break;
			case 3:
				printf("Authentication status: PeerID and Password were correct\n");
				break;
			default:
				printf("Authentication status: Unknown\n");
				break;
		} 

		printf("Inbound PeerID: %s\n",&cb.data[77]);
		
	} else {
		error();
	} //if
}; /* negot */

void cause( void ) 
{
	unsigned short disc;
  
	cb.cblock.command = PPIPE_GET_IBA_DATA;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("LAST CAUSE OF LINK FAILURE");

		memcpy(&disc,&cb.data[7],2);
		printf("Local request by termination phase: ");
		(disc & 0x0100) ? printf("Yes\n") : printf("No\n");
		printf("DCD and/or CTS dropped: ");
		(disc & 0x0200) ? printf("Yes\n") : printf("No\n");
		printf("Disabled communications locally: ");
		(disc & 0x0400) ? printf("Yes\n") : printf("No\n");
		printf("Inbound/Outbound authentication failed: ");
		(disc & 0x0800) ? printf("Yes\n") : printf("No\n");
		printf("Failed to negotiate inbound auth. protocol with peer:");
		(disc & 0x1000) ? printf(" Yes\n") : printf(" No\n");
		printf("Rejected peer's request for authentication: ");
		(disc & 0x2000) ? printf("Yes\n") : printf("No\n");
		printf("Peer rejected MRU option of config-request: ");
		(disc & 0x4000) ? printf("Yes\n") : printf("No\n");
		printf("MRU of peer was below required minumum: ");
		(disc & 0x8000) ? printf("Yes\n") : printf("No\n");
		printf("Rejected peer's LCP option(s) too many times: ");
		(disc & 0x0001) ? printf("Yes\n") : printf("No\n");
		printf("Rejected peer's IPCP option(s) too many times: ");
		(disc & 0x0002) ? printf("Yes\n") : printf("No\n");
		printf("Rejected peer's IPXCP option(s) too many times: ");
		(disc & 0x0004) ? printf("Yes\n") : printf("No\n");
	} else {
		error();
	} //if
}; /* cause */

void modem( void )
{
	unsigned char cts_dcd;
   
	cb.cblock.command = PPIPE_GET_IBA_DATA;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("MODEM STATUS");

		memcpy(&cts_dcd,&cb.data[0],1);
		printf("DCD: ");
		(cts_dcd & 0x08) ? printf("High\n") : printf("Low\n");
		printf("CTS: ");
		(cts_dcd & 0x20) ? printf("High\n") : printf("Low\n");
	} else {
		error();
	} //if
}; /* modem */

void general_stats( void ) 
{
	unsigned short tmp;
	unsigned long l_tmp;
   
	cb.cblock.command = PPP_READ_STATISTICS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner ("GENERAL STATISTICS");

		if( is_508 == TRUE ) {
			memcpy(&tmp,&cb.data[2],2);
			printf("Number of received frames discarded due to bad length: %d\n",tmp);
		} else {
			memcpy(&tmp,&cb.data[0],2);
			printf("Number of received frames discarded at the interrupt level: %d\n",tmp);
			memcpy(&tmp,&cb.data[2],2);
			printf("Number of received frames discarded at the application level: %d\n",tmp);
			memcpy(&tmp,&cb.data[4],2);
			printf("Number of received retransmitted due to aborts: %d\n",tmp);
		} 

		memcpy(&l_tmp,&cb.data[6],4);
		printf("Number of user frames transmitted: %lu\n",l_tmp);
		memcpy(&l_tmp,&cb.data[10],4);
		printf("Number of user bytes transmitted: %lu\n",l_tmp);
		memcpy(&l_tmp,&cb.data[14],4);
		printf("Number of user frames received: %lu\n",l_tmp);
		memcpy(&l_tmp,&cb.data[18],4);
		printf("Number of user bytes received: %lu\n",l_tmp);
	} else {
		error();
	} 
}; /* general_stats */

void flush_general_stats( void ) 
{
	cb.cblock.command = PPP_FLUSH_STATISTICS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result != 0 ) {
		error();
	} //if

}; /* flush_general_stats */

void comm_err( void ) 
{
	cb.cblock.command = PPP_READ_ERROR_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("COMMUNICATION ERROR STATISTICS");

		if( is_508 == TRUE ) {
			printf("Number of times receiver was halted due to full buffers: %d\n",cb.data[3]);
		} else {
			printf("Number of frames discarded at the interrupt level due to frame being too long: %d\n",cb.data[3]);
			printf("Number of transmit underruns: %d\n",cb.data[5]);
		} //if
		printf("Number of receiver overrun errors: %d\n",cb.data[0]);
		printf("Number of receiver CRC errors: %d\n",cb.data[1]);
		printf("Number of abort frames received: %d\n",cb.data[2]);
		printf("Number of abort frames transmitted: %d\n",cb.data[4]);
		printf("Number of missed transmit underrun interrupts: %d\n",cb.data[6]);
		printf("Number of IRQ timeouts: %d\n",cb.data[7]);
		printf("Number of times DCD changed state: %d\n",cb.data[8]);
		printf("Number of times CTS changed state: %d\n",cb.data[9]);
	} else {
		error();
	} //if
}; /* comm_err */

void flush_comm_err( void ) 
{
	cb.cblock.command = PPP_FLUSH_ERROR_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result != 0 ) {
		error();
	} 
}; /* flush_general_stats */

void packet( void ) 
{
	unsigned short tmp, tmp2;
   
	cb.cblock.command = PPP_READ_PACKET_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("PACKET STATISTICS");

		memcpy(&tmp,&cb.data[0],2);
		printf("Number discards( bad header ): %d\n",tmp);
		memcpy(&tmp,&cb.data[2],2);
		printf("Number discards( unknown/unsupported protocol ): %d\n",
			tmp);
		memcpy(&tmp,&cb.data[4],2);
		printf("Number discards(unknown/unsupported protocol+too large for Protocol-Reject): %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&cb.data[6],2);
		memcpy(&tmp2,&cb.data[8],2);
		printf("Number of LCP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[10],2);
		memcpy(&tmp2,&cb.data[12],2);
		printf("Number of IPCP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[14],2);
		memcpy(&tmp2,&cb.data[16],2);
		printf("Number of IPXCP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[18],2);
		memcpy(&tmp2,&cb.data[20],2);
		printf("Number of PAP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[22],2);
		memcpy(&tmp2,&cb.data[24],2);
		printf("Number of CHAP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[26],2);
		memcpy(&tmp2,&cb.data[28],2);
		printf("Number of LQR packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[30],2);
		memcpy(&tmp2,&cb.data[32],2);
		printf("Number of IP packets:  %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[34],2);
		memcpy(&tmp2,&cb.data[36],2);
		printf("Number of IPX packets: %d\t\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* packet */

void flush_packet( void ) 
{
	cb.cblock.command = PPP_FLUSH_PACKET_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result != 0 ) {
		error();
	} 
}; /* flush_packet */

void lcp( void ) 
{
	unsigned short tmp, tmp2;
 
	cb.cblock.command = PPP_READ_LCP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("LCP STATISTICS");

		memcpy(&tmp,&cb.data[0],2);
		printf("Packets discarded (unknown LCP code): %d\n",tmp);
		memcpy(&tmp,&cb.data[48],2);
		printf("Received LCP packets too large: %d\n",tmp);
		memcpy(&tmp,&cb.data[50],2);
		printf("Received packets invalid or out-of-sequence Configure-Acks: %d\n",tmp);
		memcpy(&tmp,&cb.data[52],2);
		printf("Received packets invalid Configure-Naks or Configure-Rejects: %d\n",tmp);
		memcpy(&tmp,&cb.data[54],2);
		printf("Configure-Naks or Configure-Rejects with bad Identifier: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&cb.data[2],2);
		memcpy(&tmp2,&cb.data[26],2);
		printf("Number of Config-Request pkts: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[4],2);
		memcpy(&tmp2,&cb.data[28],2);
		printf("Number of Config-Ack packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[6],2);
		memcpy(&tmp2,&cb.data[30],2);
		printf("Number of Config-Nack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[8],2);
		memcpy(&tmp2,&cb.data[32],2);
		printf("Number of Config-Reject packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[10],2);
		memcpy(&tmp2,&cb.data[34],2);
		printf("Number of Term-Reqst packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[12],2);
		memcpy(&tmp2,&cb.data[36],2);
		printf("Number of Terminate-Ack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[14],2);
		memcpy(&tmp2,&cb.data[38],2);
		printf("Number of Code-Reject packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[16],2);
		memcpy(&tmp2,&cb.data[40],2);
		printf("Number of Protocol-Rej packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[18],2);
		memcpy(&tmp2,&cb.data[42],2);
		printf("Number of Echo-Request packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[20],2);
		memcpy(&tmp2,&cb.data[44],2);
		printf("Number of Echo-Reply packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[22],2);
		memcpy(&tmp2,&cb.data[46],2);
		printf("Number of Discard-Request packets: %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* lcp */

void flush_lcp( void ) 
{
	cb.cblock.command = PPP_FLUSH_LCP_STATS;
	cb.cblock.length = 0;

	DoCommand(&cb);
	if( cb.cblock.result != 0 ) {
		error();
	} 
}; /* flush_packet */

void loopback( void ) 
{
	unsigned short tmp;
   
	cb.cblock.command = PPP_READ_LCP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);
	
	if( cb.cblock.result == 0 ) {

		banner("LOOPBACK STATISTICS");

		memcpy(&tmp,&cb.data[0],2);
		printf("Looped-back link possible given Config-Req/Nak/Rej: %d\n",tmp);
		memcpy(&tmp,&cb.data[2],2);
		printf("Looped-back link detected with Echo-Request: %d\n",tmp);
		memcpy(&tmp,&cb.data[4],2);
		printf("Echo-Request received from bad source: %d\n",tmp);
		memcpy(&tmp,&cb.data[6],2);
		printf("Looped-back link detected with Echo-Reply: %d\n",tmp);
		memcpy(&tmp,&cb.data[8],2);
		printf("Echo-Reply received from bad source: %d\n",tmp);
		memcpy(&tmp,&cb.data[10],2);
		printf("Looped-back link detected with Discard-Request: %d\n",
			tmp);
		memcpy(&tmp,&cb.data[12],2);
		printf("Discard-Request received from bad source: %d\n",tmp);
		memcpy(&tmp,&cb.data[14],2);
		printf("Echo-Reply discarded( transmit collision ): %d\n",tmp);
		memcpy(&tmp,&cb.data[16],2);
		printf("Echo-Reply discarded( receive collision ): %d\n",tmp);
	
	} else {
		error();
	} 
}; /* loopback */

void flush_loopback( void ) 
{
	cb.cblock.command = PPP_FLUSH_LPBK_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result != 0 ) {
		error();
	}
 
}; /* flush_loopback */

void ipcp( void ) 
{
	unsigned short tmp, tmp2;
   
	cb.cblock.command = PPP_READ_IPCP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("IPCP STATISTICS");

		memcpy(&tmp,&cb.data[0],2);
		printf("Packets discarded (unknown IPCP code): %d\n",tmp);
		memcpy(&tmp,&cb.data[32],2);
		printf("Received IPCP packets too large: %d\n",tmp);
		memcpy(&tmp,&cb.data[34],2);
		printf("Received packets invalid or out-of-sequence Configure-Acks: %d\n",tmp);
		memcpy(&tmp,&cb.data[36],2);
		printf("Received packets invalid Configure-Naks or Configure-Rejects: %d\n",tmp);
		memcpy(&tmp,&cb.data[38],2);
		printf("Configure-Naks or Configure-Rejects with bad Identifier: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&cb.data[2],2);
		memcpy(&tmp2,&cb.data[18],2);
		printf("Number of Config-Request pkts: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[4],2);
		memcpy(&tmp2,&cb.data[20],2);
		printf("Number of Config-Ack packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[6],2);
		memcpy(&tmp2,&cb.data[22],2);
		printf("Number of Config-Nack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[8],2);
		memcpy(&tmp2,&cb.data[24],2);
		printf("Number of Config-Reject packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[10],2);
		memcpy(&tmp2,&cb.data[26],2);
		printf("Number of Term-Reqst packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[12],2);
		memcpy(&tmp2,&cb.data[28],2);
		printf("Number of Terminate-Ack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[14],2);
		memcpy(&tmp2,&cb.data[30],2);
		printf("Number of Code-Reject packets: %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* ipcp */

void flush_ipcp( void ) 
{
	cb.cblock.command = PPP_FLUSH_IPCP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result != 0 ) {
		error();
	} 
}; /* flush_ipcp */

void ipxcp( void ) 
{
	unsigned short tmp, tmp2;
  
	cb.cblock.command = PPP_READ_IPXCP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);
	if( cb.cblock.result == 0 ) {

		banner("IPXCP STATISTICS");

		memcpy(&tmp,&cb.data[0],2);
		printf("Packets discarded (unknown IPXCP code): %d\n",tmp);
		memcpy(&tmp,&cb.data[32],2);
		printf("Received IPXCP packets too large: %d\n",tmp);
		memcpy(&tmp,&cb.data[34],2);
		printf("Received packets invalid or out-of-sequence Configure-Acks: %d\n",tmp);
		memcpy(&tmp,&cb.data[36],2);
		printf("Received packets invalid Configure-Naks or Configure-Rejects: %d\n",tmp);
		memcpy(&tmp,&cb.data[38],2);
		printf("Configure-Naks or Configure-Rejects with bad Identifier: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&cb.data[2],2);
		memcpy(&tmp2,&cb.data[18],2);
		printf("Number of Config-Request pkts: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[4],2);
		memcpy(&tmp2,&cb.data[20],2);
		printf("Number of Config-Ack packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[6],2);
		memcpy(&tmp2,&cb.data[22],2);
		printf("Number of Config-Nack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[8],2);
		memcpy(&tmp2,&cb.data[24],2);
		printf("Number of Config-Reject packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[10],2);
		memcpy(&tmp2,&cb.data[26],2);
		printf("Number of Term-Reqst packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[12],2);
		memcpy(&tmp2,&cb.data[28],2);
		printf("Number of Terminate-Ack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[14],2);
		memcpy(&tmp2,&cb.data[30],2);
		printf("Number of Code-Reject packets: %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} //if
}; /* ipxcp */

void flush_ipxcp( void ) 
{
	cb.cblock.command = PPP_FLUSH_IPXCP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);
	if( cb.cblock.result != 0 ) {
		error();
	}
}; /* flush_ipxcp */

void pap( void ) 
{
	unsigned short tmp, tmp2;
   
	cb.cblock.command = PPP_READ_PAP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {


		banner("PAP STATISTICS");

		memcpy(&tmp,&cb.data[0],2);
		printf("Packets discarded (unknown PAP code): %d\n",tmp);
		memcpy(&tmp,&cb.data[16],2);
		printf("Received PAP packets too large: %d\n",tmp);
		memcpy(&tmp,&cb.data[18],2);
		printf("Received packets invalid inbound PeerID: %d\n",tmp);
		memcpy(&tmp,&cb.data[20],2);
		printf("Received packets invalid inbound Password: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&cb.data[2],2);
		memcpy(&tmp2,&cb.data[10],2);
		printf("Number of Authent-Request pkts: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[4],2);
		memcpy(&tmp2,&cb.data[12],2);
		printf("Number of Authent-Ack packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[6],2);
		memcpy(&tmp2,&cb.data[14],2);
		printf("Number of Authent-Nack packets: %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* pap */

void flush_pap( void ) 
{
	cb.cblock.command = PPP_FLUSH_PAP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);
	if( cb.cblock.result != 0 ) {
		error();
	} 
}; /* flush_pap */

void chap( void ) 
{
	unsigned short tmp, tmp2;
   
	cb.cblock.command = PPP_READ_CHAP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) {

		banner("CHAP STATISTICS");

		memcpy(&tmp,&cb.data[0],2);
		printf("Packets discarded (unknown CHAP code): %d\n",tmp);
		memcpy(&tmp,&cb.data[20],2);
		printf("Received CHAP packets too large: %d\n",tmp);
		memcpy(&tmp,&cb.data[22],2);
		printf("Received packets invalid inbound PeerID: %d\n",tmp);
		memcpy(&tmp,&cb.data[24],2);
		printf("Received packets invalid inbound Password/Secret: %d\n",
			tmp);
		memcpy(&tmp,&cb.data[26],2);
		printf("Received packets invalid inbound MD5 message digest format: %d\n",tmp);
		memcpy(&tmp,&cb.data[28],2);
		printf("Invalid inbound ID or out-of-order or unelicited responses: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&cb.data[2],2);
		memcpy(&tmp2,&cb.data[12],2);
		printf("Number of Challenge packets  : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[4],2);
		memcpy(&tmp2,&cb.data[14],2);
		printf("Number of Response packets   : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[6],2);
		memcpy(&tmp2,&cb.data[16],2);
		printf("Number of Success packets    : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&cb.data[8],2);
		memcpy(&tmp2,&cb.data[18],2);
		printf("Number of Failure packets    : %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* chap */

void flush_chap( void ) 
{
	cb.cblock.command = PPP_FLUSH_CHAP_STATS;
	cb.cblock.length = 0;
	DoCommand(&cb);

	if( cb.cblock.result != 0 ) {
		error();
	} 
}; /* flush_chap */

void usage( void ) 
{
	printf("ppipemon -i <ip-address or interface name> -u <port> -c <command>\n\n");
	printf("\tOption -i: \n");
	printf("\t\tWanpipe remote IP address must be supplied\n");
	printf("\t\t<or> Wanpipe network interface name (ex: ppp0)\n");   
	printf("\tOption -u: \n");
	printf("\t\tWanpipe UDPPORT specified in /etc/router.conf\n");
	printf("\tOption -c: \n");
	printf("\t\tCommand is split into two parts:\n"); 
	printf("\t\t\tFirst letter is a command and the rest are options:\n"); 
	printf("\t\t\tex: xm = View Modem Status\n\n");
	printf("\tSupported Commands: x=status : s=statistics : t=trace \n");
	printf("\t                    c=config : T=FT1 stats  : f=flush\n\n");
	printf("\tCommand:  Options:   Description \n");	
	printf("\t-------------------------------- \n\n");    
	printf("\tCard Status\n");
	printf("\t   x         m       Modem Status\n");
	printf("\t             n       Parameters Negotiated on Last Connection/Attempt\n");
	printf("\t             ru      Display Router UP time\n");
	printf("\t             u       PPP Timers and Counters\n");
	printf("\t             s       PPP FSM Current State\n");
	printf("\t             c       Cause for Last Disconnection\n");
	printf("\tCard Configuration\n");
	printf("\t   c         g       PPP General Configuration\n");
	printf("\t             a       Authentication Configuration\n");
	printf("\t             i       IP Configuration\n");
	printf("\t             x       IPX Configuration\n");
	printf("\tCard Statistics\n");
	printf("\t   s         g       Global Statistics\n");
	printf("\t             c       Communication Error Statistics\n");
	printf("\t             p       Packet Statistics\n");
	printf("\t             lpc     LCP Statistics\n");
	printf("\t             lo      Loopback Detection / LCP Error Statistics\n");
	printf("\t             ipc     IP Control Protocol( IPCP )Statistics\n");
	printf("\t             xpc     IPX Control Protocol( IPXCP )Statistics\n");
	printf("\t             pap     Password Authentication (PAP) Statistics\n");
	printf("\t             chp     Challenge-Handshake Auth.(CHAP) Statistics\n");
	printf("\tTrace Data\n");
	printf("\t   t         i       Trace and Interpert ALL frames\n");
	printf("\t             ip      Trace and Interpert PROTOCOL frames only\n");
	printf("\t             id      Trace and Interpert DATA frames only\n");
	printf("\t             r       Trace ALL frames, in RAW format\n");
	printf("\t             rp      Trace PROTOCOL frames only, in RAW format\n");
	printf("\t             rd      Trace DATA frames only, in RAW format\n");
	printf("\tFT1 Configuration\n");
	printf("\t   T         v       View Status\n");
	printf("\t             s       Self Test\n");
	printf("\t             l       Line Loop Test\n");
	printf("\t             d       Digital Loop Test\n");
	printf("\t             r       Remote Test\n");
	printf("\t             o       Operational Mode\n");
	printf("\tFlush Statistics\n");
	printf("\t   f         g       Global Statistics\n");
	printf("\t             c       Communication Error Statistics\n");
	printf("\t             p       Packet Statistics\n");
	printf("\t             lpc     LCP Statistics\n");
	printf("\t             lo      Loopback Detection / LCP Error Statistics\n");
	printf("\t             ipc     IP Control Protocol( IPCP )Statistics\n");
	printf("\t             xpc     IPX Control Protocol( IPXCP )Statistics\n");
	printf("\t             pap     Password Authentication (PAP) Statistics\n");
	printf("\t             chp     Challenge-Handshake Auth.(CHAP) Statistics\n");
	printf("\tDriver Statistics\n");
	printf("\t   d         ds      Display If_Send Driver Statistics\n");
	printf("\t             di      Display Interrrupt Driver Statistics\n");
	printf("\t             dg      Display General Driver Statistics\n"); 
	printf("\n\tExamples:\n");
	printf("\t--------\n\n");
	printf("\tex: ppipemon -i ppp0 -u 9000 -c xm       :View Modem Status \n");
	printf("\tex: ppipemon -i 201.1.1.2 -u 9000 -c ti  :Trace and Interpert ALL frames\n\n");
}

void line_trace(int trace_mode) 
{
	unsigned char num_frames, num_chars;
	unsigned short curr_pos = 0;
	PFRAME_DATA frame_info;
	unsigned int i, j;
	unsigned char outstr[2000];
	int recv_buff = sizeof(CBLOCK) + 100;
	fd_set ready;
	struct timeval to;
	unsigned short tmp;
 
	setsockopt( sock, SOL_SOCKET, SO_RCVBUF, &recv_buff, sizeof(int) );

        cb.cblock.command = PPIPE_DISABLE_TRACING;
        cb.cblock.length = 1;
        cb.data[0] = DISABLE_TRACING;
        DoCommand(&cb);   	

	cb.cblock.command = PPIPE_ENABLE_TRACING;
	cb.cblock.length = 1;
	cb.data[0]=0;
	if(trace_mode == TRACE_PROT){
                cb.data[0] = TRACE_SIGNALLING_FRAMES;
        }else if(trace_mode == TRACE_DATA){
                cb.data[0] = TRACE_DATA_FRAMES;
        }else{
                cb.data[0] = TRACE_SIGNALLING_FRAMES | 
			           TRACE_DATA_FRAMES; 
	}
	DoCommand(&cb);

	if( cb.cblock.result == 0 ) { 
		printf("Starting trace...(Press ENTER to exit)\n");
	} else if( cb.cblock.result == 0xCD ) {
		printf("Cannot Enable Line Tracing from Underneath.\n");
		return;
	} else if( cb.cblock.result == 0x01 ) {
		printf("Starting trace...(although it's already enabled!)\n");
		printf("Press ENTER to exit.\n");
	} else {
		printf("Failed to Enable Line Tracing. Return code: 0x%02X\n", 
			cb.cblock.result );
		return;
	}

	for(;;) {
		FD_ZERO(&ready);
		FD_SET(0,&ready);
		to.tv_sec = 1;
		to.tv_usec = 0;
	
		if(select(1,&ready, NULL, NULL, &to)) {
			break;
		} /* if */

		cb.cblock.command = PPIPE_GET_TRACE_INFO;
		cb.cblock.length = 0;
		DoCommand(&cb);

		if (cb.cblock.result == 0 && cb.cblock.length) { 
			/* if we got something back then get number of frames */
			num_frames = cb.data[0] >> 2;
			for ( i=0; i<num_frames; i++) {
				
				frame_info = (PFRAME_DATA)(cb.data + curr_pos);

				/*  frame type */
				if (frame_info->status & 0x01) {
					sprintf(outstr,"OUTGOING\t");
				} else {
					sprintf(outstr,"INCOMING\t");
				}

				/* real length and time stamp */
				sprintf(outstr+strlen(outstr), "%d\t%d\t", 
					frame_info->real_length, 
					frame_info->time_stamp);

				/* first update curr_pos */
				curr_pos += sizeof(FRAME_DATA);
				
				if (frame_info->passed_length == 0) {
					sprintf( outstr+strlen(outstr), "the frame data is not available" );
				} else {
			 	   /* update curr_pos again */
				   curr_pos += (frame_info->real_length-1);
				   num_chars = (unsigned char)
					((frame_info->real_length <= 25)
					?frame_info->real_length:25);

				   if (raw_data) { /*  show raw data */

					for( j=0; j<num_chars; j++ ) {
						sprintf(outstr+strlen(outstr), "%02X ", (unsigned char)frame_info->data[j]);
					}
					outstr[strlen(outstr)-1] = '\0';

				   } else { /* show int data */
					memcpy(&tmp,&frame_info->data[2] ,2);

					switch(tmp) {

					case 0x2100: //IP Packet
						sprintf(outstr+strlen(outstr),"IP data packet from %d.%d.%d.%d to %d.%d.%d.%d",frame_info->data[16],frame_info->data[17],frame_info->data[18],frame_info->data[19],frame_info->data[20],frame_info->data[21],frame_info->data[22],frame_info->data[23]);
						break;
					case 0x2B00: //IPX Packet
						sprintf(outstr+strlen(outstr),"IPX Data packet");
						break;
					case 0x2180: //IPCP Packet
					case 0x2B80: //IPXCP Packet
					case 0x21C0: //LCP Packet

						switch(tmp) {
						case 0x2180:
							sprintf(outstr+strlen(outstr),"IPCP packet - ");
							break;
						case 0x2B80:
							sprintf(outstr+strlen(outstr),"IPXCP packet - ");
							break;
						case 0x21C0:
							sprintf(outstr+strlen(outstr),"LCP packet - ");
							break;
						default:
							break;
						} 
						switch(frame_info->data[4]){
						case 1:
							sprintf(outstr+strlen(outstr),"Configure REQUEST");
							break;
						case 2:
							sprintf(outstr+strlen(outstr),"Configure ACK");
							break;
						case 3:
							sprintf(outstr+strlen(outstr),"Configure NACK");
							break;
						case 4:
							sprintf(outstr+strlen(outstr),"Configure REJECT");
							break;
						case 5:
							sprintf(outstr+strlen(outstr),"Terminate REQUEST");
							break;
						case 6:
							sprintf(outstr+strlen(outstr),"Terminate ACK");
							break;
						case 7:
							sprintf(outstr+strlen(outstr),"Code REJECT");
							break;
						case 8:
							sprintf(outstr+strlen(outstr),"Protocol REJECT");
							break;
						case 9:
							sprintf(outstr+strlen(outstr),"Echo REQUEST");
							break;
						case 10:
							sprintf(outstr+strlen(outstr),"Echo REPLY");
							break;
						case 11:
							sprintf(outstr+strlen(outstr),"Discard REQUEST");
							break;
						default:
							sprintf(outstr+strlen(outstr),"Unknown type");
							break;
						} 
						break;

					case 0x23C0: //PAP
						sprintf(outstr+strlen(outstr),"PAP packet - ");
						switch(frame_info->data[4]){
						case 1:
						sprintf(outstr+strlen(outstr),"Authenticate REQUEST");
						break;
						case 2:
						sprintf(outstr+strlen(outstr),"Authenticate ACK");
						break;
						case 3:
						sprintf(outstr+strlen(outstr),"Authenticate NACK");
						break;
						default:
						sprintf(outstr+strlen(outstr),"Unknown type");
						break;
						} //switch
						break;
					case 0x25C0: //LQR
						sprintf(outstr+strlen(outstr),"Link Quality Report");
						break;
					case 0x23C2: //PAP
						sprintf(outstr+strlen(outstr),"CPAP packet - ");
						switch(frame_info->data[4]){
						case 1:
							sprintf(outstr+strlen(outstr),"Challenge");
							break;
						case 2:
							sprintf(outstr+strlen(outstr),"Response");
							break;
						case 3:
							sprintf(outstr+strlen(outstr),"Success");
							break;
						case 4:
							sprintf(outstr+strlen(outstr),"Failure");
							break;
						default:
							sprintf(outstr+strlen(outstr),"Unknown type");
							break;
						} //switch
						break;
					default:
						sprintf(outstr+strlen(outstr),"Unknown type");
						break;
					} 
				   } //if
				} //if

				/*  we have to stay on even boundary here, so 
				    update the curr_pos if needed. It's done 
				    exactly the same way in driver.
 			 	 */
 
				if (curr_pos & 0x0001) curr_pos++;
				printf("%s\n",outstr);
			} //for
		} //if
		curr_pos = 0;

		if (!(cb.data[0] &= 0x02)) { 
			sleep(TIMEOUT);
		}
	}

	cb.cblock.command = PPIPE_DISABLE_TRACING;
	cb.cblock.length = 0;
	DoCommand(&cb);
}; /* line_trace */

/* This subroutine enables the FT1 monitor on the board */

void set_FT1_monitor_status( unsigned char status) 
{
	fail = 0;
	cb.cblock.command = FT1_MONITOR_STATUS_CTRL;
	cb.cblock.length = 1;
	cb.data[0] = status; 	
	DoCommand(&cb);
	
	if( cb.cblock.result != 0 && status){
		fail = 1;
		printf("This cblock.command is only possible with S508/FT1 board!");
	}
} /* set_FT1_monitor_status */


/* Subroutine for changing modes on a FT1 boards */

void set_FT1_mode( void )
{
	for(;;){ 
		cb.cblock.command = SET_FT1_MODE;
		cb.cblock.length = 0;
		DoCommand(&cb);
		if(cb.cblock.result == 0){
			break;
		}
	}
} /* set_FT1_mode */

/* Read the data for status of all the lights */

void read_FT1_status( void )
{
	int i;
	unsigned long delay;
	struct timeval tv;

	i = gettimeofday(&tv,NULL);
	delay = tv.tv_usec;

	for(;;){
		i = gettimeofday(&tv,NULL);
		if(abs((tv.tv_usec - delay)) > 90000 )
			break;  
	}

	cb.cblock.command = PPIPE_FT1_READ_STATUS;
	cb.cblock.length = 0;
	DoCommand(&cb); 

	if( cb.cblock.result == 0 ){
		par_port_A_byte = cb.data[0];
		par_port_B_byte = cb.data[1];
	}

} /* read_FT1_status */

/* Display the status of all the lights */

void view_FT1_status( void )
{
	off_counter = 0;
	red_counter = 0;
	green_counter = 0;
	loop_counter = 0;

	/* check for INS light */
	for(;;){
		read_FT1_status();
		if((par_port_B_byte & 0x30) == 0x30)
			off_counter++;
		if((par_port_B_byte & 0x10) == 0x00)
			red_counter++;
		if((par_port_B_byte & 0x20) == 0x00)
			green_counter++;

		/* Is the INS light red ? */
		if(red_counter != 0 && off_counter != 0 && loop_counter ==30) {
			printf("Unit is not IN SERVICE\n");
			break;
		}
		/* Is the INS light green ? */ 	
		if(green_counter != 0 && off_counter == 0 && loop_counter ==30){
			printf("Unit is IN SERVICE\n");
			break;
		}
	
		/* Is the INS light flashing green ? */
		if(green_counter != 0 && off_counter != 0 && loop_counter ==30){
			printf("INS is flashing green\n");
			break;
		}
	
		/* Is the INS light OFF ? */
		if(off_counter != 0 && red_counter == 0 && green_counter == 0 
							&& loop_counter == 30){
			printf("INS is off\n");
			break;
		}
	
		loop_counter++;
    	 }	
     
	/* check for ERR light */
	off_counter = 0;
	red_counter = 0;
	green_counter = 0;
	loop_counter = 0;

	for(;;){
		read_FT1_status();
		if((par_port_B_byte & 0x0C) == 0x0C)
			off_counter++;
		if((par_port_B_byte & 0x08) == 0x00)
			red_counter++;
		if((par_port_B_byte & 0x04) == 0x00)
			green_counter++;

		/* Is the ERR light off ? */
		if(off_counter != 0 && red_counter == 0 && green_counter == 0 
							&& loop_counter == 30){
			printf("No Line ERROR being received or Valid Line\n");
			break;
		}
		
		/* Is the ERR light red ? */
		if(red_counter != 0 && off_counter == 0 && green_counter == 0 
							&& loop_counter == 30){
			printf("Line ERROR being received or Invalid Line\n");
			break;
		}

		loop_counter++;
     	}

	/* check TXD light */
	loop_counter = 0;
	off_counter = 0;
	green_counter = 0;

	for(;;){
		read_FT1_status();
		if((par_port_B_byte & 0x02) == 0x02)
			off_counter++;
		if((par_port_B_byte & 0x02) == 0x00)
			green_counter++;
	
		/* Is the TXD light off ? */
		if(off_counter != 0 && green_counter == 0 && loop_counter ==20){
			printf("Transmit data is not present\n");	
			break;
		}
	
		/* Is the TXD light flashing green ? */
		if(off_counter != 0 && green_counter != 0 && loop_counter ==20){
			printf("Transmit data is present \n");
			break;
		}
		loop_counter++;
     	}

	/* check RXD light */
	loop_counter = 0;
	off_counter = 0;
	green_counter = 0;

	for(;;){
		read_FT1_status();
		if((par_port_B_byte & 0x01) == 0x01)
			off_counter++;
		if((par_port_B_byte & 0x01) == 0x00)
			green_counter++;

		/* Is the RXD light off ? */
		if(off_counter != 0 && green_counter == 0 && loop_counter ==20){
			printf("Receive data is not present\n");	
			break;
		}

		/* Is the RXD light flashing green ? */
		if(off_counter != 0 && green_counter != 0 && loop_counter ==20){
			printf("Receive data is present\n");
			break;
		}
		loop_counter++;
     	}

}/* view_FT1_status */


void FT1_operational_mode(void)
{
	printf("Operational Mode has been selected\n");
	printf("Putting S508/FT1 in operational mode....");
	loop_counter = 0;
	off_counter = 0;
	red_counter = 0;
	green_counter = 0;

	for(;;){
		read_FT1_status();
     		/* ST light is OFF */
     		if((par_port_B_byte & 0xc0) == 0xc0 ){
			off_counter++;
     		}
     	
		/* ST light is GREEN */
     		if((par_port_B_byte & 0x40) == 0x00){
			green_counter++;
			red_counter = 0;
     		}
     
		/* ST light is RED */
     		if((par_port_B_byte & 0x80) == 0x00){
			red_counter++;
			green_counter = 0;
   		}
		/* Is ST light off ? - then check next light */
		if(off_counter != 0 && red_counter == 0 && green_counter == 0 
							&& loop_counter == 20){
			break;
		}
	
		/* Is ST light red or green ? - then push button */
		if((red_counter != 0 || green_counter !=0) && loop_counter==20){
			set_FT1_mode();
			break;
		}

   		loop_counter++;
     	} /* end of for */
       
	loop_counter = 0;
	off_counter = 0;
	red_counter = 0;
	green_counter = 0;

	for(;;){
   		read_FT1_status();
  	 	/* DL light is OFF */
   		if((par_port_A_byte & 0x08) == 0x08){
			off_counter++;
		}
       		/* DL light is RED */
   		if((par_port_A_byte & 0x08) == 0x00){
			red_counter++;
		}

		/* Is DL light off ? - then check next light */
		if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
			break;
		}
	
		/* Is DL light red ? - then push button */
		if(red_counter != 0 && loop_counter == 20){
			set_FT1_mode();
			break;
		}
      		loop_counter++; 
     	}	 
     
	loop_counter = 0;
     	off_counter = 0;
     	red_counter = 0;
     	green_counter = 0;
     
	for(;;){
   		read_FT1_status();
   		/* LL light is off */
   		if((par_port_A_byte & 0x04) == 0x04){
			off_counter++;
		}
   		
		if((par_port_A_byte & 0x04) == 0x00){
			red_counter++;
		}
	
		/* Is LL light off ? - then check next light */
		if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
			break;
		}
	
		/* Is LL light red ? - then push button */
		if(red_counter != 0 && loop_counter == 20){
			set_FT1_mode();
			break;
		}
        
		loop_counter++;
     	}
     
	loop_counter = 0;
     	off_counter = 0;
     	red_counter = 0;
     	green_counter = 0;
     
	for(;;){
   		read_FT1_status();
		/* RT light is OFF */
   		if((par_port_A_byte & 0x03) == 0x03){
			off_counter++;
		}
		/* RT light is RED */
   		if((par_port_A_byte & 0x01) == 0x00){
			red_counter++;
		}
		/* RT light is GREEN */
   		if((par_port_A_byte & 0x02) == 0x00){
   			green_counter++;
		}

		/* Is RT light off ? - then it is in operational mode */ 
		if(off_counter != 0 && red_counter == 0 && green_counter == 0 
							&& loop_counter == 20){
			printf("Done!\n");
			break;
		}

		/* Is RT light flashing red or green ? - then push button */	
		if((red_counter != 0 || green_counter != 0) && off_counter != 0
 							&& loop_counter == 20){
			set_FT1_mode();
			printf("Done!\n");
			break;
		}

		/* Is RT light solid green ? - then can't push the button */
		if(off_counter == 0 && green_counter != 0 && loop_counter ==20){
			printf("Failed!\n");
			printf("Remote End is running Remote Test\n");
			printf("Exit Remote Test at remote end\n");
			break;
		}

		loop_counter++;
     	}        	

} /* FT1_operational_mode */


void FT1_self_test(void)
{
     	int started = 0; 
     	int selftest = 0;
 
     	set_FT1_mode();
     	off_counter = 0;
    	green_counter = 0;
     	red_counter = 0;
     	loop_counter = 0;
     	printf("Self Test has been selected\n");
     
	for(;;){
	  	read_FT1_status();
	  	/* ST light is OFF */
	  	if((par_port_B_byte & 0xc0) == 0xc0){
			off_counter++;
	  	}
	  
		/* ST light is GREEN */
	  	if((par_port_B_byte & 0x40) == 0x00){
			green_counter++;
	  	}
	  
		/* ST light is RED */
	  	if((par_port_B_byte & 0x80) == 0x00){
			red_counter++;
	  	}
         
		/* Is ST light flashing red ? - if not then push button */
		if(red_counter == 0 && loop_counter == 3){
			set_FT1_mode();    
			off_counter = 0;
			red_counter = 0;
			green_counter = 0;
			loop_counter = -1;
     			printf("Selecting Self Test....\r");
			selftest++;
	  		if( selftest == 10){
				  printf("\t\t\tFailed!\n");
				  printf("Self Test will not operate while connected to a valid line\n");
		 		 FT1_operational_mode();
				  break;
	      		}
	  	}
	 
		/* Is ST light flashing red ? */ 
		if(red_counter != 0 && off_counter != 0 && (loop_counter % 2)){
			printf("Performing Self Test....\r");
	        	started = 1;
	  	}
	  
		/* Is ST light flashing red ? */
		if(green_counter != 0 && off_counter != 0 && started){
	   		printf("\t\t\tDone!\n");
			FT1_operational_mode();
			break;
	  	}
          
		loop_counter++;	  
     	}/* end of for */     
} /* FT1_self_test */


void FT1_dl_test( void )
{
	int dl_test=0;

     	set_FT1_mode();
     	off_counter = 0;
     	red_counter = 0;
     	loop_counter = 0;
     
	printf("Bi-directional Digital Loop has been selected\n"); 
     
	for(;;){
		read_FT1_status();
		if((par_port_A_byte & 0x08) == 0x08){
			off_counter++;  
		}
	
		if((par_port_A_byte & 0x08) == 0x00){
			red_counter++;
		}

		/* Is DL light flashing red ? - if not then push button */	
		if(red_counter == 0 && loop_counter == 20){
			set_FT1_mode();
			off_counter = 0;
			red_counter = 0;
			loop_counter = -1;
			printf("Selecting Digital Loop Test....\r");
			dl_test++;
		
			if(dl_test==10){
	 			printf("\t\t\t\tFailed\n");
				printf("Remote End might be running Remote Test\n");
				break;
			}
		}
	
		/* Is DL light flashing red ? -if yes then DL test in progress*/
		if(red_counter != 0){
			off_counter = 0;
			red_counter = 0;
			green_counter = 0;
			loop_counter = 0;
			printf("Performing Digital Loop Test....\r");
			
			for(;;){
				read_FT1_status();
				printf("Performing Digital Loop Test....\r");
				/* check INS light */
				if((par_port_B_byte & 0x30) == 0x30)
					off_counter++;
				if((par_port_B_byte & 0x10) == 0x00){
					red_counter++;
					green_counter = 0;
				}
				if((par_port_B_byte & 0x20) == 0x00){
					green_counter++;
					red_counter = 0;
				}

				/* Is INS light red ? - if yes then DL test 
				   failed */
				if(red_counter != 0 && (par_port_B_byte & 0x08) 
								== 0x00 ){
					printf("\t\t\t\tFailed!\n");
					printf("Bi-directional Digital Loop test has failed\n");
					printf("Either the unit is not connected or the line is Invalid\n");
					break;
				}
			
				/* Is INS light green ? - if yes then DL test 
				   passed */
				if(green_counter != 0  && (par_port_B_byte & 
					0x0C) == 0x0C && loop_counter == 100 ){
					printf("\t\t\t\tDone!\n");
					printf("Bi-directional Digital Loop test has been successfully\n");
					printf("completed\n");
					break;
		
				}
			
				loop_counter++;
			} /* end of for */
			break;	
		} 
		loop_counter++;
     	} /* end of for */

} /* FT1_dl_test */

void FT1_ll_test( void )
{
    	int ll_test = 0;

    	set_FT1_mode();	
    	off_counter = 0;
    	red_counter = 0;
    	loop_counter = 0;
    
	printf("Line Loop Test has been selected\n");
    
	for(;;){
		read_FT1_status();
	
		if((par_port_A_byte & 0x04) == 0x04){
			off_counter++;
		}
		if((par_port_A_byte & 0x04) == 0x00){
			red_counter++;
		}

		/* Is LL light not flashing red ? - if not then push button */
		if(red_counter == 0 && off_counter != 0 && loop_counter == 20){
			set_FT1_mode();
			off_counter = 0;
			red_counter = 0;
			loop_counter = -1;
			printf("Selecting Line Loop Test....\r");
			ll_test++;
			if(ll_test == 10){
				printf("\t\t\t\tFailed!\n");
				printf("Line Loop Test will not operate while connected to a valid line\n");
				FT1_operational_mode();
		 		break;	
			}
		}
	
		/* Is LL light flashing red ? - if yes then it is selected */
		if(red_counter != 0){	
			off_counter = 0;
			red_counter = 0;
			loop_counter = 0;
		
			for(;;){
				printf("Performing Line Loop Test....\r");
				read_FT1_status();	
			
				/* check INS light */
				if((par_port_B_byte & 0x30) == 0x30)
					off_counter++;
			
				if((par_port_B_byte & 0x10) == 0x00){
					red_counter++;
					green_counter = 0;
				}
			
				if((par_port_B_byte & 0x20) == 0x00){
					green_counter++;
					red_counter = 0;
				}
			
				/* Is INS light green ? - if yes then test 
				   passed */
				if(green_counter != 0 && red_counter == 0){
					printf("\t\t\t\tDone!\n");
					printf("Line Loop Test has been successfully completed\n");
					break;
				}
				
				/* Is INS light red ? - if yes then test 
				   failed  */
				if(red_counter != 0 && green_counter == 0 && loop_counter == 100){
					printf("\t\t\t\tFailed!\n");
					break;
				}
			
				loop_counter++;
			} /* end of for */
			break;
		} /* end of if */
		loop_counter++;
    	} /* end of for */

} /* FT1_ll_test */

void FT1_rt_test( void )
{
    	off_counter = 0;
   	red_counter = 0;
    	green_counter = 0;	  	
    	loop_counter = 0;
    	printf("Remote Test has been selected\n");
    	
	for(;;){
		read_FT1_status();
		if((par_port_A_byte & 0x03) == 0x03)
			off_counter++;
		if((par_port_A_byte & 0x01) == 0x00)
			red_counter++;
		if((par_port_A_byte & 0x02) == 0x00)
			green_counter++;

		/* Is RT light red ? - if not then push button */
		if(red_counter == 0 && loop_counter == 20){
			set_FT1_mode();
			off_counter = 0;
			red_counter = 0;
			green_counter = 0;
			loop_counter = -1;
			printf("Selecting Remote Test....\r");		
		}

		/* Is RT light flashing green ? - if yes then RT test is 
	 	   selected */
		if(green_counter != 0 && loop_counter == 10){
			loop_counter = 0;
	   		off_counter = 0;
	   
			for(;;) {
				read_FT1_status();
				if((par_port_A_byte & 0x03) == 0x03)
					off_counter++;

				/* RT test was already started at the remote
				   end */
	        		if(off_counter == 0 && loop_counter == 20){
		   			printf("Remote End is currently sending Remote Test data\n");
		   			printf("Exit from the Remote Test mode at the remote end\n");
		   			break;
				}
				/* RT test has started */
				if(off_counter != 0 && loop_counter == 20) {
		   			printf("This unit is currently in Remote Testing Mode\n");
		   			break;
				}
	        		loop_counter++;
	  		}
	   		break;
		}
	
		if(red_counter != 0){
			printf("Waiting for a valid response from remote end....\r");
			loop_counter = 0;
			for(;;){
				read_FT1_status();
				if((par_port_A_byte & 0x02) == 0x00){
					printf("\t\t\t\t\t\tDone!\n");
					printf("Valid response has been received from remote end\n");
					printf("Remote Test has been successfully completed\n");
					break;	
				}
				if((par_port_B_byte & 0x08) == 0x00){
					printf("\t\t\t\t\t\tFailed!\n");
					printf("The test can only be successful between two Sangoma\n");
					printf("S508/FT1 units configured the SAME WAY\n");

					break;
				}
			} /* end of for */
			break;
		}
    		loop_counter++;	
    	} /* end of for */	
} /* FT1_rt_test */

void ppp_driver_stat_ifsend( void )
{
	if_send_stat_t *if_stats;
	cb.cblock.command = PPIPE_DRIVER_STAT_IFSEND;
	cb.cblock.length = 0;
	cb.data[0] = 0;
	DoCommand(&cb);
      	if_stats = (if_send_stat_t *)&cb.data[0];


      	printf("                                 Total number of If_Send entries:  %ld\n", 
  		if_stats->if_send_entry);
      	printf("                       Number of If_Send entries with SKB = NULL:  %ld\n",
 		if_stats->if_send_skb_null);
      	printf("Number of If_Send entries with broadcast addressed pkt discarded:  %ld\n",
		if_stats->if_send_broadcast);
      	printf("Number of If_Send entries with multicast addressed pkt discarded:  %ld\n",
		if_stats->if_send_multicast);
      	printf("             Number of If_Send entries with CRITICAL_RX_INTR set:  %ld\n",
		if_stats->if_send_critical_ISR);
      	printf("   Number of If_Send entries with Critical set and pkt discarded:  %ld\n",
		if_stats->if_send_critical_non_ISR);
      	printf("                  Number of If_Send entries with Device Busy set:  %ld\n",
		if_stats->if_send_tbusy);
      	printf("              Number of If_Send entries with Device Busy Timeout:  %ld\n",
		if_stats->if_send_tbusy_timeout);
      	printf("           Number of If_Send entries with PTPIPE MONITOR Request:  %ld\n",
		if_stats->if_send_PIPE_request);
      	printf("                 Number of If_Send entries with WAN Disconnected:  %ld\n",
		if_stats->if_send_wan_disconnected);
      	printf("                   Number of If_Send entries with Protocol Error:  %ld\n",
		if_stats->if_send_protocol_error);
      	printf("       Number of If_Send entries with Transmit Interrupt Enabled:  %ld\n",
		if_stats->if_send_tx_int_enabled);
	printf("              Number of If_Send entries with Adapter Send Failed:  %ld\n",
		if_stats->if_send_bfr_not_passed_to_adptr);
      	printf("              Number of If_Send entries with Adapter Send passed:  %ld\n",
		if_stats->if_send_bfr_passed_to_adptr);
// 	printf("               Number of times host irq left disabled in If_Send:  %d\n", 
//		if_stats. ); 

} /* ppp_driver_stat_ifsend */

void ppp_driver_stat_intr( void )
{
	rx_intr_stat_t *rx_stat;
	global_stats_t *g_stat;
      	cb.cblock.command = PPIPE_DRIVER_STAT_INTR;
     	cb.cblock.length = 0;
      	cb.data[0] = 0;
      	DoCommand(&cb);
	g_stat = (global_stats_t *)&cb.data[0];
	rx_stat = (rx_intr_stat_t *)&cb.data[sizeof(global_stats_t)];
	
      
      	printf("                                    Number of ISR entries:   %ld\n", 
		g_stat->isr_entry );
      	printf("                  Number of ISR entries with Critical Set:   %ld\n", 
		g_stat->isr_already_critical );
      	printf("                              Number of Receive Interrupt:   %ld\n", 
		g_stat->isr_rx );
      	printf("                             Number of Transmit Interrupt:   %ld\n", 
		g_stat->isr_tx );
      	printf("              Number of ISR entries for Interrupt Testing:   %ld\n", 
		g_stat->isr_intr_test );
      	printf("                             Number of Spurious Interrupt:   %ld\n", 
		g_stat->isr_spurious );
      	printf("       Number of times Transmit Interrupts Enabled in ISR:   %ld\n", 
		g_stat->isr_enable_tx_int );
      	printf("         Number of Receive Interrupts with Corrupt Buffer:   %ld\n", 
		g_stat->rx_intr_corrupt_rx_bfr );
      	printf("              Number of Receive Interrupts with No socket:   %ld\n", 
		rx_stat->rx_intr_no_socket );
      	printf("  Number of Receive Interrupts for PTPIPE MONITOR Request:   %ld\n",
		rx_stat->rx_intr_PIPE_request);
      	printf(" Number of Receive Interrupts with Buffer Passed to Stack:   %ld\n",
		rx_stat->rx_intr_bfr_passed_to_stack);
        printf(" Number of Receive Inter. with Buffer NOT Passed to Stack:   %ld\n",
		rx_stat->rx_intr_bfr_not_passed_to_stack);
      	printf("     Number of Receive Interrupts with Device not started:   %ld\n", 
		g_stat->rx_intr_dev_not_started );
      	printf("    Number of Transmit Interrupts with Device not started:   %ld\n", 
		g_stat->tx_intr_dev_not_started );
     
}


void ppp_driver_stat_gen( void )
{
	pipe_mgmt_stat_t *p_stat;
	global_stats_t *g_stat;
      	cb.cblock.command = PPIPE_DRIVER_STAT_GEN;
      	cb.cblock.length = 0;
      	cb.data[0] = 0;
      	DoCommand(&cb);
	p_stat = (pipe_mgmt_stat_t *)&cb.data[0];
	g_stat = (global_stats_t *)&cb.data[sizeof(pipe_mgmt_stat_t)];
     
      	printf("           Number of PTPIPE Monitor call with Kmalloc error:  %ld\n",
		p_stat->UDP_PIPE_mgmt_kmalloc_err );
      	printf("      Number of PTPIPE Monitor call with Adapter Type error:  %ld\n",
		p_stat->UDP_PIPE_mgmt_adptr_type_err );
      	printf("         Number of PTPIPE Monitor call with Direction Error:  %ld\n",
		p_stat->UDP_PIPE_mgmt_direction_err );
      	printf(" Number of PTPIPE Monitor call with Adapter Command Timeout:  %ld\n",
		p_stat->UDP_PIPE_mgmt_adptr_cmnd_timeout );
      	printf("      Number of PTPIPE Monitor call with Adapter Command OK:  %ld\n",
		p_stat->UDP_PIPE_mgmt_adptr_cmnd_OK );
      	printf("  Number of PTPIPE Monitor call with pkt passed to Adapter:   %ld\n",
		p_stat->UDP_PIPE_mgmt_passed_to_adptr );
      	printf("    Number of PTPIPE Monitor call with pkt passed to Stack:   %ld\n",
		p_stat->UDP_PIPE_mgmt_passed_to_stack);
      	printf("              Number of PTPIPE Monitor call with no socket:   %ld\n",
		p_stat->UDP_PIPE_mgmt_no_socket);
      	printf("                                    Number of Poll Entries:   %ld\n",
		g_stat->poll_entry);
      	printf("                  Number of Poll Entries with Critical set:   %ld\n",
		g_stat->poll_already_critical);
      	printf("                          Number of Poll Entries Processed:   %ld\n",
		g_stat->poll_processed);
      	printf("            Number of times host irq left disabled in Poll:   %ld\n", 
		g_stat->poll_host_disable_irq);

} /* ppp_driver_stat_ifsend */

void flush_driver_stats( void )
{
      	cb.cblock.command = PPIPE_FLUSH_DRIVER_STATS;
      	cb.cblock.length = 0;
      	cb.data[0] = 0;
      	DoCommand(&cb);

      	printf("All Driver Statistics are Flushed.\n");

}

void ppp_router_up_time( void )
{
     	unsigned long time;
     	cb.cblock.command = PPIPE_ROUTER_UP_TIME;
     	cb.cblock.length = 0;
     	cb.data[0] = 0;
     	DoCommand(&cb);
     
     	time = *(unsigned long*)&cb.data[0];

     	if (time < 3600) {
		if (time<60) 
     			printf("    Router UP Time:  %ld seconds\n", time);
		else
     			printf("    Router UP Time:  %ld minute(s)\n", (time/60));
     	}else
     		printf("    Router UP Time:  %ld hour(s)\n", (time/3600));
			
      
}

void banner (char *title){
	
	int len,i;
	
	len = strlen(title);
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	printf("\n\t\t%s",title);
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	printf("\n\n");
	

}

int main(int argc, char* argv[])
{
	int proceed;
	char command;
	char *opt;	

   	printf("\n");
   	if( argc > 2 ) {
     
		init( argc, argv);
     		proceed = MakeConnection();
		command = cmd[0];
		opt   = (char *) &cmd[1];
     		
		if(proceed == TRUE){
      			switch(command) {
				
      				case 'x':
					if (!strcmp(opt,"m")){
						modem();
					}else if (!strcmp(opt,"u")){
						timers();	
					}else if (!strcmp(opt,"n")){
						negot();
					}else if (!strcmp(opt,"c")){
						cause();
					}else if (!strcmp(opt,"s")){
						state();
					}else if (!strcmp(opt,"ru")){
	 					ppp_router_up_time();
					}else{
						printf("ERROR: Invalid Status Command 'x', Type ppipemon <cr> for help\n\n");
					}
	 				break;


				case 'c':
					if (!strcmp(opt,"i")){
						ip_config();	
					}else if (!strcmp(opt,"x")){
						ipx_config();
					}else if (!strcmp(opt,"a")){
						authent();
					}else if (!strcmp(opt,"g")){
						general_conf();	
					}else{
						printf("ERROR: Invalid Configuration Command 'c', Type ppipemon <cr> for help\n\n");		
					}
					break;

      				case 's':
					if (!strcmp(opt,"g")){
	 					general_stats();
					}else if (!strcmp(opt,"c")){
						comm_err();
					}else if (!strcmp(opt,"p")){
						packet();
					}else if (!strcmp(opt,"lo")){
						loopback();
					}else if (!strcmp(opt,"pap")){
						pap();
					}else if (!strcmp(opt,"chp")){
						chap();			
					}else if (!strcmp(opt,"ipc")){
						ipcp();
					}else if (!strcmp(opt,"xpc")){
						ipxcp();			
					}else if (!strcmp(opt,"lpc")){
						lcp();
					}else{
						printf("ERROR: Invalid Statistics Command 's', Type ppipemon <cr> for help\n\n");
					}
	 				break;

      				case 't':
	 				if(!strcmp(opt,"i" )){
						line_trace(TRACE_ALL);
					}else if (!strcmp(opt, "ip")){
						line_trace(TRACE_PROT);
					}else if (!strcmp(opt, "id")){
						line_trace(TRACE_DATA);
					}else if (!strcmp(opt, "r")){
	   					raw_data = TRUE;
						line_trace(TRACE_ALL);
					}else if (!strcmp(opt, "rp")){
						raw_data = TRUE;
						line_trace(TRACE_PROT);
					}else if (!strcmp(opt, "rd")){
						raw_data = TRUE;
						line_trace(TRACE_DATA);
					}else{
						printf("ERROR: Invalid Trace Command 't', Type ppipemon <cr> for help\n\n");
					}
	 				break;
      				case 'd':
	 				if(!strcmp(opt,"s" )){
	 					ppp_driver_stat_ifsend();
	 				}else if (!strcmp(opt,"i" )){ 
						ppp_driver_stat_intr();	 
	 				}else if (!strcmp(opt,"g" )){ 
						ppp_driver_stat_gen();	 
					}else{
						printf("ERROR: Invalid Driver Stat Command 't', Type ppipemon <cr> for help\n\n");
					}
					break;
      				case 'f':
	 				if(!strcmp(opt,"g" )){	
	       					flush_general_stats();
	      	 				general_stats();
					}else if (!strcmp(opt,"c" )){
	       					flush_comm_err();
	       					comm_err();
					}else if (!strcmp(opt,"p" )){
	       					flush_packet();
	       					packet();
					}else if (!strcmp(opt,"l" )){
						flush_lcp();
	       					lcp();
					}else if (!strcmp(opt,"lo" )){
	       					flush_loopback();
	       					loopback();
					}else if (!strcmp(opt,"icp" )){
	       					flush_ipcp();
	       					ipcp();
					}else if (!strcmp(opt,"ixp" )){
	       					flush_ipxcp();
	       					ipxcp();
					}else if (!strcmp(opt,"pap" )){
	       					flush_pap();
	       					pap();
					}else if (!strcmp(opt,"chp" )){
	       					flush_chap();
	       					chap();
					}else if (!strcmp(opt,"d" )){
	       					flush_driver_stats();
	 				} else {
						printf("ERROR: Invalid Flush Command 'f', Type ppipemon <cr> for help\n\n");
	 				}
	 				break;
      				case 'T':
					if (!strcmp(opt,"v" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							view_FT1_status();
	    					 }
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"s" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){	 	
							FT1_self_test();
	    	 				}
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"l" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
	    						FT1_ll_test();
             					}
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"d" )){
             					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_dl_test();
	     					}
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"r" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_rt_test();
	     					}
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"o" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_operational_mode();
	    		 			}
						set_FT1_monitor_status(0x00);
					} else{
						printf("ERROR: Invalid FT1 Command 'T', Type ppipemon <cr> for help\n\n");
					} 
        				break; 
      				default:
					printf("ERROR: Invalid Command, Type ppipemon <cr> for help\n\n");

      			} //switch
     		} 
     		close( sock );
   	} else {
      		usage();
   	} //if
   printf("\n");

   return 0;
}; //main

/*
 * EOF ppipemon.c
 */
