/*
 * (c) Copyright 1992 by Panagiotis Tsirigotis
 * All rights reserved.  The file named COPYRIGHT specifies the terms 
 * and conditions for redistribution.
 */

static char RCSid[] = "$Id: access.c,v 1.3 1999/10/13 02:51:03 bbraun Exp $" ;

#include <syslog.h>
#include <time.h>
#include <signal.h>

#ifdef LIBWRAP
#include <tcpd.h>
int deny_severity = LOG_INFO;
int allow_severity = LOG_INFO;
#endif

#include "connection.h"
#include "service.h"
#include "server.h"
#include "state.h"
#include "addr.h"
#include "access.h"

char *inet_ntoa() ;
time_t time() ;

void msg() ;

#ifdef HAVE_LOADAVG
extern double xgetloadavg() ;
#endif

struct name_value access_code_names[] =
	{
		{ "address",				(int) AC_ADDRESS			},
		{ "time",					(int) AC_TIME				},
		{ "fork",					(int) AC_FORK				},
		{ "service_limit",		(int) AC_SERVICE_LIMIT	},
		{ "per_source_limit",	(int) AC_PER_SOURCE_LIMIT	},
		{ "process_limit",		(int) AC_PROCESS_LIMIT	},
		{ "libwrap",            (int) AC_LIBWRAP        },
      { "load",               (int) AC_LOAD           },
      { "connections per second", (int) AC_CPS        },
		{ CHAR_NULL,				1								},
		{ "UNKNOWN",				0								}
	} ;

/* Signal handler for re-enabling a service that has been disabled. */
PRIVATE void alrm_ena()
{
	int i;
	time_t nowtime;

	nowtime = time(NULL);
	for( i=0; i < pset_count( SERVICES(ps) ); i++ ) {
		struct service *sp;
		struct service_config *scp;

		sp = pset_pointer( SERVICES(ps), i);
		scp = SVC_CONF( sp );

		if( sp->svc_state == SVC_DISABLED ) {
			if( scp->sc_time_reenable <= nowtime ) {
				/* re-enable the service */
				if( svc_activate(sp) == OK ) {
					syslog(LOG_ERR, "Activating service %s\n", scp->sc_name);
				} else {
					syslog(LOG_ERR, "Error activating service %s\n", scp->sc_name);
				} /* else */
			}
		}
	} /* for */
} /* alrm_ena */

/*
 * Returns OK if the IP address in sinp is acceptable to the access control
 * lists of the specified service.
 */
PRIVATE status_e remote_address_check( sp, sinp )
	register struct service		*sp ;
#ifdef INET6
	struct sockaddr_in6 		*sinp ;
#else
	struct sockaddr_in 			*sinp ;
#endif
{
	/*
	 * of means only_from, na means no_access
	 */
#ifdef INET6
	struct in6_addr         of_match ;
	struct in6_addr         na_match ;
	register bool_int		of_matched  = 0;
	register bool_int		na_matched  = 0;

	struct in6_addr			*addr ;

	bzero(&of_match, sizeof(of_match));
	bzero(&na_match, sizeof(na_match));
	addr = &sinp->sin6_addr ;
#else
	unsigned long			of_match ;
	unsigned long			na_match ;
	register bool_int		of_matched ;
	register bool_int		na_matched ;
	struct in_addr			*addr ;

	addr = &sinp->sin_addr ;
#endif

	if ( sp->svc_no_access != NULL )
		na_matched = addrlist_match( sp->svc_no_access, addr, &na_match ) ;
	else if ( na_matched == FERROR )
		/* There was an error matching the addresses */
		return FAILED;
	else
		na_matched = FALSE ;

	if ( sp->svc_only_from != NULL )
		of_matched = addrlist_match( sp->svc_only_from, addr, &of_match ) ;
	else if ( of_matched == FERROR )
		return FAILED;
	else
		of_matched = FALSE ;

	/*
	 * Check if the specified address is in both lists
	 */
	if ( na_matched && of_matched )
	{
		/*
		 * The greater match wins.
		 * If the matches are equal, this is an error in the service entry
		 * and we cannot allow a server to start.
		 * We do not disable the service entry (not our job).
		 */
#ifdef INET6
		int ret = memcmp(&na_match, &of_match, sizeof(na_match));
	
		if( ret == 0 )
			msg( LOG_ERR, "remote_address_check",
"Service=%s: only_from list and no_access list match equally the address %s",
				SVC_ID( sp ), xntoa( *sinp ) ) ;
		else if ( ret < 0 )
			return OK;
		else
			return FAILED;
#else
		if ( na_match == of_match )
			msg( LOG_ERR, "remote_address_check",
"Service=%s: only_from list and no_access list match equally the address %s",
				SVC_ID( sp ), xntoa( *sinp ) ) ;
		return( ( of_match > na_match ) ? OK : FAILED ) ;
#endif
	}

	if ( sp->svc_no_access != NULL && na_matched )
		return( FAILED ) ;
	if ( sp->svc_only_from != NULL && ! of_matched ) {
		return( FAILED ) ;
	}

	/*
	 * If no lists were specified, the default is to allow starting a server
	 */
	return( OK ) ;
}


/*
 * mp is the mask pointer, t is the check type
 */
#define CHECK( mp, t )		( ( (mp) == NULL ) || M_IS_SET( *(mp), t ) )

/*
 * Perform the access controls specified by check_mask.
 * If check_mask is NULL, perform all access controls
 */
access_e access_control( sp, cp, check_mask )
	register struct service		*sp ;
	register connection_s		*cp ;
	register mask_t				*check_mask ;
{
	register struct service_config	*scp = SVC_CONF( sp ) ;
#ifdef INET6
	struct sockaddr_in6					*sinp = SOCKADDRIN_NULL;
#else
	struct sockaddr_in					*sinp = SOCKADDRIN_NULL;
#endif
	int u, n;
	time_t nowtime;

#ifdef LIBWRAP
   struct request_info req;

   request_init(&req, RQ_DAEMON, scp->sc_name, RQ_FILE, cp->co_descriptor, 0);
   fromhost(&req);
   if (!hosts_access(&req))
   {
		syslog(deny_severity, "refused connect from %s", conn_addrstr(cp));
      return(AC_LIBWRAP);
   }
#endif
	/* If cps is specified, try to detect someone bombarding us, and 
	 * just disable the service.
	 */
	if( scp->sc_time_conn_max != 0 ) {
		nowtime = time(NULL);
		if( ((scp->sc_time_limit - nowtime) > 2) || ( (scp->sc_time_limit-nowtime) < 0)) {
			scp->sc_time_limit = nowtime;
			scp->sc_time_conn = 1;
		} else {
			scp->sc_time_conn ++;
			if( scp->sc_time_conn > scp->sc_time_conn_max ) {
				svc_deactivate(sp);
				/* Need to implement a restart timer */
				syslog(LOG_ERR, "Deactivating service %s due to excessive incoming connections.  Restarting in %d seconds.\n", scp->sc_name, scp->sc_time_wait);
				if( signal( SIGALRM, alrm_ena) == SIG_ERR ) {
					syslog(LOG_ERR, "Error setting up timer for reactivating service %s\n", scp->sc_name);
					return(AC_CPS);
				}
				nowtime = time(NULL);
				scp->sc_time_reenable = nowtime + scp->sc_time_wait;
				alarm(scp->sc_time_wait);
				return(AC_CPS);
			}
		}
	}


#ifdef HAVE_LOADAVG
	if ( scp->sc_max_load != 0 ) {
		if ( xgetloadavg() >= scp->sc_max_load ) {
			syslog(LOG_ERR, "refused connect from %s due to excessive load\n",
				conn_addrstr(cp));
			return( AC_LOAD );
		}
	}
#endif

	if ( CHECK( check_mask, CF_ADDRESS ) && 
				( sinp = conn_address( cp ) ) != SOCKADDRIN_NULL &&
				remote_address_check( sp, sinp ) == FAILED )
		return( AC_ADDRESS ) ;

	if ( CHECK( check_mask, CF_TIME ) &&
			SC_ACCESS_TIMES( scp ) != NULL && 
				! ti_current_time_check( SC_ACCESS_TIMES( scp ) ) )
		return( AC_TIME ) ;
	
	if ( CHECK( check_mask, CF_SERVICE_LIMIT ) &&
								SVC_RUNNING_SERVERS( sp ) >= SC_INSTANCES( scp ) )
		return( AC_SERVICE_LIMIT ) ;

	if( scp->sc_per_source != UNLIMITED )
	{
		if ( sinp == SOCKADDRIN_NULL ) sinp = conn_address ( cp ) ;
		if ( sinp != SOCKADDRIN_NULL ) {
			n = 0 ;
			for ( u = 0 ; u < pset_count( SERVERS( ps ) ) ; u++ )
			{
				struct server *serp ;
				connection_s *cop ;
				
				serp = SERP( pset_pointer( SERVERS( ps ), u ) ) ;

				if ( (SERVER_SERVICE( serp ) == sp) &&
					( cop = SERVER_CONNECTION( serp ) ) )
				{
#ifdef INET6
					if ( IN6_ARE_ADDR_EQUAL( &(cop->co_remote_address.sin6_addr), &(sinp->sin6_addr) ) ) 
#else
					if ( cop->co_remote_address.sin_addr.s_addr ==
						sinp->sin_addr.s_addr)
#endif
							n++;
				}
			}

			if ( n >= scp->sc_per_source )
				return( AC_PER_SOURCE_LIMIT ) ;
		}
	}
	
	if ( CHECK( check_mask, CF_PROCESS_LIMIT ) && ps.ros.process_limit )
	{
		unsigned processes_to_create = SC_IS_INTERCEPTED( scp ) ? 2 : 1 ;

		if ( pset_count( SERVERS( ps ) ) + processes_to_create >
																ps.ros.process_limit )
		return( AC_PROCESS_LIMIT ) ;
	}
	return( AC_OK ) ;
}

