/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/* TODO:
 * [ ] ability to choose logging options (downloaded files/connections/group
 *     updates)
 * [ ] ability to limit max log size
 * [ ] colored log entries
 * [ ] log viewer editing options (remove selected...etc)
 * [ ] View->Status Bar toggle
 */

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <time.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gnome-stock.h>

#include "gui.h"
#include "log.h"
#include "util.h"

typedef struct
{
	/* public */
	GtkWidget *window;  /* main window */
	GtkWidget *log; /* GtkCList of log contents */
	GtkWidget *appbar;
	GtkWidget *file_entry;

	/* private */
} LogViewer;

static LogViewer *log_viewer;


static void log_save_cb (GtkWidget *widget, LogViewer *lv);
static void log_save_ok_clicked (GtkWidget *widget, LogViewer *lv);
static void log_setup_appbar (void);
static void log_hide_cb (GtkWidget *widget, LogViewer *lv);
static void log_clear_cb (GtkWidget *widget, LogViewer *lv);


static GnomeUIInfo log_file_menu[] = {
//	GNOMEUIINFO_MENU_OPEN_ITEM (not_implemented, NULL),
//	GNOMEUIINFO_MENU_SAVE_ITEM (log_save_cb, NULL),
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (log_save_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_PRINT_ITEM (not_implemented, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM (log_hide_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo log_view_menu[] = {
	GNOMEUIINFO_END
};

static GnomeUIInfo log_main_menu[] = {
	GNOMEUIINFO_MENU_FILE_TREE (log_file_menu),
	GNOMEUIINFO_SUBTREE(N_("_View"), &log_view_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo log_toolbar[] = {
	GNOMEUIINFO_ITEM_STOCK (N_("Save Log"), N_("Save this log to a file."), 
				log_save_cb, GNOME_STOCK_PIXMAP_SAVE),
	GNOMEUIINFO_ITEM_STOCK (N_("Clear Log"), N_("Clear this log."),
				log_clear_cb, GNOME_STOCK_PIXMAP_CLEAR),
	GNOMEUIINFO_ITEM_STOCK (N_("Close"), N_("Close the Log Viewer."),
				log_hide_cb, GNOME_STOCK_PIXMAP_CLOSE),
	GNOMEUIINFO_END

};


/*---[ log_add ]------------------------------------------------------
 * @msg:
 *
 * add a message to the list of logged events
 *--------------------------------------------------------------------*/
void
log_add (const gchar *msg)
{
	gchar *data[2];
	const time_t cur_time = time(NULL);

	/* get our text columns */
	data[0] = g_strdup_printf ("%s", ctime(&cur_time));
	*strrchr(data[0],'\n') = '\0'; /* chop the trailing linefeed */
	data[1] = g_strdup_printf (msg);

	/* update the log */
	pan_lock();
	gtk_clist_append (GTK_CLIST(log_viewer->log), data);
	pan_unlock();

	/* cleanup */
	g_free (data[0]);
	g_free (data[1]);
}

/*---[ log_add_va ]---------------------------------------------------
 * @format:
 *
 * add a message to the log file with a variable argument list
 *--------------------------------------------------------------------*/
void
log_add_va (const gchar *format, ...)
{
	va_list args;
	gchar *msg;

	g_return_if_fail (format != NULL);

	va_start (args, format);
	msg = g_strdup_vprintf (format, args);
	va_end (args);

	log_add (msg);
	g_free (msg);
}

/* Clear out the log event list */
static void
log_clear_cb (GtkWidget *widget, LogViewer *lv)
{
	GtkCList *list = GTK_CLIST(lv->log);

	pan_lock();
	gtk_clist_freeze (list);
	gtk_clist_clear (list);
	gtk_clist_thaw (list);
	pan_unlock();
}

/* This callback hides the log window... */
static void
log_hide_cb (GtkWidget *widget, LogViewer *lv)
{
	gtk_widget_hide (log_viewer->window);
}

/*---[ log_init ]-----------------------------------------------------
 * Initialization stuff, called from pan_init()
 *--------------------------------------------------------------------*/
void
log_init (void)
{
	gchar *titles[2] = {
		_("Time"),
		_("Log Entry"),
	};

	log_viewer = (LogViewer *) g_malloc0 (sizeof(LogViewer));

	log_viewer->log = gtk_clist_new_with_titles (2, titles);
	gtk_clist_set_column_auto_resize (GTK_CLIST(log_viewer->log), 0, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(log_viewer->log), 1, TRUE);

}

/* Bring up the Save file selection dialog */
static void
log_save_cb (GtkWidget *widget, LogViewer *lv)
{
	lv->file_entry = gtk_file_selection_new (_("Save the log to file"));

	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (lv->file_entry));

	gtk_signal_connect_object (GTK_OBJECT (lv->file_entry), "destroy",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (lv->file_entry));
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (lv->file_entry)->cancel_button),
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (lv->file_entry));
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (lv->file_entry)->ok_button),
			    "clicked",
			    GTK_SIGNAL_FUNC (log_save_ok_clicked),
			    lv);

	gtk_widget_show_all (lv->file_entry);
}

static void
log_save_ok_clicked (GtkWidget *widget, LogViewer *lv)
{
	gchar *filename;
	FILE *log_dump;
	gint tot_rows, i;
	gchar *data[] = { NULL, NULL };
	gchar *log_text;

	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (lv->file_entry));

/*	g_print ("%s\n", filename); */

	gtk_clist_freeze (GTK_CLIST (lv->log));
	tot_rows = gtk_clist_append (GTK_CLIST (lv->log), data);
	gtk_clist_remove (GTK_CLIST (lv->log), tot_rows);
	gtk_clist_thaw (GTK_CLIST (lv->log));

	log_text = g_malloc0 (256);

	if ((log_dump = fopen (filename, "w")) == NULL) {
		g_print ("fopen() failed to save log");
		g_free (filename);
		return;
	}
	g_free (filename);

	for (i = 0; i < tot_rows; i++) {
		gtk_clist_get_text (GTK_CLIST(lv->log), i, 1, &log_text);
		fprintf (log_dump, "%s ", log_text);

		gtk_clist_get_text (GTK_CLIST(lv->log), i, 0, &log_text);
		fprintf (log_dump, "%s", log_text);
	}

	fclose (log_dump);
	g_free (log_text);

	gtk_widget_destroy (GTK_WIDGET (lv->file_entry));
}

static void
log_setup_appbar (void)
{
	log_viewer->appbar = gnome_appbar_new (FALSE, TRUE, GNOME_PREFERENCES_USER);
	gnome_app_set_statusbar (GNOME_APP(log_viewer->window), GTK_WIDGET (log_viewer->appbar));
}

/*---[ log_viewer_spawn ]---------------------------------------------
 * pop open a log viewer, use a generic scrolling window with a 
 * text box inside to show the log of past events
 *--------------------------------------------------------------------*/
void
log_viewer_spawn (void)
{
	GtkWidget *scrolled_window;

	if (GTK_IS_WIDGET (log_viewer->window)) {
		gtk_widget_show (GTK_WIDGET (log_viewer->window));
		gdk_window_raise (log_viewer->window->window);
		return;
	}

	log_viewer->window = gnome_app_new ("Log", _("Pan - Log Viewer"));

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);

	log_setup_appbar ();

	gnome_app_set_contents (GNOME_APP (log_viewer->window), scrolled_window);
	gnome_app_create_menus_with_data (GNOME_APP (log_viewer->window), log_main_menu, log_viewer);
	gnome_app_create_toolbar_with_data (GNOME_APP (log_viewer->window), log_toolbar, log_viewer);
	gnome_app_install_menu_hints (GNOME_APP(log_viewer->window), log_main_menu);

	gtk_window_set_default_size (GTK_WINDOW (log_viewer->window), 450, 336);

	gtk_signal_connect_object (GTK_OBJECT (log_viewer->window), "delete_event",
				   GTK_SIGNAL_FUNC (log_hide_cb),
				   GTK_OBJECT (log_viewer->window));

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);

	gtk_container_add (GTK_CONTAINER (scrolled_window), log_viewer->log);

	gtk_widget_show_all (log_viewer->window);
}
