/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#if defined(__FreeBSD__) || defined(__svr4__)
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#endif

#include <arpa/inet.h>
#include <netdb.h>
#include <pwd.h>

#ifdef AF_NEEDS_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#include "date.h"
#include "grouplist.h"
#include "log.h"
#include "message.h"
#include "message-send.h"
#include "message-window.h"
#include "nntp.h"
#include "queue.h"
#include "queue-item-send.h"
#include "rfc822.h"
#include "smtp.h"
#include "util.h"


static gchar* generate_references (Message *msg);

typedef enum
{
	OKAY,
	WARN,
	REFUSE,
}
GoodnessLevel;

static void
check_body (
	GSList** setme_errors,
	GoodnessLevel *setme_goodness,
	const gchar* body)
{
	const int GNKSA_SIG_LENGTH_MAX = 4;
	const int GNKSA_SIG_WIDTH_MAX = 80;
	gchar **sd = NULL;
	int i = 0;
	int matches = 0;
	int lines = 0;
	int unquoted_lines = 0;

	g_assert (setme_goodness!=NULL);
	g_assert (setme_errors!=NULL);

	/* check for an empty message */
	if (!body || !*body) {
		*setme_errors = g_slist_append (*setme_errors,
			g_strdup(_("ERROR: The message is empty.")));
		*setme_goodness = MAX (*setme_goodness, REFUSE);
	}

	/* check number of new lines compared to quoted lines */
	unquoted_lines = lines = 0;
	sd = g_strsplit (body, "\n", -1);
	for (i=0; sd[i]!=NULL; ++i) {
		++lines;
		if (*sd[i]!='>')
			++unquoted_lines;
	}
	if (lines && !unquoted_lines) {
		*setme_errors = g_slist_append (*setme_errors,
			g_strdup(_("ERROR: The message is entirely quoted text.")));
		*setme_goodness = MAX (*setme_goodness, REFUSE);
	}
	else if (lines && ((unquoted_lines/(double)lines)<0.25)) {
		*setme_errors = g_slist_append (*setme_errors,
			g_strdup (_("WARNING: The message is mostly quoted text.")));
		*setme_goodness = MAX (*setme_goodness, WARN);
	}

	/* check the body for lines >80 characters. */
	for (i=matches=0; i!=lines; ++i)
		if (strlen(sd[i])>80)
			++matches;
	if (matches>0) {
		if (matches==1) 
			*setme_errors = g_slist_append (*setme_errors,
				g_strdup (_("WARNING: one line is more than 80 characters wide.")));
		else
			*setme_errors = g_slist_append (*setme_errors,
				g_strdup_printf (
					_("WARNING: %d lines are more than 80 characters wide."),
					matches));
		*setme_goodness = MAX (*setme_goodness, WARN);
	}

	/* check the signature length & width */
	for (i=lines-1; i>=0; --i)
		if (!strcmp(sd[i], "-- "))
			break;
	if (i>=0) {
		int max_width = 0;
		++i; /* move past the "-- " */
		if ((lines-i)>GNKSA_SIG_LENGTH_MAX) {
			*setme_errors = g_slist_append (*setme_errors,
				g_strdup_printf (
					_("WARNING: The signature is longer "
					  "than %d lines (%d lines long)."),
					GNKSA_SIG_LENGTH_MAX, lines-i));
			*setme_goodness = MAX (*setme_goodness, WARN);
		}
		for (; i!=lines; ++i) {
			const int len = strlen(sd[i]);
			if (len>max_width)
				max_width = len;
		}
		if (max_width>GNKSA_SIG_WIDTH_MAX) {
			*setme_errors = g_slist_append (*setme_errors,
				g_strdup_printf (
					_("WARNING: The signature is wider "
					  "than %d characters (%d characters long)."),
					GNKSA_SIG_WIDTH_MAX, max_width));
			*setme_goodness = MAX (*setme_goodness, WARN);
		}
	}

	/* cleanup */
	g_strfreev (sd);
}

/**
 * Try to trim references down to an acceptable length for the NNTP server,
 * as per GNKSA rule 9.  NOTE: this does not trim in the case that the
 * first + last 3 references together are too long for the server.  In order
 * for this to happen, the references would have to be ~250 characters long
 * each. :)
 *
 * @param untrimmed "references" string
 * @return newly-allocated trimmed "references" string
 */
static gchar*
trim_references (const gchar* refs)
{
	gchar *tmp = NULL;
	gchar *retval = NULL;
	const int gnksa_cutoff = 998 - (int)strlen("References: ");
	int len;
	int id_qty;
	int i;
	int march_end = 0;
	gchar **sd = NULL;

	/* sanity check */
	g_return_val_if_fail (refs!=NULL, NULL);
	len = strlen (refs);

	/* make an array of the message-ids */
	tmp = pan_substitute(refs, ">", "");
	g_strstrip (tmp);
	pan_warn_if_fail (tmp[0]=='<');
	sd = g_strsplit (tmp+1, "<", -1);
	g_free (tmp);
	refs = tmp = NULL;

	for (id_qty=0; sd[id_qty]!=NULL; ) {
		g_strstrip (sd[id_qty]);
		++id_qty;
	}

	/* must keep last three */
	len = 0;
	if (id_qty>=3) len += strlen(sd[id_qty-3]) + 2;
	if (id_qty>=2) len += strlen(sd[id_qty-2]) + 2;
	if (id_qty>=1) len += strlen(sd[id_qty-1]) + 2;

	/* must keep first one */
	if (id_qty>=4) len += strlen(sd[0]) + 2;

	/* now add in as many as will fit */
	for (i=1; i<id_qty-3; ++i) {
		int add_len = strlen(sd[i]) + 2;
		if (len + add_len > gnksa_cutoff)
			break;
		len += add_len;
	}
	march_end = i;

	/* now we know how many fit.  build the references string. */
	retval = g_strdup ("");

	/* first one */
	if (id_qty>=4) {
		tmp = g_strconcat (retval, "<", sd[0], ">", NULL);
		g_free (retval);
		retval = tmp;
	}

	/* any in the middle that fit */
	for (i=1; i<march_end; ++i) {
		tmp = g_strconcat (retval, "<", sd[i], ">", NULL);
		g_free (retval);
		retval = tmp;
	}

	/* the last min(qty,3) */
	for (i=3; i>=1; --i) {
		if (id_qty>=i) {
			tmp = g_strconcat (retval, "<", sd[id_qty-i], ">", NULL);
			g_free (retval);
			retval = tmp;
		}
	}

	/* cleanup */
	g_strfreev (sd);
	return retval;
}


/**
 * @param msg the message being replied to.
 * @return a newly-allocated references string.
 */
static gchar*
generate_references (Message *msg)
{
	gchar *full = NULL;
	gchar *refs = NULL;

	/* sanity check */
	g_return_val_if_fail (msg!=NULL, NULL);

	/* build a full references string */
	full = g_strconcat (msg->references, msg->message_id, NULL);
	refs = trim_references (full);
	g_free (full);
	return refs;
}

#if 0
static gchar*
generate_message_id (void)
{
	unsigned long now;
	unsigned long randy;
	static unsigned char baseid[64];
	gchar *b, *t, tmp[32];
	char *chars32 = "0123456789abcdefghijklmnopqrstuv";
	static unsigned long last_now;

	while (1) {
		time((time_t *) & now);
		if (now != last_now)
			break;
		sleep (1);
	}
	last_now = now;
	srand (now + getpid());
	randy = (unsigned long) rand();

	b = baseid;
	t = tmp;
	while (now) {
		*t++ = chars32[now & 0x1F];
		now = now >> 5;
	}

	while (t > tmp) {
		t--;
		*b++ = *t;
	}
	*b++ = '.';

	t = tmp;
	while (randy) {
		*t++ = chars32[randy & 0x1F];
		randy = randy >> 5;
	}
	while (t > tmp) {
		t--;
		*b++ = *t;
	}

	*b++ = '@';
	return (char *) baseid;
}
#endif


#if 0
static gchar *
reformat_body (gchar *body)
{
	gchar *tmp;
	gchar *body2 = g_malloc0 (sizeof (body));

	while ((tmp = strtok (body, "\n"))) {
		tmp = wrap_long_lines (tmp, 78);
		g_strconcat (body2, tmp, NULL);
	}

	return body2;
}
#endif


static gint
gslist_compare_group_and_name (gconstpointer a, gconstpointer b)
{
	return g_strcasecmp(((const group_data*)a)->name, (const char*)b);
}

static void
check_followup_to (
	GSList** setme_errors,
	GoodnessLevel *setme_goodness,
	const server_data* server,
	const GSList* group_names)
{
	const GSList* l = NULL;
	const int len = g_slist_length ((GSList*)group_names);

	g_return_if_fail (server!=NULL);

	for (l=group_names; l!=NULL; l=l->next) {
		if (g_strcasecmp((gchar*)l->data, "poster") &&
		    !g_slist_find_custom (server->grouplist, l->data, gslist_compare_group_and_name))
		{
			gchar* warn = g_strdup_printf (_("ERROR: Unknown group: ``%s''."), (char*)l->data);
			*setme_errors = g_slist_append (*setme_errors, warn);
			*setme_goodness = MAX (*setme_goodness, REFUSE);
		}
	}

	if (len>5) {
		*setme_errors = g_slist_append (*setme_errors,
			g_strdup(_("WARNING: Following-Up to a large number of groups.")));
		*setme_goodness = MAX (*setme_goodness, WARN);
	}
}

static void
check_groups (
	GSList** setme_errors,
	GoodnessLevel *setme_goodness,
	const server_data* server,
	const GSList* group_names,
	gboolean followup_to_set)
{
	const GSList* l = NULL;
	const int len = g_slist_length ((GSList*)group_names);

	g_return_if_fail (server!=NULL);

	for (l=group_names; l!=NULL; l=l->next) {
		if (!g_slist_find_custom (server->grouplist, l->data, gslist_compare_group_and_name)) {
			gchar* warn = g_strdup_printf (_("ERROR: Unknown group: ``%s''."), (char*)l->data);
			*setme_errors = g_slist_append (*setme_errors, warn);
			*setme_goodness = MAX (*setme_goodness, REFUSE);
		}
	}

	if (len>5) {
		*setme_errors = g_slist_append (*setme_errors,
			g_strdup(_("WARNING: Posting to a large number of groups.")));
		*setme_goodness = MAX (*setme_goodness, WARN);
	}

	if (len>2 && !followup_to_set) {
		*setme_errors = g_slist_append (*setme_errors,
			g_strdup(_("WARNING: Crossposting to a number of groups without setting Followup-To.")));
		*setme_goodness = MAX (*setme_goodness, WARN);
	}
}



/* Find and return a list of e-mail addresses to send to */
static GSList*
get_email_rcpts (const gchar *to)
{
	GSList *l = NULL;
	ADDRESS *list = NULL;
	ADDRESS *list2 = NULL;

	/* parse the addresses */	
	RFC822Error = 0;
	list = rfc822_parse_adrlist (NULL, to);
	pan_warn_if_fail (RFC822Error==0);

	/* make copies of who the letter is to */
	for (list2=list; list2!=NULL; list2=list2->next)
		l = g_slist_append (l, g_strdup(list2->mailbox));

	/* cleanup */
	rfc822_free_address (&list);
	list = list2 = NULL;

	return l;
}


/* Find and return a list of NNTP groups to send to */
static GSList*
get_nntp_rcpts (const gchar *to)
{
	gchar *pch = g_strstrip(g_strdup(to));
	GSList *nntp_rcpts = NULL;
	gchar ** split = NULL;
	int i = 0;

	/* check simple cases */
	if (!to || !*to) {
		g_free (pch);
		return NULL;
	}

	/* split the list */
	pch = g_strdelimit (pch, " ;:", ',');
	split = g_strsplit (pch, ",", -1);
	g_free (pch);

	/* walk through the list */
	for (i=0; split[i]!=NULL; ++i)
		nntp_rcpts = g_slist_append (nntp_rcpts, g_strstrip(split[i]));

	/* cleanup */
	g_free (split);

	return nntp_rcpts;
}

/**
 * @return the number of recipients
 */
static int
check_smtp_rcpts (
	GSList** setme_errors,
	GoodnessLevel *setme_goodness,
	const gchar* text,
	gchar** setme_out_text)
{
	ADDRESS *list = NULL;
	ADDRESS *list2 = NULL;
	int qty = 0;
	
	RFC822Error = 0;
	list = rfc822_parse_adrlist (NULL, text);
	if (RFC822Error) {
		*setme_errors = g_slist_append (*setme_errors,
			g_strdup_printf (_("ERROR: %s"), RFC822Errors[RFC822Error-1]));
		*setme_goodness = MAX (*setme_goodness, REFUSE);
	}
	else {
		char buf[1024] = { '\0' };
		rfc822_write_address (buf, sizeof(buf), list);
		*setme_out_text = g_strdup(buf);
	}

	for (list2=list, qty=0; list2!=NULL; list2=list2->next)
		++qty;

	rfc822_free_address (&list);
	list = list2 = NULL;

	return qty;
}

/**
 * Note that error-checking is done elsewhere.  By the time we reach this
 * point the message is assumed to be clear to post 
 */
static gchar*
build_nntp_message (Message *msg)
{
	gchar *subject = NULL;
	gchar *from = NULL;
	gchar *newsgroups = NULL;
	gchar *followup_to = NULL;
	gchar *reply_to = NULL;
	gchar *message = NULL;
	gchar *path = NULL;
	gchar *body = NULL;
	gchar *buf = NULL;
	static const char *host = NULL;
	static const char *fqdn = NULL;
	static char *uname = NULL;
	GSList *l = NULL;
	gchar *tmp = NULL;

	g_return_val_if_fail (msg!=NULL, NULL);

	if (host == NULL)
		host = get_host_name ();
	if (fqdn == NULL)
		fqdn = get_fqdn (host);
	if (uname == NULL)
		uname = g_get_user_name ();

	/* Build "Path: " contents */
	path = g_strconcat (host, "!", uname, NULL);

	/* Build "Newsgroups: " contents */
	for (l=msg->rcpts; l!=NULL; l=l->next)
		newsgroups = pan_append (newsgroups, (gchar*)l->data, (newsgroups?',':'\0'));

	/* Build "Followup-To: " contents */
	if (GTK_IS_ENTRY(msg->window->followup_to)) {
		GSList *list = get_nntp_rcpts(gtk_entry_get_text(GTK_ENTRY(msg->window->followup_to)));
		GSList *l;
		for (l=list; l!=NULL; l=l->next)
			followup_to = pan_append (followup_to, (gchar*)l->data, (followup_to?',':'\0'));
		g_slist_foreach (list, (GFunc)g_free, NULL);
		g_slist_free (list);
	}

	/* Build "Reply-To: " contents */
	if (GTK_IS_ENTRY(msg->window->reply_to))
		reply_to = gtk_editable_get_chars (GTK_EDITABLE (msg->window->reply_to), 0, -1);

	/* Build "From: " contents */
	from = gtk_editable_get_chars (GTK_EDITABLE (msg->window->from), 0, -1);

	/* Build "Subject: " contents */
	subject = gtk_editable_get_chars (GTK_EDITABLE (msg->window->subject), 0, -1);

	/* Build "Organization: " contents */
	//organization = gtk_editable_get_chars (GTK_EDITABLE (msg->window->organization), 0, -1);

	/* manditory contents */
	message = g_strconcat ("From: ", from, "\n",
			       "Path: ", path, "\n",
			       "Newsgroups: ", newsgroups, "\n",
			       "Subject: ", subject, "\n", NULL);

	/* optional content: followup-to */
	if (followup_to !=NULL) {
		tmp = g_strconcat (message, "Followup-To: ", followup_to, "\n", NULL);
		g_free (message);
		message = tmp;
	}

	/* optional content: reply-to */
	if (reply_to != NULL) {
		tmp = g_strconcat (message, "Reply-To: ", reply_to, "\n", NULL);
		g_free (message);
		message = tmp;
	}

	g_free (path);
	g_free (from);
	g_free (followup_to);
	g_free (reply_to);
	g_free (newsgroups);
	g_free (subject);

	/* FIXME
	if (Pan.active_server->gen_msgid) {
		message = g_strconcat (tmp,"Message-ID: <", generate_message_id(), fqdn, ">\n", NULL);
		g_free (tmp);
		tmp = message;
	}
	*/
	if (msg && msg->type==NNTP_ARTICLE_REPLY) {
		gchar* pch = generate_references (msg);
		tmp = g_strconcat (message, "References: ", pch, "\n", NULL);
		g_free (pch);
		g_free (message);
		message = tmp;
	}

	tmp = gtk_editable_get_chars(GTK_EDITABLE(msg->window->body), 0, -1),
	body = g_strconcat ("\n", tmp, NULL);
	g_free (tmp);

	/* Double periods at the start of lines (RFC977) */
	tmp = pan_substitute (body, "\n.", "\n..");
	g_free (body);
	body = tmp;

	buf = generate_date ();
	tmp = g_strconcat (message,
			       "Date: ", buf, "\n",
			       //"Organization: ", organization, "\n",
			       "X-Newsreader: ", "Pan ", VERSION, "\n",
			       body, NULL);
	g_free (buf);
	g_free (message);
	g_free (body);
	message = tmp;

	return message;
}

static void
check_subject (
	GSList** setme_errors,
	GoodnessLevel *setme_goodness,
	const gchar* subject)
{
	g_assert (setme_errors!=NULL);
	g_assert (setme_goodness!=NULL);

	if (!subject || !*subject) {
		*setme_errors = g_slist_append (*setme_errors,
			g_strdup(_("ERROR: No subject specified.")));
		*setme_goodness = MAX (*setme_goodness, REFUSE);
	}
}


static int
send_done_cb (gpointer call_object, gpointer call_arg, gpointer user_data)
{
	StatusItem* item = STATUS_ITEM(call_object);
	int status = GPOINTER_TO_INT(call_arg);
	GtkWidget* send_button = GTK_WIDGET(user_data);
	Message *message = QUEUE_ITEM_SEND(item)->msg;
	GSList *lines = NULL;
	GSList *l = NULL;
	gchar *text = NULL;

	if (status & SMTP_FAILED)
		lines = g_slist_append (lines, _("E-Mail send failed."));
	if (status & NNTP_FAILED)
		lines = g_slist_append (lines, _("Usenet posting failed."));
	if (!status)
		lines = g_slist_append (lines, _("Message sent successfully."));

	for (l=lines; l!=NULL; l=l->next)
		text = pan_append (text, (const gchar*)l->data, '\n');

	pan_lock();
	gnome_ok_dialog_parented (text, GTK_WINDOW(message->window->window));
	g_message (text);
	g_free (text);

	if (!status)
	{
		gtk_widget_destroy (message->window->window);
		message_free (message);
	}
	else
	{
		gtk_widget_set_sensitive (send_button, TRUE);
	}
	pan_unlock();

	return 0;
}

static void
start_sending (GtkWidget *widget, server_data* sdata, Message *msg)
{
	PanObject *item = queue_item_send_new (sdata, msg);
	gtk_widget_set_sensitive (GTK_WIDGET(widget), FALSE);
	pan_callback_add (STATUS_ITEM(item)->done, send_done_cb, widget);
	queue_add (QUEUE_ITEM(item));
}

typedef struct {
	GtkWidget *widget;
	server_data *sdata;
	Message *msg;
} SendAnywayCallbackStruct;

static void
send_anyway_cb (gint reply, gpointer data)
{
	if (!reply) { // yes
		SendAnywayCallbackStruct *cbs = (SendAnywayCallbackStruct*)data;
		start_sending (cbs->widget, cbs->sdata, cbs->msg);
	}

	g_free (data);
}

void
send_cb (GtkWidget *widget, Message *msg)
{
	gchar *subject = NULL;
	gchar *body = NULL;
	gchar *tmp = NULL;
	GoodnessLevel goodness = OKAY;
	GSList *errors = NULL;
	GSList *groups_slist = NULL;
	server_data *server = grouplist_get_current_server();
	int group_qty = 0;
	int mail_qty = 0;
	int error_len = 0;
	gboolean followup_to_set;

	/* sanity check */
	g_return_if_fail (widget!=NULL);
	g_return_if_fail (msg!=NULL);
	g_return_if_fail (msg->window!=NULL);
	g_return_if_fail (msg->window->subject!=NULL);

	/* check the subject... */
	subject = gtk_editable_get_chars (GTK_EDITABLE (msg->window->subject), 0, -1);
	subject = g_strstrip (subject);
	check_subject (&errors, &goodness, subject);
	g_free (subject);

	/* check the body... */
	body = gtk_editable_get_chars (GTK_EDITABLE (msg->window->body), 0, -1);
	body = g_strstrip (body);
	check_body (&errors, &goodness, body);
	g_free (body);

	/* check the optional followup-to */
	followup_to_set = FALSE;
	if (GTK_IS_ENTRY(msg->window->followup_to)) {
		GSList *l = get_nntp_rcpts(gtk_entry_get_text(GTK_ENTRY(msg->window->followup_to)));
		int qty = g_slist_length (l);
		followup_to_set = qty>0;
		check_followup_to (&errors, &goodness, server, l);
		g_slist_foreach (l, (GFunc)g_free, NULL);
		g_slist_free (l);
	}

	/* check the groups... */
	groups_slist = get_nntp_rcpts(gtk_entry_get_text(GTK_ENTRY(msg->window->newsgroups)));
	group_qty = g_slist_length (groups_slist);
	check_groups (&errors, &goodness, server, groups_slist, followup_to_set);
	g_slist_foreach (groups_slist, (GFunc)g_free, NULL);
	g_slist_free (groups_slist);

	/* check the author... */
	tmp = NULL;
	check_smtp_rcpts (&errors, &goodness, gtk_entry_get_text(GTK_ENTRY(msg->window->from)), &tmp);
	if (tmp!=NULL) {
		gtk_entry_set_text (GTK_ENTRY(msg->window->from), tmp);
		g_free (tmp);
		tmp = NULL;
	}

	/* check the email recipients... */
	mail_qty = check_smtp_rcpts (
		&errors, &goodness,
		gtk_entry_get_text(GTK_ENTRY(msg->window->users)), &tmp);
	if (tmp!=NULL) {
		gtk_entry_set_text (GTK_ENTRY(msg->window->users), tmp);
		g_free (tmp);
		tmp = NULL;
	}

	/* one last error check */
	if (!group_qty && !mail_qty) {
		errors = g_slist_append (errors, g_strdup(_("ERROR: No recipients specified.")));
		goodness = MAX (goodness, REFUSE);
	}

	error_len = g_slist_length (errors);
	if (error_len)
	{
		/* post a warning dialog */
		GSList *l = NULL;
		gchar *pch = NULL;
		for (l=errors; l!=NULL; l=l->next)
			pch = pan_append (pch, (const gchar*)l->data, '\n');
		if (goodness==REFUSE)
			pan_error_dialog (pch);
		else {
			SendAnywayCallbackStruct* cbs = g_new0 (SendAnywayCallbackStruct, 1);
			const gchar *ques = error_len>1
				? _("Do you want to post anyway, despite these warnings?")
				: _("Do you want to post anyway, despite this warning?");
			pch = pan_append (pch, "\n\n", 0);
			pch = pan_append (pch, ques, 0);
			cbs->widget = widget;
			cbs->sdata = server;
			cbs->msg = msg;
			gnome_question_dialog (pch, send_anyway_cb, cbs);
		}
		g_warning (pch);
		g_free (pch);

		/* cleanup */
		g_slist_foreach (errors, (GFunc)g_free, NULL);
		g_slist_free (errors);
		errors = NULL;
	}

	if (goodness==OKAY)
		start_sending (widget, server, msg);
}

gint
message_send (StatusItem *item, PanSocket *sock, Message *msg)
{
	GSList *nntp_rcpts;
	GSList *email_rcpts;
	gchar *nntp_msg;
	gchar *tmp1, *tmp2;
	gchar *newsgroups;
	gchar *users;
	gchar *subject;
	gint retval = 0;

	g_return_val_if_fail (msg!=NULL, -1);

	subject = gtk_editable_get_chars (
		GTK_EDITABLE (msg->window->subject), 0, -1);

	/* where is this being posted... */
	newsgroups = gtk_editable_get_chars (
		GTK_EDITABLE (msg->window->newsgroups), 0, -1);
	tmp1 = g_strdup (newsgroups);
	nntp_rcpts = get_nntp_rcpts (tmp1);

	/* who is this being mailed to... */
	users = gtk_editable_get_chars (
		GTK_EDITABLE (msg->window->users), 0, -1);
	tmp2 = g_strdup (users);
	email_rcpts = get_email_rcpts (tmp2);


	if (email_rcpts) {
		gchar *leader = NULL;
		if (nntp_rcpts) {
			leader = g_strdup_printf (
				_("[This is an email copy of a Usenet posting to \"%s\"]"),
				(gchar*)nntp_rcpts->data);
		}
		msg->rcpts = email_rcpts;
		if (!smtp_send (item, msg, leader))
			log_add_va (_("E-mail Message '%s' sent."), subject);
		else {
			log_add_va (_("E-mail Message '%s' not sent."),subject);
			retval |= SMTP_FAILED;
		}
		g_free (leader);
	}


	if (nntp_rcpts) {
		msg->rcpts = nntp_rcpts;
		nntp_msg = build_nntp_message (msg);
		if (!nntp_post (sock, nntp_msg))
			log_add_va (_("NNTP Article '%s' posted."), subject);
		else
			retval |= NNTP_FAILED;

		g_free (nntp_msg);
	}


	/* clean up */
	g_free (subject);
	g_free (tmp1);
	g_free (tmp2);
	g_free (newsgroups);
	g_free (users);

	return retval;
}
