/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 TODO:
 [ ] transparent (de)compression of database (bzip2 preferably)
*/

#include <config.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <pthread.h>

#include "debug.h"
#include "pan-db.h"
extern gboolean file_exists (const gchar* filename);

/* needed for db.h with 'gcc -ansi -pedantic' */
#ifndef _BSD_SOURCE
#  define _BSD_SOURCE 1
#endif
#include <sys/types.h>
#include <sys/fcntl.h>

#ifdef PREFER_DB1
#ifdef HAVE_DB1_DB_H
# include <db1/db.h>
#else
# ifdef HAVE_DB_185_H
#  include <db_185.h>
# else
#  include <db.h>
# endif
#endif
#else
#ifdef HAVE_DB_185_H
# include <db_185.h>
#else
# ifdef HAVE_DB_H
#  include <db.h>
# else
#  include <db1/db.h>
# endif
#endif
#endif


static
gboolean
open_database (
	const gchar* filename,
	DB** setme )
{
	u_int32_t flags = 0;

	debug (DEBUG_DB, "Opening database %s", filename);

	/* set the flags... */
	flags = O_RDWR;
	if (!file_exists (filename))
		flags |= O_CREAT;

	/* open the file */
	*setme = dbopen (filename, flags, 0600, DB_BTREE, NULL);
	if ( !*setme )
		g_warning ("Can't open database ``%s'': %s (flags: %d, errno: %d).  "
			   "This is typically caused by upgrading to a new version of "
			   "Pan and linking against a different version of \"libdb\" "
			   "than had been used before.  See the Pan FAQ "
			   "at <http://www.superpimp.org/faq.html> for more information. ",
	                   filename, strerror(errno), flags, errno);

	/* return status */
	return (*setme != NULL);
}


static void
clear_dbt (
	DBT* setme)
{
	g_assert (setme!=0);
	memset (setme, 0, sizeof(DBT));
}

static void
set_dbt (
	const char* str,
	DBT* setme )
{
	g_assert (str!=0 && *str!=0);
	g_assert (setme!=0);

	clear_dbt ( setme );
	setme->data = (char*)str;
	setme->size = strlen(str) + 1;
}
	

//============================================================================

// testing
#define pthread_mutex_lock(A)
#define pthreaad_mutex_unlock(A)
#define pthreaad_mutex_trylock(A)

typedef struct
{
	DB* db;
	pthread_mutex_t lock;
	gchar* filename;
	int ref_count;
}
PanDB;

static const pthread_mutex_t init_lock = PTHREAD_MUTEX_INITIALIZER;
pan_db
pan_db_open (
	const char* filename)
{
	PanDB* pdb = g_malloc0 (sizeof(PanDB));
	pdb->filename = g_strdup (filename);
	pdb->lock = init_lock;
	pdb->ref_count = 0;
	debug (DEBUG_DB, "pan-db %p opening database '%s'", pdb, filename);
	open_database (filename, &pdb->db);
	return (pan_db)pdb;
}

void
pan_db_close (
	pan_db db)
{
	PanDB* pdb = (PanDB*)db;
	debug (DEBUG_DB, "pan-db %p closing database %s", db, pdb->filename);
	g_return_if_fail (!pdb->ref_count);
	pdb->db->close (pdb->db);
	g_free (pdb->filename);
	memset (pdb, 0, sizeof(PanDB*));
	g_free (pdb);
}

gchar*
pan_db_get_value_str (
	pan_db db,
	const char* key_str)
{
	DB* real_db = NULL;
	gchar* retval = NULL;
	DBT key, val;

	debug (DEBUG_DB,"pan-db %p looking for match to key '%s'", db, key_str);
	g_return_val_if_fail (((PanDB*)db)->db != NULL, NULL);

	real_db = ((PanDB*)db)->db;
	set_dbt (key_str, &key);
	clear_dbt (&val);
	if (!db || !real_db)
		return NULL;

	pthread_mutex_lock (&((PanDB*)db)->lock);
	if (!real_db->get (real_db, &key, &val, 0))
		retval = g_strdup ((char*)(val.data));
	pthread_mutex_unlock (&((PanDB*)db)->lock);

	debug (DEBUG_DB, "pan-db %p looking for match to key '%s' and found '%s'", db, key_str, retval);
	return retval;
}

void
pan_db_put_value_str (
	pan_db db,
	const char* key_str,
	const char* val_str )
{
	DB* real_db = NULL;
	DBT key, val;

	g_return_if_fail (db!=NULL);
	real_db = ((PanDB*)db)->db;
	g_return_if_fail (real_db);

	set_dbt (key_str, &key);
	set_dbt (val_str, &val);

	pthread_mutex_lock (&((PanDB*)db)->lock);
	real_db->put (real_db, &key, &val, 0);
	pthread_mutex_unlock (&((PanDB*)db)->lock);
}

int
pan_db_get_value_i (
	pan_db db,
	const char* key )
{
	gchar* val = pan_db_get_value_str(db, key);
	int retval = val ? atoi(val) : 0;
	g_free (val);
	return retval;
}

void
pan_db_put_value_i (
	pan_db db,
	const char* key,
	int i )
{
	char buf[32];
	sprintf (buf, "%d", i);
	pan_db_put_value_str (db, key, buf);
}

void
pan_db_add_value_i (
	pan_db db,
	const char* key,
	int i )
{
	pan_db_put_value_i (db, key, i+pan_db_get_value_i(db, key));
}

void
pan_db_erase (
	pan_db db,
	const char* key_str )
{
	DB* real_db = ((PanDB*)db)->db;
	DBT key;
	int status;
	set_dbt (key_str, &key);

	pthread_mutex_lock (&((PanDB*)db)->lock);
        status = real_db->del (real_db, &key, 0);
	pthread_mutex_unlock (&((PanDB*)db)->lock);

	if (status)
		g_warning ("Unable to erase record %s: %s", key_str, strerror(status));
}

void
pan_db_foreach (
	pan_db db,
	DBFunc func,
	gpointer user_data)
{
	DB* real_db = ((PanDB*)db)->db;
	DBT key, val;
	int status;
	u_int step = R_FIRST;

	clear_dbt ( &key );
	clear_dbt ( &val );
	pthread_mutex_lock (&((PanDB*)db)->lock);
	while ((status = real_db->seq (real_db, &key, &val, step)) == 0) {
	       (*func)(key.data, val.data, user_data);
		step = R_NEXT;
	}
	pthread_mutex_unlock (&((PanDB*)db)->lock);
}

void
pan_db_sync (
	pan_db db)
{
	DB* real_db = ((PanDB*)db)->db;

	if (!pthread_mutex_trylock (&((PanDB*)db)->lock)) {
		real_db->sync (real_db, 0);
		pthread_mutex_unlock (&((PanDB*)db)->lock);
	} else {
		g_warning (_("Didn't sync database.  This is probably okay."));
	}
}

void
pan_db_erase_all (
	pan_db db )
{
	PanDB* pdb = (PanDB*)db;
	pdb->db->close ( pdb->db );
        remove (pdb->filename);
	open_database (pdb->filename, &pdb->db);
}

//============================================================================

static GHashTable* db_hash = NULL;

pan_db
pan_db_ref (
	const char* filename )
{
	pan_db db = NULL;

	/* init the hash table if necessary */
	if ( !db_hash )
		db_hash = g_hash_table_new ( g_str_hash, g_str_equal );

	/* get the db, or create a new one if we don't have one already */
	db = g_hash_table_lookup ( db_hash, filename );
	if ( !db ) {
		db = pan_db_open ( filename );
		g_hash_table_insert ( db_hash, g_strdup(filename), db );
	}

	/* pop up the ref count and go */
	++((PanDB*)db)->ref_count;
	debug (DEBUG_DB, "pan-db %p (%s) refcount up to %d", db, filename, ((PanDB*)db)->ref_count);
	return db;
}

void
pan_db_unref (
	pan_db db )
{
	PanDB* pan_db = (PanDB*)db;

	g_return_if_fail (pan_db->ref_count >= 0);
	g_return_if_fail (db!=NULL); /* don't pass in null */
	g_return_if_fail (db_hash!=NULL); /* created in ref */
	g_return_if_fail (g_hash_table_lookup(db_hash, pan_db->filename)==db);

	debug (DEBUG_DB, "pan-db %p (%s) refcount down to %d", db, pan_db->filename, ((PanDB*)db)->ref_count-1);
	if ( !--pan_db->ref_count )
	{
		gpointer key = NULL;
		gpointer value = NULL;
		g_hash_table_lookup_extended (
				db_hash, pan_db->filename, &key, &value);
		g_hash_table_remove (db_hash, pan_db->filename);
		g_free (key);
		pan_db_close (db);
	}
}
