/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "gui.h"
#include "log.h"
#include "status-item.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

gchar*
status_item_describe (const StatusItem* item)
{
	g_return_val_if_fail (item!=NULL, NULL);
	return (*item->describe)(item);
}

StatusItem *
status_item_new (StatusItemDescribeFunc describe)
{
	StatusItem *item = g_new0 (StatusItem, 1);
        debug (DEBUG_PAN_OBJECT, "status_item_new: %p", item);
	status_item_constructor (item, status_item_destructor, describe);
	return item;
}

void
status_item_constructor (
	StatusItem *item,
	PanObjectDestructor destructor,
	StatusItemDescribeFunc describe)
{
	PanObject *obj = PAN_OBJECT(item);

	pan_object_constructor (obj, destructor);

	item->init_steps = pan_callback_new ();
	item->next_step = pan_callback_new ();
	item->progress = pan_callback_new ();
	item->set_activity = pan_callback_new ();
	item->status = pan_callback_new ();
	item->error = pan_callback_new ();
	item->done = pan_callback_new ();
	item->describe = describe;

        debug (DEBUG_PAN_OBJECT, "status_item_constructor: %p", item);
}

void
status_item_destructor (PanObject *po)
{
	StatusItem *item = STATUS_ITEM(po);
	g_return_if_fail (item != NULL);

        debug (DEBUG_PAN_OBJECT, "status_item_destructor: %p", item);

	/* clear out the callback lists */
	pan_callback_free (item->init_steps);
	pan_callback_free (item->next_step);
	pan_callback_free (item->progress);
	pan_callback_free (item->set_activity);
	pan_callback_free (item->status);
	pan_callback_free (item->error);
	pan_callback_free (item->done);

	pan_object_destructor (PAN_OBJECT(item));
}

/*****
******
*****/

void
status_item_emit_init_steps (StatusItem *item, gint steps)
{
	pan_callback_call (item->init_steps, item, GINT_TO_POINTER(steps));
}

void
status_item_emit_next_step (StatusItem *item)
{
	pan_callback_call (item->next_step, item, NULL);
}

void
status_item_emit_progress (StatusItem *item, gint out_of_100)
{
	pan_callback_call (item->progress, item, GINT_TO_POINTER(out_of_100));
}

void
status_item_emit_done (StatusItem *item, gint status)
{
	pan_callback_call (item->done, item, GINT_TO_POINTER(status));
}

void
status_item_emit_activity (StatusItem *item, gint n)
{
	pan_callback_call (item->set_activity, item, GINT_TO_POINTER(n));
}

void
status_item_emit_status (StatusItem *item, const gchar *format, ...)
{
	gchar *pch = NULL;
       	va_list args;

	g_assert (item!=NULL);
	g_return_if_fail (format!=NULL);

	va_start (args, format);
	pch = g_strdup_vprintf (format, args);
	va_end (args);
	debug(DEBUG_QUEUE,pch);
	pan_callback_call (item->status, item, pch);
	g_free (pch);
}

void
status_item_emit_error (StatusItem *item, const gchar *fmt, ...)
{
	gchar *pch = NULL;
       	va_list args;

	g_assert (item!=NULL);
	g_return_if_fail (fmt!=NULL);

	va_start (args, fmt);
	pch = g_strdup_vprintf (fmt, args);
	va_end (args);
	debug(DEBUG_QUEUE, "StatusItem %p has this error: %s", item, pch);
	pan_callback_call (item->error, item, pch);
	g_free (pch);
}

void
status_item_next_step_gfunc (gpointer gdata,
                             gpointer p_status_item)
{
	status_item_emit_next_step (STATUS_ITEM(p_status_item));
}
