#include <ctype.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <regex.h>
#include <time.h>

#include "npsummary.h"

int compare( const void *, const void * );

void sort_list_callback( GtkWidget *widget, gpointer data )
{
   static int running = 0;
   if ( running )
      return;
   else
      running = 1;

   NP_Summary *summary = ( NP_Summary *)data;

   if ( !summary->total )
      return;

   if ( widget != NULL )
   {
      gtk_toggle_button_set_active( 
            GTK_TOGGLE_BUTTON( summary->old_list_button), 
            FALSE );
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), TRUE );

      summary->old_list_button = widget;
   }

   static int header;
   if ( widget != NULL )
      header = ( int)gtk_object_get_data( GTK_OBJECT( widget ), "header");

   struct array_t
   {
      char *first, *second, *third;
      GtkStyle *first_style, *second_style, *third_style;
      int ordinal;
   }
   *array;

   array = ( array_t *)calloc( summary->total, sizeof *array );
   if ( array == NULL )
   {
      perror( "calloc" );
      exit( 1 );
   }

   int i = 0;
   for( np_thread_node_t *pointer = summary->tree;
         pointer != NULL;
         pointer = pointer->next )
   {
      if ( pointer->is_article < 0 )
         continue;

      array[ i ].ordinal = pointer->ordinal;

      switch( header )
      {
         case 0:
            array[ i ].first = pointer->subject;
            array[ i ].second = pointer->from;
            summary->convert_date( pointer->date, &array[ i ].third );

            if ( pointer->is_article )
               array[ i ].first_style = summary->read_style;
            else
               array[ i ].first_style = summary->header_style;

            array[ i ].second_style = summary->dummy_style;
            array[ i ].third_style = summary->request_style;

            break;

         case 1:
            array[ i ].first = pointer->from;
            summary->convert_date( pointer->date, &array[ i ].second );
            array[ i ].third = pointer->subject;

            if ( pointer->is_article )
               array[ i ].third_style = summary->read_style;
            else
               array[ i ].third_style = summary->header_style;

            array[ i ].first_style = summary->dummy_style;
            array[ i ].second_style = summary->request_style;

            break;

         case 2:
            summary->convert_date( pointer->date, &array[ i ].first );
            array[ i ].second = pointer->subject;
            array[ i ].third = pointer->from;

            if ( pointer->is_article )
               array[ i ].second_style = summary->read_style;
            else
               array[ i ].second_style = summary->header_style;

            array[ i ].third_style = summary->dummy_style;
            array[ i ].first_style = summary->request_style;

            break;
      }

      ++i;
   }

   qsort( array, summary->total, sizeof *array, compare );

   char *labels[ 3 ];

   static char *titles[] = { "Subject", "Poster", "Local Time" };

   gtk_clist_clear( GTK_CLIST( summary->list ));
   gtk_clist_freeze( GTK_CLIST( summary->list ));

   switch( header )
   {
      case 0:
         gtk_clist_set_column_title( GTK_CLIST( summary->list ), 0,
               titles[ 0 ] );
         gtk_clist_set_column_title( GTK_CLIST( summary->list ), 1,
               titles[ 1 ] );
         gtk_clist_set_column_title( GTK_CLIST( summary->list ), 2,
               titles[ 2 ] );
         break;

      case 1:
         gtk_clist_set_column_title( GTK_CLIST( summary->list ), 0,
               titles[ 1 ] );
         gtk_clist_set_column_title( GTK_CLIST( summary->list ), 1,
               titles[ 2 ] );
         gtk_clist_set_column_title( GTK_CLIST( summary->list ), 2,
               titles[ 0 ] );
         break;

      case 2:
         gtk_clist_set_column_title( GTK_CLIST( summary->list ), 0,
               titles[ 2 ] );
         gtk_clist_set_column_title( GTK_CLIST( summary->list ), 1,
               titles[ 0 ] );
         gtk_clist_set_column_title( GTK_CLIST( summary->list ), 2,
               titles[ 1 ] );
         break;
   }

   if ( summary->positions != NULL )
      free( summary->positions );

   if (( summary->positions = ( int *)calloc( summary->total, 
                                      sizeof *summary->positions )) == NULL )
   {
      perror( "calloc" );
      exit( 1 );
   }

   for( i = 0; i < summary->total; ++i )
   {
      summary->positions[ i ] = array[ i ].ordinal;

      switch( header )
      {
         case 0:
            labels[ 0 ] = array[ i ].first;
            labels[ 1 ] = array[ i ].second;
            summary->convert_back_date( &array[ i ].third );
            labels[ 2 ] = array[ i ].third;
            break;

         case 1:
            labels[ 0 ] = array[ i ].first;
            summary->convert_back_date( &array[ i ].second );
            labels[ 1 ] = array[ i ].second;
            labels[ 2 ] = array[ i ].third;
            break;

         case 2:
            summary->convert_back_date( &array[ i ].first );
            labels[ 0 ] = array[ i ].first;
            labels[ 1 ] = array[ i ].second;
            labels[ 2 ] = array[ i ].third;
      }

      gtk_clist_append( GTK_CLIST( summary->list ), labels );

      gtk_clist_set_cell_style( GTK_CLIST( summary->list ), i, 0, 
                                array[ i ].first_style );
      gtk_clist_set_cell_style( GTK_CLIST( summary->list ), i, 1, 
                                array[ i ].second_style );
      gtk_clist_set_cell_style( GTK_CLIST( summary->list ), i, 2,
                                array[ i ].third_style );

      switch( header )
      {
         case 0:
            free( array[ i ].third );
            break;
            
         case 1:
            free( array[ i ].second );
            break;

         case 2:
            free( array[ i ].first );
            break;
      }
   }

   gtk_clist_thaw( GTK_CLIST( summary->list ));

   free( array );

   running = 0;

   return;
}

void NP_Summary::convert_date( char *date, char **converted )
{
   static int compiled = 0;
   static regex_t regex;
   regmatch_t *matches;
   int result;

   if ( date == NULL )
   {
      if ( compiled )
         regfree( &regex );

      compiled = 0;
      return;
   }

   char buffer[ 1024 ], *wdy, *dd, *mon, *yyyy, *hh, *mm, *ss, *tzn;
   
   if ( !compiled )
   {
      // Will match slightly wonky headers as well as the kosher. 

      wdy    = "([A-Z][a-z][a-z], +)?";
      dd     = "([0-3]?[0-9]) +";
      mon    = "([A-Z][a-z][a-z]) +";
      yyyy   = "([0-9][0-9])?([0-9][0-9]) +";
      hh     = "([0-2]?[0-9])";
      mm     = ":?([0-5]?[0-9])";
      ss     = "(:([0-5]?[0-9]))?";
      tzn    = "( +[A-Z][A-Z][A-Z]| +[+-][0-2][0-9][0-5][0-9])?";

      strcpy( buffer, wdy );
      strcat( buffer, dd );
      strcat( buffer, mon );
      strcat( buffer, yyyy );
      strcat( buffer, hh );
      strcat( buffer, mm );
      strcat( buffer, ss );
      strcat( buffer, tzn );

      if ( ( result = regcomp( &regex, buffer, REG_EXTENDED )))
         return;

      compiled = 1;
   }

   if (( matches =
         ( regmatch_t *)calloc( regex.re_nsub + 1, sizeof *matches )) == NULL )
   {
      perror( "calloc" );
      exit( 1 );
   }

   if (( result = regexec( &regex, date, regex.re_nsub + 1, matches, 0 )))
   {
      if ( result == REG_NOMATCH )
      {
         if (( *converted = strdup( "" )) 
               == NULL )
         {
            perror( "strdup" );
            exit( 1 );
         }

         return;
      }
      
      return;
   }

   /* discard Wdy */

   strncpy( buffer, date + matches[ 2 ].rm_so, 
            ( result = matches[ 2 ].rm_eo - matches[ 2 ].rm_so ));
   buffer[ result ] = '\0';
   int day = atoi( buffer );

   /* month */

   strncpy( buffer, date + matches[ 3 ].rm_so,
         ( result = matches[ 3 ].rm_eo - matches[ 3 ].rm_so ));
   buffer[ result ] = '\0';
   int month = convert_month( buffer );

   /* year */

   result = 0;
   buffer[ 0 ] = '\0';
   if ( matches[ 4 ].rm_so > 0 )
   {
      strncpy( buffer, date + matches[ 4 ].rm_so,
               ( result = matches[ 4 ].rm_eo - matches[ 4 ].rm_so ));
      buffer[ result ] = '\0';
   }

   strncat( buffer, date + matches[ 5 ].rm_so,
            ( result += matches[ 5 ].rm_eo - matches[ 5 ].rm_so ));
   buffer[ result ] = '\0';

   int year;
   if ( ( year = atoi( buffer )) < 100 )
      if ( buffer[ 0 ] == '9' )
         year += 1900;
      else
         year += 2000;

   /* hours */

   strncpy( buffer, date + matches[ 6 ].rm_so,
            ( result = matches[ 6 ].rm_eo - matches[ 6 ].rm_so ));
   buffer[ result ] = '\0';
   int hours = atoi( buffer );

   /* minutes */

   strncpy( buffer, date + matches[ 7 ].rm_so,
            ( result = matches[ 7 ].rm_eo - matches[ 7 ].rm_so ));
   buffer[ result ] = '\0';
   int minutes = atoi( buffer );

   /* seconds */

   strncpy( buffer, date + matches[ 9 ].rm_so,
            ( result = matches[ 9 ].rm_eo - matches[ 9 ].rm_so ));
   buffer[ result ] = '\0';
   int seconds = atoi( buffer );

   /* timezone */

   if ( matches[ 10 ].rm_so > 0 )
   {
      strncpy( buffer, date + matches[ 10 ].rm_so + 1,
               ( result = matches[ 10 ].rm_eo - matches[ 10 ].rm_so - 1 ));
      buffer[ result ] = '\0';
   }
   else
      strcpy( buffer, "GMT" );

   free( matches );

   /* convert to GMT */

   convert_zone( buffer );

   char second_buffer[ 1024 ];
   if ( strncmp( buffer, "unknown", 7 ))
   {
      strcpy( second_buffer, buffer );

      strncpy( buffer, second_buffer, 3 );
      buffer[ 3 ] = '\0';

      hours += atoi( buffer );

      strncpy( buffer, second_buffer + 3 , 3 );
      buffer[ 3 ] = '\0';

      minutes += atoi( buffer );

      /* convert to locale time */

      convert_localtime( &hours, &minutes, &seconds, &day, &month, &year );

      snprintf( buffer, sizeof buffer, "%4d%02d%02d%02d%02d%02d", 
                year, month, day, hours, minutes, seconds );
   }
   else
      strcpy( buffer, " (unknown timezone)" );

   if (( *converted = strdup( buffer )) == NULL )
   {
      perror( "strdup" );
      exit( 1 );
   }

   return;
}

void NP_Summary::convert_localtime( int *hours, int *minutes, int *seconds,
                                int *day, int *month, int *year )
{
   time_t t;
   div_t d;

   time( &t );
   localtime( &t );

   d = div( timezone, 3600 );

   *hours -= d.quot;
   *minutes -= d.rem / 60;

   adjust_time( hours, minutes, seconds, day, month, year );

   return;
}

int NP_Summary::convert_month( char *pointer )
{
   int month;

   switch( *pointer )
   {
   case 'A':
      switch( *( pointer + 1 ))
      {
      case 'p':
         month = 4;
         break;

      default:
         month = 8;
         break;
      }

      break;

   case 'D':
      month = 12;
      break;

   case 'F':
      month = 2;
      break;

   case 'J':
      switch( *( pointer + 1 ))
      {
      case 'a':
         month = 1;
         break;

      default:
         switch( *( pointer + 2 ))
         {
         case 'l':
            month = 7;
            break;

         default:
            month = 6;
            break;
         }
      }
      break;

   case 'M':
      switch( *( pointer + 2 ))
      {
      case 'r':
         month = 3;
         break;

      default:
         month = 5;
         break;
      }

      break;

   case 'N':
      month = 11;
      break;

   case 'O':
      month = 10;
      break;

   default:
      month = 9;
      break;
   }

   return month;
}

void NP_Summary::convert_zone( char *buffer )
{
   char *zone;

   if ( buffer[ 0 ] == '+' || buffer[ 0 ] == '-' )
   {
      buffer[ 5 ] = buffer[ 4 ];
      buffer[ 4 ] = buffer[ 3 ];
      buffer[ 3 ] = buffer[ 0 ] = (( buffer[ 0 ] == '+' ) ? '-' : '+' );
      buffer[ 6 ] = '\0';
      return;
   }
   else
      switch( buffer[ 0 ] )
      {
      case 'C':
         switch( buffer[ 1 ] )
         {
         case 'D':
            zone = "-06-00";
            break;

         default:
            zone = "-07-00";
            break;
         }

         break;

      case 'E':
         switch( buffer[ 1 ] )
         {
         case 'D':
            zone = "-00-00";
            break;

         default:
            zone = "-06-00";
            break;
         }

         break;

      case 'G':
         zone = "+00+00";
         break;

      case 'M':
         switch( buffer[ 1 ] )
         {
         case 'D':
            zone = "-07-00";
            break;

         default:
            zone = "-08-00";
            break;
         }

         break;

      case 'P':
         switch( buffer[ 1 ] )
         {
         case 'D':
            zone = "-08-00";
            break;

         default:
            zone = "-09-00";
            break;
         }
         break;

      default:
         zone = "unknown";
         break;
      }

   strcpy( buffer, zone );

   return;
}

void NP_Summary::adjust_time( int *hours, int *minutes, int *seconds,
                              int *day, int *month, int *year )
{
   if ( *minutes < 0 )
   {
      *minutes += 60;
      ( *hours )--;
   }
   else
      if ( *minutes > 59 )
      {
         *minutes -= 60;
         ( *hours )++;
      }

   if ( *hours < 0 )
   {
      *hours += 24;
      ( *day )--;
   }
   else
      if ( *hours > 23 )
      {
         *hours -= 24;
         ( *day )++;
      }

   if ( *day < 1 )
   {
      switch( *month )
      {
      case 3:
         /* 
            Determine leap year.
            Don't use this program after a hundred years :)
         */

         if ( ( div( *year, 4 )).rem )
            *day = 28;
         else
            *day = 29;

         break;

      case 5:
      case 7:
      case 10:
      case 12:
         *day = 30;
         break;

      default:
         *day = 31;
         break;
      }

      ( *month )--;
   }
   else
      if ( *day > 28 )
      {
         switch( *month )
         {
         case 2:
            if ( ( div( *year, 4 )).rem )
               *day = 1;
            else
               *day = 29;

            ( *month )++;
            break;

         case 4:
         case 6:
         case 9:
         case 11:
            if ( *day > 30 )
            {
               *day -= 30;
               ( *month )++;
            }
            break;

         default:
            if ( *day > 31 )
            {
               *day -= 31;
               ( *month )++;
            }
            break;
         }
      }

   if ( *month < 1 )
   {
      *month += 12;
      ( *year )--;
   }
   else
      if ( *month > 12 )
      {
         *month -= 12;
         ( *year )++;
      }

   return;
}


/* 
 * Converts a numerical representation of a month into its three-letter
 * abbreviation.
 */

void NP_Summary::convert_back_month( int number, char *month )
{
   switch( number )
   {
      case 1:
         strcpy( month, "Jan" );
         break;

      case 2:
         strcpy( month, "Feb" );
         break;

      case 3:
         strcpy( month, "Mar" );
         break;

      case 4:
         strcpy( month, "Apr" );
         break;

      case 5:
         strcpy( month, "May" );
         break;

      case 6:
         strcpy( month, "Jun" );
         break;

      case 7:
         strcpy( month, "Jul" );
         break;

      case 8:
         strcpy( month, "Aug" );
         break;

      case 9:
         strcpy( month, "Sep" );
         break;

      case 10:
         strcpy( month, "Oct" );
         break;

      case 11:
         strcpy( month, "Nov" );
         break;

      case 12:
         strcpy( month, "Dec" );
         break;
   }

   return;
}

/*
 * Renders converted date string into a proper Usenet date header.
 */

int NP_Summary::convert_back_date( char **date )
{
   char *temp, month[ 4 ], buffer[ 25 ];
   time_t t;
   struct tm *btime;


   if ( !strcmp( *date, " (unknown timezone)" ))
      return -1;

   if (( temp = strdup( *date )) == NULL )
   {
      perror( "strdup" );
      exit( 1 );
   }

   free( *date );

   time( &t );
   btime = localtime( &t );

   buffer[ 0 ] = temp[ 4 ];
   buffer[ 1 ] = temp[ 5 ];
   buffer[ 2 ] = '\0';

   convert_back_month( atoi( buffer ), month );

   /* day */

   buffer[ 0 ] = temp[ 6 ];
   buffer[ 1 ] = temp[ 7 ];
   buffer[ 2 ] = ' ';

   /* month */

   buffer[ 3 ] = month[ 0 ];
   buffer[ 4 ] = month[ 1 ];
   buffer[ 5 ] = month[ 2 ];
   buffer[ 6 ] = ' ';

   /* year */

   buffer[ 7 ] = temp[ 0 ];
   buffer[ 8 ] = temp[ 1 ];
   buffer[ 9 ] = temp[ 2 ];
   buffer[ 10 ] = temp[ 3 ];
   buffer[ 11 ] = ' ';

   /* HH:MM:SS */

   buffer[ 12 ] = temp[ 8 ];
   buffer[ 13 ] = temp[ 9 ];
   buffer[ 14 ] = ':';
   buffer[ 15 ] = temp[ 10 ];
   buffer[ 16 ] = temp[ 11 ];
   buffer[ 17 ] = ':';
   buffer[ 18 ] = temp[ 12 ];
   buffer[ 19 ] = temp[ 13 ];
   buffer[ 20 ] = ' ';
   buffer[ 21 ] = '\0';

   free( temp );

   /* TZN */

   if ( tzname[ btime->tm_isdst ] != NULL )
      strcat( buffer, tzname[ btime->tm_isdst ] );
   else
      strcat( buffer, "(local time)" );

   if (( *date = strdup( buffer )) == NULL )
   {
      perror( "strdup" );
      exit( 1 );
   }

   return 0;
}
