/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <string.h>

#include "detect_scsi.h"
#include "vector_commands.h"
#include "common_gtk.h"
#include "command.h"
#include "linebuffer.h"
#include "locks.h"
#include "cdrecord_options.h"
#include "contractions.h"
#include "globals.h"

static char *get_cdrecord_device(config_cdr_data *cdr_info) {
	char device[13];

	if (0 != strlen(gtk_entry_get_text(GTK_ENTRY(cdr_info->device)))) {
		return g_strdup(gtk_entry_get_text(GTK_ENTRY(cdr_info->device)));
	} else {
		g_snprintf(device, 13, "dev=%i,%i,%i",
		gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->scsibus)),
		gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->target)),
		gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->lun)));
		return g_strdup(device);
	}
	g_assert_not_reached();
	return NULL;
}

static int do_cdrecord_shortcommand(gchar *command, config_cdr_data *cdr_info) {
	char *cdr_cmd, *tmp;
	int result;

	cdr_cmd = string_append(cdrecord_path, NULL);
	cdr_cmd = string_append(cdr_cmd, command);
	cdr_cmd = string_append(cdr_cmd, " ");

	tmp = get_cdrecord_device(cdr_info);
	cdr_cmd = string_append(cdr_cmd, tmp);
	free(tmp);

	if (0 != strlen(gtk_entry_get_text(GTK_ENTRY(cdr_info->driver)))) {
		cdr_cmd = string_append(cdr_cmd, " driver=");
		cdr_cmd = string_append(cdr_cmd, gtk_entry_get_text(GTK_ENTRY(cdr_info->driver)));
	}

	cdr_cmd = string_append(cdr_cmd, " 1>/dev/null 2>/dev/null");

	result = system(cdr_cmd);
	free(cdr_cmd);

	return result;
}

static void cdrecord_auto_detect_swab(config_cdr_data *cdr_info) {
	cmd_v *cdr_cmd;
	char *tmp;

	cdr_cmd = get_new_cmd();
	add_option_to_cmd(cdr_cmd, cdrecord_path);
	add_option_to_cmd(cdr_cmd, "-checkdrive");
	tmp = get_cdrecord_device(cdr_info);
	add_option_to_cmd(cdr_cmd, tmp);
	free(tmp);

	if (!(is_running())) {
		FILE *f;
		char buf[BUF_S];
		int swab = FALSE;

		if (NULL == (f = popen_r_stdout(cdr_cmd))) {
			alert_user_of_error(_(" Could not detect if swab is to be used or not. "));
			g_warning("%s %i: Could not execute command", __FILE__, __LINE__);
		} else {
			while (NULL != fgets(buf, BUF_S, f)) {
				if (NULL != strstr(buf, "SWABAUDIO"))
					swab = TRUE;
			}

			if (swab) {
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cdr_info->swab_audio), TRUE);
			} else {
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cdr_info->swab_audio), FALSE);
			}
		}
            not_running();
      } else {
		alert_user_of_error(_(" Could not detect if swab is to be used or not. "));
	}
	destroy_cmd(cdr_cmd);
}

void cdrecord_checkdrive(GtkWidget *widget, gpointer data) {
	GtkWidget *dialog_window, *label, *ok;
	config_cdr_data *cdr_info = (config_cdr_data *) data;
	if (is_running())
		return;

	dialog_window = gtk_dialog_new();
	gtk_widget_show(dialog_window);

	if (do_cdrecord_shortcommand(" -checkdrive", cdr_info))
		label = gtk_label_new(_("cdrecord failed to recognice selected drive\n"
					"(wrong scsi settings or no permission to device)!"));
	else
		label = gtk_label_new(_("    Settings ok!    "));

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog_window)->vbox), label, TRUE,
			TRUE, 10);
	gtk_widget_show(label);

	ok = gtk_button_new_with_label(_("OK"));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog_window)->action_area), ok, TRUE, TRUE, 10);
	gtk_signal_connect(GTK_OBJECT(ok), "clicked",
			GTK_SIGNAL_FUNC(close_dialog), (gpointer) dialog_window);
	GTK_WIDGET_SET_FLAGS (ok, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(ok);
	gtk_widget_show(ok);

	not_running();
}

/* Tries to autodect and cycle trought scsi devices via /proc/scsi/scsi
 * Another way to do it would be to use the output from cdrecord -scanbus */
void cdrecord_auto_detect_drive(GtkWidget *widget, gpointer data) {
	FILE *fd;
	char *sptr;
	int channel, id, lun;
	char line1[BUF_S], line2[BUF_S], line3[BUF_S];
	static int cycle_dev = 1; /* which device we want this time */
	int times_trough = 0, no_of_cd_dev = 0;
	config_cdr_data *cdr_info = (config_cdr_data *) data;

	/* first get no of available scsi cd-r devices */
	if (NULL == (fd = fopen("/proc/scsi/scsi", "r")))
		return;
	fgets(line1, BUF_S, fd);
	while (1) {
		if (NULL == fgets(line1, BUF_S, fd))
			break;
		if (NULL == fgets(line2, BUF_S, fd))
			break;
		if (NULL == fgets(line3, BUF_S, fd))
			break;
		if (strstr(line3, "CD-ROM") || strstr(line3, "WORM"))
			no_of_cd_dev++;
	}
	fclose(fd);
	
	if (NULL == (fd = fopen("/proc/scsi/scsi", "r")))
		return;

	fgets(line1, BUF_S, fd);
	do {
		if (NULL == fgets(line1, BUF_S, fd)) {
			fclose(fd);
			return;
		}
		if (NULL == fgets(line2, BUF_S, fd)) {
			fclose(fd);
			return;
		}
		if (NULL == fgets(line3, BUF_S, fd)) {
			fclose(fd);
			return;
		}
		if (strstr(line3, "CD-ROM") || strstr(line3, "WORM"))
			times_trough++;
	} while (times_trough < cycle_dev); /* when done lineX has the info on the device we want */
	fclose(fd);

	sptr = line1;
	if (NULL == (sptr = strstr(sptr, "Host: scsi")))
		return;
	channel = atol(&sptr[10]);

	if (NULL == (sptr = strstr(sptr, "Id")))
		return;
	id = atol(&sptr[4]);

	if (NULL == (sptr = strstr(sptr, "Lun")))
		return;
	lun = atol(&sptr[5]);
/*
	gtk_adjustment_set_value(GTK_ADJUSTMENT(info->scsibus), channel);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(info->target), id);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(info->lun), lun);
*/
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(cdr_info->scsibus), channel);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(cdr_info->target), id);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(cdr_info->lun), lun);

	cdrecord_auto_detect_swab(cdr_info);

	cycle_dev %= no_of_cd_dev;
	cycle_dev++;
}

void cdrecord_reset_drive(GtkWidget *widget, gpointer data) {
	GtkWidget *dialog_window, *label, *ok;
	config_cdr_data *cdr_info = (config_cdr_data *) data;

	if (is_running())
		return;

	dialog_window = gtk_dialog_new();
	gtk_widget_show(dialog_window);

	if (do_cdrecord_shortcommand(" -reset", cdr_info))
		label = gtk_label_new(_("cdrecord failed to reset selected drive\n"
					"(wrong scsi settings, no permission to device or\n"
					"the device did not accept the reset sequence)!"));
	else
		label = gtk_label_new(_("Device successfully resetted"));

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog_window)->vbox), label, TRUE,
			TRUE, 10);
	gtk_widget_show(label);

	ok = gtk_button_new_with_label(_("OK"));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog_window)->action_area), ok, TRUE, TRUE, 10);
	gtk_signal_connect(GTK_OBJECT(ok), "clicked",
			GTK_SIGNAL_FUNC(close_dialog), (gpointer) dialog_window);
	GTK_WIDGET_SET_FLAGS (ok, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(ok);
	gtk_widget_show(ok);

	not_running();
}
