/* The Type of Track representing a raw cd track coming directly from cdrom */

#include <gtk/gtk.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>

#include "tracks.h"
#include "precachetrack.h"
#include "preferences.h"
#include "datacopydlg.h"
#include "cddrives.h"
#include "layoutconfig.h"

/* uncomment for debugging */
/* #define DEBUG */

/* information handed to the several functions used via callback to 
 * preread a track */

typedef struct 
{
	tracks_trackinfo *dc;
	precachetrack_info *info;

	int readtrack;
	int writetrack;
	int readcount;
	char buf[BUFSIZE];
	
	void (*callback)(tracks_trackinfo*,int);
	int inputwatch;

	datacopydlg_dlginfo *dlg;
} precachetrack_copyinfo;
	

int precachetrack_openpipe(void*trackinfo)
{	
	precachetrack_info *i;
	
	i=(precachetrack_info*)((tracks_trackinfo*)trackinfo)->data;

	if (i->filedes==-1)
	  {
		  i->filedes=open(i->tempfile,O_RDONLY);
		  if (i->filedes==-1)
		      perror ("error opening data track");	
	  }
	;
	return i->filedes;
}
;

void precachetrack_closepipe(void*trackinfo)
{
	precachetrack_info *i;
	
	i=(precachetrack_info*)((tracks_trackinfo*)trackinfo)->data;
	if (i->filedes!=-1)
	  {		  
		  close(i->filedes);
		  i->filedes=-1;
	  }
	;
	
}
;

int precachetrack_tracksize(void*trackinfo)
{
	precachetrack_info *i;
	struct stat buf;
	
	i=(precachetrack_info*)((tracks_trackinfo*)trackinfo)->data;
	
	stat(i->tempfile,&buf);
		
	return buf.st_size;
}
;

int precachetrack_dataavail(void*trackinfo)
{
	precachetrack_info *i;

	i=(precachetrack_info*)((tracks_trackinfo*)trackinfo)->data;

	return (i->filedes!=-1);
}
;	

/* FIXME: add function tracks_invalidate() to tracks.h to regulate
 * the availability of the tracks located on a cdrom */
int  precachetrack_valid(void*trackinfo) 
{
	return 1;
}
;

void precachetrack_destroy(void*trackinfo)
{
	precachetrack_info *i;
	
	i=(precachetrack_info*)((tracks_trackinfo*)trackinfo)->data;
	
	if (precachetrack_dataavail(trackinfo))
	    close (i->filedes);
	
	remove(i->tempfile);
	
	/* free the client claimed by the precachetrack */
	tracks_unclaim(i->client);
	
	free(i);
}
;


void precachetrack_donecb(precachetrack_copyinfo *copy)
{
	gdk_input_remove(copy->inputwatch);
	
#ifdef DEBUG
	printf ("precachetrack_donecb: closing track %s\n",(char*)&copy->info->client->name);
#endif	
	tracks_closepipe(copy->info->client);
	if (copy->writetrack!=-1)
	    close(copy->writetrack);
	
        layoutconfig_widget_hide(copy->dlg->messagebox,"datacopydialog");
	datacopydlg_destroy(copy->dlg);

   	/* report the new track to the caller,who can now replace the
	 * precachetracks client by the precachetrack itself */
	copy->callback(copy->dc,1);   
   	/* unclaim() our local copy of the precachetrack.
	 * it should have been claimed by the caller by now so this is
	 * safe and simply decreases the users-counter by one...
	 * If the caller did not claim the track,however,it is assumed
	 * that it has thought better of it and does not need that track
	 * any longer - the structure will hence be destroyed,including
	 * the buffer containing the precached data */
	tracks_unclaim(copy->dc);
	free(copy);
}
;

void precachetrack_stopcb(GtkWidget *w,precachetrack_copyinfo *copy)
{
	gdk_input_remove(copy->inputwatch);

#ifdef DEBUG
	printf ("precachetrack_stopcb: closing track %s\n",(char*)&copy->info->client->name);
#endif
	tracks_closepipe(copy->info->client);
	if (copy->writetrack!=-1)
	    close(copy->writetrack);

        layoutconfig_widget_hide(copy->dlg->messagebox,"datacopydialog");
	datacopydlg_destroy(copy->dlg);
	
	/* degrade precache requirements of current track */
	if (copy->info->client->precacherequired>0)
	    copy->info->client->precacherequired--;

   	/* return NULL instead of the new precachetrack and 0 to
	 * show that the operation was not successful */
	copy->callback(NULL,0);
	
	/* free the incomplete precache track,
	 * this does also unclaim() the client track for the precaching
	 * routines */
	tracks_unclaim(copy->dc);

	free(copy);
}
;	

void precachetrack_newdatacb(gpointer data,gint source,
			     GdkInputCondition condition)
{
	precachetrack_copyinfo *copy;
	
	copy=(precachetrack_copyinfo*)data;

	
	copy->readcount=read(copy->readtrack,&copy->buf,BUFSIZE);
	
	/* finish precaching when no more data could be read */
	if (copy->readcount>0)
	  {
		  write(copy->writetrack,&copy->buf,copy->readcount);
		  
		  datacopydlg_newdatablock(copy->dlg,copy->readcount);		  			    
	  }
	else
	    precachetrack_donecb(copy);
};		   
	


tracks_trackinfo *precachetrack_create(tracks_trackinfo *client,
				       void (*callback)(tracks_trackinfo*,int)
				       )
{
	char copyname[1024];
	
	precachetrack_copyinfo *copyinfo;
	copyinfo=(precachetrack_copyinfo*)malloc(sizeof(precachetrack_copyinfo));
	#ifdef DEBUG
	    printf ("precachetrack: allocated preachetrack_copyinfo at %p\n",
		    copyinfo);
	#endif
	
	copyinfo->info=(precachetrack_info*)malloc(sizeof(precachetrack_info));

	copyinfo->info->tempfile=tempnam(TEMPDIR,"tcache");	
	copyinfo->info->client=client;	
	copyinfo->info->filedes=-1;
	copyinfo->callback=callback;

	/* claim the client */
	tracks_claim(client);

	strcpy((char*)&copyname,(char*)&copyinfo->info->client->name);
	strcat((char*)&copyname," (precached)");	
	copyinfo->dc=tracks_create((char*)&copyname,
				   copyinfo->info->client->tracktype,none,
				   precachetrack_openpipe,
				   precachetrack_closepipe,
				   precachetrack_tracksize,
				   precachetrack_dataavail,
				   precachetrack_valid,
				   precachetrack_destroy,
				   copyinfo->info);

	copyinfo->dlg=datacopydlg_create("precaching file...",
					 GTK_SIGNAL_FUNC(precachetrack_stopcb),
					 copyinfo,
					 1, /* only one "thread" */
					 
					 DATACOPYDLG_SHOWTHROUGHPUT|
					 DATACOPYDLG_SHOWTIME_REMAINING|
					 DATACOPYDLG_SHOWTIME_ELAPSED|
					 DATACOPYDLG_SHOWPROGRESS|
					 DATACOPYDLG_SHOWPROGRESSINTITLE,
					 "",
					 "",					 
					 tracks_tracksize(client)

					 );
        layoutconfig_widget_show(copyinfo->dlg->messagebox,"datacopydialog");
   
	
 	copyinfo->writetrack=open(copyinfo->info->tempfile,O_CREAT|O_RDWR|O_TRUNC,0600);
#ifdef DEBUG
	printf ("opening track %s for precaching,mem location %p.\n",
		(char*)&copyinfo->info->client->name,
		copyinfo->info->client);
#endif
	copyinfo->readtrack=tracks_openpipe(copyinfo->info->client);
	/* wait for the permission to read at first */
	copyinfo->inputwatch=gdk_input_add(copyinfo->readtrack,
					   GDK_INPUT_READ,
					   precachetrack_newdatacb,
					   copyinfo);
		  	
	return copyinfo->dc;

}
;

	
