#!/usr/local/bin/perl
#
# This script is used to take "CSV" output from Sun Microsystems' IMPACT
# tool (really Tab separated, but Impact calls it CSV) and import it into
# the Pilot Address book.
#
# This tool communicates with the pilot address book using the pre-beta
# version of SyncAB included in this directory.  Before you can use this
# script you must setup SyncAB.
# Initial setup:
# 1. Copy SyncAB.pm from contrib/alan-harder into the main pilotmgr directory.
# 2. Run PilotManager, activate and configure SyncAB.
# 3. Configure SyncAB to use CSV format.
# 4. Do a sync.  This will copy all your pilot addresses onto your desktop.
#    Now you are ready to use ImpactImport.
# 
# How to use this script:
# 1. Make sure the path on the first line of this file points to perl5.
# 2. Use Impact and save a "CSV" file output (really Tab separated).
# 3. Follow instructions below to configure this script to match for
#    format of your output from Impact.
# 4. Execute ImpactImport.pl [-c <category>] [<impact_file>]
#    If no impact_file is specified, script reads from standard input.
#    Default category for new records is Unfiled.
#

# This is the mapping of the Impact fields (in order) to Pilot fields.
# You can edit this list using the field labels listed below in $CSVorder.
#
@fieldMap = (
		'lastname',
		'firstname',
		'title',
		'company',
		'phone1',
		'phone2',
		'address',
		'zip',
		'city'
	    );

# These are the phone labels to use for fields phone1-phone5.  Enter the
# phone types in order here.. the default (english) numbers are
# 0=Work 1=Home 2=Fax 3=Other 4=Email 5=Main 6=Pager 7=Mobile
#
$phoneTypes = '0 1 2 3 4';

###########################################################################
# Order of fields in SyncAB CSV file - do not edit!
#
$CSVorder = [ 'lastname', 'firstname', 'company',
	      'phone1', 'phone2', 'phone3', 'phone4', 'phone5',
	      'address', 'city', 'state', 'zip', 'country', 'title',
	      'custom1', 'custom2', 'custom3', 'custom4', 'note',
	      'whichphone', 'phonetypes', 'category' ];

# Read PilotManager SyncAB prefs to find output filename
#
eval `cat $ENV{HOME}/.pilotmgr/SyncAB/SyncAB.prefs`;
$outfile = defined $PREFS->{'CSVFile'} ? $PREFS->{'CSVFile'}
				       : "$ENV{HOME}/.csvAddr";

# Find ID for next new record
#
$id = 0;
if (open(FD, "<$outfile"))
{
    while (<FD>)
    {
	$_ =~ s/,.*$//;
	$id = $_+1 if ($_ >= $id);
    }
    close(FD);
}

# Check for category for filing new records, default=Unfiled
#
$defaultCategory = 0;
$catName = 'Unfiled';
if (@ARGV && $ARGV[0] eq '-c')
{
    shift;
    $cat = shift;
    if (open(FD, "<$ENV{HOME}/.pilotmgr/SyncAB/pilot.appinfo"))
    {
	<FD>;
	for ($i = 0; <FD>; $i++)
	{
	    chomp;
	    if (/^$cat$/i)
	    {
		$catName = $_;
		$defaultCategory = $i;
		last;
	    }
	}
	close(FD);
    }
}
elsif (@ARGV && $ARGV[0] eq '-h')
{
    print "Usage: $0 [-c <category>] [<impact_csv_file>]\n",
	  "  Category for new records defaults to Unfiled.\n",
	  "  If no Impact file is specified, reads from standard input.\n";
    exit;
}

# Open Impact file, or STDIN if none specified
#
if (@ARGV)
{
    open(IF, "<$ARGV[0]") || die "Unable to open input file: $ARGV[0]\n";
}
else
{
    open(IF, "<&STDIN");
}

# Open output file
#
open(OF, ">>$outfile") || die "Unable to open output file: $outfile\n";

# Read input, write output
#
while (<IF>)
{
    chomp;
    @fields = split("\t", $_);
    $rec = { 'whichphone' => 0, 'phonetypes' => $phoneTypes,
	     'category' => $defaultCategory };

    foreach $i ($[..$#fieldMap)
    {
	$rec->{$fieldMap[$i]} = $fields[$i];
    }

    print OF $id++;
    foreach $fld (@$CSVorder)
    {
	print OF ',';
	print OF &StrToCSV($rec->{$fld}) if defined $rec->{$fld};
    }
    print OF "\n";
}
close(IF);
close(OF);

print "Done.\n" if $DEBUG;



sub StrToCSV
{
    my ($str) = @_;

    $str =~ s/(\\*)(n|\n)/'\\' x (2*length($1)) . ($2 eq 'n' ? 'n' : '\\n')/ge;
    if ($str =~ /[,"]/)
    {
	$str =~ s/"/""/g;
	$str = '"' . $str . '"';
    }

    return $str;
}

