/*
 *  Driver for Trident 4DWave DX/NX AudioPCI soundcard
 *  audio@tridentmicro.com
 *  Fri Feb 19 15:55:28 MST 1999
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/trident.h"
#include "../include/initval.h"

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_dac_frame_size_dx[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 64};
int snd_adc_frame_size_dx[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 64};
int snd_dac_frame_size_nx[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
int snd_adc_frame_size_nx[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 64};
int snd_pcm_channels[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 8};
int snd_wavetable_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 2048};
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for Trident 4DWave PCI soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for Trident 4DWave PCI soundcard.");
MODULE_PARM(snd_dac_frame_size_dx, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dac_frame_size_dx, "DAC frame size in kB for Trident 4DWave DX PCI soundcard.");
MODULE_PARM(snd_adc_frame_size_dx, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_adc_frame_size_dx, "ADC frame size in kB for Trident 4DWave DX PCI soundcard.");
MODULE_PARM(snd_dac_frame_size_nx, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dac_frame_size_nx, "DAC frame size in kB for Trident 4DWave NX PCI soundcard.");
MODULE_PARM(snd_adc_frame_size_nx, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_adc_frame_size_nx, "ADC frame size in kB for Trident 4DWave NX PCI soundcard.");
MODULE_PARM(snd_pcm_channels, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_pcm_channels, "Number of hardware channels assigned for PCM.");
MODULE_PARM(snd_wavetable_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_wavetable_size, "Maximum memory size in kB for wavetable synth.");

struct snd_trident {
	struct pci_dev *pci;
	snd_irq_t *irqptr;
	snd_dma_t *dma1ptr;	/* DAC frame */
	snd_dma_t *dma2ptr;	/* ADC frame */
	snd_card_t *card;
	trident_t *trident;
	snd_pcm_t *pcm;		/* ADC/DAC */
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
};

static struct snd_trident *snd_trident_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_trident_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_trident_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static int snd_trident_detect(snd_card_t * card, struct snd_trident *scard,
			      unsigned short device)
{
	if ((scard->pci = pci_find_device(PCI_VENDOR_ID_TRIDENT,
					  device,
					  scard->pci)) == NULL)
		return -ENODEV;
#ifdef NEW_PCI
	if (snd_register_ioport(card, scard->pci->resource[0].start, 0x100, "Trident 4DWave PCI", NULL) < 0)
#else
	if (snd_register_ioport(card, scard->pci->base_address[0] & ~PCI_BASE_ADDRESS_SPACE, 0x100, "Trident 4DWave PCI", NULL) < 0)
#endif
		goto __nodev;
	return 0;
      __nodev:
	snd_unregister_ioports(card);
	return -ENODEV;
}

static void snd_trid_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_trident *scard = (struct snd_trident *) dev_id;

	if (!scard || !scard->trident)
		return;
	snd_trident_interrupt(scard->trident);
}

static int snd_trident_resources(snd_card_t * card,
				 struct snd_trident *scard, int dev)
{
	int err;
	int dacSize;
	int adcSize;

	if (card->type == SND_CARD_TYPE_TRID4DWAVEDX) {
		dacSize = snd_dac_frame_size_dx[dev];
		adcSize = snd_adc_frame_size_dx[dev];
	} else {
		dacSize = snd_dac_frame_size_nx[dev];
		adcSize = snd_adc_frame_size_nx[dev];
	}

	if ((err = snd_register_interrupt(card,
				   "Trident 4DWave PCI", scard->pci->irq,
				    SND_IRQ_TYPE_PCI, snd_trid_interrupt,
				       scard, NULL, &scard->irqptr)) < 0)
		return err;

	if ((err = snd_register_dma_channel(card,
				     "Trident 4DWave PCI - DAC frame", 0,
					    TRIDENT_DMA_TYPE, dacSize,
					    NULL, &scard->dma1ptr)) < 0)
		return err;
	scard->dma1ptr->multi = 1;	/* enable multi-alloc */

	if ((err = snd_register_dma_channel(card,
				     "Trident 4DWave PCI - ADC frame", 1,
					    TRIDENT_DMA_TYPE, adcSize,
					    NULL, &scard->dma2ptr)) < 0)
		return err;

	return 0;
}

static int snd_trident_probe(int dev, struct snd_trident *scard)
{
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
	snd_rawmidi_t *rmidi = NULL;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_trident_use_inc, snd_trident_use_dec);
	if (!card)
		return -ENOMEM;
	card->static_data = scard;
	scard->pci = NULL;
	do {
		if (!snd_trident_detect(card, scard, PCI_DEVICE_ID_TRIDENT_4DWAVE_DX)) {
			card->type = SND_CARD_TYPE_TRID4DWAVEDX;
			break;
		}
	} while (scard->pci);
	if (scard->pci == NULL) {
		do {
			if (!snd_trident_detect(card, scard, PCI_DEVICE_ID_TRIDENT_4DWAVE_NX)) {
				card->type = SND_CARD_TYPE_TRID4DWAVENX;
				break;
			}
		} while (scard->pci);
	}
	if (scard->pci == NULL) {
		snd_card_free(card);
		return -ENODEV;
	}
	if (snd_trident_resources(card, scard, dev) < 0) {
		snd_card_free(card);
		return -ENODEV;
	}
	scard->trident = snd_trident_create(card, scard->pci,
					    scard->dma1ptr, scard->dma2ptr,
					    scard->irqptr, snd_pcm_channels[dev], 
					    snd_wavetable_size[dev]);
	if (!scard->trident)
		goto __nodev;
	mixer = snd_trident_mixer(scard->trident);
	if (!mixer)
		goto __nodev;
	pcm = snd_trident_pcm(scard->trident);
	if (!pcm)
		goto __nodev;
	rmidi = snd_mpu401_uart_new_device(card, MPU401_HW_TRID4DWAVE,
					   scard->trident->midi_port,
					   scard->irqptr->irq);
	if (!rmidi)
		goto __nodev;
	snd_trident_rawmidi(scard->trident, (mpu401_t *) rmidi->private_data);

#ifdef CONFIG_SND_SEQUENCER
	if (snd_trident_attach_synthesizer(scard->trident) < 0) {
		//if (midi_uart)
		//	snd_rawmidi_unregister(midi_uart);
		//midi_uart = NULL;
		goto __nodev;
	}
#endif

	if (snd_mixer_register(mixer, 0) < 0) {
#ifdef CONFIG_SND_SEQUENCER
		snd_trident_detach_synthesizer(scard->trident);
#endif
		goto __nodev;
	}
	if (snd_pcm_register(pcm, 0) < 0) {
#ifdef CONFIG_SND_SEQUENCER
		snd_trident_detach_synthesizer(scard->trident);
#endif
		snd_mixer_unregister(mixer);
		mixer = NULL;
		goto __nodev;
	}
	if (snd_rawmidi_register(rmidi, 0) < 0) {
#ifdef CONFIG_SND_SEQUENCER
		snd_trident_detach_synthesizer(scard->trident);
#endif
		snd_pcm_unregister(pcm);
		pcm = NULL;
		snd_mixer_unregister(mixer);
		mixer = NULL;
		goto __nodev;
	}
	strcpy(card->abbreviation, "4DWave");
	strcpy(card->shortname, "Trident 4DWave PCI");
	sprintf(card->longname, "%s %s at 0x%lx, irq %i",
		card->shortname,
		card->type == SND_CARD_TYPE_TRID4DWAVEDX ? "4DWAVEDX" : "4DWAVENX",
#ifdef NEW_PCI
		scard->pci->resource[0].start,
#else
		scard->pci->base_address[0] & ~PCI_BASE_ADDRESS_SPACE,
#endif
		scard->irqptr->irq);

	if (!snd_card_register(card)) {
		scard->card = card;
		scard->mixer = mixer;
		scard->pcm = pcm;
		scard->rmidi = scard->trident->rmidi = rmidi;
		return 0;
	}
	goto __nodev;

      __nodev:
	if (mixer)
		snd_mixer_free(mixer);
	if (pcm)
		snd_pcm_free(pcm);
	if (rmidi)
		snd_rawmidi_free(rmidi);
	if (scard->trident) {
		snd_trident_free(scard->trident);
		scard->trident = NULL;
	}
	snd_card_free(card);
	return -ENXIO;
}

static int snd_trid_free(int dev)
{
	struct snd_trident *scard;
	snd_pcm_t *pcm;

	scard = snd_trident_cards[dev];
	snd_trident_cards[dev] = NULL;
	if (scard) {
		snd_card_unregister(scard->card);
#ifdef CONFIG_SND_SEQUENCER
		snd_trident_detach_synthesizer(scard->trident);
#endif
		if (scard->rmidi) {
			scard->trident->rmidi = NULL;
			snd_rawmidi_unregister(scard->rmidi);
		}
		if (scard->mixer) {
			snd_mixer_unregister(scard->mixer);
		}
		if (scard->pcm) {
			pcm = scard->pcm;
			scard->pcm = NULL;	/* turn off interrupts */
			snd_pcm_unregister(pcm);
		}
		if (scard->trident) {
			snd_trident_free(scard->trident);
			scard->trident = NULL;
		}
		snd_card_free(scard->card);
		snd_kfree(scard);
	}
	return 0;
}

int init_module(void)
{
	int dev, cards;
	struct snd_trident *scard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		scard = (struct snd_trident *)
		    snd_kcalloc(sizeof(struct snd_trident), GFP_KERNEL);
		if (!scard)
			continue;
		if (snd_trident_probe(dev, scard) < 0) {
			snd_kfree(scard);
			break;
		}
		snd_trident_cards[dev] = scard;
		cards++;
	}
	if (!cards) {
		snd_printk("Trident 4DWave PCI soundcard #%i not found or device busy\n", dev + 1);
		return -ENODEV;
	}
	return 0;
}

void cleanup_module(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_trid_free(dev);
}
