/*
 * OSS compatible sequencer driver
 *
 * OSS compatible i/o control
 *
 * Copyright (C) 1998,99 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../../../include/driver.h"
#include "seq_oss_device.h"
#include "seq_oss_readq.h"
#include "seq_oss_writeq.h"
#include "seq_oss_timer.h"
#include "seq_oss_synth.h"
#include "seq_oss_midi.h"
#include "seq_oss_event.h"

int
snd_seq_oss_ioctl(seq_oss_devinfo_t *dp, unsigned int cmd, unsigned long carg)
{
	int dev, val;
	oss_synth_info_t inf;
	oss_midi_info_t minf;
	unsigned char ev[8];
	void *arg = (void*)carg;
	snd_seq_event_t tmpev;

	switch (cmd) {
	case SND_OSS_SNDCTL_TMR_TIMEBASE:
	case SND_OSS_SNDCTL_TMR_TEMPO:
	case SND_OSS_SNDCTL_TMR_START:
	case SND_OSS_SNDCTL_TMR_STOP:
	case SND_OSS_SNDCTL_TMR_CONTINUE:
	case SND_OSS_SNDCTL_TMR_METRONOME:
	case SND_OSS_SNDCTL_TMR_SOURCE:
	case SND_OSS_SNDCTL_TMR_SELECT:
		debug_printk(("seq_oss_ioctl: timer\n"));
		/* NOP */
		return 0;

	case SND_OSS_SNDCTL_SEQ_PANIC:
		debug_printk(("seq_oss_ioctl: panic\n"));
		snd_seq_oss_reset(dp);
		return -EINVAL;

	case SND_OSS_SNDCTL_SEQ_SYNC:
		debug_printk(("seq_oss_ioctl: sync\n"));
		if (! is_write_mode(dp->file_mode) || ! dp->writeq)
			return 0;
		while (snd_seq_oss_writeq_sync(dp->writeq))
			;
		if (signal_pending(current))
			return -EINTR;
		return 0;

	case SND_OSS_SNDCTL_SEQ_RESET:
		debug_printk(("seq_oss_ioctl: reset\n"));
		snd_seq_oss_reset(dp);
		return 0;

	case SND_OSS_SNDCTL_SEQ_TESTMIDI:
		debug_printk(("seq_oss_ioctl: test midi\n"));
		dev = snd_ioctl_in((long*)arg);
		return snd_seq_oss_midi_open(dp, dev, dp->file_mode);

	case SND_OSS_SNDCTL_SEQ_GETINCOUNT:
		debug_printk(("seq_oss_ioctl: get in count\n"));
		if (!dp->readq || ! is_read_mode(dp->file_mode))
			return 0;
		return snd_ioctl_out((long*)arg, dp->readq->qlen);

	case SND_OSS_SNDCTL_SEQ_GETOUTCOUNT:
		debug_printk(("seq_oss_ioctl: get out count\n"));
		if (! is_write_mode(dp->file_mode) || ! dp->writeq)
			return 0;
		return snd_ioctl_out((long*)arg, snd_seq_oss_writeq_get_free_size(dp->writeq));

	case SND_OSS_SNDCTL_SEQ_GETTIME:
		debug_printk(("seq_oss_ioctl: get time\n"));
		return snd_ioctl_out((long*)arg, snd_seq_oss_timer_get_tick(dp->timer));

	case SND_OSS_SNDCTL_SEQ_CTRLRATE:
		debug_printk(("seq_oss_ioctl: ctrl rate\n"));
		/* if *arg == 0, just return the current rate */
		if (snd_ioctl_in((long*)arg))
			return -EINVAL;
		return snd_ioctl_out((long*)arg, SND_SEQ_OSS_CTRLRATE);

	case SND_OSS_SNDCTL_SEQ_RESETSAMPLES:
		debug_printk(("seq_oss_ioctl: reset samples\n"));
		dev = snd_ioctl_in((long*)arg);
		return snd_seq_oss_synth_ioctl(dp, dev, cmd, carg);

	case SND_OSS_SNDCTL_SEQ_NRSYNTHS:
		debug_printk(("seq_oss_ioctl: nr synths\n"));
		return snd_ioctl_out((long*)arg, dp->max_synthdev);

	case SND_OSS_SNDCTL_SEQ_NRMIDIS:
		debug_printk(("seq_oss_ioctl: nr midis\n"));
		return snd_ioctl_out((long*)arg, dp->max_mididev);

	case SND_OSS_SNDCTL_SYNTH_MEMAVL:
		debug_printk(("seq_oss_ioctl: mem avail\n"));
		dev = snd_ioctl_in((long*)arg);
		val = snd_seq_oss_synth_ioctl(dp, dev, cmd, carg);
		return snd_ioctl_out((long*)arg, val);

	case SND_OSS_SNDCTL_FM_4OP_ENABLE:
		debug_printk(("seq_oss_ioctl: 4op\n"));
		dev = snd_ioctl_in((long*)arg);
		snd_seq_oss_synth_ioctl(dp, dev, cmd, carg);
		return 0;

	case SND_OSS_SNDCTL_SYNTH_INFO:
	case SND_OSS_SNDCTL_SYNTH_ID:
		debug_printk(("seq_oss_ioctl: synth info\n"));
		if (copy_from_user(&inf, arg, sizeof(inf)))
			return -EFAULT;
		if (snd_seq_oss_synth_make_info(dp, inf.device, &inf) < 0)
			return -EINVAL;
		if (copy_to_user(arg, &inf, sizeof(inf)))
			return -EFAULT;
		return 0;

	case SND_OSS_SNDCTL_SEQ_OUTOFBAND:
		debug_printk(("seq_oss_ioctl: out of bound\n"));
		if (copy_from_user(ev, arg, 8))
			return -EFAULT;
		memset(&tmpev, 0, sizeof(tmpev));
		snd_seq_oss_fill_addr(dp, &tmpev, dp->addr.port, dp->addr.client);
		tmpev.time.tick = 0;
		if (! snd_seq_oss_process_event(dp, (evrec_t*)ev, &tmpev)) {
			snd_seq_oss_dispatch(dp, &tmpev, 0, 0);
		}
		return 0;

	case SND_OSS_SNDCTL_MIDI_INFO:
		debug_printk(("seq_oss_ioctl: midi info\n"));
		if (copy_from_user(&minf, arg, sizeof(minf)))
			return -EFAULT;
		if (snd_seq_oss_midi_make_info(dp, minf.device, &minf) < 0)
			return -EINVAL;
		if (copy_to_user(arg, &minf, sizeof(minf)))
			return -EFAULT;
		return 0;

	case SND_OSS_SNDCTL_SEQ_THRESHOLD:
		debug_printk(("seq_oss_ioctl: threshold\n"));
		if (! is_write_mode(dp->file_mode))
			return 0;
		val = snd_ioctl_in((long*)arg);
		if (val < 1)
			val = 1;
		if (val >= dp->writeq->maxlen)
			val = dp->writeq->maxlen - 1;
		snd_seq_oss_writeq_set_output(dp->writeq, val);
		return 0;

	case SND_OSS_SNDCTL_MIDI_PRETIME:
		debug_printk(("seq_oss_ioctl: pretime\n"));
		if (!dp->readq || !is_read_mode(dp->file_mode))
			return 0;
		val = snd_ioctl_in((long*)arg);
		if (val <= 0)
			val = -1;
		else
			val = (HZ * val) / 10;
		dp->readq->pre_event_timeout = val;
		return snd_ioctl_out((long*)arg, val);

	default:
		debug_printk(("seq_oss_ioctl: others\n"));
		if (! is_write_mode(dp->file_mode))
			return -EIO;
		return snd_seq_oss_synth_ioctl(dp, 0, cmd, carg);
	}
	return 0;
}

