// textinput.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/button.h>
#include <InterViews/shape.h>
#include <InterViews/textdisplay.h>
#include <InterViews/world.h>
#include <string.h>
#include "localdefs.h"
#include "textinput.h"
#include "queryvalue.h"

TextInput::TextInput(ButtonState* s, const char* samp, QueryValue *value)
		: StringEditor(s, samp, "\007\015"), myValue(value) {
	initialize();
}

TextInput::TextInput(const char* name, ButtonState* s, const char* samp,
	QueryValue *value) : StringEditor(s, samp, "\007\015"), myValue(value) {
	SetInstance(name);
	initialize();
}

void
TextInput::initialize() {
	head = this;
	next = this;
	SetClassName("TextInput");
	setText();
}

// in response to Update() message from its buttonstate, set the QueryValue
// from the text, and then retrieve new text from QueryValue

void
TextInput::Update() {
	Super::Update();
	int stateValue;
	subject->GetValue(stateValue);
	if(stateValue == Yes)
		doFinish();
}

// this is redefined in subclasses to allow string value to be computer from
// sources other than the QueryValue instance

const char *
TextInput::stringValue() {
	return value()->get();		// set string from QueryValue string
}

// this is redefined in subclasses to constrain the value passed to the 
// QueryValue object

int
TextInput::setCurrent(const char* text) {
	return value()->set(text) == nil;
}

void
TextInput::setText() {
	Super::Message(stringValue());
}

boolean
TextInput::setValue() { return value()->set(Text()) == nil; }

void
TextInput::Reconfig() {
	Super::Reconfig();
	Shape s = *GetShape();
	s.Rigid();
	Reshape(s);
}

boolean
TextInput::valueChanged() {
	return (strcmp(Text(), value()->get()) != 0);
}

void
TextInput::doFinish() {
	if(valueChanged()) {
		setCurrent(Text());
		setText();
		setValue();
	}
}

void
TextInput::selectNext() {
	Select(strlen(Text()));		// to remove highlighting if present
	display->CaretStyle(NoCaret);
	next->Edit();
}

boolean
TextInput::HandleChar (char c) {
	register boolean retval = false;
	if(c == '\t') {		// for <tab> char
		doFinish();
		// switch to next TextInput if it exists
		if(next != this) selectNext();
		return true;
	}
	else if(checkChar(c)) {
		retval = Super::HandleChar(c);
		if (strchr(done, c) != nil)	// user finished with this item
			doFinish();
	}
	else {
		GetWorld()->RingBell(50);
	}
	return retval;
}

boolean
TextInput::checkChar(char c) {
	return value()->checkCharacter(c);
}

TextInput *
TextInput::tail() {
	TextInput *t = head;
	do {
		t = t->next;
	} while(t->next != head);
	return t;
}

void
TextInput::appendTo(TextInput *t) {
	tail()->next = t;
	t->next = t->head = head;
}
