// wavheader.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author 	
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "datafile.h"
#include "valuesetter.h"
#include "wavheader.h"

WAVSoundHeader::WAVSoundHeader(DataType format, int rate, int chans)
	: SoundHeader(format, rate, chans, 0, 0),
		sampleFormat(FormatStruct::WAVE_FORMAT_PCM) {
	header_type = Wave;
}

boolean
WAVSoundHeader::isMagic() {
	return (magic() == RIFF || magic() == FFIR);
}

// this routine is biased towards bigendian, since "RIFF" in chars is equal
// to the value in the macro FFIR when on bigendian machines

boolean
WAVSoundHeader::magicIsSwapped() {
	return magic() == RIFF || needsByteSwapping();
}

int
WAVSoundHeader::readInfo(DataFile* file) {
	file->seek(0);
	int status = false;
	RIFFChunk rchunk;
	if(rchunk.read(file)) {
		if(rchunk.isValid()) {
			long totalLength = rchunk.size();
			StandardChunk chunk;
			long chunkSize = 0;
			do {		// search for format chunk
				status = file->skip(chunkSize).good() && chunk.read(file);
				chunkSize = chunk.size();
			} while(status && !chunk.isA("fmt "));
			if(!status)	// reached EOF
				Application::alert("readInfo:  missing format chunk");
			else {		// found it -- now read format information
				FormatStruct fstruct;
				if(status = fstruct.read(file))
					readFmt(fstruct);
				// skip if FMT chunksize != total data size of FormatStruct
				chunkSize = chunkSize - fstruct.readSize();
				do {		// search for data chunk
					status = file->skip(chunkSize).good() && chunk.read(file);
					chunkSize = chunk.size();
				} while(status && !chunk.isA("data"));
				if(!status)	// reached EOF
					Application::alert("readInfo:  missing data chunk");
				else {		// found it -- now finish up
					setDataOffset(int(file->tell()));
					long actualDataSize = file->dataSize() - dataOffset();
					if(actualDataSize < chunkSize) { // warn if file truncated
						char msg[120];
						sprintf(msg,
							"File data size (%ld) < header data size (%ld).",
							actualDataSize, chunkSize);
						Application::alert("Warning:  Truncated file.", msg);
						chunkSize = actualDataSize;
					}
					setDataSize(int(chunkSize));
					status = checkHeader();
				}
			}
		}
		else
			Application::alert("readInfo: Not a RIFF or not a WAVE file.");
	}
	else
		Application::error("readInfo: File read error.");
	return status;
}

void
WAVSoundHeader::readFmt(const WAVSoundHeader::FormatStruct& fmt) {
	sampleFormat = fmt.dataFormat;
	nchans = fmt.channels;
	samprate = int(fmt.sampleRate);
	switch(fmt.sampleSize) {
	case 8:
		data_type = (sampleFormat == FormatStruct::WAVE_FORMAT_PCM) ?
			UnsignedCharData : MuLawData;
		break;
	case 16:	data_type = ShortData; break;
	case 32:	data_type = IntData; break;
	default:
		data_type = NoData;
		break;
	}
}

int
WAVSoundHeader::writeInfo(DataFile *file) {
	static const long headerLength = (12 + 16 + 8);
	RIFFChunk rchunk(dataSize() + headerLength);
	int status = rchunk.write(file);
	int type = dataType();
	if(status) {
		FormatStruct fstruct(
			(type == MuLawData) ?			// sample format
				FormatStruct::WAVE_FORMAT_MULAW
				: FormatStruct::WAVE_FORMAT_PCM,
			short(nChans()),
			long(sampleRate()),
			(type==IntData) ? 32 : (type==ShortData) ? 16 : 8	// bits/samp
		);
		FormatChunk fchunk(fstruct.readSize());
		status = fchunk.write(file) && fstruct.write(file);
		if(status) {
			long dsize = dataSize();
			DataChunk dchunk(dsize);
			status = dchunk.write(file);
		}
	}
	if(status == false)
		Application::error("writeInfo:  file write error");
	return status;
}

int
WAVSoundHeader::checkHeader() {
	int status = false;
	if(sampleFormat == FormatStruct::WAVE_FORMAT_PCM
			|| sampleFormat == FormatStruct::WAVE_FORMAT_MULAW)
		status = Super::checkHeader();
	else
		Application::alert("Sorry, this program can only read and write",
			"the WAVE PCM and WAVE MULAW sample formats at this time");
	return status;
}
	
// definitions of outline nested class methods

int
WAVSoundHeader::RIFFChunk::read(DataFile* file) {
	return doRead(file, Super::valueSetters())
		&& doRead(file, valueSetters());
}

int
WAVSoundHeader::RIFFChunk::write(DataFile* file) {
	return doWrite(file, Super::valueSetters())
		&& doWrite(file, valueSetters());
}

ValueSetterBase**
WAVSoundHeader::RIFFChunk::valueSetters() {
	static ValueSetterBase* setterList[NumElements + 1];
	delete setterList[0];
	setterList[0] = newValueSetter(&typeId);
	return setterList;
}

//********

WAVSoundHeader::FormatStruct::FormatStruct(
		short fmt, short chans, long sr, short sampsize)
			: dataFormat(fmt), channels(chans),
			sampleRate(sr), sampleSize(sampsize) {
	bytesPerSecond = (sr * chans * sampleSize + 7) / 8;
	blockAlign = (chans * sampleSize + 7) / 8;
}

ValueSetterBase**
WAVSoundHeader::FormatStruct::valueSetters() {
	static ValueSetterBase* setterList[NumElements + 1];
	delete setterList[0];
	setterList[0] = newValueSetter(&dataFormat);
	delete setterList[1];
	setterList[1] = newValueSetter(&channels);
	delete setterList[2];
	setterList[2] = newValueSetter(&sampleRate);
	delete setterList[3];
	setterList[3] = newValueSetter(&bytesPerSecond);
	delete setterList[4];
	setterList[4] = newValueSetter(&blockAlign);
	delete setterList[5];
	setterList[5] = newValueSetter(&sampleSize);
	return setterList;
}
