/* Copyright (c) 1997-1999 Miller Puckette.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

/* Pd side of the Pd/Pd-gui interface. */

#include "m_imp.h"
#ifdef UNIX
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#ifndef __linux__
#include <bstring.h>
#endif
#include <sys/time.h>
#endif
#ifdef NT
#include <io.h>
#include <fcntl.h>
#include <process.h>
#include <winsock.h>
typedef int pid_t;
#define EADDRINUSE WSAEADDRINUSE
#endif
#include <stdarg.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>

extern char pd_version[];

typedef struct _fdpoll
{
    int fdp_fd;
    t_fdpollfn fdp_fn;
    void *fdp_ptr;
} t_fdpoll;

#define INBUFSIZE 4096
#define OUTBUFSIZE PIPE_BUF
#define OUTBUFMESSADVISE 1024
#define OUTBUFMESSOVERFLOW 1024
#define OUTBUFTOT (OUTBUFSIZE + OUTBUFMESSOVERFLOW)

struct _socketreceiver
{
    char *sr_inbuf;
    int sr_inhead;
    int sr_intail;
    void *sr_owner;
    t_socketnotifier sr_notifier;
};

static int sys_nfdpoll;
static t_fdpoll *sys_fdpoll;
static int sys_maxfd;
static int sys_guisock;

static t_binbuf *inbinbuf;
static t_socketreceiver *sys_socketreceiver;
extern int sys_addhist(int phase);

void sys_sockerror(char *s)
{
#ifdef NT
    int err = WSAGetLastError();
    if (err == 10054) return;
#endif
#ifdef UNIX
    int err = errno;
#endif
    fprintf(stderr, "%s: %s (%d)\n", s, strerror(err), err);
}

void sys_addpollfn(int fd, t_fdpollfn fn, void *ptr)
{
    int nfd = sys_nfdpoll;
    int size = nfd * sizeof(t_fdpoll);
    t_fdpoll *fp;
    sys_fdpoll = (t_fdpoll *)t_resizebytes(sys_fdpoll, size,
    	size + sizeof(t_fdpoll));
    fp = sys_fdpoll + nfd;
    fp->fdp_fd = fd;
    fp->fdp_fn = fn;
    fp->fdp_ptr = ptr;
    sys_nfdpoll = nfd + 1;
    if (fd >= sys_maxfd) sys_maxfd = fd + 1;
}

void sys_rmpollfn(int fd)
{
    int nfd = sys_nfdpoll;
    int i, size = nfd * sizeof(t_fdpoll);
    t_fdpoll *fp;
    for (i = nfd, fp = sys_fdpoll; i--; fp++)
    {
    	if (fp->fdp_fd == fd)
    	{
    	    while (i--)
    	    {
    	    	fp[0] = fp[1];
    	    	fp++;
    	    }
	    sys_fdpoll = (t_fdpoll *)t_resizebytes(sys_fdpoll, size,
    		size - sizeof(t_fdpoll));
	    sys_nfdpoll = nfd - 1;
	    return;
    	}
    }
    post("warning: %d removed from poll list but not found", fd);
}

static int sys_domicrosleep(int microsec, int pollem)
{
    struct timeval timout;
    int i, didsomething = 0;
    t_fdpoll *fp;
    timout.tv_sec = 0;
    timout.tv_usec = microsec;
    if (pollem)
    {
    	fd_set readset, writeset, exceptset;
	FD_ZERO(&writeset);
	FD_ZERO(&readset);
	FD_ZERO(&exceptset);
    	for (fp = sys_fdpoll, i = sys_nfdpoll; i--; fp++)
    	    FD_SET(fp->fdp_fd, &readset);
    	select(sys_maxfd+1, &readset, &writeset, &exceptset, &timout);
	for (i = 0; i < sys_nfdpoll; i++)
    	    if (FD_ISSET(sys_fdpoll[i].fdp_fd, &readset))
	{
    	    (*sys_fdpoll[i].fdp_fn)(sys_fdpoll[i].fdp_ptr, sys_fdpoll[i].fdp_fd);
    	    didsomething = 1;
	}  
	return (didsomething);
    }
    else
    {
    	select(0, 0, 0, 0, &timout);
	return (0);
    }
}
static double sys_lasttimeslept;

void sys_microsleep(int microsec)
{
    sys_lasttimeslept = sys_getrealtime();
    sys_domicrosleep(microsec, 1);
}

int sys_usecsincelastsleep(void)
{
    return ((sys_getrealtime() - sys_lasttimeslept) * 1000000);
}
t_socketreceiver *socketreceiver_new(void *owner, t_socketnotifier notifier)
{
    t_socketreceiver *x = (t_socketreceiver *)getbytes(sizeof(*x));
    x->sr_inhead = x->sr_intail = 0;
    x->sr_owner = owner;
    x->sr_notifier = notifier;
    if (!(x->sr_inbuf = malloc(INBUFSIZE))) bug("t_socketreceiver");;
    return (x);
}

void socketreceiver_free(t_socketreceiver *x)
{
    free(x->sr_inbuf);
    freebytes(x, sizeof(*x));
}

    /* this is in a separately called subroutine so that the buffer isn't
    sitting on the stack while the messages are getting passed. */
static int socketreceiver_doread(t_socketreceiver *x)
{
    char messbuf[INBUFSIZE], *bp = messbuf;
    int indx;
    int inhead = x->sr_inhead;
    int intail = x->sr_intail;
    char *inbuf = x->sr_inbuf;
    if (intail == inhead) return (0);
    for (indx = intail; indx != inhead; indx = (indx+1)&(INBUFSIZE-1))
    {
    	    /* if we hit a semi that isn't preceeded by a \, it's a message
    	    boundary.  LATER we should deal with the possibility that the
    	    preceeding \ might itself be escaped! */
    	char c = *bp++ = inbuf[indx];
    	if (c == ';' && (!indx || inbuf[indx-1] != '\\'))
    	{
    	    intail = (indx+1)&(INBUFSIZE-1);
    	    binbuf_text(inbinbuf, messbuf, bp - messbuf);
    	    if (sys_debuglevel & DEBUG_MESSDOWN)
    	    {
    	    	write(2,  messbuf, bp - messbuf);
    	    	write(2, "\n", 1);
    	    }
    	    x->sr_inhead = inhead;
    	    x->sr_intail = intail;
    	    return (1);
    	}
    }
    return (0);
}

static void socketreceiver_getudp(int fd)
{
    char buf[INBUFSIZE];
    int ret = recv(fd, buf, INBUFSIZE, 0);
    if (ret < 0)
    {
	sys_sockerror("recv");
	sys_rmpollfn(fd);
	sys_closesocket(fd);
    }
    else if (ret > 0)
    {
    	binbuf_text(inbinbuf, buf, ret);
    	binbuf_eval(inbinbuf, 0, 0, 0);
    }
}

void socketreceiver_read(t_socketreceiver *x, int fd)
{
    if (x)  /* x nonzero: streaming protocol */
    {
	char *semi;
	int readto =
	    (x->sr_inhead >= x->sr_intail ? INBUFSIZE : x->sr_intail-1);
	int ret;

    	    /* the input buffer might be full.  If so, drop the whole thing */
	if (readto == x->sr_inhead)
	{
    	    fprintf(stderr, "pd: dropped message from gui\n");
    	    x->sr_inhead = x->sr_intail = 0;
    	    readto = INBUFSIZE;
	}
	else
	{
	    ret = recv(fd, x->sr_inbuf + x->sr_inhead,
	    	readto - x->sr_inhead, 0);
	    if (ret < 0)
	    {
		sys_sockerror("recv");
		if (x == sys_socketreceiver) sys_bail();
		else
		{
	    	    if (x->sr_notifier) (*x->sr_notifier)(x->sr_owner);
	    	    sys_rmpollfn(fd);
	    	    sys_closesocket(fd);
		}
	    }
	    else if (ret == 0)
	    {
		if (x == sys_socketreceiver)
		{
    	    	    fprintf(stderr, "pd: exiting\n");
    	    	    sys_bail();
		}
		else
		{
	    	    post("EOF on socket %d\n", fd);
	            if (x->sr_notifier) (*x->sr_notifier)(x->sr_owner);
	    	    sys_rmpollfn(fd);
	    	    sys_closesocket(fd);
		}
	    }
	    else
	    {
    		x->sr_inhead += ret;
    		if (x->sr_inhead >= INBUFSIZE) x->sr_inhead = 0;
    		while (socketreceiver_doread(x))
    		    binbuf_eval(inbinbuf, 0, 0, 0);
 	    }
	}
    }
    else socketreceiver_getudp(fd);   	/* x is zero: datagram protocol */
}

void sys_closesocket(int fd)
{
#ifdef UNIX
    close(fd);
#endif
#ifdef NT
    closesocket(fd);
#endif
}


void sys_gui(char *s)
{
    int length = strlen(s), written = 0, res, histwas = sys_addhist(4);
    if (sys_debuglevel & DEBUG_MESSUP) fprintf(stderr, "%s",  s);
    
    while (1)
    {
    	res = send(sys_guisock, s + written, length, 0);
    	if (res < 0)
    	{
    	    perror("pd output pipe");
    	    sys_bail();
    	}
    	else
    	{
    	    written += res;
    	    if (written >= length)
	    	break;
    	}
    }
    sys_addhist(histwas);
}

/* LATER should do a bounds check -- but how do you get printf to do that?
    See also rtext_senditup() in this regard */

void sys_vgui(char *fmt, ...)
{
    int result, i;
    char buf[2048];
    va_list ap;

    va_start(ap, fmt);
    vsprintf(buf, fmt, ap);
    sys_gui(buf);
    va_end(ap);
}

static pid_t childpid;

#define FIRSTPORTNUM 5400

/* -------------- signal handling for UNIX -------------- */

#ifdef UNIX
typedef void (*sighandler_t)(int);

static void sys_signal(int signo, sighandler_t sigfun)
{
    struct sigaction action;
    action.sa_flags = 0;
    action.sa_handler = sigfun;
    memset(&action.sa_mask, 0, sizeof(action.sa_mask));
#if !defined(IRIX) && !defined(__alpha__)
    action.sa_restorer = 0;
#endif
    if (sigaction(signo, &action, 0) < 0) perror("sigaction");
}

static void sys_virtualalarmhandler(int n)
{
    fprintf(stderr, "Pd cpu timeout -- sleeping\n");
    sys_microsleep(50000);
    sys_setvirtualalarm();
}

static void sys_exithandler(int n)
{
    fprintf(stderr, "Pd: signal %d\n", n);
    sys_bail();
}

void sys_setvirtualalarm( void)
{
    struct itimerval gonzo,  bonzo;
    gonzo.it_interval.tv_sec = 0;
    gonzo.it_interval.tv_usec = 0;
    gonzo.it_value.tv_sec = 0;
    gonzo.it_value.tv_usec = 500000;
    setitimer(ITIMER_VIRTUAL, &gonzo, &bonzo);
    sys_signal(SIGVTALRM, sys_virtualalarmhandler);
}

void sys_setitimer( void)
{
    signal(SIGVTALRM, sys_virtualalarmhandler);
}

#endif

int sys_startgui(const char *guipath)
{
    char cmdbuf[4*MAXPDSTRING];
    struct sockaddr_in server;
    int msgsock;
    char buf[15];
    int len = sizeof(server);
    int ntry = 0, portno = FIRSTPORTNUM;
    int xsock;
#ifdef UNIX
    int pipeto[2], pipefrom[2];
#endif

#ifdef NT
    short version = MAKEWORD(2, 0);
    WSADATA nobby;

    if (WSAStartup(version, &nobby)) sys_sockerror("WSAstartup");
#endif

    /* create an empty FD poll list */
    sys_fdpoll = (t_fdpoll *)t_getbytes(0);
    sys_nfdpoll = 0;
    
    /* create a socket */
    xsock = socket(AF_INET, SOCK_STREAM, 0);
    if (xsock < 0) sys_sockerror("socket");
    server.sin_family = AF_INET;
    server.sin_addr.s_addr = INADDR_ANY;

    /* assign server port number */
    server.sin_port = portno;

    /* name the socket */
    while (bind(xsock, (struct sockaddr *)&server, sizeof(server)) < 0)
    {
#ifdef NT
    	int err = WSAGetLastError();
#endif
#ifdef UNIX
    	int err = errno;
#endif
    	if ((ntry++ > 20) || (err != EADDRINUSE))
    	{
	    perror("binding stream socket");
	    exit(1);
    	}
    	server.sin_port = ++portno;
    }
    
    if (sys_verbose) fprintf(stderr, "port %d\n", portno);
    
    sys_socketreceiver = socketreceiver_new(0, 0);

    inbinbuf = binbuf_new();

#ifdef UNIX
    signal(SIGHUP, sys_exithandler);
    signal(SIGINT, sys_exithandler);
    signal(SIGQUIT, sys_exithandler);
    signal(SIGILL, sys_exithandler);
    signal(SIGIOT, sys_exithandler);
    signal(SIGBUS, sys_exithandler);
    signal(SIGFPE, SIG_IGN);
    signal(SIGILL, sys_exithandler);
    signal(SIGSEGV, sys_exithandler);
    signal(SIGPIPE, SIG_IGN);
    signal(SIGALRM, SIG_IGN);
    signal(SIGTERM, SIG_IGN);
#if !defined(IRIX) && !defined(__alpha__)
    signal(SIGSTKFLT, sys_exithandler);
#endif
    signal(SIGCHLD, SIG_IGN);
    signal(SIGVTALRM, sys_virtualalarmhandler);
    if (pipe(pipeto) < 0) sys_sockerror("pipe");
    childpid = fork();
    if (childpid < 0)
    {
        if (errno) perror("sys_startgui");
    	else fprintf(stderr, "sys_startgui failed\n");
    	return (1);
    }
    else if (childpid) 	/* we're the parent */
    {
    	    /* now that we've spun off the child process we can promote
	    our process's priority, if we happen to be root.  */
	if (sys_hipriority)
	{
	    if (!getuid() || !geteuid())
	    {
	    	post("setting realtime priority...\n");
    	    	sys_set_priority();
	    }
	    else
	    {
	    	post("realtime setting failed because not root\n");
		sys_hipriority = 0;
	    }
	}
	seteuid(getuid());  	/* lose setuid priveliges */
    }
    else    	    	    	/* we're the child */
    {
    	seteuid(getuid());  	/* lose setuid priveliges */
    	dup2(pipeto[0], 0);
    	close(pipeto[0]);
    	close(pipeto[1]);

    	sprintf(cmdbuf,
    	    "TCL_LIBRARY=%s/tcl/library TK_LIBRARY=%s/tk/library %s %d\n",
    	    	sys_libdir->s_name, sys_libdir->s_name, guipath, portno);
    	if (sys_verbose) fprintf(stderr, "%s", cmdbuf);
    	execl("/bin/sh", "sh", "-c", cmdbuf, 0);
    	perror("pd: exec");
    	_exit(1);
    }
#endif
#ifdef NT
    {
    	    /* in NT land "guipath" is unused; we just do everything from
	    the libdir. */
    	char scriptbuf[MAXPDSTRING+30], wishbuf[MAXPDSTRING+30], portbuf[80];
    	int spawnret;

    	fprintf(stderr, "%s\n", sys_libdir->s_name);
    	
    	strcpy(scriptbuf, sys_libdir->s_name);
    	strcat(scriptbuf, "/src/u_main.tk");
    	sys_bashfilename(scriptbuf, scriptbuf);
    	
		sprintf(portbuf, "%d", portno);

    	strcpy(wishbuf, sys_libdir->s_name);
    	strcat(wishbuf, "/bin/wish80.exe");
    	sys_bashfilename(wishbuf, wishbuf);
    	
     	spawnret = _spawnl(P_NOWAIT, wishbuf, "wish80", scriptbuf, portbuf, 0);
    	if (spawnret < 0)
    	{
    	    perror("spawnl");
    	    fprintf(stderr, "%s: couldn't load TCL\n", wishbuf);
    	    exit(1);
    	}
    	if (!SetPriorityClass(GetCurrentProcess(), HIGH_PRIORITY_CLASS))
    	    fprintf(stderr, "pd: couldn't set high priority class\n");
    }
#endif

    if (sys_verbose)
    	fprintf(stderr, "Waiting for connection request... \n");
    if (listen(xsock, 5) < 0) sys_sockerror("listen");

    sys_guisock = accept(xsock, (struct sockaddr *) &server, &len);
    close(xsock);
    if (sys_guisock < 0) sys_sockerror("accept");
    sys_addpollfn(sys_guisock, (t_fdpollfn)socketreceiver_read,
    	sys_socketreceiver);

    if (sys_verbose)
    	fprintf(stderr, "... connected\n");

    sys_vgui("pdtk_pd_startup {%s}\n", pd_version); 
    return (0);

}

int sys_poll_togui(void)
{
    /* LATER use this to flush output buffer to gui */
    return (0);
}

int sys_pollgui(void)
{
    return (sys_domicrosleep(0, 1) || sys_poll_togui());
}

/* LATER try to save dirty documents */
void sys_bail(void)
{
    sys_close_audio_and_midi();
    exit(0);
}

void glob_quit(void *dummy)
{
    sys_vgui("exit\n");
    close(sys_guisock);
    sys_bail(); 
}

