/*************************************************************/
/* parse.c: midi parser                                      */
/* parsed midibytes und ruft befehle auf                     */
/*            V 1.00 (feb 91)             by winfried ritsch */
/*************************************************************/
#define debug 0

/* include headers */
#include <stdlib.h>
#include "s_linux_mididefs.h"
#include "s_linux_midiparse.h"

/* protos */
void cmd_2byte(MIDIParser *p);
void cmd_3byte(MIDIParser *p);
void cmd_sysex(MIDIParser *p);

/* Parser gedaechtnis */
#define CMD_NO 	0
#define CMD_GOOD 	1
#define CMD_SYSEX 2

MIDIParser *mp_parse_init(int port)
{
  MIDIParser *p;
  /*alloc memory for parser */

  p = malloc(sizeof(MIDIParser));

  if(p==NULL) /* no memory */
	 return NULL;

  p->port_no = port;

  p->p_kanal = p->p_befehl = 0;
  p->smpte_status = MP_SMPTE_STOP;
  p->frame = p->sec = p->min = p->hr = 0;

  p->cmd_stat = CMD_NO; /* running status */
  p->cmd_byte = 0;      /* aktuelle nummer der message bytes */
  p->cmd_len = 0;       /* laenges der midi message */

  p->cmd_2byte = cmd_2byte;
  p->cmd_3byte = cmd_3byte;
  p->cmd_sysex = cmd_sysex;

  /* user routines */
  p->noteonoff = NULL;
  /* or */
  p->noteon = NULL;
  p->noteoff = NULL;

  p->controlchange = NULL;
  p->programchange = NULL;
  p->pitchbend = NULL;
  p->aftertouch = NULL;
  p->polyaftertouch = NULL;
  /* system messages */
  p->mode = NULL;
  p->quarterframe = NULL;
  p->mode = NULL;
  p->songposition = NULL;
  p->songselect = NULL;
  p->tunerequest = NULL;
  p->sysex = NULL;
  p->realtime = NULL;

  return p;
}


void mp_parse_exit(MIDIParser *p)
{
free(p);
return;
}

/* byte fuer byte */
int mp_parse(MIDIParser *p,byte b)
{

  /* -------------- Vorfilter ------------------ */

  if(b & 0x80){ /*status byte*/

	 if(b >= 0xF8){         /* Realtime sofort verarbeiten */
		
		if(p->realtime)p->realtime(p->port_no,b);
		return(MP_REALTIME);
	 };


	 /* SYSTEM Exclusive MESSAGE */					
	 if(b == 0xF0){
		p->cmd_stat = CMD_SYSEX;
		p->cmd_byte = 1;
		p->cmd_len = -1;
		/*		if(debug)printf("[sysex:start]\n");*/
		return(MP_SYSEX);
	 }

	 /* filter 1 byte cmds direct */

	 if(b == 0xF6){                   /* tune req */
		if(p->tunerequest)
		  p->tunerequest(p->port_no); 

		return(MP_SYSTEM);
	 };


	 if(b == 0xF7){                   /* end of SYSEX */
		p->cmd_len = p->cmd_byte -1;

		if(p->cmd_sysex)
		  p->cmd_sysex(p); 

		return(MP_EOS);
	 };

	 

	 /* SYSTEM COMMON MESSAGES */
	 if(b > 0xF0){
		p->cmd_stat = CMD_GOOD;
		p->cmd_byte = 1;

		if(b >= 0xF4){
		  p->cmd_len = 0;   /* 1 byte messages */
		  p->cmd_byte = 0;
		}
		else if(b == 0xF2)
		  p->cmd_len = 2;  /* 3 byte message */
		else
		  p->cmd_len = 1;  /* 1 byte message */

		p->p_befehl = b;

		return(MP_COMMON);
	 }

					/* CHANNEL MESSAGES */
	 p->p_kanal = (b & 0x0F);
	 p->p_befehl = (b & 0xF0);

	 p->cmd_stat = CMD_GOOD;
	 p->cmd_len = ((p->p_befehl & 0xE0) == 0xC0) ? 1 : 2;  

	 p->cmd_byte = 1; /* nchstes byte */          		
	 return(MP_CHANNEL);
  }; /* if status */

  /* --------- DATA Bytes ------------- */
	
  switch(p->cmd_byte){ /* welches byte als naechstes */

  case 0:  /* erstes byte */

	 if(p->cmd_len == 0) /* darf nie passieren */
		return(MP_ERROR);

	 p->cmd_byte = 1; /* running status */

  case 1:	/* erstes Datenbyte */

	 p->p_data[0] = b;	

	 if(p->cmd_len == 1){                 /* ein Byte befehl ? */
		p->cmd_byte = 0;                   /* message ende */

		if(p->cmd_2byte != NULL)
		  p->cmd_2byte(p);  /* befehl ausfhren */
		return(MP_DATA);
	 };
	 

	 p->cmd_byte = 2;
	 return(MP_DATA);	 

  case 2:  /* zweites Datenbyte */

	 p->p_data[1] = b;

	 if(p->cmd_stat == CMD_GOOD){

		p->cmd_byte = 0;

		if(p->cmd_3byte != NULL)
		  p->cmd_3byte(p);

		return(MP_DATA);
	 };
		  

	 p->cmd_byte = 3; /* sonst nichts tun */
	 return(MP_DATA);
	
  default: /* nur SYSEX */

	 if(p->cmd_byte-1 > MAX_DATA) /* nur blocks von 256 erlaubt */
		return(MP_ERROR);

	 p->p_data[p->cmd_byte -1] = b;
	 p->cmd_byte++;

	 return(MP_DATA);	 
  }  /* switch cmd_byte */
}

void cmd_3byte(MIDIParser *p)
{
  switch(p->p_befehl){

  case MD_NOTE_OFF :
	 if(p->noteonoff)
		p->noteonoff(p->port_no,(int) p->p_kanal,
						 (int) p->p_data[0],0,(int) p->p_data[1]);
	 else 
		if(p->noteoff)
		  p->noteoff(p->port_no,(int) p->p_kanal,
						 (int) p->p_data[0],(int) p->p_data[1]);
	 break;
	 

  case MD_NOTE_ON  :

	 if(p->noteonoff)
		p->noteonoff(p->port_no,(int) p->p_kanal,
						 (int) p->p_data[0],(int) p->p_data[1],0);
	 else 
		if(p->noteon)
		  p->noteoff(p->port_no,(int) p->p_kanal,
						 (int) p->p_data[0],(int) p->p_data[1]);

	 break;


  case MD_POLY_TOUCH     :
	 if(p->polyaftertouch)
		(p->polyaftertouch)(p->port_no,(int) p->p_kanal,
								  (int) p->p_data[0],(int) p->p_data[1]);
	 break;


  case MD_CONTROL_CHANGE :

	 if(p->p_data[0] > MD_MODE_MESSAGES){
		if(p->mode)
		  p->mode(p->port_no,(int) p->p_data[0],(int) p->p_data[1]);
		break;
	 };

	 if(p->controlchange)
		p->controlchange(p->port_no,(int) p->p_kanal,
							  (int) p->p_data[0],(int) p->p_data[1]);
	 break;

  case MD_PITCH_BEND :
	 if(p->pitchbend)
		p->pitchbend(p->port_no,(int) p->p_kanal,
						 (int) p->p_data[0] + (((int) p->p_data[1])<<7));
	 break;

  case MD_SYS_SONG_POSITION :
	 if(p->songposition)
		p->songposition(p->port_no,(int) p->p_data[0]+(((int) p->p_data[1])<<7));

	 break;

  default:
	 /*
	 sprintf(stderr,"midiparser: should never happen: unknown 3 Byte MIDI cmd");
	 */
	 return;
  }

}

#include <stdio.h>
void cmd_2byte(MIDIParser *p)
{
  switch(p->p_befehl){

  case MD_CHANNEL_TOUCH : 

	 if(p->aftertouch)
		(p->aftertouch)(p->port_no,(int) p->p_kanal,(int)p->p_data[0]); 
	 return;

  case MD_PROGRAM_CHANGE :
	 if(p->programchange)
		(p->programchange)(p->port_no,(int) p->p_kanal,(int) p->p_data[0]);
	 return;

  case MD_SYS_MTC_QUARTER :
	 if(p->quarterframe == NULL)
		return;

	 switch(p->smpte_status){

	 case MP_SMPTE_STOP :
		if((p->p_data[0] & 0xF0) == 0x00){
		  p->smpte_status = MP_SMPTE_FORWARD;
		  p->smpte_frameidx = 0x00;
		  p->smpte_count = 0;
		  p->frame = p->p_data[0] & 0x0F;
		}
		else if((p->p_data[0] & 0xF0) == 0x70){
		  p->smpte_status = MP_SMPTE_REVERSE;
		  p->smpte_frameidx = 0x70;
		  p->smpte_count = 0;
		  p->frame = p->p_data[0] & 0x0F;
		};
		return;
	 
	 
	 case MP_SMPTE_FORWARD :
		p->smpte_frameidx = 	(p->smpte_frameidx +0x10) & 0x70;
		break;

	 case MP_SMPTE_REVERSE :
		p->smpte_frameidx = 	(p->smpte_frameidx - 0x10) & 0x70;
	 break;

	 default:
		return;
	 };

	 if((p->p_data[0] & 0x70) != p->smpte_frameidx){		  
		  p->smpte_status = MP_SMPTE_STOP; /* error - for resync stop*/
		  p->smpte_count = 0;
		  return;
		}


	 /* frame boundary */
	 if(p->smpte_status == MP_SMPTE_FORWARD && (p->p_data[0] & 0x30) == 0x00){

		p->smpte_count++;

		if(p->smpte_count > 1) /* enough valid data */
		  p->quarterframe(p->port_no,p->frame,p->sec,p->min,p->hr);
	 };

	 /* get data */
	 switch((p->p_data[0] & 0x70)){
		
	 case 0x00:
		p->frame = (p->frame & 0x30) + (p->p_data[0] & 0x0f);
		break;
	 case 0x10:
		p->frame = (p->frame & 0x0f) + ((p->p_data[0] & 0x01)<<4);
		break;
	 case 0x20:
		p->sec = (p->sec & 0x30) + (p->p_data[0] & 0x0f);
		break;
	 case 0x30:
		p->sec = (p->sec & 0x0f) + ((p->p_data[0] & 0x03)<<4);
		break;
	 case 0x40:
		p->min = (p->min & 0x30) + (p->p_data[0] & 0x0f);
		break;
	 case 0x50:
		p->min = (p->min & 0x0f) + ((p->p_data[0] & 0x03)<<4);
		break;
	 case 0x60:
		p->hr = (p->hr & 0x30) + (p->p_data[0] & 0x0f);
		break;
	 case 0x70:
		p->hr = (p->hr & 0x0f) + ((p->p_data[0] & 0x03)<<4);
		break;
	 }
	 
	 /* frame boundary */
	 if(p->smpte_status == MP_SMPTE_REVERSE && (p->p_data[0] & 0x30) == 0x00){

		p->smpte_count++;

		if(p->smpte_count > 1) /* enough valid data */
		  p->quarterframe(p->port_no,p->frame,p->sec,p->min,p->hr);
	 };
	 


  case MD_SYS_SONG_SELECT :

	 if(p->songselect)
		(p->songselect)(p->port_no,(int) p->p_data[0]);
	 return;

  default:
	 /*	
	 sprintf(stderr,"midiparser: should never happen: unknown 3 Byte MIDI cmd");
	  */
	 return;

  }
}

void cmd_sysex(MIDIParser *p)
{
  /*filter out smpte messages FULL Time */
  if(p->p_data[0] == MD_SYSEX_RT && 
	  p->p_data[2] == MD_SYSEX_RT_MTC && 
	  p->p_data[3] == MD_SYSEX_RT_MTC_FULL){
	 
	 /* ignore channel data[1] */
	 p->hr = p->p_data[4] & 0x0f;
	 p->min = p->p_data[5] & 0x0f;
	 p->sec = p->p_data[6] & 0x0f;
	 p->frame = p->p_data[7] & 0x0f;
	 p->smpte_count = 2; /* next quarter frame run */
	 /* hopefully direction is ok */
  };
  
  /* not implemented yet */
  if(p->sysex)
	 p->sysex(p->port_no,p->cmd_byte-1,p->p_data); 
} 
