;;; examples of Guile extensions to Snd
;;;
;;;        contents
;;;
;;; snd.html examples made harder to break
;;; 'info' from extsnd.html using format
;;; correlation
;;; XEmacs-like "Buffers" menu
;;; set fft-size based on current time domain window size
;;; example of abort?
;;; play sound n times
;;; make a system call
;;; make dot size dependent on number of samples being displayed
;;; auto-save
;;; move window left edge to mark upon 'm' key
;;; flash selected data red and green
;;; show bomb icon
;;; play-section, play-selection
;;; delete selected portion and smooth the splice
;;; mix with result at original peak amp
;;; mapping extensions (map arbitrary single-channel function over various channel collections)
;;;     do-chans, do-all-chans, do-sound-chans
;;;     every-sample?
;;;     sort-samples
;;;     swap-channels
;;; interp-env (CLM-style envelope interpolation)
;;; mix mono sound into stereo sound panning according to env
;;; fft-edit -- FFT based editing
;;; comb-filter
;;;
;;;
;;;     ---- the rest are CLM-based ----
;;; comb-filter, notch-filter, formant-filter
;;; echo (delays)
;;; ring-modulation, am
;;; src-related sound effects (src, rand-interp, etc)
;;; compand (array-interp)
;;; shift pitch keeping duration constant (src+granulate)
;;; tempo change via envelope (granulate)
;;; cross-synthesis (using a formant bank)
;;; voiced->unvoiced (formants)
;;; convolution (convolve)
;;; reverb (all-pass etc)
;;; scissor-tailed flycatcher (waveshaping)
;;; fm-violin (FM and various other generators, #&key args)
;;; digital zipper "crossfade" (file->sample)
;;; FOF voice synthesis (wave-train, #&optional args)
;;; phase vocoder
;;; mix with envelope


(use-modules (ice-9 debug)) ;comment this out in guile versions before 1.3.2

;;; -------- snd.html examples made harder to break --------
;;; this mainly involves keeping track of the current sound/channel

(define region-rms
  (lambda (n)
    (let* ((data (region-samples 0 0 n))
	   (len (vector-length data))
	   (sum 0.0))
      (do ((i 0 (1+ i))) ((= i len) (sqrt (/ sum len)))
	(set! sum (+ sum (* (vector-ref data i) (vector-ref data i))))))))

(define window-samples
  (lambda (snd chn)
    (let ((wl (left-sample snd chn))
	  (wr (right-sample snd chn)))
      (samples wl (+ 1 (- wr wl)) snd chn))))

(define display-energy
  (lambda (snd chn y0 y1)
    (let* ((ls (left-sample snd chn))
	   (rs (right-sample snd chn))
	   (data (samples ls (+ 1 (- rs ls)) snd chn))
	   (len (vector-length data))
	   (sr (srate snd)))
      (do ((i 0 (1+ i))) ((= i len))
	(vector-set! data i (* (vector-ref data i) (vector-ref data i))))
      (graph data "energy" (/ ls sr) (/ rs sr) 0.0 (* y1 y1) snd chn))))
      ;; the y1 use causes our energy graph to reflect the vertical sliders

;(add-hook! graph-hook display-energy)

(define vct-display-energy
  (lambda (snd chn y0 y1)
    (let* ((ls (left-sample))
           (rs (right-sample))
           (data (samples->vct ls (+ 1 (- rs ls)) snd chn))
           (len (vct-length data))
           (sr (srate snd)))
      (vct-multiply! data data)
      (graph data "energy" (/ ls sr) (/ rs sr) 0.0 (* y1 y1) snd chn))))


(define unsaved-edits?
  (lambda (ind)
    (and (< ind (max-sounds))
	 (or (and (ok? ind)
		  (> (vector-ref (edits ind) 0) 0)
		  (report-in-minibuffer "there are unsaved edits")
		  #t)
	     (unsaved-edits? (+ ind 1))))))

;(add-hook! exit-hook (lambda () (report-in-minibuffer "") (unsaved-edits? 0)))

(define no-startup-file?
  (lambda (ind file)
    (if (= ind (max-sounds))
	(begin
	  (write (string-append "can't open " file) (current-error-port))
	  (newline (current-error-port))
	  #t)
	(if (ok? ind)
	    #f
	    (no-startup-file? (+ ind 1) file)))))

;(add-hook! start-hook (lambda (file) (if (> (string-length file) 0) (no-startup-file? 0 file) #f)))


(define fft-peak
  (lambda (snd chn scale)
    (if (and (ffting) (= (fft-style) normal-fft))
	(let ((samps (transform-samples snd chn)))
	  (if samps
	      (let* ((len (vector-length samps))
		     (mx (vector-ref samps 0))
		     (peak (do ((i 1 (+ i 1))) ((= i len) (/ (* 2 mx) (fft-size)))
			     (let ((val (abs (vector-ref samps i))))
			       (if (> val mx) (set! mx val))))))
		(report-in-minibuffer (number->string peak) snd)))))
      #f))
	
;(add-hook! fft-hook fft-peak)


;;; -------- 'info' from extsnd.html using format --------

(use-modules (ice-9 format))

(define finfo
  (lambda (file)
    (format #f "~A: chans: ~D, srate: ~D, ~A, ~A, len: ~1,3F"
	    file
	    (sound-chans file)
	    (sound-srate file)
	    (sound-type-name (sound-header-type file))
	    (sound-format-name (sound-data-format file))
	    (/ (sound-samples file)
	       (* (sound-chans file) (sound-srate file))))))


;;; -------- Correlation --------
;;;
;;; correlation of channels in a stereo sound
;;; (in this example, nearly all the compute time is going into the spectral multiply)

(define window-samples
 (lambda (snd chn)
  (let ((wl (left-sample snd chn))
        (wr (right-sample snd chn)))
   (samples wl (+ 1 (- wr wl)) snd chn))))

(define correlate
 (lambda (snd chn y0 y1)
   (if (= (channels snd) 2)
       (let* ((data1 (window-samples snd 0))
	      (data2 (window-samples snd 1))
	      (len1 (vector-length data1))
	      (len2 (vector-length data2))
	      (len (if (> len1 len2) len2 len1))
	      (pow2 (ceiling (/ (log len) (log 2))))
	      (fftlen (inexact->exact (expt 2 pow2)))
	      (fftlen2 (/ fftlen 2))
	      (fftscale (/ 1.0 fftlen))
	      (rl1 (make-vector fftlen 0.0))
	      (im1 (make-vector fftlen 0.0))
	      (rl2 (make-vector fftlen 0.0))
	      (im2 (make-vector fftlen 0.0))
	      (data3 (make-vector fftlen2 0.0)))
	 (do ((i 0 (1+ i))) ((= i len)) 
	   (vector-set! rl1 i (vector-ref data1 i))
	   (vector-set! rl2 i (vector-ref data2 i)))
	 (fft rl1 im1 1)
	 (fft rl2 im2 1)
	 (do ((i 0 (1+ i))) ((= i fftlen))
	   (let ((tempr1 (vector-ref rl1 i))
		 (tempi1 (vector-ref im1 i))
		 (tempr2 (vector-ref rl2 i))
		 (tempi2 (vector-ref im2 i)))
	     (vector-set! rl1 i (+ (* tempr1 tempr2) (* tempi1 tempi2)))
	     (vector-set! im1 i (- (* tempr1 tempi2) (* tempr2 tempi1)))))
	 (fft rl1 im1 -1)
	 (do ((i 0 (1+ i))) ((= i fftlen2)) (vector-set! data3 i (* (vector-ref rl1 i) fftscale)))
	 (graph data3 "lag time" 0 fftlen2))
       (report-in-minibuffer "correlate wants stereo input"))))

;(add-hook! graph-hook correlate)

;;; the (much faster) vct version of this is:

(define vct-correlate
 (lambda (snd chn y0 y1)
   (if (= (channels snd) 2)
       (let* ((ls (left-sample snd 0))
	      (rs (right-sample snd 0))
	      (ilen (+ 1 (- rs ls)))
	      (pow2 (ceiling (/ (log ilen) (log 2))))
	      (fftlen (inexact->exact (expt 2 pow2)))
	      (fftlen2 (/ fftlen 2))
	      (fftscale (/ 1.0 fftlen))
	      (rl1 (samples->vct ls fftlen snd 0))
	      (rl2 (samples->vct ls fftlen snd 1))
	      (im1 (make-vct fftlen))
	      (im2 (make-vct fftlen)))
	 (fft rl1 im1 1)
	 (fft rl2 im2 1)
	 (let* ((tmprl (vct-copy rl1))
		(tmpim (vct-copy im1))
		(data3 (make-vct fftlen2)))
	   (vct-multiply! tmprl rl2)     ; (* tempr1 tempr2)
	   (vct-multiply! tmpim im2)     ; (* tempi1 tempi2)
	   (vct-multiply! im2 rl1)       ; (* tempr1 tempi2)
	   (vct-multiply! rl2 im1)       ; (* tempr2 tempi1)
	   (vct-add! tmprl tmpim)        ; add the first two
	   (vct-scale! rl2 -1.0)
	   (vct-add! im2 rl2)            ; subtract the 4th from the 3rd
	   (fft tmprl im2 -1)
	   (vct-add! data3 tmprl)        ; copy into data3 (which is half the size of tmprl)
	   (vct-scale! data3 fftscale)   ; scale by fftscale
	   (graph data3 "lag time" 0 fftlen2)))
       (report-in-minibuffer "vct-correlate wants stereo input"))))

;;; The inner let body could also be:
;;;
;;;	   (graph
;;;	    (vct-scale! 
;;;	     (vct-add! data3
;;;		       (fft (vct-add! (vct-multiply! tmprl rl2) (vct-multiply! tmpim im2))
;;;			    (vct-add! (vct-multiply! im2 rl1) (vct-scale! (vct-multiply! rl2 im1) -1.0))
;;;			    -1))
;;;	     fftscale) "lag time" 0 fftlen2)))



;;; -------- Buffers Menu --------
;;; patterned after the XEmacs Buffers menu

(define buffer-menu (add-to-main-menu "Buffers"))

(define open-buffer
  (lambda (filename)
    (add-to-menu buffer-menu 
		 filename 
		 (string-append "(select-sound (find-sound " "\"" filename "\"" "))"))))

(define close-buffer 
  (lambda (snd)
    (remove-from-menu buffer-menu (file-name snd))))

;;; here we're adding this menu handling code to whatever is already happening at open/close-hook time

;(add-hook! open-hook open-buffer)
;(add-hook! close-hook close-buffer)


;;; -------- set fft-size based on current time domain window size

(add-hook! graph-hook 
	   (lambda (snd chn y0 y1)
	     (if (and (ffting) (= (fft-style) normal-fft))
		 (set-fft-size
		  (expt 2 (ceiling (/ (log (- (right-sample) (left-sample))) 
				      (log 2.0))))))))


;;; -------- abort? example (Anders Vinjar)

(define (locate-zero limit)
  (do ((n (cursor 0 0) (1+ n))
       (samps (frames 0)))
      ((or (>= n samps)
	    (abort?)
	    (< (+ (abs (sample n 0 0)) (abs (sample n 0 1))) limit))
       (begin
	(set-cursor n 0 0)
	n))))


;;; -------- play sound n times -- (pl 3) for example.

(define plays 0)

(define pl1
  (lambda (snd)
    (if (= plays 0)
	(remove-hook! stop-playing-hook pl1)
      (begin
	(set! plays (- plays 1))
	(play 0 snd)))))

(define (pl n) 
  (set! plays (- n 1))
  (add-hook! stop-playing-hook pl1)
  (play))

(bind-key (char->integer #\p) 0 "(pl (max 1 (prefix-arg)))" #t)


;;; -------- make a system call from the listener
;;;
;;;   (shell "df") for example -- there's probably a more elegant way to do this is in Scheme
;;; or to play a sound whenever a file is closed:
;;;   (set! close-hook "(shell \"sndplay wood16.wav\")")

(use-modules (ice-9 popen))  

(define shell 
  (lambda (cmd)
    (let* ((str "")
	   (fil (open-pipe cmd "r")))
      (do ((val (read-char fil) (read-char fil))) 
	  ((eof-object? val))
	(set! str (string-append str (string val))))
      (close-pipe fil)
      str)))


;;; -------- make dot size dependent on number of samples being displayed
;;; 
;;; this could be extended to set graph-style to graph-lines if many samples are displayed, etc

(define auto-dot
  (lambda (snd chn y0 y1)
    (let ((dots (- (right-sample snd chn)
		   (left-sample snd chn))))
      (if (> dots 100) 
	  (set-dot-size 1)
	(if (> dots 50)
	    (set-dot-size 2)
	  (if (> dots 25)
	      (set-dot-size 3)
	    (set-dot-size 5)))))))
    
;(add-hook! graph-hook auto-dot)


;;; -- current time 
;(strftime "%H:%M" (localtime (current-time)))


;;; -------- auto-save 
;;;
;;; we use the "in" function to glance at the state of edits every "interval" seconds
;;; a simple example of "in" is:
;;;    (in 5000 "(report-in-minibuffer \"boo!\")")
;;; which will be print "boo" 5 seconds after it is evaluated.
;;;
;;; in this somewhat minimal implementation, the user calls (auto-save secs)
;;; for example, (auto-save 100)
;;; which causes the function auto-save to run every "secs" seconds
;;; If a sound is found with a channel with more than 10 unsaved edits,
;;;  it is saved on the /tmp directory with the name #<filename># -- that
;;;  is, test.snd's current state is saved as /tmp/#test.snd#.
;;;
;;; useful extensions: delete the temp file when associated file is closed (close-hook addition)
;;;                    check that auto-saved version is actually out-of-date
;;;                    check that there's enough disk space
;;;                    check at open for more recent auto-save version

(define auto-save
  (lambda (interval)
    (let ((lim (max-sounds)))
      (do ((i 0 (1+ i)))
	  ((= i lim))
	(if (ok? i)
	    (let ((max-edits 0)
		  (chans (channels i)))
	      (do ((chn 0 (1+ chn)))
		  ((= chn chans))
		(let ((eds (vector-ref (edits i chn) 0)))
		  (if (> eds max-edits) (set! max-edits eds))))
	      (if (> max-edits 10)
		  (begin
		   (report-in-minibuffer "auto-saving..." i)
		   (save-sound-as (string-append "/tmp/#" (short-file-name i) "#") i))))))
      (in (* 1000 interval) (string-append "(auto-save " (number->string interval) ")")))))



;;; -------- move window left edge to mark upon 'm'
;;;
;;; in large sounds, it can be pain to get the left edge of the window
;;; aligned with a specific spot in the sound.  In this code, we assume
;;; the desired left edge has a mark, and the 'm' key (without control)
;;; will move the window left edge to that mark.

(define first-mark-in-window-at-left
  (lambda ()
    (let* ((keysnd (or (selected-sound) 0))
	   (keychn (or (selected-channel keysnd) 0))
	   (current-left-sample (left-sample keysnd keychn))
	   (num-marks (marks keysnd keychn)))
      (if (= num-marks 0)
	  (report-in-minibuffer "no marks!")
	(if (= num-marks 1)
	    (begin
	      (set-left-sample (mark-sample 0) keysnd keychn)
	      cursor-update-display)
          (do ((i 0 (1+ i)))
	      ((or (= i num-marks) 
		   (<= current-left-sample (mark-sample i)))
	       (if (or (= i num-marks) 
		       (> (mark-sample i) (right-sample keysnd keychn)))
		   (report-in-minibuffer "no mark in window")
		 (begin
		   (set-left-sample (mark-sample i) keysnd keychn)
		   cursor-update-display)))))))))

(bind-key (char->integer #\m) 0 "(first-mark-in-window-at-left)")



;;; -------- flash selected data red and green
;;;

(define red (make-color 1 0 0))
(define green (make-color 0 1 0))
(define data-red? #t)
(set-selected-data-color red)

(define flash-selected-data
  (lambda (interval)
    (if (selected-sound)
	(begin
	  (set-selected-data-color (if data-red? green red))
	  (set! data-red? (not data-red?))
	  (in interval (lambda () (flash-selected-data interval)))))))


;;; -------- show bomb icon

(define show-bomb 
  (lambda (n speed) 
    (if (> n 0) 
	(begin 
	  (bomb) 
	  (in speed (lambda () (show-bomb (- n 1) speed))))
	(bomb 0 #f))))

; (show-bomb 20 200)


;;; -------- play portion of sound, play current selection (as opposed to C-x p which plays region 0)

(use-modules (ice-9 optargs))

(define play-section
  (lambda* (beg end #&optional (snd 0))
    (if (and (ok? snd) (> end beg))
	(let* ((chans (channels snd))
	       (edited (make-vector chans)))
	  (do ((chn 0 (1+ chn)))
	      ((= chn chans))
	    (let ((len (frames snd chn)))
	      (if (< end len) 
		  (begin
		    (vector-set! edited chn #t)
		    (set-squelch-update #t snd chn)
		    (delete-samples end (- len end) snd chn))
		(vector-set! edited chn #f))))
	  (play-and-wait beg snd)
	  (do ((chn 0 (1+ chn)))
	      ((= chn chans))
	    (if (vector-ref edited chn) 
		(begin
		  (undo 1 snd chn)
		  (set-squelch-update #f snd chn))))))))

(define play-selection
  (lambda ()
    ;; this is kinda brute force
    (save-selection "tmp.snd" next-sound-file snd-16-linear (region-srate 0) "")
    (play-and-wait "tmp.snd")
    (delete-file "tmp.snd")))


;;; -------- delete selected portion and smooth the splice

(define delete-selection-and-smooth
  (lambda ()
    (if (selection-length)
	(let ((beg (selection-beg))
	      (len (selection-length)))
	  (do ((i 0 (1+ i)))
	      ((= i (max-sounds)) #f)
	    (if (ok? i)
		(do ((j 0 (1+ j)))
		    ((= j (channels i)) #f)
		  (if (selection-member i j)
		      (let ((smooth-beg (max 0 (- beg 16))))
			(delete-samples beg len i j)
			(smooth smooth-beg 32 i j))))))))))


;;; -------- mix with result at original peak amp

(define normalized-mix 
  (lambda (filename beg in_chan snd chn)
    ;; like mix but returns result with same peak amp as before (return scaler)
    (let ((original-max-amp (maxamp snd chn)))
      (mix filename beg in_chan snd chn)
      (let ((new-max-amp (maxamp snd chn)))
	(if (not (= original-max-amp new-max-amp))
	    (let ((scaler (/ original-max-amp new-max-amp))
		  (old-sync (syncing snd)))
	      (set-syncing #f snd)
	      (scale-by scaler snd chn)
	      (set-syncing old-sync snd)
	      scaler)
	    1.0)))))


;;; -------- mapping extensions (map arbitrary single-channel function over various channel collections)
;;;

(define do-all-chans
  (lambda (proc args origin)
    (do ((i 0 (1+ i)))
	((= i (max-sounds)) #f)
      (if (ok? i)
	  (do ((j 0 (1+ j)))
	      ((= j (channels i)) #f)
	    (map-chan (apply proc args) #f #f origin i j))))))

(define update-graphs
  (lambda ()
    (do ((i 0 (1+ i)))
	((= i (max-sounds)) #f)
      (if (ok? i)
	  (do ((j 0 (1+ j)))
	      ((= j (channels i)) #f)
	    (update-graph i j))))))

(define do-chans
  (lambda (proc args origin)
    (if (syncing)
	(do ((i 0 (1+ i)))
	    ((= i (max-sounds)) #f)
	  (if (and (ok? i) (syncing i))
	      (do ((j 0 (1+ j)))
		  ((= j (channels i)) #f)
		(map-chan (apply proc args) #f #f origin i j))))
	(map-chan (apply proc args)))))

(define do-sound-chans
  (lambda (proc args origin)
    (let ((ind (selected-sound)))
      (do ((j 0 (1+ j)))
	  ((= j (channels ind)) #f)
      (map-chan (apply proc args) #f #f origin ind j)))))


(define every-sample?
  (lambda (proc)
    (let ((baddy (scan-chan (lambda (y) (if y (not (proc y)) #f)))))
      (if baddy (set-cursor (cadr baddy)))
      (not baddy))))

(define sort-samples
  (lambda (nbins)
    (let ((bins (make-vector nbins 0)))
      (lambda (y)
	(if y
	    (let ((bin (inexact->exact (floor (* (abs y) nbins)))))
	      (vector-set! bins bin (+ (vector-ref bins bin) 1))
	      #f)
	    bins)))))

(define swap-channels
  (lambda ()
    (if (= (channels) 2)
	(map-across-sound-chans
	 (lambda (data chans)
	   (if data 
	       (let ((chan0-sample (vector-ref data 0)))
		 (vector-set! data 0 (vector-ref data 1))
		 (vector-set! data 1 chan0-sample)
		 data)
	       #f))
	 #f #f "swap-channels")
	(string-append (short-file-name) " is not stereo!"))))


;;; -------- interp-env
;;;
;;; (interp-env .3 '(0 0 .5 1 1 0) -> .6

(define interp-env                      ;env is list of x y breakpoint pairs, interpolate at x returning y
  (lambda args                          ;  (x env &optional (base 1.0)
    (let ((x (car args))
	  (env (cadr args))
	  (base (if (null? (cddr args)) #f (caddr args))))
      (cond ((null? env) 0.0)		;no data -- return 0.0
	    ((or (<= x (car env))	;we're sitting on x val (or if < we blew it)
		 (null? (cddr env)))	;or we're at the end of the list
	     (cadr env))		;so return current y value
	    ((> (caddr env) x)		;x <= next env x axis value
	     (if (or (= (cadr env) (cadddr env))
		     (and base (= base 0.0)))
		 (cadr env)		;y1=y0, so just return y0 (avoid endless calculations below)
		 (if (or (not base) (= base 1.0))
		     (+ (cadr env)	;y0+(x-x0)*(y1-y0)/(x1-x0)
			(* (- x (car env))
			   (/ (- (cadddr env) (cadr env))
			      (- (caddr env) (car env)))))
		     (+ (cadr env)
			(* (/ (- (cadddr env) (cadr env))
			      (- base 1.0))
			   (- (expt base (/ (- x (car env))
					    (- (caddr env) (car env))))
			      1.0))))))
	    (else (interp-env x (cddr env))))))) ;go on looking for x segment



;;; -------- mix mono sound into stereo sound panning according to env

(define pan-mono-to-stereo
  ;; this function assumes we've currently syncd together one mono and one stereo sound,
  ;; then mixes the mono into the stereo following the panning envelope;
  ;; the latter is 0 when we want all the mono in chan 0, 1 for all in chan 1.
  ;; We'll assume for simplicity that the envelope goes from 0 to 1 along the x axis.
  (lambda (panning-envelope)
    (letrec ((find-nchannel-sound (lambda (ind chans) 
				      (if (< ind (max-sounds))
					  (if (and (ok? ind) (syncing ind) (= (channels ind) chans)) 
					      ind 
					      (find-nchannel-sound (1+ ind) chans))
					  (begin
					    (report-in-minibuffer 
					     (string-append "can't find any syncd " 
							    (number->string chans) 
							    " channel sound!"))
					    #f)))))
      (let* ((mono-sound (find-nchannel-sound 0 1))
	     (stereo-sound (and mono-sound (find-nchannel-sound 0 2))))
	(if stereo-sound
	    (let ((samps (max (frames stereo-sound) (frames mono-sound)))
		  (current-samp 0)
		  (mono (if (> stereo-sound mono-sound) 0 2))
		  (stereo (if (> stereo-sound mono-sound) 1 0)))
	      (map-across-chans
	       (lambda (data chans)
		 (if data
		     (let ((samp (vector-ref data mono))
			   (y (interp-env (/ current-samp samps) panning-envelope)))
		       (set! current-samp (1+ current-samp))
		       (vector-set! data mono #f) ;don't edit the mono file
		       (vector-set! data stereo (+ (vector-ref data stereo) (* (- 1.0 y) samp)))
		       (vector-set! data (1+ stereo) (+ (vector-ref data (1+ stereo)) (* y samp)))
		       data)
		   #f)))))))))


;;; -------- FFT-based editing
;;;

(define fft-edit
  ;; fft entire sound, remove all energy below 'bottom' Hz and all above 'top' Hz,
  ;; then unfft.  We're assuming we can fit the entire fft into memory (we could
  ;; easily reduce this by a factor of 2 by repacking the data in and out).
  (lambda (bottom top)
    (let* ((sr (srate))
	   (len (frames))
	   (fsize (expt 2 (ceiling (/ (log len) (log 2.0)))))
	   (rdata (samples->vct 0 fsize))
	   (idata (make-vct fsize))
	   (lo (round (/ bottom (/ sr fsize))))
	   (hi (round (/ top (/ sr fsize)))))
      (fft rdata idata 1)
      (do ((i 0 (1+ i))
	   (j (- fsize 1) (1- j)))
	  ((= i lo))
	(vct-set! rdata i 0.0)
	(vct-set! rdata j 0.0)
	(vct-set! idata i 0.0)
	(vct-set! idata j 0.0))
      (do ((i hi (1+ i))
	   (j (- fsize hi) (1- j)))
	  ((= i (/ fsize 2)))
	(vct-set! rdata i 0.0)
	(vct-set! rdata j 0.0)
	(vct-set! idata i 0.0)
	(vct-set! idata j 0.0))
      (fft rdata idata -1)
      (vct-scale! rdata (/ 1.0 fsize))
      (set-samples 0 (1- len) rdata))))


;;; -------- comb-filter

(define comb-filter
  (lambda (scaler size)
    (let ((delay-line (make-vector size 0.0))
	  (delay-loc 0))
      (lambda (x)
	(if x
	    (let ((result (vector-ref delay-line delay-loc)))
	      (vector-set! delay-line delay-loc (+ x (* scaler result)))
	      (set! delay-loc (1+ delay-loc))
	      (if (= delay-loc size) (set! delay-loc 0))
	      result)
	    #f)))))

; (map-chan (comb-filter .8 32))

;;; the same thing using the CLM module is:

(define comb-filter 
  (lambda (scaler size)
    (let ((cmb (make-comb scaler size)))
      (lambda (x) (if x (comb cmb x))))))

;;; by using filters at harmonically related sizes, we can get chords:

(define comb-chord
  (lambda (scaler size amp)
    (let ((c1 (make-comb scaler size))
	  (c2 (make-comb scaler (* size .75)))
	  (c3 (make-comb scaler (* size 1.2))))
      (lambda (x) (if x (* amp (+ (comb c1 x) (comb c2 x) (comb c3 x))))))))

; (map-chan (comb-chord .95 100 .3))
; (map-chan (comb-chord .95 60 .3))

;;; or change the comb length via an envelope:

(define max-envelope
  (lambda (e mx)
    (if (null? e)
	mx
      (max-envelope (cddr e) (max mx (abs (cadr e)))))))

(define zcomb
  (lambda (scaler size pm)
    (let ((cmb (make-comb scaler size :max-size (+ size 1 (max-envelope pm 0))))
	  (penv (make-env :envelope pm :end (frames))))
      (lambda (x) (if x (comb cmb x (env penv)))))))

; (map-chan (zcomb .8 32 '(0 0 1 10)))

;;; or notch: 

(define notch-filter 
  (lambda (scaler size)
    (let ((cmb (make-notch scaler size)))
      (lambda (x) (if x (notch cmb x))))))

; (map-chan (notch-filter .8 32))

;;; or formant:

(define formant-filter
  (lambda (radius frequency)
    (let ((frm (make-formant radius frequency)))
      (lambda (x) (if x (formant frm x))))))

; (map-chan (formant-filter .01 2400))

;;; to impose several formants, just add them in parallel:

(define formants
  (lambda (r1 f1 r2 f2 r3 f3)
    (let ((fr1 (make-formant r1 f1))
	  (fr2 (make-formant r2 f2))
	  (fr3 (make-formant r3 f3)))
      (lambda (x)
	(if x (+ (formant fr1 x)
		 (formant fr2 x)
		 (formant fr3 x)))))))

; (map-chan (formants .01 900 .02 1800 .01 2700))

;;; to get a moving formant:

(define moving-formant
  (lambda (radius move)
    (let ((frm (make-formant radius (cadr move)))
	  (menv (make-env :envelope move :end (frames))))
      (lambda (x)
	(if x
	    (let ((val (formant frm x)))
	      (mus-set-frequency frm (env menv))
	      val))))))

; (map-chan (moving-formant .01 '(0 1200 1 2400)))


;;; -------- echo

(define echo 
  (lambda (scaler secs)
    (let ((del (make-delay (round (* secs (srate))))))
      (lambda (inval)
	(if inval (+ inval (delay del (* scaler (+ (tap del) inval)))))))))

; (map-chan (echo .5 .5) 0 44100)

;;; here is a multi-channel version:

(define echoes
  ;; since map-across-sound-chans is expected here as the caller, I'll embed it
  (lambda (scaler secs decay-time)
    (let* ((len (channels))
	   (dels (make-vector len))
	   (total-len (+ (frames) (round (* (srate) decay-time)))))
      (do ((i 0 (1+ i)))
	  ((= i len))
	(vector-set! dels i (make-delay (round (* secs (srate))))))
      (map-across-sound-chans
       (lambda (invals chans)
	 (if invals
	     (do ((i 0 (1+ i)))
		 ((= i chans) invals)
	       (let ((inval (vector-ref invals i))
		     (del (vector-ref dels i)))
		 (vector-set! invals i (+ inval (delay del (* scaler (+ (tap del) inval)))))))))
       0 total-len
       (string-append "(echoes "
		      (number->string scaler) " "
		      (number->string secs) " "
		      (number->string decay-time) ")")))))

; (echoes .5 .75 2.0)

;;; here is a version that modulates the echos:

(define zecho 
  (lambda (scaler secs frq amp)
    (let* ((os (make-oscil frq))
	   (len (round (* secs (srate))))
	   (del (make-delay len :max-size (+ len amp 1))))
      (lambda (inval)
	(if inval 
	    (+ inval 
	       (delay del 
		      (* scaler (+ (tap del) inval))
		      (* amp (oscil os)))))))))

; (map-chan (zecho .5 .75 6 10.0) 0 65000)

;;; or low-pass filter the echoes:

(define flecho 
  (lambda (scaler secs)
    (let* ((flt (make-fir-filter :order 4 :xcoeffs (list->vct '(.125 .25 .25 .125))))
	   (del (make-delay  (round (* secs (srate))))))
      (lambda (inval)
	(if inval 
	    (+ inval 
	       (delay del 
		      (fir-filter flt (* scaler (+ (tap del) inval))))))))))

; (map-chan (flecho .5 .9) 0 75000)


;;; -------- ring-mod and am
;;;
;;; CLM instrument is ring-modulate.ins

(define ring-mod
  (lambda (freq gliss-env)
    (let* ((os (make-oscil :frequency freq))
	   (len (frames))
	   (genv (make-env :envelope gliss-env :end len)))
      (lambda (inval)
	(if inval (* (oscil os (env genv)) inval))))))

; (map-chan (ring-mod 100 '(0 0 1 0)))
; (map-chan (ring-mod 10 (list 0 0 1 (hz->radians 100))))

(define am 
  (lambda (freq) 
    (let ((os (make-oscil freq))) 
      (lambda (inval) 
	(if inval 
	    (amplitude-modulate 1.0 inval (oscil os)))))))

; (map-chan (am 440))


;;; -------- hello-dentist
;;;
;;; CLM instrument version is in clm.html

(define hello-dentist 
  (lambda (frq amp)
    (let* ((rn (make-rand-interp :frequency frq :amplitude amp))
	   (i 0)
	   (j 0)
	   (len (frames))
	   (in-data (samples->vct 0 len))
	   (out-data (make-vct (round (* len (+ 1.0 (* 2 amp))))))
	   (rd (make-src :srate 1.0 
			 :input (lambda (dir) 
				  (let ((val (vct-ref in-data i)))
				    ;; here we could use (sample i) without in-data
				    (set! i (+ i dir)) 
				    val)))))
      (do ()
	  ((= i len))
	(vct-set! out-data j (src rd (rand-interp rn)))
	(set! j (+ j 1)))
      (vct->samples 0 j out-data))))

; (hello-dentist 40.0 .1)

;;; a very similar function uses oscil instead of rand-interp, giving
;;; various "Forbidden Planet" sound effects:

(define fp
  (lambda (sr osamp osfrq)
    (let* ((os (make-oscil osfrq))
	   (sr (make-src :srate sr))
	   (len (frames))
	   (inctr 0)
	   (out-data (make-vct len)))
      (do ((i 0 (1+ i)))
	  ((= i len))
	(vct-set! out-data i
		  (src sr (* osamp (oscil os))
		       (lambda (dir)
			 (let ((val (sample inctr)))
			   (set! inctr (+ inctr dir))
			   val)))))
      (vct->samples 0 len out-data))))

; (fp 1.0 .3 20)
	    

;;; -------- compand

(define compand
  (lambda ()
    (let* ((tbl (make-vct 17)))
      ;; we'll fill this by hand with some likely-looking companding curve
      ;; (we're eye-balling the curve on p55 of Steiglitz's "a DSP Primer")
      (vct-set! tbl 8 0.0)
      (vct-set! tbl 7 -.25) (vct-set! tbl 9  .25) 
      (vct-set! tbl 6 -.45) (vct-set! tbl 10 .45) 
      (vct-set! tbl 5 -.6)  (vct-set! tbl 11 .6)
      (vct-set! tbl 4 -.72) (vct-set! tbl 12 .72)
      (vct-set! tbl 3 -.82) (vct-set! tbl 13 .82)
      (vct-set! tbl 2 -.90) (vct-set! tbl 14 .90)
      (vct-set! tbl 1 -.96) (vct-set! tbl 15 .96)
      (vct-set! tbl 0 -1.0) (vct-set! tbl 16 1.0)
      (lambda (inval)
	(if inval
	    (let ((index (+ 8.0 (* 8.0 inval))))
	      (array-interp tbl index 17)))))))

; (map-chan (compand))

;;; since there's no state in this function, it can be used without change
;;; in any of the mapping functions (unlike echo, for example)

;;; it is easy to test functions like this in the listener:
;;;
;;;    >((compand) 0.0)
;;;    0.0
;;;    >((compand) 1.0)
;;;    1.0
;;;    >((compand) .1)
;;;    0.200000047683716
;;;    >((compand) .99)
;;;    0.996800001335146
;;;    >((compand) .95)
;;;    0.984000006675728


;;; -------- shift pitch keeping duration constant
;;;
;;; both src and granulate take a function argument to get input whenever it is needed.
;;; in this case, src calls granulate which reads the currently selected file.
;;; CLM version is in expsrc.ins

(define expsrc 
  (lambda (rate)
    (let* ((gr (make-granulate :expansion rate))
	   ;; this can be improved by messing with make-granulate's hop and length args
	   (sr (make-src :srate rate))
	   (inctr 0))
      (lambda (inval)
	(if inval
	    (src sr 0.0
		 (lambda (dir)
		   (granulate gr
			      (lambda (dir)
				(let ((val (sample inctr)))
				  (set! inctr (+ inctr dir))
				  val))))))))))

;;; the next (expsnd) changes the tempo according to an envelope; the new duration
;;; will depend on the expansion envelope -- we integrate it to get
;;; the overall expansion, then use that to decide the new length.

(define integrate-envelope
  (lambda (e sum)
    (if (or (null? e) (null? (cddr e)))
	sum
      (integrate-envelope (cddr e) (+ sum (* (+ (cadr e) (cadddr e)) .5 (- (caddr e) (car e))))))))

(define max-x
  (lambda (e)
    (if (null? (cddr e))
	(car e)
      (max-x (cddr e)))))

(define expsnd
  (lambda (gr-env)
    (let* ((dur (/ (* (/ (frames) (srate)) (integrate-envelope gr-env 0.0)) (max-x gr-env)))
	   (gr (make-granulate :expansion (cadr gr-env) :jitter 0))
	   (ge (make-env :envelope gr-env :duration dur))
	   (sound-len (round (* (srate) dur)))
	   (len (max sound-len (frames)))
	   (out-data (make-vct len))
	   (inctr 0))
      (do ((i 0 (1+ i)))
	  ((= i sound-len))
	(vct-set! out-data i 
		  (granulate gr (lambda (dir)
				  (let ((val (sample inctr)))
				    (set! inctr (+ inctr 1))
				    val))))
	(mus-set-increment gr (env ge)))
      (vct->samples 0 len out-data))))

; (expsnd '(0 1 2 .4))
; (expsnd '(0 .5 2 2.0))


;;; -------- cross-synthesis
;;;
;;; CLM version is in clm.html

(define cross-synthesis
  (lambda (cross-snd amp fftsize r)
    ;; cross-snd is the index of the other sound (as opposed to the map-chan sound)
    (let* ((freq-inc (/ fftsize 2))
	   (fdr (make-vct fftsize))
	   (fdi (make-vct fftsize))
	   (spectr (make-vct freq-inc))
	   (diffs (make-vct freq-inc))
	   (inctr 0)
	   (ctr freq-inc)
	   (radius (/ r fftsize))
	   (bin (/ (srate) fftsize))
	   (formants (make-vector freq-inc)))
      (do ((i 0 (1+ i)))
	  ((= i freq-inc))
	(vector-set! formants i (make-formant radius (* i bin))))
      (lambda (inval)
	(if inval
	    (let ((outval 0.0))
	      (if (= ctr freq-inc)
		  (begin
		   (do ((i 0 (1+ i)))
		       ((= i fftsize))
		     (vct-set! fdr i (sample inctr cross-snd))
		     (set! inctr (+ inctr 1)))
		   (set! inctr (- inctr freq-inc))
		   (clear-array fdi)
		   (mus-fft fdr fdi fftsize 1)
		   (rectangular->polar fdr fdi)
		   ;; (spectrum fdr fdi #f fftsize 2) is equivalent to the last 3 lines
		   (do ((i 0 (1+ i)))
		       ((= i freq-inc))
		     (vct-set! diffs i (/ (- (vct-ref fdr i) (vct-ref spectr i)) freq-inc)))
		   (set! ctr 0)))
	      (set! ctr (+ ctr 1))
	      (vct-add! spectr diffs)
	      (do ((i 0 (1+ i)))
		  ((= i freq-inc))
		(set! outval (+ outval (* (vct-ref spectr i) (formant (vector-ref formants i) inval)))))
	      (* amp outval)))))))

; (map-chan (cross-synthesis 1 .5 128 6.0))

;;; similar ideas can be used for spectral cross-fades, etc -- for example:

(define voiced->unvoiced
  (lambda (amp fftsize r tempo)
    (let* ((freq-inc (/ fftsize 2))
	   (fdr (make-vct fftsize))
	   (fdi (make-vct fftsize))
	   (spectr (make-vct freq-inc))
	   (diffs (make-vct freq-inc))
	   (noi (make-rand (/ (srate) 3)))
	   (inctr 0)
	   (ctr freq-inc)
	   (radius (/ r fftsize))
	   (bin (/ (srate) fftsize))
	   (len (frames))
	   (outlen (inexact->exact (/ len tempo)))
	   (hop (inexact->exact (* freq-inc tempo)))
	   (out-data (make-vct (max len outlen)))
	   (formants (make-vector freq-inc))
	   (old-peak-amp 0.0)
	   (new-peak-amp 0.0))
      (do ((i 0 (1+ i)))
	  ((= i freq-inc))
	(vector-set! formants i (make-formant radius (* i bin))))
      (call-with-current-continuation ; setup non-local exit (for C-g interrupt)
       (lambda (break)                ;   now (break value) will exit the call/cc returning value
	 (do ((k 0 (1+ k)))           ; this is our actual loop 
	     ((= k outlen))
	   (let ((outval 0.0))
	     (if (= ctr freq-inc)
		 (begin
		  (if (abort?)               ; let interface run
		      (break "interrupted")) ;   if C-g exit the loop returning th string "interrupted"
		  (do ((i 0 (1+ i))
		       (curctr inctr (1+ curctr)))
		      ((= i fftsize))
		    (let ((samp (sample curctr)))
		      (if (> (abs samp) old-peak-amp) (set! old-peak-amp (abs samp)))
		      (vct-set! fdr i samp)))
		  (spectrum fdr fdi #f fftsize 2)
		  (set! inctr (+ hop inctr))
		  (do ((i 0 (1+ i)))
		      ((= i freq-inc))
		    (vct-set! diffs i (/ (- (vct-ref fdr i) (vct-ref spectr i)) freq-inc)))
		  (set! ctr 0)))
	     (set! ctr (+ ctr 1))
	     (vct-add! spectr diffs)
	     (do ((i 0 (1+ i)))
		 ((= i freq-inc))
	       (set! outval (+ outval (* (vct-ref spectr i) (formant (vector-ref formants i) (rand noi))))))
	     (if (> (abs outval) new-peak-amp) (set! new-peak-amp (abs outval)))
	     (vct-set! out-data k outval)))
	 (vct-scale! out-data (* amp (/ old-peak-amp new-peak-amp)))
	 (vct->samples 0 (max len outlen) out-data)
	 (play))))))

;;; this example also shows how to let the rest of Snd run during a long computation,
;;;  and a simple way to jump out of a loop if C-g is typed.
; (voiced->unvoiced 1.0 256 2.0 2.0)



;;; -------- convolution example

(define cnvtest
  ;; returns new max sample
  (lambda (snd0 snd1 amp)
    (let* ((flt-len (frames snd0))
	   (total-len (+ flt-len (frames snd1)))
	   (cnv (make-convolve :filter (samples->vct 0 flt-len snd0)))
	   (inctr 0)
	   (max-samp 0.0)
	   (out-data (make-vct total-len)))
      (do ((i 0 (1+ i)))
	  ((= i total-len))
	(let ((samp (* amp (convolve cnv
				     (lambda (dir)
				       (let ((val (sample inctr snd1)))
					 (set! inctr (+ inctr 1))
					 val))))))
	  (if (> (abs samp) max-samp) (set! max-samp (abs samp)))
	  (vct-set! out-data i samp)))
      (vct->samples 0 total-len out-data snd1)
      (if (> max-samp 1.0) (set-y-bounds (- max-samp) max-samp snd1))
      max-samp)))

; (cnvtest 0 1 .1)


;;; -------- reverb (1-channel in this example)
;;;
;;; CLM version is jcrev.ins

(define jc-reverb
  (lambda (decay-dur low-pass volume amp-env)
    (let* ((allpass1 (make-all-pass -0.700 0.700 1051))
	   (allpass2 (make-all-pass -0.700 0.700  337))
	   (allpass3 (make-all-pass -0.700 0.700  113))
	   (comb1 (make-comb 0.742 4799))
	   (comb2 (make-comb 0.733 4999))
	   (comb3 (make-comb 0.715 5399))
	   (comb4 (make-comb 0.697 5801))
	   (outdel1 (make-delay (round (* .013 (srate)))))
	   (comb-sum 0.0)
	   (comb-sum-1 0.0)
	   (comb-sum-2 0.0)
	   (all-sums 0.0)
	   (delA 0.0)
	   (delB 0.0)
	   (dur (+ decay-dur (/ (frames) (srate))))
	   (envA (if amp-env (make-env :envelope amp-env :scaler volume :duration dur) #f))
	   (len (round (* dur (srate)))))
      (map-chan
       (lambda (inval)
	 (if inval
	     (let ((allpass-sum (all-pass allpass3 (all-pass allpass2 (all-pass allpass1 inval)))))
	       (set! comb-sum-2 comb-sum-1)
	       (set! comb-sum-1 comb-sum)
	       (set! comb-sum 
		     (+ (comb comb1 allpass-sum)
			(comb comb2 allpass-sum)
			(comb comb3 allpass-sum)
			(comb comb4 allpass-sum)))
	       (if low-pass
		   (set! all-sums (+ (* .25 (+ comb-sum comb-sum-2)) (* .5 comb-sum-1)))
		 (set! all-sums comb-sum))
	       (+ inval
		  (if envA
		      (* (env envA) (delay outdel1 all-sums))
		    (* volume (delay outdel1 all-sums)))))))
       0 (round (* dur (srate)))))))

; (jc-reverb 2.0 #f .1 #f)


;;; -------- scissor-tailed flycatcher
;;;
;;; mix a scissor-tailed flycatcher call into the current sound
;;; CLM version is bigbird.ins (see bird.ins and bird.clm for lots more)

(define sum-partials
  (lambda (lst sum)
    (if (null? lst)
	sum
      (sum-partials (cddr lst) (+ sum (cadr lst))))))

(define scale-partials
  (lambda (lst scl newlst)
    (if (null? lst)
	newlst
      (scale-partials (cddr lst) scl (append newlst (list (car lst) (* scl (cadr lst))))))))

(define normalize-partials
  (lambda (lst)
    (scale-partials lst (/ 1.0 (sum-partials lst 0.0)) '())))

(define bigbird
  (lambda (start dur frequency freqskew amplitude
		 freq-envelope amp-envelope partials
		 lpcoeff)
    (let* ((gls-env (make-env freq-envelope (hz->radians freqskew) dur))
	   (os (make-oscil :frequency frequency))
	   (fil (make-one-pole lpcoeff (- 1.0 lpcoeff)))
	   (coeffs (partials->polynomial (normalize-partials partials)))
	   (amp-env (make-env amp-envelope amplitude dur))
	   (len (round (* (srate) dur)))
	   (beg (round (* (srate) start)))
	   (out-data (make-vct len)))
      (do ((i 0 (1+ i)))
	  ((= i len))
	(vct-set! out-data i
		  (+ (sample (+ i beg))
		     (one-pole fil (* (env amp-env)
				      (polynomial coeffs
						  (oscil os (env gls-env))))))))
      (vct->samples beg len out-data))))

(define scissor
  (lambda (begin-time)
    (let ((scissorf '(0 0  40 1  60 1  100 0)))
      (bigbird begin-time 0.05 1800 1800 .2 
	       scissorf 
	       '(0 0  25 1  75 1  100 0) 
	       '(1 .5  2 1  3 .5  4 .1  5 .01)
	       1.0))))

; (scissor 2.0)


;;; -------- fm-violin
;;;
;;; here we're using the keyword stuff in guile/ice-9/optargs.scm
;;; CLM version is v.ins, C version is in sndlib.html

(use-modules (ice-9 optargs))

(define (keyword->symbol kw)
  (let ((sym (symbol->string (keyword-dash-symbol kw))))
    (string->symbol (substring sym 1 (string-length sym)))))

(define pi 3.141592653589793)

(define fm-violin 
  (lambda* (startime dur frequency amplitude #&key
	    (fm-index 1.0)
	    (amp-env '(0 0  25 1  75 1  100 0))
	    (periodic-vibrato-rate 5.0) (random-vibrato-rate 16.0)
	    (periodic-vibrato-amplitude 0.0025) (random-vibrato-amplitude 0.005)
	    (noise-amount 0.0) (noise-freq 1000.0)
	    (ind-noise-freq 10.0) (ind-noise-amount 0.0)
	    (amp-noise-freq 20.0) (amp-noise-amount 0.0)
	    (gliss-env '(0 0  100 0)) (glissando-amount 0.0) 
	    (fm1-env '(0 1  25 .4  75 .6  100 0)) (fm1-rat 1.0) (fm1-index #f) 
	    (fm2-env '(0 1  25 .4  75 .6  100 0)) (fm2-rat 3.0)	(fm2-index #f) 
	    (fm3-env '(0 1  25 .4  75 .6  100 0)) (fm3-rat 4.0) (fm3-index #f)
	    (base 1.0)
	    (reverb-amount 0.01)
	    (degree #f) (distance 1.0) (degrees #f)
	    #&allow-other-keys)
    (let* ((beg (floor (* startime (srate))))
	   (len (floor (* dur (srate))))
	   (end (+ beg len))
	   (frq-scl (hz->radians frequency))
	   (modulate (not (zero? fm-index)))
	   (maxdev (* frq-scl fm-index))
	   (logfreq (log frequency))
	   (sqrtfreq (sqrt frequency))
	   (index1 (or fm1-index (min pi (* maxdev (/ 5.0 logfreq)))))
	   (index2 (or fm2-index (min pi (* maxdev 3.0 (/ (- 8.5 logfreq) (+ 3.0 (* frequency .001)))))))
	   (index3 (or fm3-index (min pi (* maxdev (/ 4.0 sqrtfreq)))))
	   (easy-case (and (zero? noise-amount)
			   (equal? fm1-env fm2-env)
			   (equal? fm1-env fm3-env)
			   (= fm1-rat (floor fm1-rat))
			   (= fm2-rat (floor fm2-rat))
			   (= fm3-rat (floor fm3-rat))))
	   (coeffs (and easy-case modulate
			(partials->polynomial
			 (list fm1-rat index1
			       (floor (/ fm2-rat fm1-rat)) index2
			       (floor (/ fm3-rat fm1-rat)) index3))))
	   (norm (or (and easy-case modulate 1.0) index1))
	   (carrier (make-oscil frequency))
	   (fmosc1  (and modulate (make-oscil (* fm1-rat frequency))))
	   (fmosc2  (and modulate (or easy-case (make-oscil (* fm2-rat frequency)))))
	   (fmosc3  (and modulate (or easy-case (make-oscil (* fm3-rat frequency)))))
	   (ampf  (make-env amp-env :scaler amplitude :base base :duration dur))
	   (indf1 (and modulate (make-env fm1-env norm :duration dur)))
	   (indf2 (and modulate (or easy-case (make-env fm2-env index2 :duration dur))))
	   (indf3 (and modulate (or easy-case (make-env fm3-env index3 :duration dur))))
	   (frqf (make-env gliss-env (* glissando-amount frq-scl) :duration dur))
	   (pervib (make-triangle-wave periodic-vibrato-rate (* periodic-vibrato-amplitude frq-scl)))
	   (ranvib (make-rand-interp random-vibrato-rate (* random-vibrato-amplitude frq-scl)))
	   (fm-noi (if (not (= 0.0 noise-amount))
		       (make-rand noise-freq (* pi noise-amount))
		       #f))
	   (ind-noi (if (and (not (= 0.0 ind-noise-amount)) (not (= 0.0 ind-noise-freq)))
			(make-rand-interp ind-noise-freq ind-noise-amount)
			#f))
	   (amp-noi (if (and (not (= 0.0 amp-noise-amount)) (not (= 0.0 amp-noise-freq)))
			(make-rand-interp amp-noise-freq amp-noise-amount)
			#f))
	   (vib 0.0) 
	   (modulation 0.0)
	   (loc (make-locsig :degree (or degree degrees (random 90.0)) :reverb reverb-amount :distance distance))
	   (fuzz 0.0)
	   (ind-fuzz 1.0)
	   (amp-fuzz 1.0)
	   (out-data (make-vct len)))
      (do ((i 0 (1+ i)))
	  ((= i len))
	(if (not (= 0.0 noise-amount))
	    (set! fuzz (rand fm-noi)))
	(set! vib (+ (env frqf) (triangle-wave pervib) (rand-interp ranvib)))
	(if ind-noi (set! ind-fuzz (+ 1.0 (rand-interp ind-noi))))
	(if amp-noi (set! amp-fuzz (+ 1.0 (rand-interp amp-noi))))
	(if modulate
	    (if easy-case
		(set! modulation
		      (* (env indf1) 
			 (polynomial coeffs (oscil fmosc1 vib)))) ;(* vib fm1-rat)??
		(set! modulation
		      (+ (* (env indf1) (oscil fmosc1 (+ (* fm1-rat vib) fuzz)))
			 (* (env indf2) (oscil fmosc2 (+ (* fm2-rat vib) fuzz)))
			 (* (env indf3) (oscil fmosc3 (+ (* fm3-rat vib) fuzz)))))))
	(vct-set! out-data i 
		  (+ (sample (+ i beg))
		     (* (env ampf) amp-fuzz
			(oscil carrier (+ vib (* ind-fuzz modulation)))))))
      (vct->samples beg len out-data))))

; (fm-violin 0 1 440 .1 #:fm-index 2.0)

;;; this keyword syntax is different from the clm2scm module's in that 
;;;   they've prepended the "#" for some reason.  I like Common Lisp's
;;;   better.


;;; -------- zipper "crossfade"
;;;
;;; create the 'digital zipper' effect
;;; a not-very-debonair way to fade out file1 and fade in file2
;;; CLM version in zipper.ins

;(use-modules (ice-9 optargs))

;(define max-envelope
;  (lambda (e mx)
;    (if (null? e)
;	mx
;      (max-envelope (cddr e) (max mx (abs (cadr e)))))))

(define zipper 
  (lambda* (beg dur file1 file2 ramp-envelope frame-size #&optional (ramp-envelope-base 1.0) (frame-envelope #f))
  ;; pan between file1 and file2 using a "zipper-like" effect
  ;; ramp-env at 0 => all file1, at 1 => all file2, in between a mixture
  ;; frame-size is the basic speed at which the mixture takes place (dependent also on frame-env)
  ;; ramp-env-base is the base of the panning envelope (for exponential pans)
  ;; frame-env affects the frame size -- don't let it get to 0!!
  (let* ((f1 (make-file->sample file1))
	 (f2 (make-file->sample file2))
	 (start (floor (* (srate) beg)))
	 (len (floor (* (srate) dur)))
	 (end (+ start len))
	 (tframe-envelope (or frame-envelope '(0 1 1 1)))
	 (fe (make-env tframe-envelope :scaler (* (srate) frame-size) :duration dur))
	 (maxframe (max-envelope tframe-envelope 0.0))
	 (ctr1 0)
	 (ctr2 0)
	 (frame-loc 0)
	 (ramp-loc 0.0)
	 (rampe (make-env ramp-envelope :duration dur :base ramp-envelope-base))
	 (trigger 0)
	 (frame-samples (1+ (ceiling (* (srate) maxframe frame-size))))
	 (frame (make-vct frame-samples))
	 (frame1 (make-vct frame-samples))
	 (frame2 (make-vct frame-samples))
	 (cursamples 0)
	 (low-start (/ 20.0 (srate)))
	 (high-start (- 1.0 low-start))
	 (out-data (make-vct len)))
    (do ((i 0 (1+ i)))
	((= i len))
      (let ((insamp (sample (+ start i))))
	(set! ramp-loc (env rampe))
	(set! frame-samples (floor (env fe)))
	;; fe's duration assumes it's being called on every sample, but we only need this value every once in a while
	(if (<= ramp-loc low-start)
	    (begin
	      (vct-set! out-data i (+ insamp (file->sample f1 ctr1)))
	      (set! ctr1 (+ ctr1 1)))
	    (if (>= ramp-loc high-start)
		(begin
		  (vct-set! out-data i (+ insamp (file->sample f2 ctr2)))
		  (set! ctr2 (+ ctr2 1)))
		;; else we're in the ramp phase
		;;  read frame if we're within its bounds
		(if (= trigger 0)
		    (vct-set! out-data i (+ insamp (vct-ref frame frame-loc)))
		    ;; now get next portion of the ramp
		    (begin
		      (set! cursamples frame-samples)
		      (let* ((changept (floor (* cursamples ramp-loc)))
			     (samp1 (/ 1.0 (- 1.0 ramp-loc)))
			     (samp2 (/ 1.0 ramp-loc)))
			(do ((k 0 (1+ k)))
			    ((= k cursamples))
			 (vct-set! frame1 k (file->sample f1 ctr1))
			 (set! ctr1 (+ ctr1 1))
			 (vct-set! frame2 k (file->sample f2 ctr2))
			 (set! ctr2 (+ ctr2 1)))
			;; now resample each dependent on location in ramp (samp1 and samp2 are increments)
			(clear-array frame)
			(let ((start-ctr 0.0))
			  (do ((k 0 (1+ k)))
			      ((= k changept))
			    (let* ((ictr (floor start-ctr))
				   (y0 (vct-ref frame2 ictr))
				   (y1 (vct-ref frame2 (+ ictr 1))))
			      (vct-set! frame k (+ y0 (* (- y1 y0) (- start-ctr ictr))))
			      (set! start-ctr (+ start-ctr samp2)))))
			(let ((start-ctr 0.0)
			      (m changept))
			  (do ((k 0 (1+ k)))
			      ((= k (- cursamples changept)))
			    (let* ((ictr (floor start-ctr))
				   (y0 (vct-ref frame1 ictr))
				   (y1 (vct-ref frame1 (+ ictr 1))))
			      (vct-set! frame m (+ y0 (* (- y1 y0) (- start-ctr ictr))))
			      (set! start-ctr (+ start-ctr samp1))
			      (set! m (+ m 1)))))
			(vct-set! out-data i (+ insamp (vct-ref frame 0))))))))
	(set! frame-loc (+ frame-loc 1))
	(set! trigger 0)
	(if (>= frame-loc cursamples)
	    (begin
	      (set! frame-loc 0)
	      (set! trigger 1)))))
    (vct->samples start len out-data)
    )))

;;; this is also good if the same file is used twice -- sort of like a CD player gone berserk

; (zipper 0 1 "fyow.snd" "now.snd" '(0 0 1 1) .05)
; (zipper 0 3 "mb.snd" "fyow.snd" '(0 0 1.0 0 1.5 1.0 3.0 1.0) .025)


;;; -------- FOF example

(define two-pi (* 2 3.141592653589793))

(define fofins 
  (lambda* (beg dur frq amp vib f0 a0 f1 a1 f2 a2 #&optional (ae '(0 0 25 1 75 1 100 0)))
    (let* ((start (floor (* beg (srate))))
	   (len (floor (* dur (srate))))
	   (ampf (make-env :envelope ae :scaler amp :duration dur))
	   (frq0 (hz->radians f0))
	   (frq1 (hz->radians f1))
	   (frq2 (hz->radians f2))
	   (foflen (if (= (srate) 22050) 100 200))
	   (vibr (make-oscil :frequency 6))
	   (win-freq (/ two-pi foflen))
	   (foftab (make-vct foflen))
	   (wt0 (make-wave-train :wave foftab :frequency frq))
	   (out-data (make-vct len)))
    (do ((i 0 (1+ i)))
	((= i foflen))
      (vct-set! foftab i (* (+ (* a0 (sin (* i frq0)))
			       (* a1 (sin (* i frq1)))
			       (* a2 (sin (* i frq2))))
			    .5 (- 1.0 (cos (* i win-freq))))))
    (do ((i 0 (1+ i)))
	((= i len))
      (vct-set! out-data i
		(+ (sample (+ i start))
		   (* (env ampf) 
		      (wave-train wt0 (* vib (oscil vibr)))))))
    (vct->samples start len out-data))))

; (fofins 0 1 270 .2 .001 730 .6 1090 .3 2440 .1)


;;; -------- phase vocoder --------
;;;
;;; this is a translation of Michael Klingbeil's pvoc.ins in CLM

(define ifloor (lambda (n) (inexact->exact (floor n))))
(define pi 3.141592653589793)

(define pvoc
  (lambda* (#&key
	   (fftsize 512) (overlap 4) (time 1.0)
	   (pitch 1.0) (gate 0.0) (hoffset 0.0))
    (let* ((len (frames))
	   (filptr 0)           ; index into the file
	   (pi2 (* 2 pi))       ; handy constant
	   (sr (srate))
	   (N fftsize)          ; fft size
	   (N2 (ifloor (/ N 2)))  ;; half the fft size
	   (Nw fftsize) ;; window size -- currently restricted to the fftsize
	   (D (ifloor (/ fftsize overlap))) ; decimation factor (how often do we take an fft)
	   (interp (* (ifloor (/ fftsize overlap)) time)) ; interpolation factor how often do we synthesize
	   (windowsum 0.0)        ; for window normalization
	   ;; take a resynthesis gate specificed in dB, convert to linear amplitude
	   (syngate (if (= 0.0 gate) 0.0 (expt 10 (/ (- (abs gate)) 20))))
	   (poffset (hz->radians hoffset))
	   (window (make-vct Nw)) ; array for the window
	   (fdr (make-vct N))     ; buffer for real fft data
	   (fdi (make-vct N))     ; buffer for imaginary fft data
	   (lastphase (make-vct N2)) ;; last phase change
	   (lastamp (make-vct N2)) ;; last sampled amplitude
	   (lastfreq (make-vct N2)) ;; last sampled frequency
	   (ampinc (make-vct N2)) ;; amplitude interpolation increment
	   (freqinc (make-vct N2)) ;; frequency interpolation increments
	   ;; expresses the fundamental in terms of radians per OUTPUT sample
	   (fundamental (/ pi2 N))
	   (output interp)      ; count of samples that have been output
	   (resynth-oscils (make-vector N2))  ; synthesis oscillators
	   (nextpct 10.0)       ; how often to print out the percentage complete message
	   (outlen (ifloor (* time len)))
	   (out-data (make-vct (max len outlen))))
      ;; setup oscillators
      (do ((i 0 (1+ i)))
	  ((= i N2))
	(vector-set! resynth-oscils i (make-oscil :frequency 0)))
      ;; set-up the analysis window here
      (set! windowsum 0.0)
      ;; create a Hamming window (Moore p. 251)
      (do ((k 0 (1+ k)))
	  ((= k Nw))
	(let ((val (- 0.54 (* 0.46 (cos (* 2 pi (/ k (- Nw 1))))))))
	  (vct-set! window k val)
	  (set! windowsum (+ windowsum val))))
      ;; normalize window
      (set! windowsum (/ 2.0 windowsum))
      ;; loop over normalizing the window
      (vct-scale! window windowsum)
      (call-with-current-continuation
       (lambda (break)
	 (do ((i 0 (1+ i)))
	     ((>= i outlen))
	   ;; begin the master run loop
	   (if (>= output interp) ;; if all the samples have been output then do the next frame
	       (let ((buffix (modulo filptr N)))
					; buffix is the index into the input buffer
					; it wraps around circularly as time increases in the input
		 (if (abort?) (break "interrupted"))
		 (set! output 0)       ; reset the output sample counter
		 ;; save the old amplitudes and frequencies
		 (vct-fill! lastamp 0.0)
		 (vct-fill! lastfreq 0.0)
		 (vct-add! lastamp fdr)
		 (vct-add! lastfreq fdi)
		 (do ((k 0 (1+ k)))
		     ((= k N))
		   ;; apply the window and then stuff into the input array
		   (vct-set! fdr buffix (* (vct-ref window k) (sample filptr)))
		   (set! filptr (1+ filptr))
		   ;; increment the buffer index with wrap around
		   (set! buffix (1+ buffix))
		   (if (>= buffix N) (set! buffix 0)))
		 ;; rewind the file for the next hop
		 (set! filptr (- filptr (- N D)))
		 ;; no imaginary component input so zero out fdi
		 (vct-fill! fdi 0.0)
		 ;; compute the fft
		 (mus-fft fdr fdi N 1)
		 ;; now convert into magnitude and interpolated frequency
		 (do ((k 0 (1+ k)))
		     ((= k N2))
		   (let* ((a (vct-ref fdr k))
			  (b (vct-ref fdi k))
			  (mag (* (sqrt (+ (* a a) (* b b)))))
			  (phase 0)
			  (phasediff 0))
		     (vct-set! fdr k mag)    ;; current amp stored in fdr
		     ;; mag is always positive
		     ;; if it is zero then the phase difference is zero
		     (if (> mag 0)
			 (begin
			  (set! phase (- (atan b a)))
			  (set! phasediff (- phase (vct-ref lastphase k)))
			  (vct-set! lastphase k phase)
			  ;; frequency wrapping from Moore p. 254
			  (if (> phasediff pi) (do () ((<= phasediff pi)) (set! phasediff (- phasediff pi2))))
			  (if (< phasediff (- pi)) (do () ((>= phasediff (- pi))) (set! phasediff (+ phasediff pi2))))))
		     ;; current frequency stored in fdi
		     ;; scale by the pitch transposition
		     (vct-set! fdi k 
			       (* pitch (+ (/ (* phasediff sr) (* D sr))
					   (* k fundamental)
					   poffset)))
		     ;; resynthesis gating
		     (if (< (vct-ref fdr k) syngate) (vct-set! fdr k 0.0))
		     ;; take (vct-ref lastamp k) and count up to (vct-ref fdr k)
		     ;; interpolating by ampinc
		     (vct-set! ampinc k (/ (- (vct-ref fdr k) (vct-ref lastamp k)) interp))
		     ;; take (vct-ref lastfreq k) and count up to (vct-ref fdi k)
		     ;; interpolating by freqinc
		     (vct-set! freqinc k (/ (- (vct-ref fdi k) (vct-ref lastfreq k)) interp))))))
	   ;; loop over the partials interpolate frequency and amplitude
	   (vct-add! lastamp ampinc)
	   (vct-add! lastfreq freqinc)
	   (let ((sum 0.0))
	     (do ((k 0 (1+ k)))
		 ((= k N2))
	       (let ((curamp (vct-ref lastamp k)))
		 (if (> curamp 0.0)
		   (set! sum (+ sum (* curamp (oscil (vector-ref resynth-oscils k) (vct-ref lastfreq k))))))))
	     (vct-set! out-data i sum))
	   (set! output (1+ output)))
	 (vct->samples 0 (max len outlen) out-data))))))


;;;-------- mix with envelope on mixed-in file
;;;
;;; there are lots of ways to do this; this version uses functions from Snd, CLM, and Sndlib.

(define enveloped-mix
  (lambda (filename beg env)
    (let ((len (sound-frames filename))
	  (tmpfil (open-sound-output "/tmp/tmp.snd" 22050 1 snd-16-linear next-sound-file ""))
	  (mx (make-mixer 1 1.0))
	  (envs (make-vector 1))
	  (inenvs (make-vector 1)))
      (close-sound-output tmpfil 0)
      (vector-set! inenvs 0 (make-env env :end len))
      (vector-set! envs 0 inenvs)
      (mus-mix "/tmp/tmp.snd" filename 0 len 0 mx envs)
      (mix "/tmp/tmp.snd" beg)
      (delete-file "/tmp/tmp.snd"))))

;(enveloped-mix "pistol.snd" 0 '(0 0 1 1 2 0))
