# Copyright (C) 1998, DGA - part of the Transcriber program
# distributed under the GNU General Public License (see COPYING file)

################################################################
# Read single-level segmentations

# Read any file
proc ReadFile {fileName} {
   global v

   set channel [open $fileName r]
   # Read with chosen encoding (intended for tcl/tk 8.1 only)
   if {[info exists v(encoding)] && $v(encoding) != ""} {
      fconfigure $channel -encoding $v(encoding)
   }
   set text [read -nonewline $channel]
   close $channel
   return $text
}


# Read TIMIT format
proc ReadSegmtTimit {content} {
   set segmt {}
   foreach line [string trim [split $content "\n"]] {
      if {$line == ""} continue
      set begin [expr [lindex $line 0]/16000.0]
      set end   [expr [lindex $line 1]/16000.0]
      set text  [lrange $line 2 end]
      lappend segmt [list $begin $end $text]
   }
   return $segmt
}

# Read OGI Lola format
proc ReadSegmtLola {content} {
   set mpf 10.0
   set header 1
   set segmt {}
   foreach line [string trim [split $content "\n"]] {
      if {$line == ""} continue
      if {$header} {
	 switch -glob -- $line {
	    "MillisecondsPerFrame:*" {
	       set mpf [lindex $line 1] }
	    "END OF HEADER" {
	       set header 0
	    }
	 }
      } else {
	 set begin [expr [lindex $line 0]*$mpf/1000.0]
	 set end   [expr [lindex $line 1]*$mpf/1000.0]
	 set text  [lrange $line 2 end]
	 lappend segmt [list $begin $end $text]
      }
   }
   return $segmt
}

# Read xwaves label format
proc ReadSegmtLab {content} {
   set end 0.0
   set header 1
   set segmt {}
   foreach line [string trim [split $content "\n"]] {
      if {$line == ""} continue
      if {$header} {
	 switch -glob -- $line {
	    "\#" {
	       set header 0
	    }
	 }
      } else {
	 set begin $end
	 set end   [lindex $line 0]
	 set text  [lrange $line 2 end]
	 lappend segmt [list $begin $end $text]
      }
   }
   return $segmt
}

# Read Limsi LBL
proc ReadSegmtLbl {content} {
   set segmt {}
   set begin 0.0
   foreach line [string trim [split $content "\n"]] {
      if {$line == ""} continue
      set end   [lindex $line 0]
      if {$end > $begin} {
	 lappend segmt [list $begin $end $text]
      }
      set text  [lrange $line 1 end]
      set begin $end
   }
   return $segmt
}

# Open a new independant segmentation file
# This function is not yet interfaced (has to be called manually)
proc OpenSegmt {} {
   global v

   # Choose one empty seg
   set i 3
   while {[info exists v(trans,seg$i)]} {
      incr i
   }
   set seg "seg$i"

   set types [subst {
      {"Label file"   {$v(ext,lbl)}}
      {"All files"   {*}}
   }]
   set name [tk_getOpenFile -filetypes $types -initialdir $v(trans,path) \
		 -title [Local "Open segmentation file"]]
   if {$name != ""} {
      set format [string trimleft [file extension $name] "."]
      switch -exact -- $format {
	 "lab" -
	 "esps"  { set v(trans,$seg) [ReadSegmtLab [ReadFile $name]] }
	 "lbl"  { set v(trans,$seg) [ReadSegmtLbl [ReadFile $name]] }
	 "lola"  { set v(trans,$seg) [ReadSegmtLola [ReadFile $name]] }
	 "phn" - "wrd" - "txt" -
	 "timit" { set v(trans,$seg) [ReadSegmtTimit [ReadFile $name]] }
	 default { error [concat [Local "Unknown format"] $format] }
      }
      foreach wavfm $v(wavfm,list) {
	 CreateSegmentWidget $wavfm $seg -fg $v(color,fg-sync) -full $v(color,bg-sync)
      }
   }
}

# Create a transcription from a segmentation
proc SegmtToTrans {segmt} {
   global v

   # Newly created transcription must follow the DTD
   ::xml::dtd::read $v(file,dtd)

   if {$v(sig,name) != ""} {
      set t0 $v(sig,min)
      set t2 $v(sig,max)
   } else {
      set t0 [lindex [lindex $segmt 0] 0]
      set t2 [lindex [lindex $segmt end] 1]
   }
   set v(trans,root) [::xml::element "Trans"]
   set id0 [::xml::element "Episode" {} -in $v(trans,root)]
   set id1 [::xml::element "Section" [list "type" "report" "startTime" $t0 "endTime" $t2] -in $id0]
   set id2 [::xml::element "Turn" [list "startTime" $t0 "endTime" $t2] -in $id1]
   foreach s $segmt {
      set t1 [lindex $s 0]
      if {$t0 < $t1} {
	 ::xml::element "Sync" [list "time" $t0] -in $id2
	 ::xml::data "" -in $id2
      }
      ::xml::element "Sync" [list "time" $t1] -in $id2
      ::xml::data [lindex $s 2] -in $id2
      set t0 [lindex $s 1]
   }
   if {$t0 < $t2} {
      ::xml::element "Sync" [list "time" $t0] -in $id2
      ::xml::data "" -in $id2
   }
}

################################################################
# Read transcriptions

# Read LDC typ (added field for topic, <bg> for Background)
proc ReadTransTyp {content} {
   global v

   array set typ2ML {"sn" "nontrans" "sr" "report" "sf" "filler"}

   ::xml::dtd::read $v(file,dtd)
   set v(trans,root) [::xml::element "Trans"]
#    if {[info exists v(file,speakers)]} {
#       $v(trans,root) addChilds [::xml::parser::read_file $v(file,speakers) -dtdname $v(file,dtd)]
#       ::speaker::register
#    }
   set episode [::xml::element "Episode" {} -in $v(trans,root)]

   # Transcription begins implicitly with a "nontrans" section
   set begin 0
   set text ""
   set type "sn"
   set topic ""
   set speaker ""
   set sec ""
   set tur ""
   set lines [split $content "\n"]
   if {$v(sig,name) != ""} {
      set lines [concat $lines [list "<sn $v(sig,max)>"]]
   }
   foreach line $lines {
      if [regexp {<([a-z][a-z0-9]*) ([0-9.]+)( ([^>]+))?>( <<((male|female|child), ((I|O), )?)?(.*)>>)?} \
	      $line match code time hasattrib attrib hasname hasgender gender hasnativ nativ name] {
	 if {[info exists begin] && $time>0} {
	    if {$time<$begin} {
	       continue
	       #error "Segments are not in right order ($time<$begin)" 
	    } elseif {$time == $begin} {
	       # This can happen for near values with the 1ms precision
	       #continue
	    }
	    if {![catch {set t $typ2ML($type)}]} {
	       if {$tur != ""} {
		  $tur setAttr "endTime" $begin
	       }
	       set tur ""
	       if {$sec != ""} {
		  $sec setAttr "endTime" $begin
	       }
	       set sec [::xml::element "Section" "type $t" -in $episode]
	       if {$topic != ""} {
		  $sec setAttr "topic" $topic
	       }
	       $sec setAttr "startTime" $begin
	    } else {
	       set t ""
	    }
	    if {$speaker != "" || $t != "" || $type == "e1" || $type == "e2"} {
	       if {$type == "e2"} {
		  set speaker [lindex $overspk 0]
	       } elseif {$type == "e1"} {
		  set speaker [lindex $overspk 1]
	       }
	       if {$tur != ""} {
		  $tur setAttr "endTime" $begin
		  if {$type == "o"} {
		     set speaker [concat [$tur getAttr "speaker"] $speaker]
		     set overspk $speaker
		     #regexp ".*SPEAKER1: ?(.*) SPEAKER2: ?(.*)" $text all t1 t2
		     #set text "\[1] $t1 \[2] $t2"
		  }
	       }
	       set tur [::xml::element "Turn" {} -in $sec]
	       if {$speaker != ""} {
		  $tur setAttr "speaker" $speaker
	       }
	       $tur setAttr "startTime" $begin
	    }
	    if {$type == "bg"} {
	       set m [expr [llength $bg]-1]
	       set bgTyp [lrange $bg 0 [expr $m-1]]
	       set bgLvl [lrange $bg $m end]
	       set attrs [list "time" $begin "type" $bgTyp "level" $bgLvl]
	       set sync [::xml::element "Background" $attrs -in $tur]
	    } else {
	       set sync [::xml::element "Sync" "time $begin" -in $tur]
	    }
	    if {$time > $begin} {
	       if {$text != ""} {::xml::data $text -in $tur}
	    }
	 } else {
	    # Just in case first line is not marked as a section
	    if {[lsearch [array names typ2ML] $code] < 0} {
	       set code "sr"
	    }
	 }
	 set begin $time
	 set type  $code
	 if {$type != "bg"} {
	    set topic [::topic::create $attrib]
	 } else {
	    set bg [string tolower $attrib]
	    set topic ""
	 }
	 regsub -all "_" $name " " name
	 set speaker ""
	 foreach onename [split $name "+"] {
	    set onename [string trim $onename]
	    lappend speaker [::speaker::create $onename "" $gender]
	 }
	 set text ""
      } else {
	 if {$text != ""} {
	    append text " "
	 }
	 append text $line
      }
   }
   if [info exists begin] {
      if {$tur != ""} {
   	 $tur setAttr "endTime" $begin
      }
      if {$sec != ""} {
   	 $sec setAttr "endTime" $begin
      }
   }
}

# Transcription in XML format
proc ReadTransXML {name} {
   global v

   # Newly created transcription must follow the DTD
   ::xml::dtd::read $v(file,dtd)
   set v(trans,root) [::xml::parser::read_file $name -keepdtd 1]
   ::speaker::register
   ::topic::register
}

# Read transcription file in several formats
# Called from: OpenTransFile, RevertTrans, StartWith
proc ReadTrans {name {soundFile ""} {format ""}} {
   global v
   
   # First, try to rescue from last autosaved file if it exists.
   AutoRescue $name

   if {$format == ""} {
      set format [string trimleft [file extension $name] "."]
   }
   DisplayMessage [Local "Cleaning up memory..."]; update
   CloseTrans -nosave

   # Try to open associated sound file as early as possible for non-xml files
   if {$format != "xml" && $format !="trs"} {
      LookForSignal $name $soundFile
   }

   DisplayMessage [Local "Reading transcription file..."]; update
   switch -exact -- $format {
      "typ"   { ReadTransTyp [ReadFile $name] }
      "lab" -
      "esps"  { SegmtToTrans [ReadSegmtLab [ReadFile $name]] }
      "lbl"   { SegmtToTrans [ReadSegmtLbl [ReadFile $name]] }
      "lola"  { SegmtToTrans [ReadSegmtLola [ReadFile $name]] }
      "phn" - "wrd" - "txt" -
      "timit" { SegmtToTrans [ReadSegmtTimit [ReadFile $name]] }
      default { ReadTransXML $name; set format "trs" }
   }
   set v(trans,name) $name
   UpdateShortName
   set v(trans,format) $format
   set v(trans,saved) 0
   set v(trans,path) [file dirname $name]
   InitModif
   GetVersion
   NormalizeTrans
   DisplayTrans
   TraceOpen

   # Try to open automatically sound file else ask user
   if {$format == "xml" || $format == "trs"} {
      LookForSignal $name $soundFile
   } else {
      # For newly created transcriptions, keep info about signal basename
      UpdateFilename
   }
   if {$v(sig,name) == ""} {
      tk_messageBox -type ok -icon warning -message \
	  [concat [Local "Please open signal for transcription"] $name]
      OpenAudioFile
      if {$v(sig,name) == ""} {
	 EmptySignal
      }
   }
}

# Open transcription file through selection box
proc OpenTransFile {} {
   global v

   if [catch {SaveIfNeeded} err] return
   set types [subst {
      {"Transcription"   {$v(ext,trs)}}
      {"Label file"   {$v(ext,lbl)}}
      {"XML format" {.xml .trs}}
      {"LDC format" {.typ}}
      {"ESPS/xwaves" {.lab}}
      {"OGI lola" {.lola}}
      {"Limsi label" {.lbl}}
      {"TIMIT" {.phn .wrd .txt}}
      {"All files"   {*}}
   }]
   set name [tk_getOpenFile -filetypes $types -initialdir $v(trans,path) \
		 -title [Local "Open transcription file"]]
   if {$name != ""} {
      if {[catch {
	 ReadTrans $name
      } error]} {
	 tk_messageBox -message $error -type ok -icon error
	 NewTrans $v(sig,name)
	 return
      }
   }
}

# Open transcription or sound file through selection box at startup
proc OpenTransOrSoundFile {} {
   global v

   set types [subst {
      {"All files"   {*}}
      {"Transcription"   {$v(ext,trs) $v(ext,lbl)}}
      {"Audio files" {$v(ext,snd)}}
   }]
   set name [tk_getOpenFile -filetypes $types -initialdir $v(trans,path) \
		 -title [Local "Open transcription or audio file"]]
   if {$name != ""} {
      if {[catch {
	 set ext [file extension $name]
	 if {[lsearch -exact [lindex [lindex $types 1] 1] $ext] >= 0} {
	    ReadTrans $name
	 } elseif {[lsearch -exact [lindex [lindex $types 2] 1] $ext] >= 0
		   || [SoundFileType $name] != "RAW"} {
	    NewTrans $name
	 } else {
	    tk_messageBox -message "Type of $name unknown" -type ok -icon error
	    NewTrans "<empty>"
	 }
      } error]} {
	 tk_messageBox -message "$error" -type ok -icon error
      } else {
	 return
      }
   }
   NewTrans "<empty>"
}

################################################################
# Write transcriptions

# By default, extend typ format with topic and background infos
# Do not convert overlapping speech to standard typ format  <o> <e.>
# but as: <t> <<speaker A + speaker B>> \n [1] ... [2] ...
proc WriteTransTyp {name {extend 1}} {
   global v
   array set ML2typ {"nontrans" "sn" "report" "sr" "filler" "sf"}

   set topic ""
   set channel [open $name w]
   set episode [$v(trans,root) getChilds "element" "Episode"]
   foreach sec [$episode getChilds "element" "Section"] {
      set type $ML2typ([$sec getAttr "type"])
      set time [$sec getAttr "startTime"]
      if {$extend} {
	 set topic [$sec getAttr topic]
	 if {$topic != ""} {
	    set topic " [::topic::get_atts $topic]"
	 }
      }
      if {$time > 0 || $type != "sn" || $extend} {
	 puts -nonewline $channel [format "<$type %.3f$topic>" $time]
      }
      set turns [$sec getChilds "element" "Turn"]
      for {set nt 0} {$nt < [llength $turns]} {incr nt} {
	 set tur [lindex $turns $nt]
	 set spk [$tur getAttr "speaker"]
	 if {$spk != ""} {
	    set spk [::speaker::name $spk]
	 }
	 if {[string index $type 0] != "s"} {
	    set time [$tur getAttr "startTime"]
	    puts -nonewline $channel [format "<t %.3f>" $time]
	 }
	 if {$spk != ""} {
	    puts $channel " <<$spk>>"
	 } elseif {$time > 0 || $type != "sn" || $extend} {
	    puts $channel ""
	 }
	 set type "t"
	 set do_nl 0
	 foreach chn [$tur getChilds] {
	    if {[$chn class] == "data"} {
	       set text [$chn getData]
	       if {$text != ""} {
		  puts -nonewline $channel $text
		  set do_nl 1
	       }
	    } elseif {[$chn class] == "element"} {
	       switch [$chn getType] {
	       "Sync" {
		  if {$type != "t"} {
		     set time [$chn getAttr "time"]
		     if {$do_nl} {puts $channel ""; set do_nl 0}
		     puts $channel [format "<b %.3f>" $time]
		  }
		  set type "b"
	       }
	       "Background" {
		  set time [$chn getAttr "time"]
		  set bgTyp [$chn getAttr "type"]
		  set bgLvl [$chn getAttr "level"]
		  # Background saved as extension of typ format
		  if {$extend} {
		     if {$do_nl} {puts $channel ""; set do_nl 0}
		     puts $channel [format "<bg %.3f $bgTyp $bgLvl>" $time]
		  }
		  set type "b"
	       }
	       "Who" {
		  set nb [$chn getAttr "nb"]
		  if {$nb > 1} { puts $channel "" }
		  puts -nonewline $channel "SPEAKER$nb: "
		  set do_nl 1
	       }
	       "Event" - "Comment" {
		  puts -nonewline $channel [StringOfEvent $chn]
		  set do_nl 1
	       }
	       }
	    }
	 }
	 if {$do_nl} {puts $channel ""; set do_nl 0}
      }
   }
   close $channel
}

proc WriteTransSTM {name} {
   global v

   #;; LABEL "F0" "Baseline//Broadcast//Speech" ""
   #;; LABEL "F1" "Spontaneous//Broadcast//Speech" ""
   #;; LABEL "F2" "Speech Over//Telephone//Channels" ""
   #;; LABEL "F3" "Speech in the//Presence of//Background Music" ""
   #;; LABEL "F4" "Speech Under//Degraded//Acoustic Conditions" ""
   #;; LABEL "F5" "Speech from//Non-Native//Speakers" ""
   #;; LABEL "FX" "All other speech" ""

   set head ""
   set time ""
   set bgLvl "off"
   set bgTyp ""
   set nxt ""
   set channel [open $name w]
   #set base [$v(trans,root) getAttr "audio_filename"]
   set base [file root [file tail $name]]
   set episode [$v(trans,root) getChilds "element" "Episode"]
   foreach sec [$episode getChilds "element" "Section"] {
      foreach tur [$sec getChilds "element" "Turn"] {
	 set turncond "f0"
	 if {[$tur getAttr "mode"] == "spontaneous"} {
	    set turncond "f1"
	 }
	 if {[$tur getAttr "channel"] == "telephone"} {
	    set turncond "f2"
	 }
	 if {[$tur getAttr "fidelity"] == "low"} {
	    set turncond "f4"
	 }
	 set spk [$tur getAttr "speaker"]
	 set gender ""
	 set scope "global"
	 if {$spk != ""} {
	    if {[llength $spk] == 1} {
	       catch {
		  set atts [::speaker::get_atts $spk]
		  set gender [lindex $atts 2]
		  set scope [lindex $atts 5]
		  if {[lindex $atts 3] == "nonnative"} {
		     if {$turncond == "f4"} {
			set turncond "fx"
		     } else {
			set turncond "f5"
		     }
		  }
	       }
	    } else {
	       set turncond "f4"
	    }
	    set spk [::speaker::name $spk]
	    if {$scope != "global"} {
	       set spk "$base $spk"
	    }
	 }
	 foreach chn [$tur getChilds] {
	    if {[$chn class] == "data"} {
	       set data [$chn getData]
	       if {$nxt != ""} {
		  if {[regexp { *([^ ]+)( .*)} $data all wrd data]} {
		     append txt [format $nxt $wrd]
		  }
		  set nxt ""
	       }
	       append txt $data
	    } elseif {[$chn class] == "element"} {
	       switch [$chn getType] {
	       "Sync" - "Background" {
		  set newtime [format %.3f [$chn getAttr "time"]]
		  catch  {
		     set bgTyp [$chn getAttr "type"]
		     set bgLvl [$chn getAttr "level"]
		  }
		  if {$bgLvl == "off"} {
		     set cond $turncond
		  } elseif {$bgTyp == "music"} {
		     if {$turncond == "f4" || $turncond == "f5"} {
			set cond "fx"
		     } else {
			set cond "f3"
		     }
		  } else {
		     if {$turncond == "f5"} {
			set cond "fx"
		     } else {
			set cond "f4"
		     }
		  }
		  if {$newtime > $time || $time == ""} {
		     set time $newtime
		     if {$head != ""} {
			puts $channel "$head $time $txt"
		     }
		  }
		  set head "$base 1 \"$spk\" $time"
		  set txt "<o,$cond,$gender> "
	       }
	       "Who" {
		  set nb [$chn getAttr "nb"]
		  append txt " \[$nb] "
	       }
	       "Comment" {
		  set desc [$chn getAttr "desc"]
		  append txt "<comment>$desc</comment>"
	       }
	       "Event" {
		  set desc [$chn getAttr "desc"]
		  set type [$chn getAttr "type"]
		  set extn [$chn getAttr "extent"]
		  if {$type == "noise"} {
		     set f(begin) "\[$desc-]"
		     set f(end) "\[-$desc]"
		     set f(instantaneous) "\[$desc]"
		  } else {
		     if {$type == "language"} {
			catch {set desc $::iso639($desc)}
		     }
		     set f(begin) "<$type=$desc>"
		     set f(end) "</$type>"
		     set f(instantaneous) "$f(begin) $f(end)"
		  }		     
		  switch $extn {
		     "previous" {
			if {[regexp {(.* )([^ ]+) *} $txt all txt prv]} {
			   append txt "$f(begin) $prv $f(end)"
			}
		     }
		     "next" {
			set nxt "$f(begin) %s $f(end)"
		     }
		     "begin" - "end" - "instantaneous" {
			append txt $f($extn)
		     }
		  }
	       }
	       }
	    }
	 }
      }
   }
   set time [format %.3f [$tur getAttr "endTime"]]
   if {$head != ""} {
      puts $channel "$head $time $txt"
   }
   close $channel
}

proc WriteTransXML {name} {
   global v

   ::xml::parser::write_file $name $v(trans,root)
}

proc WriteTransOther {name} {
   tk_messageBox -type ok -icon error -message \
       [concat $name [Local "not saved (unsupported output format)"]]
}

# Write transcription to the file in specified format (or try to guess
# it from name extension); returns chosen format.
proc WriteTrans {name {format ""}} {
   global v

   if {$format == ""} {
      set format [string trimleft [file extension $name] "."]
   }
   switch -exact -- $format {
      "trs"  -
      "xml"  { WriteTransXML $name }      
      "typ"   { WriteTransTyp  $name }
      "stm"   { WriteTransSTM  $name }
      "lab" -
      "esps"  { WriteTransOther $name }
      "lola"  { WriteTransOther $name }
      "phn" - "wrd" - "txt" -
      "timit" { WriteTransOther $name }
      "default" { WriteTransOther $name }
   }
   DisplayMessage [format [Local "Transcription %s saved."] $name]
   return $format
}

################################################################
# Open and save transcriptions (user-level)

# Create new transcription. 
# If audio file not given, ask through dialog box.
proc NewTrans {{soundFile ""}} {
   global v

   if [catch {CloseTrans} err] return

   if {[catch {
      if {$soundFile == "<empty>"} {
	 EmptySignal
      } elseif {$soundFile != "" && [file readable $soundFile]} {
	 Signal $soundFile
      } else {
	 OpenAudioFile
      }
   }]} {
      if {$v(debug)} {puts $::errorInfo}
      EmptySignal
   }

   set v(trans,name) ""
   UpdateShortName
   set v(trans,format) "trs"
   set v(trans,saved) 0
   SegmtToTrans [list [list $v(sig,min) $v(sig,max) ""]]
   InitEpisode
   InitModif
   DisplayTrans
   TraceOpen
}

# Save [as] 
#  returns empty string if save failed (or was canceled)
proc SaveTrans {{as ""} {format ""}} {
   global v

   if {[GetSegmtNb seg0] <= 0} return
   if {$format == "" || $format == "xml" || $format == "trs"} {
      set format "trs"
   }
   if {$v(trans,name) != "" && $as == "" && $v(trans,format) == $format} {
      set name $v(trans,name) 
   } else {
      if {$v(trans,name) == ""} {
	 set base [$v(trans,root) getAttr "audio_filename"]
      } else {
	 set base [file root [file tail $v(trans,name)]]
      }
      if {$format == "trs"} {
	 set types {{"Transcriptions"   {.trs .xml}}}
      } else {
	 set types [subst {{"Export format"   {.$format}}}]
      }
      lappend types {"All files"   {*}}

      set name [tk_getSaveFile -filetypes $types -defaultextension ".$format" \
		    -initialfile $base.$format -initialdir $v(trans,path) \
		    -title  "Save transcription file $as"]
      if {$name != "" && [file extension $name] == ""} {
	 append name ".$format"
      }
   }
   if {$name != ""} {
      if {[file exists $name] && $v(backup,ext) != "" 
	  && ($v(trans,name) == "" || $as != "" || !$v(trans,saved))} {
	 file copy -force -- $name "$name$v(backup,ext)"
      }
      if {[HasModifs]} {
	 UpdateVersion
      }
      if [catch {
	 WriteTrans $name $format
      } res] {
	 tk_messageBox -message "$name not saved\n$res" -type ok -icon error
	 return "" 
      }
      if {$format == "trs"} {
	 set v(trans,name) $name
	 UpdateShortName
	 set v(trans,format) $res
	 set v(trans,saved) 1
	 set v(trans,path) [file dirname $name]
	 InitModif
      }
   }
   return $name
}

proc SaveIfNeeded {} {
   global v

   if {[HasModifs]} {
      set answer [tk_messageBox -message [Local "Transcription has been modified - Save before closing?"] -type yesnocancel -icon question]
      switch $answer {
	 cancel { return -code error cancel }
	 yes    { if {[SaveTrans]==""} {return -code error cancel} }
	 no     { }
      }
   }
}

proc RevertTrans {} {
   global v

   if {$v(trans,name) != "" && [HasModifs]} {
      set answer [tk_messageBox -message [Local "Warning !\nAll changes will be lost.\nReally revert from file ?"] -type okcancel -icon warning]
      if {$answer == "ok"} {
	 InitModif
	 ReadTrans $v(trans,name) $v(sig,name)
      }
   }
}

# called from: NewTrans, ReadTrans, CloseAndDestroyTrans
proc CloseTrans {{option save}} {
   global v

   if {$option=="save"} {
      SaveIfNeeded
   }
   TraceClose
   EmptyTextFrame
   ::xml::init
   set v(trans,root) ""
   ::speaker::init
   ::topic::init
   InitSegmt seg0 seg1 seg2 bg
   set v(trans,name) ""
   UpdateShortName
   InitModif
}

proc CloseAndDestroyTrans {} {
   global v

   if [catch {
      CloseTrans
   } err] {
      return -code return 
   }
   DestroyTextFrame
   DestroySegmentWidgets
}

################################################################

proc TransInfo {} {
   global v

   return [list [GetSegmtNb seg2] [llength [::topic::all_names]] [GetSegmtNb seg1] [llength [::speaker::all_names]] [GetSegmtNb seg0] [CountWordSegmt seg0]]
}

################################################################

# Normalize the transcription by "filling the holes" with sections or turns
# and creates empty data sections between non-contiguous breakpoints

proc NormalizeTrans {} {
   global v

   if {![info exist v(trans,root)] || $v(trans,root)==""} return
   set episode [$v(trans,root) getChilds "element" "Episode"]

   # Contrain sections to be a partition of the episode
   set t1 $v(sig,min)
   foreach sec [$episode getChilds "element" "Section"] {
      set t2 [$sec getAttr "startTime"]
      if {$t2 > $t1} {
	 ::xml::element "Section" [list "type" "nontrans" "startTime" $t1 "endTime" $t2] -before $sec
      }
      set t1 [$sec getAttr "endTime"]
   }
   # Don't add a new section up to the end, because we will synchronize
   # the last breakpoint to the end of signal - else it would be done with:
   #set t2 $v(sig,max)
   #if {$t2 > $t1} {
   #   ::xml::element "Section" [list "type" "nontrans" "startTime" $t1 "endTime" $t2] -in $episode
   #}

   foreach sec [$episode getChilds "element" "Section"] {
      # Constrain turns to be a partition of each section
      set t1 [$sec getAttr "startTime"]
      foreach turn [$sec getChilds "element" "Turn"] {
	 set t2 [$turn getAttr "startTime"]
	 if {$t2 > $t1} {
	    ::xml::element "Turn" [list "startTime" $t1 "endTime" $t2] \
		-before $turn
	 }
	 set t1 [$turn getAttr "endTime"]
      }
      set t2 [$sec getAttr "endTime"]
      if {$t2 > $t1} {
	 ::xml::element "Turn" [list "startTime" $t1 "endTime" $t2] -in $sec
      }

      foreach turn [$sec getChilds "element" "Turn"] {
	 set t1 [$turn getAttr "startTime"]
	 # Each turn must begin with a sync
	 set sync [lindex [$turn getChilds "element" "Sync"] 0]
	 if {$sync == "" || [$sync getAttr "time"] > $t1} {
	    ::xml::element "Sync" [list "time" $t1] -begin $turn
	 }
	 # Create data between non-contiguous breakpoints
	 foreach elem [$turn getChilds "element"] {
	    set next [$elem getBrother "element"]
	    if {$next == ""} {
	       set t2 [$turn getAttr "endTime"]
	    } else {
	       set t2 ""
	       catch {
		  set t2 [$next getAttr "time"]
	       }
	    }
	    if {$t1 == "" || $t2 == "" || $t2 > $t1} {
	       set data [$elem getBrother]
	       if {$data == "" || 
		   ([$data class] != "data" &&
		    !([$data class] == "element"  && [$data getType] == "Who"
		      && [$data getAttr "nb"] == 1))} {
		     ::xml::data "" -after $elem
	       }
	    }
	    set t1 $t2
	 }
	 # Convert data [...] to XML tags for .typ and old .xml format
	 if {$v(trans,format) == "typ" || $v(convert_events)} {
	    foreach data [$turn getChilds "data"] {
	       ConvertData $data
	    }
	 }
      }
   }
}

################################################################

# Try to guess Events from strings in .typ - very rough anyway.
proc ConvertData {data} {
   set text [$data getData]
   if {[regexp ".*SPEAKER1: ?(.*) SPEAKER2: ?(.*)" $text all t1 t2]} {
      set text "\[1]$t1\[2]$t2"
   }
   while {[regexp "^(\[^\\\[]*)\\\[(\[^]]+)](.*)$" $text all t1 evt text]} {
      $data setData $t1
      switch -regexp -- $evt {
	 ^(1|2)$ {
	    set elem [::xml::element "Who" [list "nb" $evt] -after $data]
	 }
	 ^-?(r|i|e|n|pf|bb|bg|tx|rire|sif|ch|b|conv|pap|shh|mic|jingle|musique|indicatif|top|pi|pif|nontrans)-?$ {
	    set extn "instantaneous"
	    # For backward compability: [noise-] ... [-noise] 
	    if {[regexp "^(-)?(.*\[^-])(-)?$" $evt all start evt end]} {
	       if {$start != ""} {
		  set extn "end"
	       } elseif {$end != ""} {
		  set extn "begin"
	       }
	    }
	    set elem [::xml::element "Event" \
			  [list "desc" $evt "extent" $extn] -after $data]
	 }
	 default {
	    set elem [::xml::element "Comment" [list "desc" $evt] -after $data]
	 }
      }
      set data [::xml::data $text -after $elem]
   }
}

################################################################

# Display a (previously normalized) transcription.
# Process sequentially time markers of the transcription and
# register a unique shared time for each different value.
# Create a segmentation at section, speaker and synchro level.

proc DisplayTrans {} {
   global v
   variable ::Synchro::time

   if {![info exist v(trans,root)] || $v(trans,root)==""} return
   set episode [$v(trans,root) getChilds "element" "Episode"]

   DisplayMessage [Local "Displaying transcription..."]; update

   # Init
   set min $v(sig,min)
   set max $v(sig,max)
   InitSegmt seg0 seg1 seg2 bg
   Synchro::InitTime
   InitEditor
   update

   set t0 [Synchro::NewTime $min]; set t6 $t0; set t5 $t0; set tprev $t0
   set bgTim $t0; set bgTxt ""; set bgId ""
   foreach sec [$episode getChilds "element" "Section"] {
      set t1 [Synchro::NextTimeTag $sec "startTime"]
      # Editor button
      InsertSectionButton $sec

      # register new section segment
      set t6 [Synchro::NextTimeTag $sec "endTime"]
      set top [::section::short_name $sec] 
      AddSegmt seg2 $t1 $t6 $top $sec

      set turns [$sec getChilds "element" "Turn"] 
      foreach tur $turns {
	 set t2 [Synchro::NextTimeTag $tur "startTime"]

	 # Editor button
	 InsertTurnButton $tur
	 
	 # register new turn segment
	 set t5 [Synchro::NextTimeTag $tur "endTime"]
	 set spk [::turn::get_name $tur]
	 AddSegmt seg1 $t2 $t5 $spk $tur

	 set txt ""
	 set t3 $t2
	 foreach chn [$tur getChilds] {
	    switch [$chn class] {
	    "element" {
	       switch [$chn getType] {
	       "Sync" {
		  # register new synchro segment
		  set t4 [Synchro::NextTimeTag $chn "time"]
		  if {$time($t4)>$time($t3)} {
		     AddSegmt seg0 $t3 $t4 $txt $id
		     # Test overlap with previous turn for display
		     if {$time($tprev) > $time($t3)} {
			ChangeSyncButton $idprev over1
			ChangeSyncButton $id over2
		     }
		  }
		  set t3 $t4
		  set id $chn
		  set txt ""
		  InsertSyncButton $id
		  # for first Background breakpoint
		  if {$bgId == ""} {
		     set bgId $chn
		  }
	       }
	       "Background" {
		  set t4 [Synchro::NextTimeTag $chn "time"]
		  if {$time($t4) > $time($bgTim)} {
		     AddSegmt bg $bgTim $t4 $bgTxt $bgId
		  }
		  set bgTim $t4
		  set bgId $chn
		  foreach {bgTxt img} [ReadBackAttrib $chn] {}
		  InsertImage $chn $img
	       }
	       "Event" - "Comment" {
		  InsertEvent $chn
		  append txt [StringOfEvent $chn]
	       }
	       "Who" {
		  if {[$chn getAttr "nb"] > 1} {
		     append txt "\n"
		  }
		  InsertWho $chn
	       }
	       }
	    }
	    "data" {
	       append txt [$chn getData]
	       InsertData $chn
	    }
	    }
	 }
	 
	 # register last synchro segment
	 if {$time($t5)>$time($t3)} {
	    AddSegmt seg0 $t3 $t5 $txt $id
	    # Test overlap with previous turn for display
	    if {$time($tprev) > $time($t3)} {
	       ChangeSyncButton $idprev over1
	       ChangeSyncButton $id over2
	    }
	 }
	 # For overlapping speech
	 set idprev $id
	 set tprev $t5
      }
   }
   if {$time($t6) > $time($bgTim)} {
      AddSegmt bg $bgTim $t6 $bgTxt $bgId
   }
   #set t7 [Synchro::NewTime $max]

   # For demo purposes only
   if {[info exists v(demo)]} {
      DestroyTextFrame
      DestroySegmentWidgets
      CreateSegmentWidget .snd.w seg0 -fg $v(color,fg-sync) -full $v(color,bg-sync) -height 1 -high $v(color,hi-sync)
      destroy .demo
      frame .demo -bd 2 -relief raised
      pack .demo -expand true -fill both -side top
      text .demo.txt -wrap word  -width 40 -height 15 \
	  -fg $v(color,fg-text) -bg $v(color,bg-text) \
	  -font {courier 24 bold} -yscrollcommand [list .demo.ysc set]
      scrollbar .demo.ysc -orient vertical -command [list .demo.txt yview]
      pack .demo.txt -side left -fill both -expand true
      pack .demo.ysc -side right -fill y
      bind .demo.txt <BackSpace> {.demo.txt delete 1.0 end}
      return
   }

   # Create widgets if necessary
   CreateAllSegmentWidgets

   HomeEditor
   DisplayMessage ""
}

################################################################

# Construct text description associated with Sync tag
proc TextFromSync {bp} {
   set txt ""
   for {set tag [$bp getBrother]} {$tag != ""} {set tag [$tag getBrother]} {
      switch [$tag class] {
	 "data" {
	    append txt [$tag getData]
	 }
	 "element" {
	    switch [$tag getType] {
	       "Sync" {
		  break
	       }
	       "Background" {
		  #append txt " * "
	       }
	       "Who" {
		  set nb [$tag getAttr "nb"]
		  if {$nb > 1} {
		     append txt "\n"
		  }
	       }
	       "Event" - "Comment" {
		  append txt [StringOfEvent $tag]
	       }
	    }
	 }
      }
   }
   return $txt
}

# Get BP from which depends current tag
proc SyncBefore {tag} {
   return [$tag getBrother "element" "Sync" -1]
}
