/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "xmms.h"
#include "defskin/main.xpm"
#include "defskin/cbuttons.xpm"
#include "defskin/titlebar.xpm"
#include "defskin/shufrep.xpm"
#include "defskin/text.xpm"
#include "defskin/volume.xpm"
#include "defskin/monoster.xpm"
#include "defskin/playpaus.xpm"
#include "defskin/numbers.xpm"
#include "defskin/nums_ex.xpm"
#include "defskin/posbar.xpm"
#include "defskin/pledit.xpm"
#include "defskin/eqmain.xpm"
#include "defskin/eq_ex.xpm"

#include <ctype.h>

Skin *skin;

static gint skin_default_viscolor[24][3] =
{
	{9,34,53},
	{10,18,26},
	{0,54,108},
	{0,58,116},
	{0,62,124},
	{0,66,132},
	{0,70,140},
	{0,74,148},
	{0,78,156},
	{0,82,164},
	{0,86,172},
	{0,92,184},
	{0,98,196},
	{0,104,208},
	{0,110,220},
	{0,116,232},
	{0,122,244},
	{0,128,255},
	{0,128,255},
	{0,104,208},
	{0,80,160},
	{0,56,112},
	{0,32,64},
	{200, 200, 200}

};

static void setup_skin_masks(void)
{
	if (cfg.player_visible)
	{
		if (cfg.doublesize)
			gtk_widget_shape_combine_mask(mainwin, get_skin_pixmap(cfg.player_shaded ? SKIN_MASK_SHADE_DS : SKIN_MASK_MAIN_DS), 0, 0);
		else
			gtk_widget_shape_combine_mask(mainwin, get_skin_pixmap(cfg.player_shaded ? SKIN_MASK_SHADE : SKIN_MASK_MAIN), 0, 0);
	}
	if (cfg.eq_doublesize_linked && cfg.doublesize)
		gtk_widget_shape_combine_mask(equalizerwin, get_skin_pixmap(cfg.equalizer_shaded ? SKIN_MASK_EQ_SHADE_DS : SKIN_MASK_EQ_DS), 0, 0);
	else
		gtk_widget_shape_combine_mask(equalizerwin, get_skin_pixmap(cfg.equalizer_shaded ? SKIN_MASK_EQ_SHADE : SKIN_MASK_EQ), 0, 0);

}

static GdkBitmap *create_default_mask(GdkWindow * parent, gint w, gint h)
{
	GdkBitmap *ret;
	GdkGC *gc;
	GdkColor pattern;

	ret = gdk_pixmap_new(parent, w, h, 1);
	gc = gdk_gc_new(ret);
	pattern.pixel = 1;
	gdk_gc_set_foreground(gc, &pattern);
	gdk_draw_rectangle(ret, gc, TRUE, 0, 0, w, h);
	gdk_gc_destroy(gc);

	return ret;
}

void init_skins(void)
{
	gint i;

	skin = (Skin *) g_malloc0(sizeof (Skin));
	skin->def_main = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_main);
	skin->def_cbuttons = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_cbuttons);
	skin->def_titlebar = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_titlebar);
	skin->def_shufrep = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_shufrep);
	skin->def_text = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_text);
	skin->def_volume = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_volume);
	skin->def_monostereo = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_monoster);
	skin->def_playpause = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_playpaus);
	skin->def_numbers = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_numbers);
	skin->def_nums_ex = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_nums_ex);
	skin->def_posbar = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_posbar);
	skin->def_pledit = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_pledit);
	skin->def_eqmain = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_eqmain);
	skin->def_eq_ex = gdk_pixmap_create_from_xpm_d(mainwin->window, NULL, NULL, skin_eq_ex);

	skin->def_pledit_normal.red = 0x2400;
	skin->def_pledit_normal.green = 0x9900;
	skin->def_pledit_normal.blue = 0xffff;
	gdk_color_alloc(gdk_colormap_get_system(), &skin->def_pledit_normal);
	skin->def_pledit_current.red = 0xffff;
	skin->def_pledit_current.green = 0xee00;
	skin->def_pledit_current.blue = 0xffff;
	gdk_color_alloc(gdk_colormap_get_system(), &skin->def_pledit_current);
	skin->def_pledit_normalbg.red = 0x0A00;
	skin->def_pledit_normalbg.green = 0x1200;
	skin->def_pledit_normalbg.blue = 0x0A00;
	gdk_color_alloc(gdk_colormap_get_system(), &skin->def_pledit_normalbg);
	skin->def_pledit_selectedbg.red = 0x0A00;
	skin->def_pledit_selectedbg.green = 0x1200;
	skin->def_pledit_selectedbg.blue = 0x4A00;
	gdk_color_alloc(gdk_colormap_get_system(), &skin->def_pledit_selectedbg);
	for (i = 0; i < 24; i++)
	{
		skin->vis_color[i][0] = skin_default_viscolor[i][0];
		skin->vis_color[i][1] = skin_default_viscolor[i][1];
		skin->vis_color[i][2] = skin_default_viscolor[i][2];
	}
	skin->def_mask = create_default_mask(mainwin->window, 275, 116);
	skin->def_mask_ds = create_default_mask(mainwin->window, 550, 232);
	skin->def_mask_shade = create_default_mask(mainwin->window, 275, 14);
	skin->def_mask_shade_ds = create_default_mask(mainwin->window, 550, 28);

	setup_skin_masks();

	create_skin_window();
}

static guchar hex_char_to_int(gchar c)
{
	if(c>= '0' && c <= '9')
		return c - '0';
	return 10 + (toupper(c) - 'A');
}

GdkColor *load_skin_color(const gchar * path, const gchar * file, const gchar * section, const gchar * key)
{
	gchar *filename, *value;
	GdkColor *color = NULL;
	gint base = 0;

	filename = find_file_recursively(path, file);
	if (filename)
	{
		value = read_ini_string(filename, section, key);
		if (value)
		{
			color = g_malloc(sizeof (GdkColor));
			g_strchug(g_strchomp(value));
			if (value[0] == '#')
				base++;
			else
				base = 0;
			color->red = (hex_char_to_int(value[base]) << 12 | hex_char_to_int(value[base] + 1) << 8) | 0xFF;
			color->green = (hex_char_to_int(value[base + 2]) << 12 | hex_char_to_int(value[base + 3] + 1) << 8) | 0xFF;
			color->blue = (hex_char_to_int(value[base + 4]) << 12 | hex_char_to_int(value[base + 5] + 1) << 8) | 0xFF;

			gdk_color_alloc(gdk_colormap_get_system(), color);
			g_free(value);
		}
		g_free(filename);
	}
	return color;
}

GdkPixmap *load_skin_pixmap(const gchar * path, const gchar * file, GdkPixmap *def_pixmap)
{
	char *filename;
	GdkPixmap *pixmap = NULL, *ret = NULL;
	gint w,h, pw, ph;
	static GdkGC *skin_gc = NULL;

	filename = find_file_recursively(path, file);
	if (filename)
	{
		pixmap = read_bmp(filename);
		if(!pixmap)
			return NULL;
		gdk_window_get_size(def_pixmap, &w, &h);
		ret = gdk_pixmap_new(mainwin->window, w, h,
				     gdk_rgb_get_visual()->depth);
		if(!skin_gc)
			skin_gc = gdk_gc_new(mainwin->window);
		gdk_draw_pixmap(ret, skin_gc, def_pixmap, 0, 0, 0, 0, w, h);
		gdk_window_get_size(pixmap, &pw, &ph);
		if(pw > w)
			pw = w;
		if(ph > h)
			ph = h;
		gdk_draw_pixmap(ret, skin_gc, pixmap, 0, 0, 0, 0, pw, ph);
		gdk_pixmap_unref(pixmap);		
		g_free(filename);
	}
	return ret;
}

GdkBitmap *skin_create_transparent_mask(const gchar * path, const gchar * file, const gchar * section, GdkWindow * window, gint width, gint height, gboolean doublesize)
{
	gchar *filename;

	GdkBitmap *mask = NULL;
	GdkGC *gc = NULL;
	GdkColor pattern;
	GdkPoint *gpoints;

	gboolean created_mask = FALSE, inited_mask = FALSE;
	GArray *num, *point;
	gint i, j, k;

	if (path)
	{
		filename = find_file_recursively(path, file);
		if (filename)
		{
			if ((num = read_ini_array(filename, section, "NumPoints")) != NULL)
			{
				if ((point = read_ini_array(filename, section, "PointList")) != NULL)
				{
					mask = gdk_pixmap_new(window, width, height, 1);
					gc = gdk_gc_new(mask);

					pattern.pixel = 0;
					gdk_gc_set_foreground(gc, &pattern);
					gdk_draw_rectangle(mask, gc, TRUE, 0, 0, width, height);
					pattern.pixel = 1;
					gdk_gc_set_foreground(gc, &pattern);
					inited_mask = TRUE;

					j = 0;
					for (i = 0; i < num->len; i++)
					{
						if ((point->len - j) >= (g_array_index(num, gint, i) * 2))
						{
							created_mask = TRUE;
							gpoints = g_malloc(g_array_index(num, gint, i) * sizeof (GdkPoint));
							for (k = 0; k < g_array_index(num, gint, i); k++)
							{
								gpoints[k].x = g_array_index(point, gint, j + k * 2) * (1 + doublesize);
								gpoints[k].y = g_array_index(point, gint, j + k * 2 + 1) * (1 + doublesize);
							}
							j += k * 2;
							gdk_draw_polygon(mask, gc, TRUE, gpoints, g_array_index(num, gint, i));
							g_free(gpoints);
						}
					}
					g_array_free(num, TRUE);
				}
				g_array_free(point, TRUE);
			}
			g_free(filename);
		}
	}
	if (!inited_mask)
		return NULL;

	if (!created_mask)
		gdk_draw_rectangle(mask, gc, TRUE, 0, 0, width, height);

	gdk_gc_destroy(gc);

	return mask;
}

void load_skin_viscolor(const gchar * path, const gchar * file)
{
	FILE *f;
	gint i, c;
	gchar line[256], *filename;
	GArray *a;

	for (i = 0; i < 24; i++)
	{
		skin->vis_color[i][0] = skin_default_viscolor[i][0];
		skin->vis_color[i][1] = skin_default_viscolor[i][1];
		skin->vis_color[i][2] = skin_default_viscolor[i][2];
	}

	filename = find_file_recursively(path, file);
	if (filename)
	{
		if ((f = fopen(filename, "r")) != NULL)
		{
			for (i = 0; i < 24; i++)
			{
				if (fgets(line, 255, f))
				{
					a = string_to_garray(line);
					if (a->len > 2)
					{
						for (c = 0; c < 3; c++)
							skin->vis_color[i][c] = g_array_index(a, gint, c);
					}
					g_array_free(a, TRUE);
				}
				else
					break;

			}
			fclose(f);
		}
		g_free(filename);
	}
}

void free_skin(void)
{
	gint i;

	if (skin->main)
		gdk_pixmap_unref(skin->main);
	if (skin->cbuttons)
		gdk_pixmap_unref(skin->cbuttons);
	if (skin->titlebar)
		gdk_pixmap_unref(skin->titlebar);
	if (skin->shufrep)
		gdk_pixmap_unref(skin->shufrep);
	if (skin->text)
		gdk_pixmap_unref(skin->text);
	if (skin->volume)
		gdk_pixmap_unref(skin->volume);
	if (skin->balance)
		gdk_pixmap_unref(skin->balance);
	if (skin->monostereo)
		gdk_pixmap_unref(skin->monostereo);
	if (skin->playpause)
		gdk_pixmap_unref(skin->playpause);
	if (skin->numbers)
		gdk_pixmap_unref(skin->numbers);
	if (skin->posbar)
		gdk_pixmap_unref(skin->posbar);
	if (skin->pledit)
		gdk_pixmap_unref(skin->pledit);
	if (skin->eqmain)
		gdk_pixmap_unref(skin->eqmain);
	if (skin->eq_ex)
		gdk_pixmap_unref(skin->eq_ex);
	if (skin->mask_main)
		gdk_bitmap_unref(skin->mask_main);
	if (skin->mask_main_ds)
		gdk_bitmap_unref(skin->mask_main_ds);
	if (skin->mask_shade)
		gdk_bitmap_unref(skin->mask_shade);
	if (skin->mask_shade_ds)
		gdk_bitmap_unref(skin->mask_shade_ds);
	if (skin->mask_eq)
		gdk_bitmap_unref(skin->mask_eq);
	if (skin->mask_eq_ds)
		gdk_bitmap_unref(skin->mask_eq_ds);
	if (skin->mask_eq_shade)
		gdk_bitmap_unref(skin->mask_eq_shade);
	if (skin->mask_eq_shade_ds)
		gdk_bitmap_unref(skin->mask_eq_shade_ds);
	if (skin->pledit_normal)
		g_free(skin->pledit_normal);
	if (skin->pledit_current)
		g_free(skin->pledit_current);
	if (skin->pledit_normalbg)
		g_free(skin->pledit_normalbg);
	if (skin->pledit_selectedbg)
		g_free(skin->pledit_selectedbg);
	skin->main = NULL;
	skin->cbuttons = NULL;
	skin->titlebar = NULL;
	skin->shufrep = NULL;
	skin->text = NULL;
	skin->volume = NULL;
	skin->balance = NULL;
	skin->monostereo = NULL;
	skin->playpause = NULL;
	skin->numbers = NULL;
	skin->posbar = NULL;
	skin->pledit = NULL;
	skin->eqmain = NULL;
	skin->eq_ex = NULL;
	skin->pledit_normal = NULL;
	skin->pledit_current = NULL;
	skin->pledit_normalbg = NULL;
	skin->pledit_selectedbg = NULL;
	skin->mask_main = NULL;
	skin->mask_main_ds = NULL;
	skin->mask_shade = NULL;
	skin->mask_shade_ds = NULL;
	skin->mask_eq = NULL;
	skin->mask_eq_ds = NULL;
	skin->mask_eq_shade = NULL;
	skin->mask_eq_shade_ds = NULL;
	for (i = 0; i < 24; i++)
	{
		skin->vis_color[i][0] = skin_default_viscolor[i][0];
		skin->vis_color[i][1] = skin_default_viscolor[i][1];
		skin->vis_color[i][2] = skin_default_viscolor[i][2];
	}
}

static void _load_skin(const gchar * path, gboolean force)
{
	char *ending, *tmp = NULL, *tempdir, *unzip, *tar;

	if(!force)
	{
		if (skin->path && path)
			if (!strcmp(skin->path, path))
				return;
		if (!skin->path && !path)
			return;
		free_skin();
	}
	if (path)
	{
		if (strncmp(path, P_tmpdir, strlen(P_tmpdir)))
		{
			skin->path = g_realloc(skin->path, strlen(path) + 1);
			strcpy(skin->path, path);
		}
		ending = strrchr(path, '.');
		if (ending)
		{
			if (!strcasecmp(ending, ".zip") || !strcasecmp(ending, ".wsz") ||
			    !strcasecmp(ending, ".tgz") || !strcasecmp(ending, ".gz") ||
			    !strcasecmp(ending, ".bz2") || !strcasecmp(ending, ".tar"))
			{

				tempdir = tempnam(NULL, NULL);

				unzip = getenv("UNZIPCMD");
				if (!unzip)
					unzip = "unzip";
				tar = getenv("TARCMD");
				if (!tar)
					tar = "tar";
				if (!strcasecmp(ending, ".zip") || !strcasecmp(ending, ".wsz"))
				{
					tmp = g_strdup_printf("%s >/dev/null -o -j \"%s\" -d %s", unzip, path, tempdir);
				}
				if (!strcasecmp(ending, ".tgz") || !strcasecmp(ending, ".gz"))
				{
					tmp = g_strdup_printf("%s >/dev/null xzf \"%s\" -C %s", tar, path, tempdir);
					mkdir(tempdir, 0755);	/* Permissions: 755 */
				}
				if (!strcasecmp(ending, ".bz2"))
				{
					tmp = g_strdup_printf("%s >/dev/null xIf \"%s\" -C %s", tar, path, tempdir);
					mkdir(tempdir, 0755);	/* Permissions: 755 */
				}
				if (!strcasecmp(ending, ".tar"))
				{
					tmp = g_strdup_printf("%s >/dev/null xf \"%s\" -C %s", tar, path, tempdir);
					mkdir(tempdir, 0755);	/* Permissions: 755 */
				}
				system(tmp);
				g_free(tmp);
				load_skin(tempdir);
				del_directory(tempdir);
				free(tempdir);
				return;
			}
		}
		skin->main = load_skin_pixmap(path, "main.bmp", skin->def_main);
		skin->cbuttons = load_skin_pixmap(path, "cbuttons.bmp", skin->def_cbuttons);
		skin->titlebar = load_skin_pixmap(path, "titlebar.bmp", skin->def_titlebar);
		skin->shufrep = load_skin_pixmap(path, "shufrep.bmp", skin->def_shufrep);
		skin->text = load_skin_pixmap(path, "text.bmp", skin->def_text);
		skin->volume = load_skin_pixmap(path, "volume.bmp", skin->def_volume);
		skin->balance = load_skin_pixmap(path, "balance.bmp", skin->def_volume);
		skin->monostereo = load_skin_pixmap(path, "monoster.bmp", skin->def_monostereo);
		skin->playpause = load_skin_pixmap(path, "playpaus.bmp", skin->def_playpause);
		skin->numbers = load_skin_pixmap(path, "nums_ex.bmp", skin->def_nums_ex);
		if (!skin->numbers)
			skin->numbers = load_skin_pixmap(path, "numbers.bmp", skin->def_numbers);
		skin->posbar = load_skin_pixmap(path, "posbar.bmp", skin->def_posbar);
		skin->pledit = load_skin_pixmap(path, "pledit.bmp", skin->def_pledit);
		skin->eqmain = load_skin_pixmap(path, "eqmain.bmp", skin->def_eqmain);
		skin->eq_ex = load_skin_pixmap(path, "eq_ex.bmp", skin->def_eq_ex);
		skin->pledit_normal = load_skin_color(path, "pledit.txt", "text", "normal");
		skin->pledit_current = load_skin_color(path, "pledit.txt", "text", "current");
		skin->pledit_normalbg = load_skin_color(path, "pledit.txt", "text", "normalbg");
		skin->pledit_selectedbg = load_skin_color(path, "pledit.txt", "text", "selectedbg");
		skin->mask_main = skin_create_transparent_mask(path, "region.txt", "Normal", mainwin->window, 275, 116, FALSE);
		skin->mask_main_ds = skin_create_transparent_mask(path, "region.txt", "Normal", mainwin->window, 550, 232, TRUE);
		skin->mask_shade = skin_create_transparent_mask(path, "region.txt", "WindowShade", mainwin->window, 275, 14, FALSE);
		skin->mask_shade_ds = skin_create_transparent_mask(path, "region.txt", "WindowShade", mainwin->window, 550, 28, TRUE);
		skin->mask_eq = skin_create_transparent_mask(path, "region.txt", "Equalizer", equalizerwin->window, 275, 116, FALSE);
		skin->mask_eq_ds = skin_create_transparent_mask(path, "region.txt", "Equalizer", equalizerwin->window, 550, 232, TRUE);
		skin->mask_eq_shade = skin_create_transparent_mask(path, "region.txt", "EqualizerWS", equalizerwin->window, 275, 14, FALSE);
		skin->mask_eq_shade_ds = skin_create_transparent_mask(path, "region.txt", "EqualizerWS", equalizerwin->window, 550, 28, TRUE);
		
		load_skin_viscolor(path, "viscolor.txt");
	}
	else
	{
		if (skin->path)
			g_free(skin->path);
		skin->path = NULL;
	}

	setup_skin_masks();

	draw_main_window(TRUE);
	draw_playlist_window(TRUE);
	draw_equalizer_window(TRUE);
	playlistwin_update_list();
}

void load_skin(const gchar * path)
{
	_load_skin(path, FALSE);
}

void reload_skin(void)
{
	_load_skin(skin->path, TRUE);
}

void cleanup_skins(void)
{
	free_skin();
}

GdkPixmap *get_skin_pixmap(SkinIndex si)
{
	GdkPixmap *ret = NULL;

	switch (si)
	{
		case SKIN_MAIN:
			ret = skin->main;
			if (!ret)
				ret = skin->def_main;
			break;
		case SKIN_CBUTTONS:
			ret = skin->cbuttons;
			if (!ret)
				ret = skin->def_cbuttons;
			break;
		case SKIN_TITLEBAR:
			ret = skin->titlebar;
			if (!ret)
				ret = skin->def_titlebar;
			break;
		case SKIN_SHUFREP:
			ret = skin->shufrep;
			if (!ret)
				ret = skin->def_shufrep;
			break;
		case SKIN_TEXT:
			ret = skin->text;
			if (!ret)
				ret = skin->def_text;
			break;
		case SKIN_VOLUME:
			ret = skin->volume;
			if (!ret)
				ret = skin->def_volume;
			break;
		case SKIN_BALANCE:
			ret = skin->balance;
			if (!ret)
				ret = get_skin_pixmap(SKIN_VOLUME);
			break;
		case SKIN_MONOSTEREO:
			ret = skin->monostereo;
			if (!ret)
				ret = skin->def_monostereo;
			break;
		case SKIN_PLAYPAUSE:
			ret = skin->playpause;
			if (!ret)
				ret = skin->def_playpause;
			break;
		case SKIN_NUMBERS:
			ret = skin->numbers;
			if (!ret)
				ret = skin->def_numbers;
			break;
		case SKIN_POSBAR:
			ret = skin->posbar;
			if (!ret)
				ret = skin->def_posbar;
			break;
		case SKIN_PLEDIT:
			ret = skin->pledit;
			if (!ret)
				ret = skin->def_pledit;
			break;
		case SKIN_EQMAIN:
			ret = skin->eqmain;
			if (!ret)
				ret = skin->def_eqmain;
			break;
		case SKIN_EQ_EX:
			ret = skin->eq_ex;
			if (!ret)
				ret = skin->def_eq_ex;
			break;
		case SKIN_MASK_MAIN:
			ret = skin->mask_main;
			if (!ret)
				ret = skin->def_mask;
			break;
		case SKIN_MASK_MAIN_DS:
			ret = skin->mask_main_ds;
			if (!ret)
				ret = skin->def_mask_ds;
			break;
		case SKIN_MASK_EQ:
			ret = skin->mask_eq;
			if (!ret)
				ret = skin->def_mask;
			break;
		case SKIN_MASK_EQ_DS:
			ret = skin->mask_eq_ds;
			if (!ret)
				ret = skin->def_mask_ds;
			break;
		case SKIN_MASK_SHADE:
			ret = skin->mask_shade;
			if (!ret)
				ret = skin->def_mask_shade;
			break;
		case SKIN_MASK_SHADE_DS:
			ret = skin->mask_shade_ds;
			if (!ret)
				ret = skin->def_mask_shade_ds;
			break;
		case SKIN_MASK_EQ_SHADE:
			ret = skin->mask_eq_shade;
			if (!ret)
				ret = skin->def_mask_shade;
			break;
		case SKIN_MASK_EQ_SHADE_DS:
			ret = skin->mask_eq_shade_ds;
			if (!ret)
				ret = skin->def_mask_shade_ds;
			break;
	}
	return ret;
}

GdkColor *get_skin_color(SkinColorIndex si)
{
	GdkColor *ret = NULL;

	switch (si)
	{
		case SKIN_PLEDIT_NORMAL:
			ret = skin->pledit_normal;
			if (!ret)
				ret = &skin->def_pledit_normal;
			break;
		case SKIN_PLEDIT_CURRENT:
			ret = skin->pledit_current;
			if (!ret)
				ret = &skin->def_pledit_current;
			break;
		case SKIN_PLEDIT_NORMALBG:
			ret = skin->pledit_normalbg;
			if (!ret)
				ret = &skin->def_pledit_normalbg;
			break;
		case SKIN_PLEDIT_SELECTEDBG:
			ret = skin->pledit_selectedbg;
			if (!ret)
				ret = &skin->def_pledit_selectedbg;
			break;
	}
	return ret;
}

void get_skin_viscolor(guchar vis_color[24][3])
{
	gint i;

	for (i = 0; i < 24; i++)
	{
		vis_color[i][0] = skin->vis_color[i][0];
		vis_color[i][1] = skin->vis_color[i][1];
		vis_color[i][2] = skin->vis_color[i][2];
	}
}

gboolean is_new_skin(gchar * old_skin)
{
	if (!old_skin && skin->path)
		return TRUE;
	if (old_skin && !skin->path)
		return TRUE;
	if (!old_skin && !skin->path)
		return FALSE;
	if (strcmp(old_skin, skin->path))
		return TRUE;
	return FALSE;
}
