// Aspell check functions
// Copyright 1999 by Kevin Atkinson under the terms of the LGPL

#ifndef as_check__
#define as_check__

#include "manager.hh"
#include "token.hh"
#include "filter.hh"
#include "itr_equal.hh"
#include "string_map.hh"
#include "itemize.hh"
#include "exception.hh"
#include "file_exceps.hh"
#include "tl_string_map.hh"

namespace afilter {
  struct MapReplReadError;
}

namespace aspell {

  using namespace autil;

  struct UnknownFilter : public Exception {
    string filter;
    UnknownFilter() {}
    UnknownFilter(const string & f);
  };

  struct BadMapFile : public BadFileFormat {
    MapReplReadError * maperr;
    BadMapFile(const string & file, MapReplReadError * e)
      : BadFileFormat(file), maperr(e) {}
  };

  FilterItrPart * get_filter_itr_throw(const string & name, const ConfigData &);

  template <typename Itr, typename EndF = itr_equal<Itr> >
  class CheckState {
  private:
    TokenItr itr;
    typedef FilterItrRootClass<Itr,EndF> RootClass;
    Manager * manager_;

  public:
    CheckState(Manager & s, const string & file_name = "") 
      : itr(s.lang()), manager_(&s)
    {
      start_over(file_name);
    }
    void start_over(const string & file_name = "");
    
    Manager & manager() {return *manager_;}
  
    const Itr & word_begin() const 
    {
      return static_cast<const RootClass *>
	(itr.word_begin().root())->true_itr();
    }
    const Itr & word_end()   const 
    {
      return static_cast<const RootClass *>
	(itr.word_end().root())->true_itr();
    }
    bool is_word() const {return itr.is_word();}
    const string & word() const {return itr.word();}
    bool at_end() const {return itr.at_end();}

    void advance() {itr.advance();}

    void add(FilterItrPart * i) {itr.add(i);}
    void add(const string & name) 
    {
      add(get_filter_itr_throw(name,manager_->config()));
    }

    void reset() {itr.reset();}
  
    void restart(const Itr & i)                 {itr.restart(RootClass(i));}
    void restart(const Itr & i, const EndF & e) {itr.restart(RootClass(i,e));}
    void restart(const Itr & i, const Itr & e)  {itr.restart(RootClass(i,e));}

    void backup() {itr.backup();}

    void scan(const Itr & stop) {itr.scan(RootClass(stop));}
  };

  template <typename Itr, typename EndF>
  void CheckState<Itr, EndF>::start_over(const string & file_name) {
    
    Config options = manager_->config();
    
    int i = file_name.rfind('.');
    string extension;
    
    if (i != string::npos) {
      extension.assign(file_name, i+1);
      for (i = 0; i != extension.size(); ++i) {
	extension[i] = tolower(extension[i]);
      }
    }
    if (extension == "tex") {
      options.insert("mode", "tex");
    } else {
      ToLowerStringMap sgml_extns;
      options.retrieve_list("sgml-extension", sgml_extns);
      if (sgml_extns.have(extension))
	options.insert("mode", "sgml");
    }
    
    itr.start_over();
    itr.add(new RootClass()); 
    StringMap filters;
    options.retrieve_list("filter", filters);
    StringMap::Emul els = filters.elements();
    while (!els.at_end())
      add(els.next().first);
  }

  struct DoNothing {
    void operator () () const {}
  };

  //FIXME: deal with conversion properly.
  //       ie. if the filter is already there don't convert
  template <class Itr, class EndF, class StatusF>
  void check (CheckState<Itr,EndF> & state,
	      const StatusF & print_status) 
  {
    while (!state.at_end() && 
	   (!state.is_word() || state.manager().check(state.word()))
	   )
      {
	if (state.is_word()) print_status();
	state.advance();
      }
  }

  template <class Itr, class EndF>
  inline void check (CheckState<Itr, EndF> & state) 
  {
    check(state, DoNothing());
  }
}
#endif
