## -*- perl -*-
## ----------------------------------------------------------------------
## DebianDoc_SGML/Format/HTML: SGML conversion specification for HTML output format
## ----------------------------------------------------------------------
## Copyright (C) 1998,1999,2000 Ardo van Rangelrooij
## Copyright (C) 1996 Ian Jackson
##
## This is free software; see the GNU General Public Licence
## version 2 or later for copying conditions.  There is NO warranty.
## ----------------------------------------------------------------------

## ----------------------------------------------------------------------
## package interface definition
package DebianDoc_SGML::Format::HTML;
use strict;
use vars qw( @ISA @EXPORT );
use Exporter;
@ISA = ( 'Exporter' );
@EXPORT = qw ();

## ----------------------------------------------------------------------
## import packages
use I18N::LangTags qw( locale2language_tag );
use File::Basename;
use File::Spec;
use SGMLS::Output;
use Text::Format;
use URI;

## ----------------------------------------------------------------------
## file name definitions
my $content = '';
if ( $DebianDoc_SGML::Format::Driver::opt_c )
{
    my $language_tag =
	lc( locale2language_tag( $DebianDoc_SGML::Format::Driver::locale ) );
    $language_tag = 'en' if $language_tag eq undef;
    my $charset = ".$DebianDoc_SGML::Format::Driver::i18n{'charset'}"
	if length( $DebianDoc_SGML::Format::Driver::i18n{'charset'} )
	    && $DebianDoc_SGML::Format::Driver::locale =~ m/\./;
    $content = ".$language_tag$charset";
}
my $basename = $DebianDoc_SGML::Format::Driver::opt_b;
my $prefix = '';
if ( $DebianDoc_SGML::Format::Driver::opt_b =~ m,/, )
{
    $basename = dirname( $DebianDoc_SGML::Format::Driver::opt_b );
    $prefix = basename( $DebianDoc_SGML::Format::Driver::opt_b ) . '-';
}
my $topname = length( $DebianDoc_SGML::Format::Driver::opt_t )
    ? $DebianDoc_SGML::Format::Driver::opt_t : 'index';
my $extension = length( $DebianDoc_SGML::Format::Driver::opt_e )
    ? ".$DebianDoc_SGML::Format::Driver::opt_e" : '.html';

## ----------------------------------------------------------------------
## directory definition and creation
my $directory = "$basename$extension";
-d "$directory" || mkdir( "$directory", 0777 )
    || die "cannot make directory \`$directory': $!\n";

## ----------------------------------------------------------------------
## layout definition
$DebianDoc_SGML::Format::Driver::indent_level = 0;
my $text = new Text::Format;
$text->columns( 79 );
$text->firstIndent( 0 );
$text->extraSpace( 1 );

## ----------------------------------------------------------------------
## global variables
use vars qw( $toclevel $need_hr $currentfile $cfilesrid );
use vars qw( $chainchapt $chaptsrid %fchainsrid %bchainsrid );
use vars qw( @footnotes @footnote_cfilesrid );
use vars qw( @comments @comment_editors @comment_cfilesrid );
undef $chainchapt;
$toclevel = -1;

## ----------------------------------------------------------------------
## book output subroutines
## ----------------------------------------------------------------------
sub _output_start_book
{
}
sub _output_end_book
{
    _output_footnotes() if ( @footnotes );
    _output_comments() if ( @comments );
}

## ----------------------------------------------------------------------
## title page output subroutines
## ----------------------------------------------------------------------
sub _output_titlepag
{
    _start_file( '' );
    _html_head( $DebianDoc_SGML::Format::Driver::title, 1 );
    output( "\n" );
    output( "<hr>\n" );
    output( "\n" );
    output( "<h1>$DebianDoc_SGML::Format::Driver::title</h1>\n" );
    if ( length( $DebianDoc_SGML::Format::Driver::abstract )
	 || length( $DebianDoc_SGML::Format::Driver::copyright ) )
    {
	output( "\n" );
	output( "<hr>\n" );
    }
    if ( length( $DebianDoc_SGML::Format::Driver::abstract ) )
    {
	output( "\n" );
	output( "<h2><a name=\"abstract\">$DebianDoc_SGML::Format::Driver::i18n{'abstract'}</a></h2>\n" );
	output( $DebianDoc_SGML::Format::Driver::abstract );
    }
    if ( length( $DebianDoc_SGML::Format::Driver::copyright ) )
    {
	output( "\n" );
	output( "<h2><a name=\"copyright\">$DebianDoc_SGML::Format::Driver::i18n{'copyright notice'}</a></h2>\n" );
	output( $DebianDoc_SGML::Format::Driver::copyright );
    }
    if ( length( $DebianDoc_SGML::Format::Driver::abstract )
	 || length( $DebianDoc_SGML::Format::Driver::copyright ) )
    {
	output( "\n" );
    }
    output( "</p>\n" );
}
sub _output_copyrightsummaries
{
    output( "\n" );
    output( "<p>\n" );
    output( join( "<br>\n", @_ ), "\n" );
}

## ----------------------------------------------------------------------
## table of contents output subroutines
## ----------------------------------------------------------------------
sub _output_toc
{
    output( "\n" );
    output( "<hr>\n" );
    output( "\n" );
    output( "<h2><a name=\"contents\">$DebianDoc_SGML::Format::Driver::i18n{'contents'}</a></h2>\n" );
    output( "\n" );
    output( "<ul>\n" );
    output( $_[0] );
    output( "</ul>\n" x ( $toclevel + 2 ) );
    _html_tail( 1 );
    _finish_file();
}
sub _output_tocentry
{
    return if $_[1] > $DebianDoc_SGML::Format::Driver::toc_detail;
    if ( $toclevel > $_[1] )
    {
	output( "</ul>\n" x ( $toclevel - $_[1] ) );
    }
    elsif ( $toclevel < $_[1] )
    {
	output( "<ul>\n" x ( $_[1] - $toclevel ) );
    }
    output( '<li>' );
    if ( $_[1] == -1 )
    {
	$chaptsrid = $_[3];
	if ( length( $chainchapt ) )
	{
	    $fchainsrid{$chainchapt} = $chaptsrid;
	    $bchainsrid{$chaptsrid} = $chainchapt;
	}
	$chainchapt = $chaptsrid;
	_output_ref( "$_[2] $_[0]", '', $chaptsrid );
    }
    else
    {
	_output_ref( "$_[2]", '', $chaptsrid, $_[3] );
	output( " $_[0]" );
    }
    output( "\n" );
    $toclevel = $_[1];
}

## ----------------------------------------------------------------------
## section output subroutines
## ----------------------------------------------------------------------
sub _output_chapter
{
    _start_file( $cfilesrid );
    output( $_[0] );
    _html_tail( 1 );
    _finish_file();
}
sub _output_appendix
{
    _start_file( $cfilesrid );
    output( $_[0] );
    _html_tail( 1 );
    _finish_file();
}
sub _output_heading
{
    if ( $_[1] < 0 )
    {
	$cfilesrid = $_[3];
        _html_head( "$DebianDoc_SGML::Format::Driver::title - $_[0]", 1 );
	output( "\n" );
	output( "<hr>\n" );
	output( "\n" );
        output( "<h1>\n" );
	output( "$DebianDoc_SGML::Format::Driver::title<br>\n" );
	$_[2] =~ m/^[A-Z]$/
	    ? output( $DebianDoc_SGML::Format::Driver::i18n{ 'appendix' } )
	    : output( $DebianDoc_SGML::Format::Driver::i18n{ 'chapter' } );
	output( " $_[2] $_[0]\n" );
	output( "</h1>\n" );
	output( "\n" );
	output( "<hr>\n" );
	$need_hr = 0;
    }
    else
    {
	if ( $need_hr )
	{
	    output( "</p>\n" );
	    output( "\n" );
	    output( "<hr>\n" );
	}
	else
	{
	    $need_hr = 1;
	}
	output( "\n" );
	output( "<h" . ( $_[1] + 2 ) . ">\n" );
	output( "<a name=\"$_[3]\">" );
	output( "$_[2] $_[0]" );
        output( "</a>\n" );
        output( "</h" . ( $_[1] + 2 ) . ">\n" );
    }
}
sub _start_file
{
    $currentfile = $prefix . ( length( $_[0] ) ? $_[0] : $topname )
	. $content . $extension;
    push_output( 'file', File::Spec->catfile( "$directory", "$currentfile" ) );
}
sub _finish_file
{
    pop_output();
}
sub _html_head
{
    my ( $pagetitle, $np_links ) = @_;
    $pagetitle =~ s/\<[^<>]*\>//g; ### WRONG !!!
    output( "<!doctype html public \"-//W3C//DTD HTML 4.0//EN\">\n" );
    output( "\n" );
    output( "<html>\n" );
    output( "\n" );
    output( "<head>\n" );
    output( "\n" );
    output( "<meta http-equiv=\"content-type\" content=\"text/html; "
	    . "charset=$DebianDoc_SGML::Format::Driver::i18n{'charset'}\">\n" )
	if length( $DebianDoc_SGML::Format::Driver::i18n{'charset'} );
    output( "\n" );
    output( "<title>$pagetitle</title>\n" );
    output( "\n" );
    output( "</head>\n" );
    output( "\n" );
    output( "<body>\n" );
    _navigation( $np_links );
}
sub _html_tail
{
    my ( $np_links ) = @_;
    output( "</p>\n" ); ### ??? !!!
    _navigation( $np_links );
    output( "\n" );
    output( "<hr>\n" );
    output( "\n" );
    output( "$DebianDoc_SGML::Format::Driver::title<br>\n" );
    output( "\n" );
    output( "<address>\n" );
    output( "$DebianDoc_SGML::Format::Driver::version<br>\n" )
	if length( $DebianDoc_SGML::Format::Driver::version );
    output( join( "<br>\n", @DebianDoc_SGML::Format::Driver::authors ), "\n" );
    output( "</address>\n" );
    output( "\n" );
    output( "</body>\n" );
    output( "\n" );
    output( "</html>\n" );
    output( "\n" );
}
sub _navigation
{
    my ( $np_links ) = @_;
    output( "\n" );
    output( "<hr>\n" );
    output( "\n" );
    if ( $np_links && defined( $bchainsrid{$cfilesrid} ) )
    {
	output( "[ " );
	_output_ref( $DebianDoc_SGML::Format::Driver::i18n{'previous'},
		     '', $bchainsrid{$cfilesrid} );
	output( " ]\n" );
    }
    if ( length( $DebianDoc_SGML::Format::Driver::abstract ) )
    {
	output( "[ " );
	_output_ref( $DebianDoc_SGML::Format::Driver::i18n{'abstract'},
		     '', $topname, 'abstract' );
	output( " ]\n" );
    }
    if ( length( $DebianDoc_SGML::Format::Driver::copyright ) )
    {
	output( "[ " );
	_output_ref( $DebianDoc_SGML::Format::Driver::i18n{'copyright notice'},
		     '', $topname, 'copyright' );
    	output( " ]\n" );
    }
    output( "[ " );
    _output_ref( $DebianDoc_SGML::Format::Driver::i18n{'contents'},
		 '', $topname, 'contents' );
    output( " ]\n" );
    if ( $np_links && defined( $fchainsrid{ $cfilesrid } ) )
    {
	output( "[ " );
	_output_ref( $DebianDoc_SGML::Format::Driver::i18n{'next'},
		     '', $fchainsrid{ $cfilesrid } );
	output( " ]\n" );
    }
}

## ----------------------------------------------------------------------
## paragraph output subroutines
## ----------------------------------------------------------------------
sub _output_p
{
    if ( length( $_[0] ) )
    {
	output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
	output( "<p>\n" );
	output( $text->format( "$_[0]\n" ) );
	$need_hr = 1;
    }
    else
    {
	output( "\n" )
	    if ( $DebianDoc_SGML::Format::Driver::is_special
		 && ! $DebianDoc_SGML::Format::Driver::is_compact
		 && $DebianDoc_SGML::Format::Driver::will_be_compact );
    }
}

## ----------------------------------------------------------------------
## example output subroutines
## ----------------------------------------------------------------------
sub _output_example
{
    $_[0] = "     " . $_[0];
    $_[0] =~ s/\n/\n     /g;
    $_[0] =~ s/\s+$/\n/;
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "<pre>\n" );
    output( $_[0] );
    output( "</pre>\n" );
}

## ----------------------------------------------------------------------
## footnote output subroutines
## ----------------------------------------------------------------------
sub _output_footnotes
{
    my $subdoctitle = "$DebianDoc_SGML::Format::Driver::title - $DebianDoc_SGML::Format::Driver::i18n{'footnotes'}";
    _start_file( 'footnotes' );
    _html_head( $subdoctitle, 0 );
    output( "\n" );
    output( "<hr>\n" );
    output( "\n" );
    output( "<h1>$subdoctitle</h1>\n" );
    output( "\n" );
    output( "<hr>\n" );
    output( "\n" );
    my $footnoteref = 1;
    foreach my $footnote ( @footnotes )
    {
	my $cfilesrid = $footnote_cfilesrid[$footnoteref - 1];
	output( "<h2>" );
	_output_ref( $footnoteref, '', $cfilesrid, "fr$footnoteref",
		     "name=\"$footnoteref\"" );
	output( "</h2>\n" );
	output( "$footnote\n" );
	$footnoteref++;
    }
    _html_tail( 0 );
    _finish_file();
    @footnotes = ();
    @footnote_cfilesrid = ();
}
sub _output_footnote
{
    push( @footnotes, $_[0] );
    push( @footnote_cfilesrid, $cfilesrid );
    my $footnoteref = scalar( @footnotes );
    _output_ref( $footnoteref, '', 'footnotes', $footnoteref,
		 "name=\"fr$footnoteref\"", "]", "[" );
}

## ----------------------------------------------------------------------
## comment output subroutines
## ----------------------------------------------------------------------
sub _output_comments
{
    my $subdoctitle = "$DebianDoc_SGML::Format::Driver::title - $DebianDoc_SGML::Format::Driver::i18n{'comments'}";
    _start_file( 'comments' );
    _html_head( $subdoctitle, 0 );
    output( "\n" );
    output( "<hr>\n" );
    output( "\n" );
    output( "<h1>$subdoctitle</h1>\n" );
    output( "\n" );
    output( "<hr>\n" );
    output( "\n" );
    my $commentref = 1;
    foreach my $comment ( @comments )
    {
	my $editor = $comment_editors[$commentref - 1];
	my $cfilesrid = $comment_cfilesrid[$commentref - 1];
	$editor = " ($editor)" if length( $editor );
	output( "<h2>" );
	_output_ref( "c$commentref", '', $cfilesrid, "cr$commentref",
		     "name=\"c$commentref\"", $editor );
	output( "</h2>\n" );
	output( "$comment\n" );
	$commentref++;
    }
    _html_tail( 0 );
    _finish_file();
    @comments = ();
    @comment_editors = ();
    @comment_cfilesrid = ();
}
sub _output_comment
{
    push( @comments, $_[0] );
    push( @comment_editors, $_[1] );
    push( @comment_cfilesrid, $cfilesrid );
    my $commentref = scalar( @comments );
    _output_ref( "c$commentref", '', 'comments', $commentref,
		 "name=\"cr$commentref\"", "]", "[" );
}

## ----------------------------------------------------------------------
## list output subroutines
## ----------------------------------------------------------------------
sub _output_list
{
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "<ul" );
    output( " compact" ) if $DebianDoc_SGML::Format::Driver::is_compact;
    output( ">\n" );
    output( $_[0] );
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "</ul>\n" );
}
sub _output_enumlist
{
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "<ol" );
    output( " compact" ) if $DebianDoc_SGML::Format::Driver::is_compact;
    output( ">\n" );
    output( $_[0] );
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "</ol>\n" );
}
sub _output_taglist
{
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "<dl" );
    output( " compact" ) if $DebianDoc_SGML::Format::Driver::is_compact;
    output( ">\n" );
    output( $_[0] );
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "</dl>\n" );
}
sub _output_list_tag
{
}
sub _output_enumlist_tag
{
}
sub _output_taglist_tag
{
}
sub _output_list_item
{
    $_[0] =~ s/^\n//;
    $_[0] =~ s/^<p>\n//;
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "<p>\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "<li>\n" );
    output( $_[0] );
    output( "</li>\n" );
}
sub _output_enumlist_item
{
    $_[0] =~ s/^\n//;
    $_[0] =~ s/^<p>\n//;
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "<p>\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "<li>\n" );
    output( $_[0] );
    output( "</li>\n" );
}
sub _output_taglist_item
{
    $_[0] =~ s/^\n//;
    $_[0] =~ s/^<p>\n//;
    output( "\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    output( "<p>\n" ) if ! $DebianDoc_SGML::Format::Driver::is_compact;
    foreach ( @{$_[1]} )
    {
        output( "<dt>$_</dt>\n" );
    }
    output( "<dd>\n" );
    output( $_[0] );
    output( "</dd>\n" );
}

## ----------------------------------------------------------------------
## emph output subroutines
## ----------------------------------------------------------------------
sub _output_em
{
    output( "<em>$_[0]</em>" );
}
sub _output_strong
{
    output( "<strong>$_[0]</strong>" );
}
sub _output_var
{
    output( "<var>$_[0]</var>" );
}
sub _output_package
{
    output( "<code>$_[0]</code>" );
}
sub _output_prgn
{
    output( "<code>$_[0]</code>" );
}
sub _output_file
{
    output( "<code>$_[0]</code>" );
}
sub _output_tt
{
    output( "<samp>$_[0]</samp>" );
}
sub _output_qref
{
    _output_ref( $_[0], '', $_[1], $_[2] );
}

## ----------------------------------------------------------------------
## xref output subroutines
## ----------------------------------------------------------------------
sub _output_ref
{
    output( "$_[6]" ) if length( $_[6] );
    output( "<a href=\"$prefix$_[2]$content$extension" );
    output( "#$_[3]" ) if length( $_[3] ) && $_[3] ne $_[2];
    output( "\"" );
    output( " $_[4]" ) if length( $_[4] );
    output( ">$_[0]" );
    if ( length( $_[1] ) )
    {
	my ( $name, $number ) = split( / /, $_[1], 2 );
	output( ", $DebianDoc_SGML::Format::Driver::i18n{$name} $number" );
    }
    output( "</a>" );
    output( "$_[5]" ) if length( $_[5] );
}
sub _output_manref
{
    output( "<code>$_[0]($_[1])</code>" );
}
sub _output_email
{
    output( " " ) if $DebianDoc_SGML::Format::Driver::in_author;
    _output_url( "mailto:$_[0]", $_[0] );
}
sub _output_ftpsite
{
    my $url = URI->new( $_[0] );
    output( "<code>$url</code>" );
}
sub _output_ftppath
{
    _output_url( "ftp://$_[0]$_[1]", $_[1] );
}
sub _output_httpsite
{
    my $url = URI->new( $_[0] );
    output( "<code>$url</code>" );
}
sub _output_httppath
{
    _output_url( "http://$_[0]$_[1]", $_[1] );
}
sub _output_url
{
    my $url = URI->new( $_[0] );
    $_[1] = $_[0] if $_[1] eq "";
    output( "<code><a href=\"$url\">$_[1]</a></code>" );
}

## ----------------------------------------------------------------------
## data output subroutines
## ----------------------------------------------------------------------
my %cdata_sani = (
		  '<' => '&lt;',
		  '>' => '&gt;',
		  '&' => '&amp;',
		  '"' => '&quot;'
		  );
sub _cdata
{
    my ( $in ) = @_;
    my $out = '';
    while ( $in =~ m/[<>&\"]/ )
    {
        $out .= $` . $cdata_sani{ $& };
        $in = $';
    }
    $out .= $in;
    output( $out );
}
sub _sdata
{
    output( $DebianDoc_SGML::Format::Driver::sdata_mapping{ $_[0] } )
	if defined( $DebianDoc_SGML::Format::Driver::sdata_mapping{ $_[0] } );
}

## ----------------------------------------------------------------------
## don't forget this
1;

## ----------------------------------------------------------------------
