/*
 * GUI front-end
 * GDiffWindow module
 * See "gui.h" for the details of data structure.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gnome.h>
#include "gdiff.h"
#include "gui.h"
#include "guiwin.h"
#include "dirview.h"
#include "fileview.h"
#include "menu.h"
#include "guimisc.h"


/* Private function declarations */
static gint delete_event_cb(GtkWidget *w, GdkEvent *event, gpointer data);
static void switch_page_cb(GtkNotebook *notebook, GtkNotebookPage *page, guint page_num, gpointer data);
static void* get_switched_views(const GDiffWindow *gdwin, const GtkWidget *w, gboolean *ret_isdir);


/**
 * gdiff_window_new:
 * Allocate GDiffWindow, initialize it, and return its pointer.
 * No related back-end data, so what this function does is almost just
 * creating a window.
 * Now, its returned value is not used.
 * Instead, implicitly pass GDiffWindow to callback-functions of menu.
 * Input:
 * const char *geometry_string;
 * Output:
 * Return value; Allocated GDiffWindow.
 **/
GDiffWindow*
gdiff_window_new(const char *geometry_string)
{
	GDiffWindow *gdiffwin;
	GtkWidget *app;
	GtkWidget *vbox;
	GtkWidget *notebook;

	gdiffwin = g_new(GDiffWindow, 1);

	/* Initialize */
	gdiffwin->filesel = NULL;
	gdiffwin->gdirv_list = NULL;
	gdiffwin->gfilev_list = NULL;
	gdiffwin->pref = g_pref.winpref;

	/* Main window */
	app = gnome_app_new(APPNAME, _("Gtkdiff"));
	gdiffwin->app = GNOME_APP(app);
	gtk_window_set_wmclass(GTK_WINDOW(app), "gtkdiff", "gtkdiff");
	gtk_window_set_policy(GTK_WINDOW(app), TRUE, TRUE, FALSE);
	gtk_widget_set_name(app, "gtkdiff");

	if (geometry_string) {
		gint x, y, w, h;
		if (gnome_parse_geometry(geometry_string, &x, &y, &w, &h)) {
			if ((x != -1) && (y != -1))
				gtk_widget_set_uposition(GTK_WIDGET(app), x, y);
			if ((w != -1) && (h != -1))
				gtk_window_set_default_size(GTK_WINDOW(app), w, h);
		} else {
			gnome_app_error(GNOME_APP(app),
							_("Couldn't understand geometry (position and size)\n"
							  " specified on command line"));
		}
	} else {
		/* Default geometry */
		gtk_window_set_default_size(GTK_WINDOW(app), 500, 400);
		gtk_widget_set_uposition(app, 20, 20);
	}
	
	/* It's a good idea to do this for all windows. */
	/*TODO: for supporting multiple windows, to connect gdiff_window_delete*/
	gtk_signal_connect(GTK_OBJECT(app), "destroy",
					   GTK_SIGNAL_FUNC(gtkdiff_exit),
					   NULL);
	gtk_signal_connect(GTK_OBJECT(app), "delete_event",
					   GTK_SIGNAL_FUNC(delete_event_cb),
					   NULL);

	/* vbox */
	vbox = gtk_vbox_new(FALSE, 0);
	gnome_app_set_contents(GNOME_APP(app), vbox);

	/* notebook */
	notebook = gtk_notebook_new();
	gdiffwin->notebook = GTK_NOTEBOOK(notebook);
	gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
					   GTK_SIGNAL_FUNC(switch_page_cb),
					   gdiffwin);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), gdiffwin->pref.show_tabs);
	gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(notebook), FALSE);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(notebook), TRUE);
	/*???	gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);*/
	gtk_notebook_popup_enable(GTK_NOTEBOOK(notebook));
	gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
	
	/* menu */
	menu_create(gdiffwin);

	/* status bar */
	statusbar_create(gdiffwin);
	menu_install_hints_for_toolbar(gdiffwin);
	
	gtk_widget_show_all(app);

	return gdiffwin;
}

/*place holder*/
void
gdiff_window_delete(GDiffWindow *gdwin)
{
	return;
}


/* ---The followings are private functions--- */
/**
 * delete_event_cb:
 **/
static gint
delete_event_cb(GtkWidget *w, GdkEvent *event, gpointer data)
{
	return FALSE;/* do nothing, just quit. */
}

/**
 * switch_page_cb:
 * Update menu-bar, and more.
 **/
static void
switch_page_cb(GtkNotebook *notebook, GtkNotebookPage *page, guint page_num, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *w = page->child;
	Preference pref = g_pref;/* copy to local */
	void *p;
	gboolean is_dir;
	char *sbar_msg;/* status-bar */
	
	p = get_switched_views(gdwin, w, &is_dir);
	if (p == NULL)
		return;
	if (is_dir) {
		GDiffDirViews *gdviews = p;
		/* Deal with dirty-bit */
		if (gdviews->b_dirty == TRUE) {
			dirview_redisplay(gdviews);
			gdviews->b_dirty = FALSE;
		}
		/* Update menu-bar */
		pref.dvpref = gdviews->pref;
		menu_update(gdwin, &pref, MENUSTAT_DIR_VIEW);

		sbar_msg = make_current_info_msg(gdviews->dirname1, gdviews->dirname2, -1, -1, -1, -1);
	} else {
		GDiffFileViews *gfviews = p;
		/* Update menu-bar */
		pref.fvpref = gfviews->pref;
		menu_update(gdwin, &pref, MENUSTAT_FILE_VIEW);

		sbar_msg = make_current_info_msg(gfviews->filename1, gfviews->filename2, -1, -1, -1, -1);
	}

	/* Update status-bar */
	statusbar_update(gdwin, sbar_msg);
	g_free(sbar_msg);
}

/**
 * get_switched_views:
 * Called by switch_page_cb(), and return the view related to the switched page.
 * cf. gdiff_current_views() in guiviews.c
 * Input:
 * const GDiffWindow *gdwin;
 * const GtkWidget *w; switched page's widget.
 * gboolean *ret_isdir;
 * Output:
 * gboolean *ret_isdir; return TRUE if it is directory.
 * Return value; pointer to GDiffDirViews or GDiffFileViews. In no view case,  NULL.
 **/
static void*
get_switched_views(const GDiffWindow *gdwin, const GtkWidget *w, gboolean *ret_isdir)
{
	GList *list;

	for (list = gdwin->gdirv_list; list; list = list->next) {
		GDiffDirViews *gdviews = list->data;
		if (gdviews->base == w) {
			*ret_isdir = TRUE;
			return gdviews;
		}
	}
	for (list = gdwin->gfilev_list; list; list = list->next) {
		GDiffFileViews *gfviews = list->data;
		if (gfviews->base == w) {
			*ret_isdir = FALSE;
			return gfviews;
		}
	}
	
	return NULL;
}
