/*
 * Module for file view(display diff result) in one-pane mode
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <gnome.h>
#include "gdiff.h"
#include "misc.h"
#include "gui.h"
#include "fileview.h"
#include "guiwin.h"
#include "guimisc.h"
#include "rmenu.h"
#include "style.h"


/* Private function declarations */
static void create_textw(GDiffFileViews *gfileviews);
static void draw_text(GDiffFileViews *gfileviews);
static void show_hide_numbers(GDiffFileViews *gfileviews, gboolean b_show);
static void create_overview(GDiffFileViews *gfileviews);
static void draw_overview(GDiffFileViews *gfileviews);
static void draw_overview_lines(GtkWidget *widget, GdkEventExpose *event, gpointer data);
static void guts_move_diff(GDiffFileViews *gfileviews, MoveDiff mv_diff);



/**
 * onepane_create_widgets:
 * Called from gdiff_fileviews_new().
 * Create widgets and show them.
 * The contents(diff results) are not shown here, which is up to onepane_display().
 * Input:
 * GDiffFileViews *gfileviews;
 * Output:
 * GDiffFileViews *gfileviews; updated.
 **/
void
onepane_create_widgets(GDiffFileViews *gfileviews)
{
	GDiffWindow *gdwin = gfileviews->gdwin;
	GtkWidget *hbox;
	GtkWidget *label;
	gint page;
	
	/* hbox */
	hbox = gtk_hbox_new(FALSE, 0);
	gfileviews->pane.one.hbox = GTK_HBOX(hbox);
	gfileviews->base = GTK_WIDGET(gfileviews->pane.one.hbox);/* Virtual base widget. */

	/* Add to the notebook */
	label = make_label_for_notebook(gfileviews->filename1, gfileviews->filename2, ONE_PANE_VIEW);
	gfileviews->pane.one.label = GTK_LABEL(label);
	gtk_notebook_append_page(gdwin->notebook, hbox, label);

	/* text area */
	create_textw(gfileviews);
	/* overview area */
	create_overview(gfileviews);
	
	set_style_fileviews(gfileviews, ONE_PANE);
	gtk_widget_show(hbox);
	/* Is this a proper way ? */
	page = gtk_notebook_page_num(gdwin->notebook, hbox);
	gtk_notebook_set_page(gdwin->notebook, page);

	rmenu_create(gfileviews);
}

/**
 * onepane_display:
 * Show the diff result in one-pane mode.
 * Input:
 * GDiffFileViews *gfileviews;
 * Output:
 * GDiffFileViews *gfileviews; updated.
 **/
void
onepane_display(GDiffFileViews *gfileviews)
{
	DiffFiles *files = gfileviews->dfiles;

	if (is_files_different(files, TRUE) == FALSE) {
		gtk_widget_hide(GTK_WIDGET(gfileviews->base));
		return;
	}

	draw_text(gfileviews);
	draw_overview(gfileviews);
	if (gfileviews->pref.show_line_num == TRUE) {
		show_hide_numbers(gfileviews, TRUE);
	}
}

/**
 * onepane_show_linenum:
 * Show(Hide) the line numbers on text widget.
 * Input:
 * GDiffFileViews *gfileviews;
 * gboolean to_show; If TRUE, show. if FALSE, hide.
 * Output:
 * GDiffFileViews *gfileviews; updated.
 **/
void
onepane_show_linenum(GDiffFileViews *gfileviews, gboolean to_show)
{
	if (to_show == TRUE && gfileviews->pref.show_line_num == FALSE) {
		show_hide_numbers(gfileviews, TRUE);
		gfileviews->pref.show_line_num = TRUE;
	} else if (to_show == FALSE && gfileviews->pref.show_line_num == TRUE) {
		show_hide_numbers(gfileviews, FALSE);
		gfileviews->pref.show_line_num = FALSE;
	}	
}

/**
 * onepane_move_diff:
 * Move to a difference, such as next, previous, first or last.
 **/ 
void
onepane_move_diff(GDiffFileViews *gfileviews, MoveDiff mv_diff)
{
	char *msg = NULL;

	guts_move_diff(gfileviews, mv_diff);

	/* Update status-bar */
	if (mv_diff == MOVED_REL_NEXT || mv_diff == MOVED_REL_PREV) {
		msg = make_current_info_msg(gfileviews->filename1, gfileviews->filename2, -1, -1, -1, -1);
	} else {
		const DiffLines *dlines;

		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		if (dlines) {
			msg = make_current_info_msg(gfileviews->filename1, gfileviews->filename2,
										dlines->between[FIRST_FILE].begin,
										dlines->between[FIRST_FILE].end,
										dlines->between[SECOND_FILE].begin,
										dlines->between[SECOND_FILE].end);
		}
	}
	if (msg) {
		statusbar_update(gfileviews->gdwin, msg);
		g_free(msg);
	}
}



/* ---The followings are private functions--- */
/**
 * create_textw:
 * Create GtkText widget.
 **/
static void
create_textw(GDiffFileViews *gfileviews)
{
	GtkHBox *hbox = gfileviews->pane.one.hbox;
	GtkWidget *text;
	GtkWidget *scrollwin;

	scrollwin = gtk_scrolled_window_new(NULL, NULL);
	/* XXX: I can't use horizontal scrollbar,
	   because of the current text widget limitation. */
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
								   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(hbox), scrollwin, TRUE, TRUE, 0);
	gtk_widget_show(scrollwin);
	
	text = gtk_text_new(NULL, NULL);
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_text_set_word_wrap(GTK_TEXT(text), FALSE);
	gtk_text_set_line_wrap(GTK_TEXT(text), gfileviews->pref.line_wrap);
	gtk_container_add(GTK_CONTAINER(scrollwin), text);
	gtk_widget_grab_focus(text);
	gtk_widget_show(text);

	gfileviews->pane.one.text = GTK_TEXT(text);
}

/**
 * draw_text:
 * Draw text with coloring different parts.
 * This function also takes care of merging two text.
 **/
static void
draw_text(GDiffFileViews *gfileviews)
{
	DiffFiles *files = gfileviews->dfiles;
	const FileInfo *fi1 = dfiles_get_fileinfo(files, FIRST_FILE, TRUE);
	const FileInfo *fi2 = dfiles_get_fileinfo(files, SECOND_FILE, TRUE);
	GtkText *text = gfileviews->pane.one.text;
	GdkFont *font = GTK_WIDGET(text)->style->font;
	const char *pt1;
	const char *pt2;
	const char *text_last1;/* for calculation of the length left */
	const char *text_last2;/* for calculation of the length left */
	int line1;
	GList *node;/* node of DiffLines' list */
	FontProp fprop;
	GdkColor *fg_color_f1 = &gfileviews->pref.diff_fg[FIRST_FILE];
	GdkColor *bg_color_f1 = &gfileviews->pref.diff_bg[FIRST_FILE];
	GdkColor *fg_color_f2 = &gfileviews->pref.diff_fg[SECOND_FILE];
	GdkColor *bg_color_f2 = &gfileviews->pref.diff_bg[SECOND_FILE];
	BufInfo binfo;

	if (fi1->text == NULL || fi2->text == NULL) /* Maybe, file has been deleted. */
		return;

 	fprop.font = font;

	line1 = 1;
	pt1 = fi1->text;
	text_last1 = fi1->text + fi1->lenb;
	text_last2 = fi2->text + fi2->lenb;
	gtk_text_freeze(text);
	for (node = files->dlines_list; node; node = node->next) {
		DiffLines *dlines;
		int begin1, end1;
		int begin2, end2;
		int lenb;
		
		dlines = node->data;
		/* Use local variables for ease to read */
		begin1 = dlines->between[FIRST_FILE].begin;
		end1 = dlines->between[FIRST_FILE].end;
		begin2 = dlines->between[SECOND_FILE].begin;
		end2 = dlines->between[SECOND_FILE].end;

		if (end1 != 0) {
			fprop.fg = NULL;
			fprop.bg = NULL;
			binfo.buf = pt1;
			binfo.lenb = text_last1 - pt1;
			lenb = draw_text_lines(text, &fprop, &binfo, begin1 - line1);
			pt1 += lenb;
		}
		switch (dlines->difftype) {
		case CHANGE:
			fprop.fg = fg_color_f1;
			fprop.bg = bg_color_f1;
			binfo.buf = pt1;
			binfo.lenb = text_last1 - pt1;
			lenb = draw_text_lines(text, &fprop, &binfo, end1-begin1+1);
			pt1 += lenb;

			fprop.fg = fg_color_f2;
			fprop.bg = bg_color_f2;
			pt2 = skip_n_lines(fi2->text, begin2-1, fi2->lenb);/*XXX performance hit?*/
			if (pt2) {
				binfo.buf = pt2;
				binfo.lenb = text_last2 - pt2;
				draw_text_lines(text, &fprop, &binfo, end2-begin2+1);
			}
			break;
		case F2ONLY:
			fprop.fg = NULL;
			fprop.bg = NULL;
			if (end1 != 0) {
				binfo.buf = pt1;
				binfo.lenb = text_last1 - pt1;
				lenb = draw_text_lines(text, &fprop, &binfo, end1-begin1+1);
				pt1 += lenb;
			}
			fprop.fg = fg_color_f2;
			fprop.bg = bg_color_f2;
			pt2 = skip_n_lines(fi2->text, begin2-1, fi2->lenb);/*XXX performance hit?*/
			if (pt2) {
				binfo.buf = pt2;
				binfo.lenb = text_last2 - pt2;
				draw_text_lines(text, &fprop, &binfo, end2-begin2+1);
			}
			break;
		case F1ONLY:
			fprop.fg = fg_color_f1;
			fprop.bg = bg_color_f1;
			binfo.buf = pt1;
			binfo.lenb = text_last1 - pt1;
			lenb = draw_text_lines(text, &fprop, &binfo, end1-begin1+1);
			pt1 += lenb;
			break;
		default:/* not used */
			break;
		}
		line1 = end1 + 1;
	}
	/* Draw the remained part */
#ifdef DEBUG
	if (line1 >= fi1->nline) {
		g_print("Problem?\n");
	}
#endif
	fprop.fg = NULL;
	fprop.bg = NULL;
	binfo.buf = pt1;
	binfo.lenb = text_last1 - pt1;
	draw_text_lines(text, &fprop, &binfo, fi1->nline - line1 + 1);
	gtk_text_thaw(text);
}


/* Routines for line numbers */
/**
 * show_hide_numbers:
 * Show(or hide) line numbers on the head of each lines.
 * This takes care of two files(merged text) at the same time.
 **/
static void
show_hide_numbers(GDiffFileViews *gfileviews, gboolean b_show)
{
	DiffFiles *files = gfileviews->dfiles;
	const FileInfo *fi1 = dfiles_get_fileinfo(files, FIRST_FILE, TRUE);
	const FileInfo *fi2 = dfiles_get_fileinfo(files, SECOND_FILE, TRUE);
	GtkText *text = gfileviews->pane.one.text;
	GdkFont *font = GTK_WIDGET(text)->style->font;
	const char *pt1;
	const char *pt2;
	const char *text_last1;/* for calculation of the length left */
	const char *text_last2;/* for calculation of the length left */
	int line1;
	int pos = 0;/* position in text widget */
	int merged_nline = gfileviews->merged_nline;
	GList *node;/* node of DiffLines' list */
	FontProp fprop;
	GdkColor *fg_color_f1 = &gfileviews->pref.diff_fg[FIRST_FILE];
	GdkColor *bg_color_f1 = &gfileviews->pref.diff_bg[FIRST_FILE];
	GdkColor *fg_color_f2 = &gfileviews->pref.diff_fg[SECOND_FILE];
	GdkColor *bg_color_f2 = &gfileviews->pref.diff_bg[SECOND_FILE];
	char format_common[32];
	char format_f1[32];
	char format_f2[32];
	BufInfo binfo;
	LineFormat lformat;
	
	if (fi1->text == NULL || fi2->text == NULL) /* Maybe, file has been deleted. */
		return;

	fprop.font = font;
	fprop.fg = NULL;

	line1 = 1;
	pt1 = fi1->text;
	text_last1 = fi1->text + fi1->lenb;
	text_last2 = fi2->text + fi2->lenb;
	/* the number of columns for representing line numbers */
	lformat.n_col = calc_number_places(merged_nline);
	sprintf(format_common, "%%%dd%s", lformat.n_col, MARK_COMMON);/* e.g. "%4d  " */
	sprintf(format_f1, "%%%dd%s", lformat.n_col, MARK_FILE1);/* e.g. "%4d< " */
	sprintf(format_f2, "%%%dd%s", lformat.n_col, MARK_FILE2);/* e.g. "%4d> " */

	gtk_text_freeze(text);
	for (node = files->dlines_list; node; node = node->next) {
		DiffLines *dlines;
		int begin1, end1;
		int begin2, end2;
		int lenb;
		
		dlines = node->data;
		/* Use local variables for ease to read */
		begin1 = dlines->between[FIRST_FILE].begin;
		end1 = dlines->between[FIRST_FILE].end;
		begin2 = dlines->between[SECOND_FILE].begin;
		end2 = dlines->between[SECOND_FILE].end;

		if (end1 != 0) {
			fprop.fg = NULL;
			fprop.bg = NULL;
			binfo.buf = pt1;
			binfo.lenb = text_last1 - pt1;
			lformat.format = format_common;
			lenb = insert_remove_line_numbers(text, b_show, &pos,
											  &fprop, &binfo, 
											  line1, begin1 - line1,
											  &lformat);
			pt1 += lenb;
		}
		switch (dlines->difftype) {
		case CHANGE:
			fprop.fg = fg_color_f1;
			fprop.bg = bg_color_f1;
			binfo.buf = pt1;
			binfo.lenb = text_last1 - pt1;
			lformat.format = format_f1;
			lenb = insert_remove_line_numbers(text, b_show, &pos,
											  &fprop, &binfo,
											  begin1, end1 - begin1 + 1,
											  &lformat);
			pt1 += lenb;

			fprop.fg = fg_color_f2;
			fprop.bg = bg_color_f2;
			pt2 = skip_n_lines(fi2->text, begin2-1, fi2->lenb);
			if (pt2) {
				binfo.buf = pt2;
				binfo.lenb = text_last2 - pt2;
				lformat.format = format_f2;
				lenb = insert_remove_line_numbers(text, b_show, &pos,
												  &fprop, &binfo,
												  begin2, end2 - begin2 + 1,
												  &lformat);
			}
			break;
		case F2ONLY:
			fprop.fg = NULL;
			fprop.bg = NULL;
			if (end1 != 0) {
				binfo.buf = pt1;
				binfo.lenb = text_last1 - pt1;
				lformat.format = format_common;
				lenb = insert_remove_line_numbers(text, b_show, &pos,
												  &fprop, &binfo,
												  begin1, end1 - begin1 + 1,
												  &lformat);
				pt1 += lenb;
			}

			fprop.fg = fg_color_f2;
			fprop.bg = bg_color_f2;
			pt2 = skip_n_lines(fi2->text, begin2-1, fi2->lenb);
			if (pt2) {
				binfo.buf = pt2;
				binfo.lenb = text_last2 - pt2;
				lformat.format = format_f2;
				lenb = insert_remove_line_numbers(text, b_show, &pos,
												  &fprop, &binfo,
												  begin2, end2 - begin2 + 1,
												  &lformat);
			}
			break;
		case F1ONLY:
			fprop.fg = fg_color_f1;
			fprop.bg = bg_color_f1;
			binfo.buf = pt1;
			binfo.lenb = text_last1 - pt1;
			lformat.format = format_f1;
			lenb = insert_remove_line_numbers(text, b_show, &pos,
											  &fprop, &binfo,
											  begin1, end1 - begin1 + 1,
											  &lformat);
			pt1 += lenb;
			break;
		default:/* not used */
			break;
		}
		line1 = end1 + 1;
	}
	/* Draw the remained part */
	fprop.fg = NULL;
	fprop.bg = NULL;
	binfo.buf = pt1;
	binfo.lenb = text_last1 - pt1;
	lformat.format = format_common;
	insert_remove_line_numbers(text, b_show, &pos, &fprop, &binfo,
						line1, fi1->nline - line1 + 1, &lformat);
	gtk_text_thaw(text);
}


/* Routines for drawing overview */
/**
 * create_overview:
 * Create two GdiffOverview widgets, and drawing-area widget between them to draw lines.
 * See "gdiffoverview.[ch]" about GdiffOverview widget.
 **/
static void
create_overview(GDiffFileViews *gfileviews)
{
	GtkHBox *hbox = gfileviews->pane.one.hbox;
	GtkAdjustment *vadj;
	GtkWidget *ov1;
	GtkWidget *ov2;
	GtkWidget *darea;/* drawing area */

	
	darea = gtk_drawing_area_new();
	gtk_widget_set_name(GTK_WIDGET(darea), "drawing overview");
	vadj = GTK_TEXT(gfileviews->pane.one.text)->vadj;
	ov1 = gdiff_overview_new(vadj);
	gtk_widget_set_name(GTK_WIDGET(ov1), "file1 overview");
	gdiff_overview_set_foreground(GDIFF_OVERVIEW(ov1), &gfileviews->pref.diff_bg[FIRST_FILE]);
	ov2 = gdiff_overview_new(vadj);
	gtk_widget_set_name(GTK_WIDGET(ov2), "file2 overview");
	gdiff_overview_set_foreground(GDIFF_OVERVIEW(ov2), &gfileviews->pref.diff_bg[SECOND_FILE]);

	gtk_box_pack_start(GTK_BOX(hbox), ov1, FALSE, FALSE, 0);
	gtk_widget_show(ov1);
	gtk_box_pack_start(GTK_BOX(hbox), darea, FALSE, FALSE, 0);
	gtk_widget_show(darea);
	gtk_box_pack_start(GTK_BOX(hbox), ov2, FALSE, FALSE, 0);
	gtk_widget_show(ov2);
	
	/* adjust drawing-area */
	gtk_drawing_area_size(GTK_DRAWING_AREA(darea), WIDTH_OVERVIEW, 0);/*XXX*/
	gdiff_overview_size(GDIFF_OVERVIEW(ov1), WIDTH_OVERVIEW, 0);/*XXX*/
	gdiff_overview_size(GDIFF_OVERVIEW(ov2), WIDTH_OVERVIEW, 0);/*XXX*/
	
	gfileviews->pane.one.darea = GTK_DRAWING_AREA(darea);
	gfileviews->pane.one.overview[FIRST_FILE] = GDIFF_OVERVIEW(ov1);
	gfileviews->pane.one.overview[SECOND_FILE] = GDIFF_OVERVIEW(ov2);
}

/**
 * draw_overview:
 * Draw diff parts on overview widget.
 **/
static void
draw_overview(GDiffFileViews *gfileviews)
{
	DiffFiles *files = gfileviews->dfiles;
	GdiffOverview *ov1 = gfileviews->pane.one.overview[FIRST_FILE];
	GdiffOverview *ov2 = gfileviews->pane.one.overview[SECOND_FILE];
	GtkDrawingArea *darea = gfileviews->pane.one.darea;
	GList *node;/* node of DiffLines' list */
	int merged_nline;
	int drawn_nlines1 = 0;
	int drawn_nlines2 = 0;
	
	gfileviews->merged_nline = merged_nline = dfiles_get_merged_nlines(files);
	if (merged_nline == 0) {
		g_warning("impossible: merged_nline == 0\n");
		return;
	}
	
	for (node = files->dlines_list; node; node = node->next) {
		DiffLines *dlines;
		int begin1, end1;
		int begin2, end2;
		int draw_nline1;
		int draw_nline2;
		int line;
		
		dlines = node->data;
		/* Use local variables for ease to read */
		begin1 = dlines->between[FIRST_FILE].begin;
		end1 = dlines->between[FIRST_FILE].end;
		begin2 = dlines->between[SECOND_FILE].begin;
		end2 = dlines->between[SECOND_FILE].end;

		draw_nline1 = end1 - begin1 + 1;
		line = begin1 + drawn_nlines2;
		gdiff_overview_insert_paintrange(ov1, (gdouble)line/merged_nline, (gdouble)(line+draw_nline1)/merged_nline);

		draw_nline2 = end2 - begin2 + 1;
		line = begin2 + drawn_nlines1;
		gdiff_overview_insert_paintrange(ov2, (gdouble)line/merged_nline, (gdouble)(line+draw_nline2)/merged_nline);

		switch (dlines->difftype) {
		case CHANGE:
			drawn_nlines1 += draw_nline1;
			drawn_nlines2 += draw_nline2;
			break;
		case F1ONLY:
			drawn_nlines1 += draw_nline1;
			break;
		case F2ONLY:
			drawn_nlines2 += draw_nline2;
			break;
		default:/* not used */
			break;
		}
	}
	/* Overview drawing area */
	gtk_signal_connect(GTK_OBJECT(darea), "expose_event",
					   GTK_SIGNAL_FUNC(draw_overview_lines), gfileviews);
	
	return;
}

/**
 * draw_overview_lines:
 * Draw lines which connects two diff parts of two files.
 **/
static void
draw_overview_lines(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	GDiffFileViews *gfileviews = data;
	const DiffFiles *files = gfileviews->dfiles;
	GList *node;/* node of DiffLines' list */
	GdkWindow *w = widget->window;
	GdkGC *gc = widget->style->fg_gc[GTK_STATE_NORMAL];
	int darea_width;
	int darea_height;
	int merged_nline = gfileviews->merged_nline;
	int drawn_nlines1 = 0;
	int drawn_nlines2 = 0;

	gdk_window_get_size(w, &darea_width, &darea_height);
	for (node = files->dlines_list; node; node = node->next) {
		int y1, y2;
		DiffLines *dlines;
		int begin1, end1;
		int begin2, end2;
		int draw_nline1;
		int draw_nline2;
		int line;

		dlines = node->data;
		/* Use local variables for ease to read */
		begin1 = dlines->between[FIRST_FILE].begin;
		end1 = dlines->between[FIRST_FILE].end;
		begin2 = dlines->between[SECOND_FILE].begin;
		end2 = dlines->between[SECOND_FILE].end;

		line = begin1 + drawn_nlines2;
		draw_nline1 = end1 - begin1 + 1;
		y1 = darea_height * ((double)(line+(line+draw_nline1))/(2*merged_nline));
		line = begin2 + drawn_nlines1;
		draw_nline2 = end2 - begin2 + 1;
		y2 = darea_height * ((double)(line+(line+draw_nline2))/(2*merged_nline));
		gdk_draw_line(w, gc, 0, y1, darea_width, y2);

		switch (dlines->difftype) {
		case CHANGE:
			drawn_nlines1 += draw_nline1;
			drawn_nlines2 += draw_nline2;
			break;
		case F1ONLY:
			drawn_nlines1 += draw_nline1;
			break;
		case F2ONLY:
			drawn_nlines2 += draw_nline2;
			break;
		default:/* not used */
			break;
		}
	}
}


/* Routines for move to a difference */
/**
 * guts_move_diff:
 **/
static void
guts_move_diff(GDiffFileViews *gfileviews, MoveDiff mv_diff)
{
	GtkAdjustment *vadj = gfileviews->pane.one.text->vadj;
	DiffFiles *files = gfileviews->dfiles;
	int merged_nline = gfileviews->merged_nline;
	int cur_line1;
	const DiffLines *dlines = NULL;
	int sum_f2 = 0;

	g_return_if_fail(merged_nline != 0);

	switch (mv_diff) {
	case MOVED_NEXT:
		gfileviews->cur_dlines_list = dfiles_find_nextl_in_merge(files, gfileviews->cur_dlines_list, &sum_f2);
		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		break;
	case MOVED_PREV:
		gfileviews->cur_dlines_list = dfiles_find_prevl_in_merge(files, gfileviews->cur_dlines_list, &sum_f2);
		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		break;
	case MOVED_FIRST:
		gfileviews->cur_dlines_list = dfiles_get_firstl_in_merge(files, gfileviews->cur_dlines_list, &sum_f2);
		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		break;
	case MOVED_LAST:
		gfileviews->cur_dlines_list = dfiles_get_lastl_in_merge(files, gfileviews->cur_dlines_list, &sum_f2);
		dlines = gfileviews->cur_dlines_list ? gfileviews->cur_dlines_list->data : NULL;
		break;
	case MOVED_REL_NEXT:
		cur_line1 = guess_visible_bottom_line(vadj, merged_nline);
		dlines = dfiles_find_rel_nextl_in_merge(files, cur_line1, &sum_f2);
		break;
	case MOVED_REL_PREV:
		cur_line1 = guess_visible_top_line(vadj, merged_nline);
		dlines = dfiles_find_rel_prevl_in_merge(files, cur_line1, &sum_f2);
		break;
	}

	if (dlines) {
		gdouble value;
		gdouble max;

		max = vadj->upper - vadj->lower;
		value = (double)(dlines->between[FIRST_FILE].begin + sum_f2) / merged_nline * max - (vadj->page_size / 2);
		gtk_adjustment_set_value(vadj, value);
	}
}

