/*
 *  apcsmart.c -- The decoding of the chatty little beasts.
 *                THE LOOK-A-LIKE ( UPSlink(tm) Language )
 *
 *  apcupsd.c  -- Simple Daemon to catch power failure signals from a
 *                BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *             -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

/*
 * Parts of the information below was taken from apcd.c & apcd.h
 *
 * Definitons file for APC SmartUPS daemon
 *
 *  Copyright (c) 1995 Pavel Korensky
 *  All rights reserved
 *
 *  IN NO EVENT SHALL PAVEL KORENSKY BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 *  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF PAVEL KORENSKY
 *  HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  PAVEL KORENSKY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 *  BASIS, AND PAVEL KORENSKY HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 *  UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  Pavel Korensky          pavelk@dator3.anet.cz
 *
 *  8.11.1995
 *
 *  P.S. I have absolutely no connection with company APC. I didn't sign any
 *  non-disclosure agreement and I didn't got the protocol description anywhere.
 *  The whole protocol decoding was made with a small program for capturing
 *  serial data on the line. So, I think that everybody can use this software
 *  without any problem.
 *
 * A few words regarding the APC serial protocol
 *
 * Firstly, the cable:
 * You will need a simple 3 wires cable connected as follows:
 *
 * PC (9 pin)     APC
 *  2 RxD          2
 *  3 TxD          1
 *  5 GND          9
 *
 * Insert by Andre Hedrick
 *
 * Use APC Cables #940-0024[B,C] and #940-0095[A,C]
 * This has been tested on a SmartUPS SU700/1400RM only.
 * Will work with SMNP card in SmartAccesories Slot
 *
 * Communication protocol which is used for controlling the APC SmartUPS
 * is very simple ASCII protocol.
 * The serial communication is 2400 Bd, 8N1
 *
 * Basically, your computer send a character to the UPS and UPS answer
 * with the requested information in the form of the ASCII string terminated
 * with CR/LF.
 * In some cases, the UPS answer is the action like battery test etc.
 *
 * There are two exceptions. If the UPS is switching to the battery, the
 * characters "!!" are sent to the computer. If the UPS is switching back
 * on-line, the charater '$' is sent to the computer. If battery is low,
 * '%' is sent to the computer. When battery is recharged to more than 15%,
 * '+' is sent to the computer.
 *
 * The protocol description is not based on informations obtained from APC.
 * I tried to analyse the protocol on RS-232 port, so my description
 * is not complete.
 *
 * Command are:
 *
 * SEND  ANSWER
 *
 * a     Returns all the Possible Characters used for the UPS Control
 *       Language.  This that funny string that everyone goes, HUH?
 *
 *       3.!$%+?=#|.O+-789>@ABCDEFGKLMNOPQRSUVWXYZ'abcefgjklmnopqrsuxyz~^
 *       3.!$%+?=#|.^A+-789>@ABCDEFGKLMNOPQRSUVWXYZ'abcefgjklmnopqrsuxyz~^?
 *
 * f     Battery status in %   100 = fully loaded
 * N     Line minimum voltage (V)
 * M     Line maximum voltage (V)
 * Q     Status byte 8 bits long.
 *         - 01  UPS is in calibration
 *         - 02  UPS in shut down state, waiting for AC
 *         - 04
 *         - 08  On-line LED ON
 *         - 10  On Battery LED ON
 *         - 20
 *         - 40  Battery power is almost used up,
 *                and the UPS starts beeping without pause.
 *         - 80  Replace Battery LED ON
 *
 *   Below are Bitwise addition and results.
 *
 *         - 18  On-line & On Battery LEDs ON
 *         - 50  On Battery LED ON, Battery power is almost used up.
 *         - 88  On-line & Replace Battery LEDs ON
 *         - 90  On Battery & Replace Battery LEDs ON
 *
 *   Below are Bit Value Discriptions
 *
 *         0 - If 1, UPS is in calibration
 *         1 - If 1, UPS is sleeping
 *         2 -
 *         3 - If 1, AC is here
 *         4 - If 1, UPS is running on battery
 *         5 -
 *         6 - If 1, Battery is low
 *         7 - If 1, Replace Battery
 *
 * P     UPS load (VA)
 * F     Line frequency (Hz)
 * L     Line voltage (V)
 * O     Output voltage (V)
 * C     Inside temperature (Celsius degrees)
 * B     Battery voltage
 *
 * Some signals for UPS control
 *
 * Y     Attention
 * W     Switch on battery
 * X     Switch back on-line
 * A     Lights test
 * U     Power failure test
 * c     UPS identification ???
 *
 * One very special signal
 *
 * Z ~1500ms pause Z  Switch off the UPS completely and imediately
 *
 *
 * Special thanks to Kevin D. Smolkowski (kevins@tigger.oslc.org)
 *  who observed other control codes
 *
 * D     Runtime calibration   ON/OFF
 * V     Firmware revison
 * c     UPS Identification
 * k     UPS Warning (Power Fail, Power Fail + 30, Low Battery, OFF)
 * l     Low transfer point
 * m     Manufacture date
 * n     UPS serial number
 * o     Nominal UPS output        "115" "100" "230"
 * p     UPS turn off delay        "020" "180" "300" "600" Cycles
 * q     Low battery signal time   "02"  "05"  "07"  "10"  Miniutes
 * r     Wakeup Delay              "000" "060" "180" "300" Cycles
 * s     Sensitivity "H" "M" "L"
 * u     High transfer point
 * x     Battery replacement date
 * y     Brand "(C) APCC"
 * 7     Option switch settings (Hex) "0F" "0E" "0C" "08" "00"
 *
 * -     Change prev. settings
 *
 * To change UPS name   c - /n newname /n
 *
 *
 * Further codes, not fully tested
 *
 * S     Soft shutdown, UPS will go back on-line when AC is back
 *
 * @xxx  see example below.
 *
 * @222  Soft forced shutdown and suspend for SmartUPS Only.
 *       w.p.: UPS will go back in 22 hours and 12 minutes!!
 *       w.p.: This is a suspend command, and every digit
 *       w.p.: counts for 1/10 hour (6 minutes)
 *
 * U     Power failure test
 * G     Return 'T' if there was a power failure
 *       Return 'S' if there was not a power failure
 *       Return 'R' unknown...........
 *       UPS Test will reset the value to the 'S'
 * K     1500ms pause K Switch off UPS completely after delay
 *
 * BK Pro additions by "Christopher J. Reimer" <reimer@doe.carleton.ca>
 *
 * a     Returns all the Possible Characters used for the UPS Control
 *       Language.  This that funny string that everyone goes, HUH?
 *
 *       BackUPS Pro return string:
 *
 *       3.!$%+?=#..89>@AGKQRSUWXY'abny~
 *       3.?=#.^A89@AGKQRSUWXY'abny~
 *
 * ^a    UPS identification
 * R     Puts UPS into DUMB Mode
 *
 * 8     always reports 00
 * 9     reports FF when on line, 00 when on battery
 *       (the windows 95 monitoring software is always sending to the ups ???)
 * -     always reports NO (also reported as indicates discharged battery)
 * +     always reports NO
 * '     always reports 00
 *
 *       Newer BackUPS Pro return string:
 *
 *       3.!$%+?=#...')+-89@ABDEFGKLMNOPQRSUVWXYZ#....o++++++..-..+|..f.
 *       3.!$%+?=#|..')+-89@ABDEFGKLMNOPQRSUVWXYZ#o+++++__+|<>
 *       3.?=#.^A^N^Z')-89@ABDEFGKLMNOPQRSUVWXYZabcdefgjklmnopqrsuxyz~^?
 *
 * Additional control features.
 *
 * E     Selftest intervals in hours 336 (14 days) 168 (7 days) ON OFF
 * d     unknown 008.9 but suspected to time (in hours) since last Selftest.
 * j     UPS internal calculation of remaining runtime.
 * e     UPS minimum recharge value of batteries before restart once
 *       power has returned values vary with UPS model.
 *       "00" "15" "50" "90" or "00" "10" "25" "90"
 * g     Battery Type in Voltage 012, 024, 048
 * V     Old Firmware revison and model ID String for SmartUPS & MatrixUPS
 *         This is a three character string XYZ
 *           where X == Smart-UPS or Matrix-UPS ID Code.
 *             range 0-9 and A-P
 *               1 == unknown
 *               0 == Matrix 3000
 *               5 == Matrix 5000
 *             the rest are Smart-UPS and Smart-UPS-XL
 *               2 == 250       3 == 400       4 == 400
 *               6 == 600       7 == 900       8 == 1250
 *               9 == 2000      A == 1400      B == 1000
 *               C == 650       D == 420       E == 280
 *               F == 450       G == 700       H == 700XL
 *               I == 1000      J == 1000XL    K == 1400
 *               L == 1400XL    M == 2200      N == 2200XL
 *               O == 3000      P == 5000
 *           where Y == Possible Level of Smart Features, unknown???
 *               G == Stand Alone
 *               T == Stand Alone
 *               W == Rack Mount
 *           where Z == National Model Use Only Codes
 *               D == Domestic        115 Volts
 *               I == International   230 Volts
 *               A == Asia ??
 *               J == Japan ??        100 Volts
 *
 *      New Firmware revison and model ID String in NN.M.L is the format
 *         This returns two numbers and a character.
 *           where NN == UPS ID Code.
 *               12 == Back-UPS Pro 650
 *               13 == Back-UPS Pro 1000
 *               52 == Smart-UPS 700
 *               72 == Smart-UPS 1400
 *
 *               where NN now Nn has possible meanings.
 *                   N  == Class of UPS
 *                   1n == Back-UPS Pro
 *                   5n == Smart-UPS
 *                   7n == Smart-UPS NET
 *
 *                    n == Level of intelligence
 *                   N1 == Simple Signal, if detectable WAG(*)
 *                   N2 == Full Set of Smart Signals
 *                   N3 == Micro Subset of Smart Signals
 *
 *           where M == Possible Level of Smart Features, unknown???
 *               1 == Stand Alone
 *               8 == Rack Mount
 *               9 == Rack Mount
 *           where L == National Model Use Only Codes
 *               D == Domestic        115 Volts
 *               I == International   230 Volts
 *               A == Asia ??
 *               J == Japan ??        100 Volts
 *
 * ;     unknown 00
 * ?     unknown 0000 33AB
 * ~     unknown 40
 *
 * w.p.: Made some enhancements. The modifications are marked with "w.p."
 * w.p.: Done and tested for/with Smart-UPS 700 only.
 * w.p.: - Ensure proper restart, if mains-power returns just after
 * w.p.:   LINUX shutdown.
 * w.p.: - Check if we can talk to UPS
 * w.p.: - getline() and read() are timed-out, thus we won't get hung
 * w.p.:   in case communication problems
 * w.p.: - Fixed a minor bug in updating_logs() (fprintf() format-specifiers)
 * w.p.:
 * w.p.:  Werner Panocha, wpanocha@t-online.de
 *
 * ^N == 0x?? (unsigned char)
 *       Turn on UPS with N/A return
 *
 * ^Z == 0x1a (unsigned char)
 * APC: Jul 06 00:38:00 apcupsd-libc5: 0x1a #
 *
 * This appears to be a complex coded return for information fields
 * based on the UPS model and country that it is intended to work.
 *
 * CMD DFO RSP FSZ FVL
 * u   D   4   3   127 130 133 136
 * u   A   4   3   108 110 112 114
 * u   I   4   3   253 257 261 265
 * l   D   4   3   106 103 100 097
 * l   A   4   3   092 090 088 086
 * l   I   4   3   208 204 200 196
 * e   4   4   2   00   15  50  90
 * o   D   1   3   115
 * o   J   1   3   100
 * o   I   1   3   230
 * s   4   4   1     H   M   L   L
 * q   4   4   2    02  05  07  10
 * p   4   4   3   020 180 300 600
 * k   4   4   1     0   T   L   N
 * r   4   4   3   000 060 180 300
 * E   4   4   3   336 168  ON OFF
 *
 * CMD == UPSlink Command to issued.
 * DFO == (4)-all-countries (D)omestic (I)nternational (A)sia ?? (J)apan ??
 * RSP == Total number possible answers returned by a given CMD.
 * FSZ == Max. number of field positions to be filled.
 * FVL == Values that are returned and legal.
 *
 * WAG(*) == Wild Arse Guess...............
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <apc_config.h>
#include <apc_i18n.h>
#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

static int getline_timeout = 0;  /* w.p. timeout-flag */

/*********************************************************************/
void sig_alarm (int sig)        /* w.p. signal-handler for time-out   */
{
	getline_timeout++;
}         /* w.p. */

/*********************************************************************/
char *smart_poll (char cmd, UPSINFO *ups)
{
	static char answer[128];

	*answer=0;

	write(ups->fd, &cmd, 1);
	getline(answer, ups);

	return(answer);
}

/*********************************************************************/
int getline (char *s, UPSINFO *ups)
{
	int i = 0;
	int ending = 0;
	char c;

	/*
	 *  Install SIGALRM handler, Should eventually be done in main(),
	 *  but I do it here just for now Allow for 3 seconds
	 */

	signal(SIGALRM, sig_alarm);
	alarm(3);
	getline_timeout = 0;

	while (!ending) {
		read(ups->fd, &c, 1);
		if (getline_timeout) {   /* w.p.  Watch timeout */
			ending++;
			continue;
		}
		switch(c) {
			case UPS_ON_BATT:        /* UPS_ON_BATT = '!'   */
				ups->LineUp = 1;
				break;
			case UPS_ON_LINE:        /* UPS_ON_LINE = '$'   */
				ups->LineUp = 0;
				break;
			case BATT_LOW:           /* BATT_LOW    = '%'   */
				ups->BattUp = 1;
				break;
			case BATT_OK:            /* BATT_OK     = '+'   */
				ups->BattUp = 0;
				break;
			case UPS_GARBAGE:        /* UPS_GARBAGE = '|'   */
				break;
			case UPS_TRAILOR:        /* UPS_TRAILOR = ':'   */
				break;
			case UPS_LF:             /* UPS_LF      = '\n'  */
				break;
			case UPS_CR:             /* UPS_CR      = '\r'  */
				ending = 1;      /* This what we waited */
				break;           /* before moving on.   */
			default:
				s[i++] = c;
				break;
		}
	}
	alarm(0);                                   /* w.p. Alarm off      */
	signal(SIGALRM, SIG_IGN);                                  /* w.p. */
	s[i] = '\0';
	return(0);
}

/*********************************************************************/
void UPSlinkCheck (UPSINFO *ups)
{
	char *a;
	int comm_err = 0;
	int error_count = 0;

	/* We will get trapped into this loop, if communication with UPS
	 * is lost (maybe by a broken ore accidently unplugged cable)
	 *
	 * What else could be done in such cases?
	 * Should we send broadcasts, if this situation holds on
	 * for a longer time?
	 */

	do {
		if (strcmp(a = smart_poll('Y', ups),"SM")) {
			sleep(2);
			/*
			 *  Wait and try a recovery
			 */
			tcflush(ups->fd, TCIOFLUSH);

			if (strcmp(a = smart_poll('Y', ups),"SM")) {
				/*
				 *  Discard UPS's response, if any
				 */
				tcflush(ups->fd, TCIOFLUSH);
				error_count++;
				comm_err = 1;
				logprintf("UPSlink Comm. Error, %s != SM count %d\n",
				          a, error_count);
			} else {
				comm_err = 0;
			}
		}

		if (error_count == 100)
			system(ups->restartme);
	} while(comm_err);

	if ((error_count != 0) && (comm_err == 0)) {
		logprintf("UPSlink Comm. reestablished, %s == SM count %d\n",
		          a, error_count);
		error_count = 0;
	}
#ifdef __DEBUG
	logprintf("UPSlink Comm. Report, %s == SM count %d\n",
	          a, error_count);
#endif /* __DEBUG */
}

/*********************************************************************/
int fillUPS (UPSINFO *ups)
{
	UPSlinkCheck(ups);

	/* UPS_STATUS */
	if (ups->UPSlink_flags & 0x00000001) {
		ups->Status = strtoul(smart_poll('Q', ups),NULL,16);
		/* Use the info in the status bits */
		test_status_bits(ups);
	}

	/* ONBATT_STATUS_FLAG */
	if (ups->UPSlink_flags & 0x00000002)
		strcpy(ups->nine, smart_poll('9', ups));

	/* POWER_FAIL_HISTORY */
	if (ups->UPSlink_flags & 0x00000004)
		strcpy(ups->G, smart_poll('G', ups));

	/* UPS Self-Test Status and return on-line? */
	if (ups->UPSlink_flags & 0x00000008)
		strcpy(ups->X,smart_poll('X', ups));

	/* LINE_VOLTAGE */
	if (ups->UPSlink_flags & 0x00000010)
		ups->LineVoltage = atof(smart_poll('L', ups));

	/* UPS_LINE_MAX */
	if (ups->UPSlink_flags & 0x00000020)
		ups->LineMax = atof(smart_poll('M', ups));

	/* UPS_LINE_MIN */
	if (ups->UPSlink_flags & 0x00000040)
		ups->LineMin = atof(smart_poll('N', ups));

	/* OUTPUT_VOLTAGE */
	if (ups->UPSlink_flags & 0x00000080)
		ups->OutputVoltage = atof(smart_poll('O', ups));

	/* BATT_FULL */
	if (ups->UPSlink_flags & 0x00000100)
		ups->BatLoad = atof(smart_poll('f', ups));

	/* BATT_VOLTAGE */
	if (ups->UPSlink_flags & 0x00000200)
		ups->BattVoltage = atof(smart_poll('B', ups));

	/* UPS_LOAD */
	if (ups->UPSlink_flags & 0x00000400)
		ups->UPSLoad = atof(smart_poll('P', ups));

	/* LINE_FREQ */
	if (ups->UPSlink_flags & 0x00000800)
		ups->LineFreq = atof(smart_poll('F', ups));

	/* UPS_RUNTIME_LEFT */
	if (ups->UPSlink_flags & 0x00001000)
		ups->TimeLeft = atof(smart_poll('j', ups));

	/* UPS_TEMP */
	if (ups->UPSlink_flags & 0x00002000)
		ups->UPSTemp = atof(smart_poll('C', ups));

	/* DIP_SWITCH_SETTINGS */
	if (ups->UPSlink_flags & 0x00004000)
		ups->seven = strtoul(smart_poll('7', ups), NULL, 16);

	/* UNKNOWN */
	if (ups->UPSlink_flags & 0x00008000)
		strcpy(ups->eight, smart_poll('8', ups));

	/* UPS_ENABLE */
	smart_poll('Y', ups);
	smart_poll('Y', ups);

	return(0);
}

/*********************************************************************/
void ReportConfig (UPSINFO *ups)
{
	char msg[100];
	time_t nowtime;

	time(&nowtime);
	strftime(msg, 100, "%b %d %X", localtime(&nowtime));

	fprintf(stderr,"APC      : %s\n", msg);
	fprintf(stderr,"CABLE    : %s\n", ups->cable.long_name);
	fprintf(stderr,"UPSMODEL : %s\n", ups->mode.long_name);
	fprintf(stderr,"UPSMODE  : %s\n", ups->class.long_name);
	if (ups->sharenet.type != DISABLE)
		fprintf(stderr,"SHARE    : %s\n", ups->sharenet.long_name);

	switch(ups->mode.type) {
		case BK:
		case SHAREBASIC:
		case NETUPS:
			break;
		case BKPRO:
		case VS:
		case NBKPRO:
		case SMART:
		case SHARESMART:
		case MATRIX:
			if (ups->UPSlink_flags & 0x01000000)
				fprintf(stderr, "UPSNAME  : %s\n", ups->name);

			if (ups->UPSlink_flags & 0x40000000)
				fprintf(stderr, "REPLACE  : %s\n", ups->replace);

			if (ups->UPSlink_flags & 0x00010000) {
				switch((*ups).sensitivity[0]) {
					case 'L':
						fprintf(stderr, "SENSE    : LOW\n");
						break;
					case 'M':
						fprintf(stderr, "SENSE    : MEDIUM\n");
						break;
					case 'H':
					default :
						fprintf(stderr, "SENSE    : HIGH\n");
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00020000)
				fprintf(stderr, "WAKEUP   : %03d Cycles\n", ups->wakeup);

			if (ups->UPSlink_flags & 0x00040000)
				fprintf(stderr, "SLEEP    : %03d Cycles\n", ups->sleep);

			if (ups->UPSlink_flags & 0x00080000)
				fprintf(stderr, "LOTRANS  : %03d.0 Volts\n", ups->lotrans);

			if (ups->UPSlink_flags & 0x00100000)
				fprintf(stderr, "HITRANS  : %03d.0 Volts\n", ups->hitrans);

			if (ups->UPSlink_flags & 0x00200000)
				fprintf(stderr, "CHARGE   : %03d.0 Percent\n", ups->minon);

			if (ups->UPSlink_flags & 0x00400000) {
				switch((*ups).beepstate[0]) {
					case 'T':
						fprintf(stderr, "ALARM    : Low Batt + 30\n");
						break;
					case 'L':
						fprintf(stderr, "ALARM    : Low Batt\n");
						break;
					case 'N':
						fprintf(stderr, "ALARM    : Never\n");
						break;
					case '0':
					default :
						fprintf(stderr, "ALARM    : Always\n");
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00800000)
				fprintf(stderr, "LOWBATT  : %02d Minutes\n", ups->lowbatt);

			if (ups->UPSlink_flags & 0x00004000) {
				if (ups->seven & 0x08)
					fprintf(stderr, "DIPSW    : 0x%d", TRUE);
				else
					fprintf(stderr, "DIPSW    : 0x%d", FALSE);
				if (ups->seven & 0x04)
					fprintf(stderr, "%d", TRUE);
				else
					fprintf(stderr, "%d", FALSE);
				if (ups->seven & 0x02)
					fprintf(stderr, "%d", TRUE);
				else
					fprintf(stderr, "%d", FALSE);
				if (ups->seven & 0x01)
					fprintf(stderr, "%d\n", TRUE);
				else
					fprintf(stderr, "%d\n", FALSE);
			}
#if 0
			fprintf(stderr, "%s: %s\n", ups->argvalue, smart_poll(')', ups));
#endif
			break;
		default:
			break;
	}
}

/*********************************************************************/
void ReadUPS (UPSINFO *ups)
{
	UPSlinkCheck(ups);

	/* UPS_STATUS */
	if (ups->UPSlink_flags & 0x00000001) {
		ups->Status = strtoul(smart_poll('Q', ups),NULL,16);
		/* Use the info in the status bits */
		test_status_bits(ups);
	}

	/* ONBATT_STATUS_FLAG */
	if (ups->UPSlink_flags & 0x00000002)
		strcpy(ups->nine, smart_poll('9', ups));

	/* POWER_FAIL_HISTORY */
	if (ups->UPSlink_flags & 0x00000004)
		strcpy(ups->G, smart_poll('G', ups));

	/* UPS Self-Test Status and return on-line? */
	if (ups->UPSlink_flags & 0x00000008)
		strcpy(ups->X,smart_poll('X', ups));

	/* LINE_VOLTAGE */
	if (ups->UPSlink_flags & 0x00000010)
		ups->LineVoltage = atof(smart_poll('L', ups));

	/* UPS_LINE_MAX */
	if (ups->UPSlink_flags & 0x00000020)
		ups->LineMax = atof(smart_poll('M', ups));

	/* UPS_LINE_MIN */
	if (ups->UPSlink_flags & 0x00000040)
		ups->LineMin = atof(smart_poll('N', ups));

	/* OUTPUT_VOLTAGE */
	if (ups->UPSlink_flags & 0x00000080)
		ups->OutputVoltage = atof(smart_poll('O', ups));

	/* BATT_FULL */
	if (ups->UPSlink_flags & 0x00000100)
		ups->BatLoad = atof(smart_poll('f', ups));

	/* BATT_VOLTAGE */
	if (ups->UPSlink_flags & 0x00000200)
		ups->BattVoltage = atof(smart_poll('B', ups));

	/* UPS_LOAD */
	if (ups->UPSlink_flags & 0x00000400)
		ups->UPSLoad = atof(smart_poll('P', ups));

	/* LINE_FREQ */
	if (ups->UPSlink_flags & 0x00000800)
		ups->LineFreq = atof(smart_poll('F', ups));

	/* UPS_RUNTIME_LEFT */
	if (ups->UPSlink_flags & 0x00001000)
		ups->TimeLeft = atof(smart_poll('j', ups));

	/* UPS_TEMP */
	if (ups->UPSlink_flags & 0x00002000)
		ups->UPSTemp = atof(smart_poll('C', ups));

	/* DIP_SWITCH_SETTINGS */
	if (ups->UPSlink_flags & 0x00004000)
		ups->seven = strtoul(smart_poll('7', ups), NULL, 16);

	/* UNKNOWN */
	if (ups->UPSlink_flags & 0x00008000)
		strcpy(ups->eight, smart_poll('8', ups));

	/* SENSITIVITY */
	if (ups->UPSlink_flags & 0x00010000)
		strcpy(ups->sensitivity, smart_poll('s', ups));

	/* WAKEUP_DELAY */
	if (ups->UPSlink_flags & 0x00020000)
		ups->wakeup = atof(smart_poll('r', ups));

	/* SLEEP_DELAY */
	if (ups->UPSlink_flags & 0x00040000)
		ups->sleep = atof(smart_poll('p', ups));

	/* LOW_TRANSFER_LEVEL */
	if (ups->UPSlink_flags & 0x00080000)
		ups->lotrans = atof(smart_poll('l', ups));

	/* HIGH_TRANSFER_LEVEL */
	if (ups->UPSlink_flags & 0x00100000)
		ups->hitrans = atof(smart_poll('u', ups));

	/* UPS_BATT_CAP_RETURN */
	if (ups->UPSlink_flags & 0x00200000)
		ups->minon = atof(smart_poll('e', ups));

	/* ALARM_STATUS */
	if (ups->UPSlink_flags & 0x00400000)
		strcpy(ups->beepstate, smart_poll('k', ups));

	/* LOWBATT_SHUTDOWN_LEVEL */
	if (ups->UPSlink_flags & 0x00800000)
		ups->lowbatt = atof(smart_poll('q', ups));

	/* UPS_NAME */
	if (ups->UPSlink_flags & 0x01000000)
		strcpy(ups->name, smart_poll('c', ups));

	/* UPS_SELFTEST */
	if (ups->UPSlink_flags & 0x02000000)
		strcpy(ups->selftest, smart_poll('E', ups));

	if (ups->UPSlink_flags & 0x04000000);
	if (ups->UPSlink_flags & 0x08000000);

	/* UPS_MANUFACTURE_DATE */
	if (ups->UPSlink_flags & 0x10000000)
		strcpy(ups->birth, smart_poll('m', ups));

	/* UPS_SERIAL_NUMBER */
	if (ups->UPSlink_flags & 0x20000000)
		strcpy(ups->serial, smart_poll('n', ups));

	/* UPS_BATTERY_REPLACE */
	if (ups->UPSlink_flags & 0x40000000)
		strcpy(ups->replace, smart_poll('x', ups));

	/* UPS_BATTERY_SIZE */
	if (ups->UPSlink_flags & 0x80000000)
		ups->battery = atof(smart_poll('g', ups));
}

/**********************************************************************
 * Is this a special case for the Back-UPS Pro since,
 * getline(int getfd, char *s) tests the entire 8-bit serial read
 * regardless of the polling command charater.  This assumption is
 * based on who apcd-0.5 was designed.
 *********************************************************************/
void test_status_bits (UPSINFO *ups)
{
	if (ups->Status & 0x10)
		ups->LineUp = 1;
	else
		ups->LineUp = 0;
	if (ups->Status & 0x40)
		ups->BattUp = 1;
	else
		ups->BattUp = 0;
	if ((ups->Status & 0x0C) == 1)
		ups->LineLevel = -1;		/* LineVoltage Low */
	else if ((ups->Status & 0x0A) == 1)
		ups->LineLevel = 1;		/* LineVoltage High */
	else
		ups->LineLevel = 0;		/* LineVoltage Normal */

	if (ups->Status & 0x80)			/* Replace Battery */
		system(ups->changeme);
}
