#
# FreeDB.pm a Perl module for accessing the internet free CD DataBase
#
# Copyright (C) 1998, 1999 B. W. Fitzpatrick <fitz@red-bean.com>
#
# Authors: B. W. Fitzpatrick <fitz@red-bean.com>, 
#          Richard Martin, and J.I. van Hemert <jvhemert@cs.leidenuniv.nl>
# Maintainer: B. W. Fitzpatrick <fitz@red-bean.com>
# Created: September, 1999
# Keywords: cddb, compact-disc
#
# $Id: FreeDB.pm,v 1.12 1999/09/28 03:30:20 bwf Exp $
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you can either send email to this
# program's maintainer or write to: The Free Software Foundation,
# Inc.; 59 Temple Place, Suite 330; Boston, MA 02111-1307, USA.

package FreeDB;

use strict;

use HTTP::Request;
use LWP::UserAgent;
use URI::URL;
use Carp;

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;

@ISA = qw(Exporter AutoLoader);
# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.
@EXPORT = qw(
	
);

$VERSION = '0.6.3';

# Preloaded methods go here.

sub new {
   my $proto = shift;
   my $class = ref($proto) || $proto;
   my $self  = {};
   
   # What they're here for
   $self->{title}       = undef;
   $self->{artist}      = undef;
   $self->{disc_length}  = undef;
   $self->{discid}      = "";
   $self->{genre}       = "";
   $self->{disc_info}   = ();
   $self->{number_of_tracks}   = undef;

   $self->{extended_track_info} = ();
   $self->{track_info}          = ();
   $self->{track_lengths}       = ();
   $self->{offsets}             = ();
   $self->{detailed_track_info} = ();

   $self->{current_track_number} = 0;

   # Configuration iVars (these have setters)
   $self->{cdrom_device}    = "/dev/cdrom";
   $self->{cache_directory} = "$ENV{HOME}/.cddb";
   $self->{cddb_server}     = "http://freedb.freedb.org/cgi-bin/cddb.cgi";
   $self->{proxy_value}     = "";
   $self->{ignore_cache}    = 0;
   # Using freedb due to licensing issues.
   #$self->{cddb_server}     = "http://cddb.cddb.org/~cddb/cddb.cgi";

   # iVars that are truly internal to the class
   $self->{_hello_server}   = "hello=username+this.is.mine.org+Free-DB+0.9.2";
   $self->{_server_string}  = undef;
   $self->{_cddb_string}    = "";
   $self->{_debug}          = 0;

   # I dub thee ``object''
   bless ($self, $class);
   return $self;
}

###########################################################################
# Public methods

#These are set in our constructor

# Args:    string (optional)
# Returns: string
# Sets:    cddb_server if called with an Arg.
sub cddb_server {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "cddb_server is $tmp");
      $self->{cddb_server} = $tmp;
   }
   return $self->{cddb_server};
}

# Args:    string (optional)
# Returns: string
# Sets:    cdrom_device if called with an Arg.
sub cdrom_device {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "cdrom_device is $tmp");
      $self->{cdrom_device} = $tmp;
   }
   return $self->{cdrom_device};
}

# Args:    string (optional)
# Returns: string
# Sets:    cache_directory if called with an Arg.
sub cache_directory {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "cache_directory is $tmp");
      $self->{cache_directory} = $tmp;
   }
   return $self->{cache_directory};
}

# Args:    BOOL (optional)
# Returns: BOOL
# Sets:    ignore_cache if called with an Arg.
sub ignore_cache {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "ignore_cache is $tmp");
      $self->{ignore_cache} = $tmp;
   }
   return $self->{ignore_cache};
}

# Args:    BOOL (optional)
# Returns: BOOL
# Sets:    proxy_value if called with an Arg.
sub proxy_value {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "proxy_value is $tmp");
      $self->{proxy_value} = $tmp;
   }
   return $self->{proxy_value};
}

## Not set in our constructor

# Args:    NONE
# Returns: BOOL
# Sets:    
sub fetch {
   my $self = shift;
   my $input = shift || ""; 
   my $success;

   if ($input eq "")
   {
	   $self->_get_disc_info();
	   
	   # We might want to ignore our cache and get fresh data from the server
	   if (!$self->ignore_cache) {
	      $success = $self->_get_info_from_cache_directory;
	   }
	   if (!$success) {
	      $success = $self->_get_info_from_cddb();
	   }

	   if (!$success) {
	      return 0; # We lose
	   }

	   $self->debug(4, "Got string: $self->{_cddb_string}");
	   $self->debug(2, "Got genre: $self->{genre}");
   }
   else
   {
   	   $self->{_cddb_string} = $input;
   }	   

   ## Working to here.
   
   if (($self->{_cddb_string} ne "") && ($self->{genre} ne "")) {
      $self->_save_info_in_cache_directory(
                                          $self->{discid}, 
                                          $self->{genre}, 
                                          $self->{_cddb_string});
   }

   $self->_set_artist_and_title();
   $self->_set_tracks();

   $self->debug(4, "Got artist: $self->{artist}");
   $self->debug(4, "Got title: $self->{title}");
   return 1;
}

# Args:    string (optional)
# Returns: string
# Sets:    artist if called with an Arg.
sub artist {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "artist is $tmp");
      $self->{artist} = $tmp;
   }
   return $self->{artist};
}

# Args:    string (optional)
# Returns: string
# Sets:    discid if called with an Arg.
sub discid {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "discid is $tmp");
      $self->{discid} = $tmp;
   }
   return $self->{discid};
}

# Args:    NONE
# Returns: void
# Sets:    {disc_info}
sub disc_info
{
   my $self = shift @_;
	my $tmp = $self->{_cddb_string};
	
	my ($extend, @extd, $i);
	@extd = ($tmp =~ m%^EXTD=([^\n\r]*)%mg);
	foreach $i (@extd) {
		$extend .= "$i";
	}
   $self->{disc_info} = $extend;
}

# Args:    integer (optional)
# Returns: integer
# Sets:    number of tracks if called with an Arg.
sub number_of_tracks {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "number_of_tracks is $tmp");
      $self->{number_of_tracks} = $tmp;
   }
   return $self->{number_of_tracks};
}

# Args:    string (optional)
# Returns: string
# Sets:    title if called with an Arg.
sub title {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "title is $tmp");
      $self->{title} = $tmp;
   }
   return $self->{title};
}

# Args:    NONE
# Returns: string
# Sets:    {disc_length};
sub disc_length {
   my $self = shift @_;
	my $tmp  = $self->{_cddb_string};
	$tmp =~ /^\#\s*Disc length:\s*(\d+)/mg;
   $self->{disc_length} = $1;
   return $self->{disc_length};
}

# Args:    string (optional)
# Returns: string
# Sets:    total_tracks if called with an Arg.
sub total_tracks {
   my $self = shift;
   if (!$self->{total_tracks}) {
      $self->{total_tracks} = $self->track_info;
   }
   return $self->{total_tracks};
}

# Args:    array (optional)
# Returns: array
# Sets:    track_info if called with an Arg.
sub track_info {
   my $self = shift;
   if (@_) { 
      my @tmp = @_;
      $self->debug(3, "setting track_info to: @_");
      $self->{track_info} = \@tmp;
   }
   $self->debug(3, "track_info set to: @{$self->{track_info}}");
   return @{$self->{track_info}};
}

# Args:    array (optional)
# Returns: array
# Sets:    offsets if called with an Arg.
sub offsets {
   my $self = shift;
   if (@_) { 
      my @tmp = @_;
      $self->debug(3, "setting offsets to: @_");
      $self->{offsets} = \@tmp;
   }
   $self->debug(3, "offsets set to: @{$self->{offsets}}");
   return @{$self->{offsets}};
}

# Args:    array (optional)
# Returns: array
# Sets:    extended_track_info if called with an Arg.
sub extended_track_info {
   my $self = shift;
   if (@_) { 
    #  my @tmp = @_;
      my @tmp = map { $_ =~ s/\\n/
/g; $_} @_;
      $self->debug(3, "setting extended_track_info to: @_");
      $self->{'extended_track_info'} = \@tmp;
   }
   $self->debug(3, "extended_track_info set to: @{$self->{'extended_track_info'}}");
   return @{$self->{'extended_track_info'}};
}

# Args:    array (optional)
# Returns: array
# Sets:    detailed_track_info if called with an Arg.
sub detailed_track_info {
   my $self = shift;
   if (@_) { 
      my @tmp = @_;
      $self->debug(3, "setting detailed_track_info to: @_");
      $self->{detailed_track_info} = \@tmp;
   }
   $self->debug(3, "detailed_track_info set to: @{$self->{detailed_track_info}}");
   return @{$self->{detailed_track_info}};
}

# Args:    array (optional)
# Returns: array
# Sets:    track_lengths if called with an Arg.
sub track_lengths {
   my $self = shift;
   if (@_) { 
      my @tmp = @_;
      $self->debug(3, "setting track_lengths to: @_");
      $self->{track_lengths} = \@tmp;
   }
   $self->debug(3, "track_lengths set to: @{$self->{track_lengths}}");
   return @{$self->{track_lengths}};
}

# Args:    NONE
# Returns: int, current track number
# Sets:    increments the track number, if possible, else sets it to 0
sub next_track {
   my $self = shift;
   if ($self->{current_track_number} < $self->total_tracks()) {
      return ($self->{current_track_number} += 1);
   } 
   else {
      return 0;
   }
}

sub current_track_info {
   my $self = shift;
   return $self->{"track_info"}[$self->{current_track_number} - 1];
}

sub current_track_number {
   my $self = shift;
   if (@_) {
      my $track = ((shift @_));
      if ($track < $self->total_tracks + 1) {
         $self->{current_track_number} = $track;
         return $track;
      }
      else {
         carp "You tried to set a track number that is greater" .
             "than the total number of tracks. Tsk. Tsk. Bailing...";
         return 0
      }
   }
   return $self->{current_track_number};
}

sub current_track_number_padded {
   my $self = shift;
   my $num = $self->{"current_track_number"};
   if ($num < 10) {
      return "0$num";
   }
   return $num;
}

sub current_track_extended_info {
   my $self = shift;
   return $self->{"extended_track_info"}[$self->{"current_track_number"} - 1];
}

sub current_track_detailed_info {
   my $self = shift;
   return $self->{"detailed_track_info"}[$self->{"current_track_number"} - 1];

}

sub current_track_time_in_minutes {
   my $self = shift;
   my $tmp =  $self->{"track_lengths"}[$self->{"current_track_number"} - 1];
   my $min = int(int($tmp) / 60);
   my $sec = int($tmp) % 60;
   if ($sec < 10) { $sec = "0$sec"; }

   return "$min:$sec";
}

sub current_track_time_in_seconds {
   my $self = shift;
   return $self->{"track_lengths"}[$self->{"current_track_number"} - 1];

}

sub current_track_offset {
   my $self = shift;
   return $self->{"offsets"}[$self->{"current_track_number"} - 1];
}

sub cddb_string {
   my $self = shift;
   return $self->{_cddb_string};
}


###########################################################################
# Private methods

# Args:    NONE
# Returns: void
# Sets:    {artist} and {title}
sub _set_artist_and_title
{
   my $self = shift @_;
	my $tmp = $self->{_cddb_string};

	my (@tmp2) = ($tmp =~ m%^DTITLE=(.*)%mg);

	my ($dtitle) = "";
	my ($i) = "";
	foreach $i (@tmp2)
	{
		$dtitle .= $i;
	}

	my ($artist, $title);
	($artist, $title) = ($dtitle =~ m%([^\/]*) / (.*)%mg);

	$artist =~ s/
//g;
	$title =~ s/
//g;
	if ($artist eq "")
	{
		# Catch artist/title disks without / 
		($title) = ($tmp =~ m%^DTITLE=(.*)%m);
		($artist) = ($tmp =~ m%^DTITLE=(.*)%m);
	}


   $self->{"artist"} = $artist;
   $self->{"title"} = $title;
}

# Args:    NONE
# Returns: void
# Sets:    {extended_track_info}, {track_info}, {track_lengths}, 
#          {offsets}, and {detailed_track_info}
sub _set_tracks
{
   my $self = shift @_;
   my ($offset, $length, $i, @trackinfo);
	my $tmp = $self->{_cddb_string};

	my (@extended);
	foreach $i ($tmp =~ m%^EXTT(\d+=.*)[\n\r]*%mg)
	{
		my ($trackno, $ttitle) = split('=', $i);
		$extended[$trackno] .= $ttitle;
	}

	my (@tracks);
	foreach $i ($tmp =~ m%^TTITLE(\d+=.*)[\n\r]*%mg)
	{
		my ($trackno, $ttitle) = split('=', $i);
		$tracks[$trackno] .= $ttitle;
	}

	my (@offsets) = ($tmp =~ /^\#\s+(\d+)/mg);
	my ($disclength) = ($tmp =~ /^\#\s*Disc length:\s*(\d+)/mg);
	

	my ($lastOffset) = $offsets[0];
	my (@lengths) = ();

   $self->debug(2,"Offset[1] is: $offsets[0]");

	foreach $offset (@offsets[1..$#offsets]) {
      # some sort of weird push?
      @lengths = (@lengths, ($offset - $lastOffset));
      $lastOffset = $offset;  
	}
	
	foreach $length (@lengths) {
      $length /= 75;
	}

	@lengths = (@lengths, $disclength - ( $lastOffset / 75 ) );

   # This rounds times out properly.
   @lengths = map { my $m = $_ - int($_); if (substr($m,0,3) >= .5) { $_++; } int($_); } @lengths;

	for ($i = 0; $i <= $#tracks; $i++)
	{
		$trackinfo[$i] = join ("\n", $i + 1, $tracks[$i], $extended[$i], $lengths[$i], $offsets[$i]);
	}

   $self->extended_track_info(@extended);
   $self->track_info(@tracks);
   $self->offsets(@offsets);
   $self->detailed_track_info(@trackinfo);
   $self->track_lengths(@lengths);
   $self->number_of_tracks($#tracks + 1);

   $self->debug(3,"iVar tracks are ", $self->track_info());
}

# Gets the info that we need from the CD. 
# Args:    NONE
# Returns: BOOL
# Sets:    {discid} and _server_string()
sub _get_disc_info {
   my $self = shift @_;
	my ($result, $start, $end, $discid, $tracknumber);
   my ($track, $adr_ctrl, $format, $frame, $minute, $second);
   my $chdr = "";
   my @tracks = ();

	open (cdrom_fd, "$self->{'cdrom_device'}") or croak "Error: could not open device: \"$self->{'cdrom_device'}\"\n\nSolution: login as root and make a new group 'cdrom' in /etc/group\nthen issue these commands:\n  chgrp cdrom $self->{'cdrom_device'}\n  cdmod g+r $self->{'cdrom_device'}\n  adduser <username> cdrom\nNow logout and login as the user again.\n\n";

	# Get start and end track number
   $result = ioctl(cdrom_fd, 0x5305, $chdr);
   $self->debug(2,"RESULT IS: $result\n");
	($start, $end) = unpack('CC', $chdr);

	# Get info for each track
	for ($tracknumber = $start; $tracknumber <= $end; $tracknumber++) {
		my $tocentry = pack('C8', $tracknumber, 0, 2, 0, 0, 0, 0, 0);

		if( ioctl(cdrom_fd, 0x5306, $tocentry) < 0 ) {
			croak "Error: problems during cdrom tracks read in\n\nSolution: sorry no solution\n";
		}
		($track, $adr_ctrl, $format, $frame, $minute, $second) = unpack('C*', $tocentry);
		$tracks[$tracknumber] = (60 * $minute + $second);

	}

	# Get info for leadout
	my $tocentry = pack('C8', 0xAA, 0, 2, 0, 0, 0, 0 ,0);
	if( ioctl(cdrom_fd, 0x5306, $tocentry) < 0 )
	{
		croak "Error: problems during cdrom tracks read in\n\nSolution: sorry no solution\n";
	}
	($track, $adr_ctrl, $format, $frame, $minute, $second) = unpack('C*', $tocentry);
	$tracks[$end + 1] = (60 * $minute + $second);

	# Calculate cddb sum for all tracks and for total
	my $sum_of_all_tracks = 0;
	my $server_string = '';
	for ($tracknumber = $start; $tracknumber <= $end; $tracknumber++)
	{
		$sum_of_all_tracks += $self->_cddb_sum($tracks[$tracknumber]);	
		$server_string .= ($tracks[$tracknumber] * 75)." ";
	}

	# Calculate cddb disc id
	$discid = sprintf ("%08lx", ($sum_of_all_tracks % 0xFF) << 24 | ($tracks[$end + 1] - $tracks[$start]) << 8 | $end);

	# Construct string that can be used to query cddb
	$server_string = $discid." ".$end." ".$server_string."".$tracks[$end + 1];

	close (cdrom_fd);

	# Return disc-id, number of tracks, cddb server string and info for all tracks
   $self->debug(2, "Disk info follows...", 
                "discid is: $discid", 
                "server_string is: $server_string");

   $self->{"discid"}        = $discid;
   $self->_server_string($server_string);
   return 1;
}


# Saves a cddb record to a local file
# Args:    NONE
# Returns: BOOL
# Sets:    NOTHING
sub _save_info_in_cache_directory {
   my $self = shift @_;
   my $tmp;

   # Be paranoid and make sure that we have all the pieces
   if (($self->{"discid"}     eq "") ||
       ($self->{genre}        eq "") ||
       ($self->{_cddb_string} eq "")) {
      carp "Not saving--missing discid or genre or _cddb_string\n";
      return 0;
   }
   # Check discid and genre for non-zero length

   
	if (! -d    $self->{"cache_directory"}) {
		mkdir    $self->{"cache_directory"}, 0700;
   }
	if (! -d "$self->{'cache_directory'}/$self->{genre}") {
		mkdir "$self->{'cache_directory'}/$self->{genre}", 0700;
	}

	$self->debug(2, "Writing entry in \"$self->{'cache_directory'}/$self->{genre}/$self->{'discid'}\"");

	open (CDDB_FILE, "> $self->{'cache_directory'}/$self->{genre}/$self->{'discid'}") or croak "Error: could not open file \"$self->{'cache_directory'}/$self->{genre}/$self->{'discid'}\"\n\nSolution: make sure the directory permissions are writable for you.\n\n";

   # What exactly does this do?
   $tmp = $self->{_cddb_string};
	$tmp =~ s/^\.//gm;

	print CDDB_FILE $tmp;
	close (CDDB_FILE);
   return 1;
}

# Puts the entry into _cddb_string if we have it cached.
# Args:    NONE
# Returns: BOOL
# Sets:    {_cddb_string} on success
sub _get_info_from_cache_directory {
   my $self = shift @_;
	my $cddb_file = "";
   my $genre;

   $self->debug(2, "Looking for discid $self->{'discid'} in cache dir \"$self->{'cache_directory'}\"");
	if (-d $self->{"cache_directory"}) {
      my (@subdirs, $sub);
      opendir(DIR, $self->{"cache_directory"}) or croak "Couldn't open $self->{'cache_directory'} for reading: $!\n";
      @subdirs = readdir(DIR);
      closedir(DIR); 
      
    FILES: {
         foreach $sub (@subdirs) {
            if ($sub !~ /^\.\./) {
               my $try_file = "$self->{'cache_directory'}/$sub/$self->{'discid'}";
               $self->debug(1,"CACHE: Checking for $try_file");
               if (-e $try_file) {
                  $self->debug(1,"CACHE: GOT ONE!");
                  $cddb_file = $try_file;
                  $genre     = $sub; 
                  last FILES;
               }
            }
         }
      }
   }
   else {
      $self->debug(2,"Warning; no directory \"$self->{'cache_directory'}\" found");
	}

	my $cddb_contents = "";
	if ("$cddb_file" ne "") {
      my $tmp = $/;
      $/ = undef;
		$self->debug(2, "Using entry \"$cddb_file\"");
      
		open (CDDB_FILE, "$cddb_file");
		$cddb_contents = <CDDB_FILE>;
		close (CDDB_FILE);
      $/ = $tmp;
		$cddb_contents =~ s/\r//g;

      $self->{_cddb_string} = $cddb_contents;
      $self->{genre}        = $genre;
      return 1;
	}
	else {
      $self->debug(2, "No such entry found in \"$self->{'cache_directory'}\"");
      return 0;
   }
}

# Puts the entry into _cddb_string fresh from the server
# Args:    NONE
# Returns: BOOL
# Sets:    {_cddb_string} and {genre}on success
sub _get_info_from_cddb
{
   my $self = shift @_;
	my ($url, $request, $ua, $response, $i, $genre);
	my (@results, $result);

	$self->debug(2, "Trying server connection: \"$self->{'cddb_server'}\"");

	# Search for presence of cd in cddb database
	$url = new URI::URL("$self->{'cddb_server'}?cmd=cddb+query+$self->{_server_string}&$self->{_hello_server}&proto=3");
	$self->debug(2, "url is $url");
	$request = HTTP::Request->new ("GET", $url);
	$ua = new LWP::UserAgent;

	if ($self->proxy_value ne "") {
		$ua->proxy('http', $self->proxy_value);
	}

	$self->debug(2, "request is \"".$request->as_string."\"");
	$response = $ua->request($request);
	
	$self->debug(2, "response is ".$response->as_string);
	if (!$response->is_success) {
		carp "Error (no success while reading cddb): ".$response->message."\n";
      return 0;
	}
	if (($response->content !~ /^200/) && ($response->content !~ /^211/)) {
		carp "Error (while reading cddb): ".$response->content."\n";
		return 0;
	}
	if ($response->content =~ /^211/) {
      # TODO hmm. what can we do here as an object?
		# If multiple matches found go for interactive mode.
		my @resp_list;
		my $t = 0;
		carp "Inexact matches found, listed below\n\n";

      @resp_list = split('\n', $response->content);

		foreach $i (@resp_list) {
         last if ( $i =~ /^\./);
			carp " $t: $i\n" if $t > 0; 
			$t++;
		}
      $t--;
      
		my $key = 1;
		if ($t > 1) {
         do {
            carp "Select which match to use (1-$t): ";
            $key = getc();
         } while ( $key < 1  ||  $key > $t);
         carp "You have chosen: ".$resp_list[$key]."\n";
		}
		($genre, @results) = split(' ',$resp_list[$key]);
	}
	else
	{
		($result, $genre, @results) = split(' ',$response->content);
	}
   $self->debug(2, "result is $result");
   $self->debug(2, "genre is $genre");
   $self->debug(2, "results are ", @results);

	# Get track listing
	$url = new URI::URL("$self->{'cddb_server'}?cmd=cddb+read+$genre+$results[0]&$self->{_hello_server}&proto=3");
	$self->debug(2, "url is $url");
	$request = HTTP::Request->new ("GET", $url);
	$ua = new LWP::UserAgent;

	if ($self->proxy_value ne "") {
		$ua->proxy('http', $self->proxy_value);
	}

	$response = $ua->request($request);
	if (! ($response->is_success)) {
		carp "Error (while reading cddb): ".$response->message."\n";
      return 0;
	}
	if ($response->content !~ /^210/) {
		carp "Error (while reading cddb): ".$response->content."\n";
      return 0;
	}

   $self->debug(2, "genre is still $genre");
   $self->{genre}        = $genre;
   my $scrubby = $response->content;
   $scrubby =~ s/
//g;
   $self->{_cddb_string} = $scrubby;
   return 1;
}

# Calculations...
# Args:    number
# Returns: number
# Sets:    NOTHING
sub _cddb_sum {
   my $self  = shift @_;
	my $input = shift @_;
   my ($temp, $result);

	$temp = 10;
	$result = 0;
	while ($input > 0) {
      my $temp2;
      $temp2 = ( ($input % $temp) / ($temp / 10));
      $result += $temp2;
		$input = $input - ($input % $temp);
		$temp *= 10;
	}
	return $result;
}

# Args:    string (optional)
# Returns: string
# Sets:    _server_string if called with an Arg.
sub _server_string {
   my $self = shift;
   if (@_) { 
      my $tmp = shift;
      $self->debug(4, "Server string is $tmp");
      $tmp =~ s/ /+/g;
      $self->debug(4, "Server string is $tmp");
      $self->{_server_string} = $tmp;
   }
   return $self->{_server_string};
}

# General debugging function
# Args:    debug_value, then any number of items to be printed to STDERR
# Returns: void
# Sets:    NOTHING
sub debug {
   my $self = shift @_;
   my $debug_level = shift @_;
   if ($self->{_debug} >= $debug_level) {
      print STDERR "[DEBUG]: ", join("\n\t ", @_), "\n";
   }
}

# Autoload methods go after =cut, and are processed by the autosplit program.

1;
__END__

=head1 NAME

FreeDB - Perl extension for accessing the free internet database of CD information

=head1 SYNOPSIS

  use FreeDB;

  my $cddb = new FreeDB;
  my $success = 0; 

  $success = $cddb->fetch;

  if (!$success) {
     print "couldn't fetch cd info\n";
     exit 1;
  }
  
  print $cddb->artist . "\n";
  print $cddb->title  . "\n\n";
  
  while ($cddb->next_track) {
     print $cddb->current_track_number_padded, ". ";
     print $cddb->current_track_info;
     print " (", $cddb->current_track_time_in_minutes, ")\n";
  } 

  print "\n\n", $cddb->disc_info, "\n";

=head1 ITERATIVE METHODS

=over

=item B<next_track>

Tells FreeDB to advance to the next track on the cd. You must call
this before calling any of the current_* methods to set FreeDB on
the first track. This way you can conveniently use this method in a
while loop.  Returns the track_number or 0 when you run out of tracks.

=item B<current_track_info>

Returns the track info for the current track.

=item B<current_track_number>

Returns the number of the current track. You can give a
specific track number to set FreeDB on a particular track. 


=item B<current_track_number_padded>

Returns the number of the current track in 2 digit format. That is, 9
will print as 09 if you call this method.

=item B<current_track_extended_info>

Returns any extended info that the current track may have in it's record.

=item B<current_track_detailed_info>

Returns all relevant info for the current track-- number, info,
extended info, and time in seconds.

=item B<current_track_time_in_minutes>

Returns the track's length in minutes.

=item B<current_track_time_in_seconds>

Returns the track's length in seconds.

=item B<current_track_offset>

=back


=head1 CONFIGURATION METHODS

=over

=item B<cddb_server>

This sets and gets the full URL for the cddb server. $cddb->new() sets
this for you, but you can customize it by setting it after you call
$cddb->new() and before you call $cddb->fetch().

=item B<cdrom_device>

The device that corresponds to your cdrom drive (eg /dev/cdrom).
)$cddb->new() sets this for you, but you can customize it by setting
it after you call $cddb->new() and before you call $cddb->fetch().

=item B<cache_directory>

The directory where FreeDB will cache fetched records for you (eg
~/.cddb). $cddb->new() sets this for you, but you can customize it by
setting it after you call $cddb->new() and before you call
$cddb->fetch().

=item B<ignore_cache>

Set this to a non-zero value if you want to force FreeDB to make a
trip to the server and not attempt to load your disc's record from the
local cache. $cddb->new() sets this to 0, but you can customize it by
setting it after you call $cddb->new() and before you call
$cddb->fetch().

=item B<proxy_value>

$cddb->new() sets this to "", but if you need to go through a proxy,
you can customize it by setting it after you call $cddb->new() and
before you call $cddb->fetch().

=back


=head1 OTHER METHODS

=over

=item B<fetch>

This is the method that actually goes and gets the record (either from
the server or your local cache). It returns 1 on success or 0 on failure.
Optionally you can give a cddb entry as argument. It will use this information
and consequently does not look at the cdrom device, cache, or cddb server.

=item B<artist>

Returns the disc's artist. Big surprise there.

=item B<discid>

Returns the discid that FreeDB uses as a a filename for your locally
cached disc record.

=item B<disc_info>

Returns info from the EXTD field of the record

=item B<title>

Returns the disc's title. We're just full of surprises today.

=item B<disc_length>

Returns the disc's length (running time) in seconds.

=item B<total_tracks>

Returns the total number of tracks on the disc.

=back


=head1 ARRAY ACCESSOR METHODS

These methods return arrays containing all the info for the tracks on
the disc. They're listed here in case you want to use them, but you'll
probably do best with the current_* methods above as they provide the
easiest method for plowing through the tracks on the disc. 

=over

=item B<track_info>

Returns an array containing the track names for all tracks on the disc.

=item B<offsets>

Returns an array containing the byte offsets (Is that right?) for all
tracks on the disc.

=item B<extended_track_info>

Returns an array containing the extended info for all tracks on the
disc.

=item B<detailed_track_info>

Returns an array containing all the info for all tracks on the disc.

=item B<track_lengths>

Returns an array containing all the track lengths in seconds for all
tracks on the disc.

=back


=head1 DESCRIPTION

A Perl module to provide convenient OOP access to the free CDDB

=head1 THANKS

Thanks to J.I. van Hemert for graciously giving me permission to use
the cddb storage and retrieval code from his wonderful program
disc-cover (http://www.liacs.nl/~jvhemert/disc-cover). I originally
wrote FreeDB for use with my own cd labelling program called
cdlabelgen (http://www.red-bean.com/~bwf/software/cdlabelgen/).

=head1 AUTHOR

Code cleanup and objectification/modularization by B. W. Fitzpatrick
E<lt>F<fitz@red-bean.com>E<gt> and J.I. van Hemert
E<lt>F<jvhemert@cs.leidenuniv.nl>E<gt>. cddb routines by Richard
Martin and J.I. van Hemert.

=head1 LAST MODIFIED

$Id: FreeDB.pm,v 1.12 1999/09/28 03:30:20 bwf Exp $


=cut
