#ifndef _FTVT_H_
#define _FTVT_H_
/*
 *      Copyright (C) 1997-1999 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 * $RCSfile: ftvt.h,v $
 * $Revision: 1.4 $
 * $Date: 1999/03/05 13:13:05 $
 *
 *     This file contains the definitions for the volume table
 *     manipulation tool for the the QIC-40/80/3010/3020 floppy-tape
 *     driver "ftape" for Linux.
 */

#include <linux/ftape.h>
#include <linux/zftape.h>

#define FTVT_SIZE 128 /* bytes */

/* The following are offsets in the vtbl.  */
#define FTVT_SIG   0
#define FTVT_START 4
#define FTVT_END   6
#define FTVT_DESC  8
#define FTVT_DATE  52
#define FTVT_FLAGS 56
#define FTVT_FL_VENDOR_SPECIFIC (1<<0)
#define FTVT_FL_MUTLI_CARTRIDGE (1<<1)
#define FTVT_FL_NOT_VERIFIED    (1<<2)
#define FTVT_FL_REDIR_INHIBIT   (1<<3)
#define FTVT_FL_SEG_SPANNING    (1<<4)
#define FTVT_FL_DIRECTORY_LAST  (1<<5)
#define FTVT_FL_RESERVED_6      (1<<6)
#define FTVT_FL_RESERVED_7      (1<<7)
#define FTVT_M_NO  57
#define FTVT_EXT   58
#define FTVT_EXT_ZFTAPE_SIG     0
#define FTVT_EXT_ZFTAPE_BLKSZ  10
#define FTVT_EXT_ZFTAPE_CMAP   12
#define FTVT_EXT_ZFTAPE_QIC113 13
#define FTVT_PWD   84
#define FTVT_DIR_SIZE 92
#define FTVT_DATA_SIZE 96
#define FTVT_OS_VERSION 104
#define FTVT_SRC_DRIVE  106
#define FTVT_DEV        122
#define FTVT_RESERVED_1 123
#define FTVT_CMPR       124
#define FTVT_CMPR_UNREG 0x3f
#define FTVT_CMPR_USED  0x80
#define FTVT_FMT        125
#define FTVT_RESERVED_2 126
#define FTVT_RESERVED_3 127
/* compatability with pre revision K */
#define FTVT_K_CMPR     120 

/*  the next is used by QIC-3020 tapes with format code 6 (>2^16
 *  segments) It is specified in QIC-113, Rev. G, Section 5 (SCSI
 *  volume table). The difference is simply, that we only store the
 *  number of segments used, not the starting segment.
 */
#define FTVT_SCSI_SEGS  4 /* is a 4 byte value */

/*  one vtbl is 128 bytes, that results in a maximum number of
 *  29*1024/128 = 232 volumes.
 */
#define FTVT_MAX_VOLUMES (FT_SEGMENT_SIZE/FTVT_SIZE)
#define FTVT_IDS { "VTBL", "XTBL", "UTID", "EXVT" } /* valid ids */

typedef struct ftvt_entry {
	u_int8_t sig[4];
	union {
		struct { u_int16_t start, end; } se;
		u_int32_t scsi_segs;
	} size;
	u_int8_t  desc[44];
	u_int32_t date;
	u_int8_t vendor_specific:1;
	u_int8_t multi_cartridge:1;
	u_int8_t not_verified:1;
	u_int8_t inhibit_redirection:1;
	u_int8_t segment_spanning:1;
	u_int8_t directory_last:1;
	u_int8_t fl_reserved_6:1;
	u_int8_t fl_reserved_7:1;
	u_int8_t m_no;
	u_int8_t ext[26];
	u_int8_t pwd[8];
	u_int32_t dir_size;
	u_int64_t data_size;
	u_int16_t os_version;
	u_int8_t  source_drive[16];
	u_int8_t  device;
	u_int8_t  reserved_1;
	u_int8_t  cmpr;
	u_int8_t  format;
	u_int8_t  reserved_2;
	u_int8_t  reserved_3;
} ftvt_entry __attribute__ ((packed));

typedef struct ftvt {
	int num;
	int fmt_code;
	int start, end;
	unsigned int modified:1;
	ftvt_entry entry;
} ftvt;

#define ftvt_start     entry.size.se.start
#define ftvt_end       entry.size.se.end
#define ftvt_space     entry.size.scsi_segs
#define ftvt_sig       entry.sig
#define ftvt_desc      entry.desc
#define ftvt_date      entry.date

#define FTAPE_DEF_TAPE "/dev/rawft0"

/*  Control flags
 */
#define FTVT_VERBOSE (1 << 0)
#define FTVT_TAGGED  (1 << 1)

/* This is return by ftvt_get_ctrl
 */
typedef struct ftvt_ctrl {
	const char *version;
	unsigned long flags;
	int fmt_code;
	int first_seg;
	int last_seg;
	int max_volumes;
	const char *errstr;
    const char *drivetype;
} ftvt_ctrl;

/* exported functions
 */
extern void ftvt_set_ctrl(unsigned long flags);
extern const struct ftvt_ctrl *ftvt_get_ctrl(void);
extern int ftvt_open(const char *name, mode_t mode);
extern int ftvt_close(int tape_fd);
extern int ftvt_read(int tape_fd, ftvt *volumes, u_int8_t *buffer);
extern int ftvt_write(int tape_fd, const ftvt *volumes, u_int8_t *buffer,
					  int vtbl_cnt, int do_write);
extern void ftvt_print(const ftvt *volumes, int maxnum);
extern void ftvt_print_one(const ftvt *volume);
extern void ftvt_print_one_tagged(const ftvt *volume);
extern int ftvt_parse_tagged(ftvt *volumes);
extern int ftvt_add_volume(ftvt *volumes, int num_volumes);
extern char *ftvt_decode_date(u_int32_t timestamp);
extern int ftvt_set_date(ftvt *volumes, int maxnum,
						 const char *date, int vtbl_no);
extern int ftvt_set_label(ftvt *volumes, int maxnum,
						  const char *desc,int vtbl_no);
extern int ftvt_set_bounds(ftvt *volumes, int maxnum, 
						   int start, int end, int vtbl_no);
extern int ftvt_set_id(ftvt *volumes, int maxnum, const char *id, int vtbl_no);

#endif _FTVT_H_

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
