/*  Authors: Eric M. Ludlam <zappo@ultranet.com>
 *           Russ McManus <russell.mcmanus@gs.com>
 *
 *  gfec stands for 'guile fancy error catching'.
 *  This code is in the public domain.
 */

#include <gfec.h>

#include "util.h"


/* We assume that data is actually a char**.  the way we return
   results from this function is to malloc a fresh string, and store
   it in this pointer.  it is the caller's responsibility to do
   something smart with this freshly allocated storage. the caller can
   determine whether there was an error by initializing the char*
   passed in to NULL.  if there is an error, the char string will not
   be NULL on return. */
static SCM
gfec_catcher(void *data, SCM tag, SCM throw_args)
{
  SCM func;
  SCM result;
  char *msg = NULL;

  func = gh_eval_str("gnc:error->string");
  if (gh_procedure_p(func))
  {
    result = gh_call2(func, tag, throw_args);
    if (gh_string_p(result))
      msg = gh_scm2newstr(result, NULL);
  }

  if (msg == NULL)
  {
    msg = strdup("Error running guile function.");
    assert(msg != NULL);
  }

  *(char**)data = msg;

  return gh_int2scm(1);
}


/* the arguments to scm_internal_stack_catch:
   ------------------------------------------
   SCM tag                     : this should be SCM_BOOL_T to catch all errors.
   scm_catch_body_t body       : the function to run.
   void *body_data             : a pointer to pass to body
   scm_catch_handler_t handler : the hander function
   void *handler_data          : a pointer to pass to the handler
*/

SCM
gfec_eval_file(const char *file, gfec_error_handler error_handler)
{
  char *err_msg = NULL;
  SCM result;

  result = scm_internal_stack_catch(SCM_BOOL_T,
                                    (scm_catch_body_t) gh_eval_file,
                                    (void*) file,
                                    (scm_catch_handler_t) gfec_catcher,
                                    (void*) &err_msg);

  if (err_msg != NULL)
  {
    error_handler(err_msg);
    free(err_msg);

    return SCM_UNDEFINED;
  }

  return result;
}

SCM
gfec_eval_string(const char *str, gfec_error_handler error_handler)
{
  char *err_msg = NULL;
  SCM result;

  result = scm_internal_stack_catch(SCM_BOOL_T,
                                    (scm_catch_body_t) gh_eval_str,
                                    (void*) str,
                                    (scm_catch_handler_t) gfec_catcher,
                                    (void*) &err_msg);

  if (err_msg != NULL)
  {
    error_handler(err_msg);
    free(err_msg);

    return SCM_UNDEFINED;
  }

  return result;
}

struct gfec_apply_rec
{
  SCM proc;
  SCM arglist;
};

static void
gfec_apply_helper(void *data)
{
  struct gfec_apply_rec *apply_rec = (struct gfec_apply_rec *)data;

  gh_apply(apply_rec->proc, apply_rec->arglist);
}

SCM
gfec_apply(SCM proc, SCM arglist, gfec_error_handler error_handler)
{
  char *err_msg = NULL;
  struct gfec_apply_rec apply_rec;
  SCM result;

  apply_rec.proc = proc;
  apply_rec.arglist = arglist;

  result = scm_internal_stack_catch(SCM_BOOL_T,
                                    (scm_catch_body_t) gfec_apply_helper,
                                    (void*) &apply_rec,
                                    (scm_catch_handler_t) gfec_catcher,
                                    (void*) &err_msg);

  if (err_msg != NULL)
  {
    error_handler(err_msg);
    free(err_msg);

    return SCM_UNDEFINED;
  }

  return result;
}
