/*
    libparted
    Copyright (C) 1998-2000  Andrew Clausen  <clausen@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>

#include "fat.h"

/* for debugging */
static int
search_remap (FatOpContext* ctx, FatCluster start, FatCluster cluster)
{
	FatSpecific*	old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatCluster	i;

	PED_ASSERT (start < old_fs_info->cluster_count + 2, return 0);

	for (i = start; i < old_fs_info->cluster_count + 2; i++) {
		if (ctx->remap [i] == cluster)
			return i;
	}

	return 0;
}

/* for debugging */
static int
search_fat (FatTable* fat, FatCluster start, FatCluster cluster)
{
	FatCluster	i;

	PED_ASSERT (start < fat->cluster_count + 2, return 0);

	for (i = start; i < fat->cluster_count + 2; i++) {
		if (fat_table_get (fat, i) == cluster)
			return i;
	}

	return 0;
}

static int
needs_duplicating (FatOpContext* ctx, FatCluster cluster)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (ctx->old_fs);

	PED_ASSERT (cluster >= 2 && cluster < fs_info->cluster_count + 2,
		    return 0);

	return (fs_info->fat_flag_map [cluster] == FAT_FLAG_FILE
			&& !fat_op_context_map_static_cluster (ctx, cluster))
	       || fs_info->fat_flag_map [cluster] == FAT_FLAG_DIRECTORY;
}

static FatCluster
search_next_cluster (FatOpContext* ctx, FatCluster start)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatCluster	i;

	for (i = start; i < fs_info->cluster_count + 2; i++) {
		if (needs_duplicating (ctx, i))
			return i;
	}
	return 0;	/* all done! */
}

static int
read_marked_clusters (FatOpContext* ctx, FatCluster start, FatCluster length)
{
	FatSpecific*		fs_info = FAT_SPECIFIC (ctx->old_fs);
	int			status;
	FatCluster		i;

	PED_ASSERT (start > 1, return 0);
	PED_ASSERT (start + length - 1 < fs_info->cluster_count + 2, return 0);

	ped_exception_fetch_all ();
	status = fat_read_clusters (ctx->old_fs, fs_info->buffer,
				    start, length);
	ped_exception_leave_all ();
	if (status)
		return 1;
	ped_exception_catch ();

/* something bad happened, so read clusters one by one.  (The error may
   have occured on an unused cluster: who cares) */
	for (i = 0; i < length; i++) {
		if (ctx->buffer_map [i]) {
			if (!fat_read_cluster (ctx->old_fs,
				fs_info->buffer + i * fs_info->cluster_size,
				start + i))
				return 0;
		}
	}
	return 1;
}

static int
fetch_clusters (FatOpContext* ctx, FatCluster start)
{
	FatSpecific*	old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatCluster	fetch_length = 0;
	int		count = 0;
	int		i;

	PED_ASSERT (start > 1, return 0);

	memset (ctx->buffer_map, 0, BUFFER_SIZE);

	for (i = 0;
	     i < BUFFER_SIZE
			&& start + i < old_fs_info->cluster_count + 2;
	     i++) {
		if (needs_duplicating (ctx, start + i)) {
			ctx->buffer_map [i] = 1;
			fetch_length = i + 1;
			count++;
		}
	}

	if (!read_marked_clusters (ctx, start, fetch_length))
		return 0;

	return count;
}

/*****************************************************************************
 * here starts the write code.  All assumes that ctx->buffer_map [first] and
 * ctx->buffer_map [last] are occupied by clusters that need to be duplicated.
 *****************************************************************************/

/* finds the first cluster that is not going to get overwritten (that needs to
   get read in */
static FatCluster
get_first_underlay (FatOpContext* ctx, FatCluster new_clusters [BUFFER_SIZE],
		    int first, int last)
{
	int		old;
	FatCluster	new;

	PED_ASSERT (first <= last, return 0);

	new = new_clusters [first];
	for (old = first + 1; old <= last; old++) {
		if (!ctx->buffer_map [old])
			continue;
		new++;
		if (new_clusters [old] != new)
			return new;
	}
	return 0;
}

/* finds the last cluster that is not going to get overwritten (that needs to
   get read in */
static FatCluster
get_last_underlay (FatOpContext* ctx, FatCluster new_clusters [BUFFER_SIZE],
		   int first, int last)
{
	int		old;
	FatCluster	new;

	PED_ASSERT (first <= last, return 0);

	new = new_clusters [last];
	for (old = last - 1; old >= first; old--) {
		if (!ctx->buffer_map [old])
			continue;
		new--;
		if (new_clusters [old] != new)
			return new;
	}
	return 0;
}

/* "underlay" refers to the "static" clusters, that remain unchanged.
 * when writing large chunks at a time, we don't want to clobber these,
 * so we read them in, and write them back again.  MUCH quicker that way.
 */
static int
quick_group_write_read_underlay (FatOpContext* ctx,
				 FatCluster new_clusters [BUFFER_SIZE],
				 int first, int last)
{
	FatSpecific*	new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	FatCluster	first_underlay;
	FatCluster	last_underlay;
	FatCluster	underlay_length;

	PED_ASSERT (first <= last, return 0);

	first_underlay = get_first_underlay (ctx, new_clusters, first, last);
	if (!first_underlay)
		return 1;
	last_underlay = get_last_underlay (ctx, new_clusters, first, last);

	PED_ASSERT (first_underlay <= last_underlay, return 0);

	underlay_length = last_underlay - first_underlay + 1;
	if (!fat_read_clusters (ctx->new_fs,
				new_fs_info->buffer 
				   + (first_underlay - new_clusters [first])
					* new_fs_info->cluster_size,
				first_underlay,
				underlay_length))
		return 0;
	return 1;
}

/* quick_group_write() makes no attempt to recover from errors - just
 * does things fast.  If there is an error, slow_group_write() is
 * called.
 *    Note: we do syncing writes, to make sure there isn't any
 * error writing out.  It's rather difficult recovering from errors
 * further on.
 */
static int
quick_group_write (FatOpContext* ctx, FatCluster new_clusters [BUFFER_SIZE],
		   int first, int last)
{
	FatSpecific*		old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatSpecific*		new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	int			active_length;
	int			i;
	int			offset;
	int			cluster_size = new_fs_info->cluster_size;

	PED_ASSERT (first <= last, return 0);

	ped_exception_fetch_all ();
	if (!quick_group_write_read_underlay (ctx, new_clusters, first, last))
		goto error;

	for (i = first; i <= last; i++) {
		if (!ctx->buffer_map [i])
			continue;

		offset = new_clusters [i] - new_clusters [first];
		memcpy (new_fs_info->buffer + offset * cluster_size,
			old_fs_info->buffer + i * cluster_size,
			cluster_size);
	}

	active_length = new_clusters [last] - new_clusters [first] + 1;
	if (!fat_write_sync_clusters (ctx->new_fs, new_fs_info->buffer,
				      new_clusters [first], active_length))
		goto error;

	ped_exception_leave_all ();
	return 1;

error:
	ped_exception_catch ();
	ped_exception_leave_all ();
	return 0;
}

/* Writes clusters out, one at a time, avoiding errors on redundant writes
 * on damaged parts of the disk we already know about.  If there's an error
 * on one of the required clusters, marks it as bad, and finds a replacement.
 */
static int
slow_group_write (FatOpContext* ctx, FatCluster new_clusters [BUFFER_SIZE],
		  int first, int last)
{
	FatSpecific*		old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatSpecific*		new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	int			cluster_size = new_fs_info->cluster_size;
	int			i;

	PED_ASSERT (first <= last, return 0);

	for (i = first; i <= last; i++) {
		if (!ctx->buffer_map [i])
			continue;

		while (!fat_write_sync_cluster (ctx->new_fs,
				old_fs_info->buffer + i * cluster_size,
				new_clusters [i])) {
			fat_table_set_bad (new_fs_info->fat, new_clusters [i]);
			new_clusters [i] = fat_table_alloc_cluster
						(new_fs_info->fat);
			if (!new_clusters [i])
				return 0;
		}
	}
	return 1;
}

static int
update_remap (FatOpContext* ctx, FatCluster new_clusters [BUFFER_SIZE],
	      FatCluster read_offset, int first, int last)
{
	int		i;

	PED_ASSERT (first <= last, return 0);

	for (i = first; i <= last; i++) {
		if (ctx->buffer_map [i])
			ctx->remap [read_offset + i] = new_clusters [i];
	}

	return 1;
}

static int
group_write (FatOpContext* ctx, FatCluster new_clusters [BUFFER_SIZE],
	     FatCluster read_offset, int first, int last)
{
	PED_ASSERT (first <= last, return 0);

	if (!quick_group_write (ctx, new_clusters, first, last)) {
		if (!slow_group_write (ctx, new_clusters, first, last))
			return 0;
	}
	if (!update_remap (ctx, new_clusters, read_offset, first, last))
		return 0;
	return 1;
}

static int
write_clusters (FatOpContext* ctx, FatCluster read_offset)
{
	FatSpecific*		old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatSpecific*		new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	int			group_start;
	int			group_end;
	FatCluster		mapped_length;
	int			i;
	FatCluster		new_clusters [BUFFER_SIZE];

	PED_ASSERT (read_offset > 1, return 0);
	PED_ASSERT (read_offset < old_fs_info->cluster_count + 2, return 0);

	group_start = -1;
	for (i = 0; i < BUFFER_SIZE; i++) {
		if (!ctx->buffer_map [i])
			continue;

		new_clusters [i] = fat_table_alloc_cluster (new_fs_info->fat);
		if (!new_clusters [i])
			return 0;
		fat_table_set_eof (new_fs_info->fat, new_clusters[i]);

		if (group_start == -1)
			group_start = group_end = i;

		PED_ASSERT (new_clusters[i] >= new_clusters[group_start],
			    return 0);

		mapped_length = new_clusters[i] - new_clusters[group_start] + 1;
		if (mapped_length <= BUFFER_SIZE) {
			group_end = i;
		} else {
			/* ran out of room in the buffer, so write this group,
			 * and start a new one...
			 */
			if (!group_write (ctx, new_clusters, read_offset,
					  group_start, group_end))
				return 0;
			group_start = group_end = i;
		}
	}

	if (group_start == -1) {
		ped_exception_throw (PED_EXCEPTION_BUG, PED_EXCEPTION_CANCEL,
			"write_clusters()  no group to write!");
		return 0;
	}
	if (!group_write (ctx, new_clusters, read_offset, group_start,
			  group_end))
		return 0;
	return 1;
}

/*  default all clusters to unmoved
 */
void
init_remap (FatOpContext* ctx)
{
	FatSpecific*	old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatCluster	i;

	for (i = 2; i < old_fs_info->cluster_count + 2; i++)
		ctx->remap [i] = fat_op_context_map_static_cluster (ctx, i);
}

/*  duplicates unreachable file clusters, and all directory clusters
 */
int
fat_duplicate_clusters (FatOpContext* ctx)
{
	FatCluster		i;

	init_remap (ctx);
	for (i = search_next_cluster (ctx, 2); i;
	     i = search_next_cluster (ctx, i + BUFFER_SIZE)) {
		if (!fetch_clusters (ctx, i))
			return 0;
		if (!write_clusters (ctx, i))
			return 0;
	}
	return 1;
}

